/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.Rectangle2D;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.*;
import javax.swing.border.EmptyBorder;
import javax.swing.border.LineBorder;
import javax.swing.plaf.ComponentUI;
import javax.swing.text.*;

import de.uni_paderborn.fujaba.fsa.FSAObject;
import de.uni_paderborn.fujaba.fsa.listener.AscendDescendMouseHandler;


/**
 * Grab UI that has a TextComponent to edit the value of JBend.data
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.15 $
 */
public class TextGrabUI extends GrabUI
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static String EDITOR = "de.uni_paderborn.fujaba.fsa.swing.TextGrabUI:Editor";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JBend bend;


   /**
    * Get the bend attribute of the TextGrabUI object
    *
    * @return   The bend value
    */
   public JBend getBend()
   {
      return bend;
   }


   /**
    * Sets the bend attribute of the TextGrabUI object
    *
    * @param bend  The new bend value
    * @return      No description provided
    */
   public boolean setBend (JBend bend)
   {
      if (bend != this.bend)
      {
         if (this.bend != null)
         {
            this.bend.setUI (null);
         }
         this.bend = bend;
         if (bend != null)
         {
            if (bend.getUI() != this)
            {
               bend.setUI (this);
            }
         }
         return true;
      }
      return false;
   }

   // The Component that the Editor uses for editing
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JTextComponent editor;


   /**
    * Get the editor attribute of the TextGrabUI object
    *
    * @return   The editor value
    */
   public JTextComponent getEditor()
   {
      return editor;
   }


   /**
    * Sets the editor attribute of the TextGrabUI object
    *
    * @param editor  The new editor value
    * @return        No description provided
    */
   public boolean setEditor (JTextComponent editor)
   {
      if (editor != this.editor)
      {
         if (this.editor != null)
         {
            uninstallComponents();
         }
         this.editor = editor;
         if (editor != null)
         {
            configureEditor();
         }

         if (bend != null)
         {
            bend.putClientProperty (EDITOR, editor);

            if (editor != null)
            {
               installComponents();
            }

            bend.revalidate();
            bend.repaint();
         }
         return true;
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String text;


   /**
    * the text of the grab.
    *
    * @return   The text value
    */
   public String getText()
   {
      return this.text;
   }


   /**
    * the text of the grab. taken from grab.data
    *
    * @param text  The new text value
    * @return      No description provided
    */
   public boolean setText (String text)
   {
      if (this.text == null && text != null || this.text != null && !this.text.equals (text))
      {
         String oldText = this.text;
         this.text = text;
         setEditorText (text);
         firePropertyChange (bend, "text", oldText, text);
         return true;
      }
      return false;
   }


   /**
    * the text of the TextField
    *
    * @return   The editorText value
    */
   protected String getEditorText()
   {
      String text =  (editor == null ? null : editor.getText());

      if (text == null)
      {
         text = "";
      }

      return text;
   }


   /**
    * the text of the TextField
    *
    * @param text  The new editorText value
    */
   protected void setEditorText (String text)
   {
      if (editor != null)
      {
         if (text == null)
         {
            text = "";
         }

         String editorText = getEditorText();
         if (!editorText.equals (text))
         {
            editor.setText (text);
            if (bend != null)
            {
               bend.revalidate();
               bend.repaint();
            }
            else
            {
               editor.revalidate();
            }
         }
      }
   }

   // Listeners that are attached to the JTextComponent
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FocusListener editorFocusListener;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PropertyChangeListener propertyChangeListener;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private MouseListener mouseListener;

   //========================
   // begin UI Initialization
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param c  No description provided
    * @return   No description provided
    */
   public static ComponentUI createUI (JComponent c)
   {
      return new TextGrabUI();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param c  No description provided
    */
   public void installUI (JComponent c)
   {
      setBend ((JBend) c);
      bend.setLayout (createLayoutManager());

      installDefaults();

      installComponents();
      installListeners();

      bend.setRequestFocusEnabled (true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param c  No description provided
    */
   public void uninstallUI (JComponent c)
   {
      bend.setLayout (null);

      uninstallComponents();
      uninstallListeners();
      uninstallDefaults();

      mouseListener = null;
      editorFocusListener = null;
      propertyChangeListener = null;
      bend = null;
   }


   /**
    * Installs the default colors, default font, default renderer, and default editor into
    * the JTextComponent.
    */
   protected void installDefaults()
   {
      bend.setOpaque (true);
      bend.setForeground (FSAObject.COLOR_FOREGROUND);
      bend.setBackground (FSAObject.COLOR_BACKGROUND);
      bend.setBorder (new LineBorder (FSAObject.COLOR_FOREGROUND));
      if (editor == null)
      {
         setEditor (createEditor());
         editor.setBorder (new EmptyBorder (0, 0, 0, 0));
      }
      bend.putClientProperty (EDITOR, editor);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void installListeners()
   {
      if ( (propertyChangeListener = createPropertyChangeListener()) != null)
      {
         bend.addPropertyChangeListener (propertyChangeListener);
      }
      if ( (mouseListener = createMouseListener()) != null)
      {
         AscendDescendMouseHandler.addMouseListener (bend, mouseListener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void uninstallDefaults()
   {
      bend.putClientProperty (EDITOR, null);
      setEditor (null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void uninstallListeners()
   {
      if (propertyChangeListener != null)
      {
         bend.removePropertyChangeListener (propertyChangeListener);
      }
      if (mouseListener != null)
      {
         AscendDescendMouseHandler.removeMouseListener (bend, mouseListener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected PropertyChangeListener createPropertyChangeListener()
   {
      return new PropertyChangeHandler();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected MouseListener createMouseListener()
   {
      return new MouseHandler();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected LayoutManager createLayoutManager()
   {
      return new BendLayoutManager();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected JTextComponent createEditor()
   {
      if (bend.getClientProperty (EDITOR) != null && bend.getClientProperty (EDITOR) instanceof JTextComponent)
      {
         return (JTextComponent) bend.getClientProperty (EDITOR);
      }
      return new JTextField();
   }

   //
   // end UI Initialization
   //======================


   //======================
   // begin Inner classes
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.15 $
    */
   public class MouseHandler extends MouseAdapter
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mouseClicked (MouseEvent e)
      {
         if (e.getClickCount() > 1 && !isEditorVisible (bend))
         {
            setEditorVisible (bend, true);
         }
      }
   }


   /**
    * This listener watches for bound properties that have changed in the JTextComponent. It
    * looks for the model and editor being swapped-out and updates appropriately. It also looks
    * for changes in the editable, enabled, and maximumRowCount properties. This inner class
    * is marked &quot;public&quot; due to a compiler bug. This class should be treated as a
    * &quot;protected&quot; inner class. Instantiate it only within subclasses of <FooUI>.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.15 $
    */
   public class PropertyChangeHandler implements PropertyChangeListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void propertyChange (PropertyChangeEvent e)
      {
         String propertyName = e.getPropertyName();
         if (propertyName.equals ("enabled"))
         {
            if (bend.isEnabled())
            {
               if (editor != null)
               {
                  editor.setEnabled (true);
               }
            }
            else
            {
               if (editor != null)
               {
                  editor.setEnabled (false);
               }
            }
            bend.repaint();
         }
         else if (propertyName.equals ("font"))
         {
            if (editor != null)
            {
               editor.setFont (bend.getFont());
            }
            bend.revalidate();
            bend.repaint();
         }
         else if (propertyName.equals ("foreground"))
         {
            if (editor != null)
            {
               editor.setForeground (bend.getForeground());
            }
            bend.repaint();
         }
         else if (propertyName.equals ("background"))
         {
            if (editor != null)
            {
               editor.setBackground (bend.getBackground());
            }
            bend.repaint();
         }
         else if (propertyName.equals ("opaque"))
         {
            if (editor != null)
            {
               editor.setOpaque (bend.isOpaque());
            }
            bend.repaint();
         }
         else if (propertyName.equals (JComponent.TOOL_TIP_TEXT_KEY))
         {
            updateToolTipTextForChildren();
         }
         else if (propertyName.equals ("data"))
         {
            String editorText = getEditorText();
            String newText = getText();
            if (!editorText.equals (newText))
            {
               setEditorText (newText);
               firePropertyChange (bend, "text", e.getOldValue(), newText);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void updateToolTipTextForChildren()
   {
      Component[] children = bend.getComponents();
      for (int i = 0; i < children.length; ++i)
      {
         if (children[i] instanceof JComponent)
         {
             ((JComponent) children[i]).setToolTipText (bend.getToolTipText());
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.15 $
    */
   public class BendLayoutManager implements LayoutManager
   {
      /**
       * Access method for an one to n association.
       *
       * @param name  The object added.
       * @param comp  The object added.
       */
      public void addLayoutComponent (String name, Component comp)
      {
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param comp  No description provided
       */
      public void removeLayoutComponent (Component comp)
      {
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param parent  No description provided
       * @return        No description provided
       */
      public Dimension preferredLayoutSize (Container parent)
      {
         return parent.getPreferredSize();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param parent  No description provided
       * @return        No description provided
       */
      public Dimension minimumLayoutSize (Container parent)
      {
         return parent.getMinimumSize();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param parent  No description provided
       */
      public void layoutContainer (Container parent)
      {
         Rectangle cvb;

         if (editor != null)
         {
            cvb = rectangleForCurrentValue();
            editor.setBounds (cvb);
         }
      }
   }

   //
   // end Inner classes
   //====================


   //===============================
   // begin Sub-Component Management
   //

   /**
    * The editor and arrow button are added to the JTextComponent here.
    */
   protected void installComponents()
   {
      if (isEditorVisible (bend))
      {
         addEditor();
      }
      else
      {
         removeEditor();
      }
   }


   /**
    * The editor and/or arrow button are removed from the JTextComponent here. This method
    * calls removeAll() on the JTextComponent just to make sure that everything gets removed.
    */
   protected void uninstallComponents()
   {
      if (editor != null)
      {
         unconfigureEditor();
      }
      bend.removeAll(); // Just to be safe.

   }


   /**
    * Adds the editor to the JTextComponent.
    */
   public void addEditor()
   {
      removeEditor();
      if (editor == null)
      {
         setEditor (createEditor());
      }
      configureEditor();
      bend.add (editor);
   }


   /**
    * Removes the editor from the JTextComponent. It also calls unconfigureEditor()
    */
   public void removeEditor()
   {
      if (editor != null)
      {
         unconfigureEditor();
         bend.remove (editor);
      }
   }


   /**
    * Configures the editor by setting its font and adding listeners.
    */
   protected void configureEditor()
   {
      editor.setFont (bend.getFont());
      editor.setBorder (new EmptyBorder (0, 0, 0, 0));
      editor.setOpaque (bend.isOpaque());
      editor.setForeground (bend.getForeground());
      editor.setBackground (bend.getBackground());
      editor.setText (getText());
      installKeyboardActions();

      if (editorFocusListener == null)
      {
         editorFocusListener =
            new FocusAdapter()
            {
               public void focusGained (FocusEvent e)
               {
               }


               public void focusLost (FocusEvent e)
               {
                  editor.setText (getText());
                  setEditorVisible (bend, false);
               }
            };
      }
      editor.addFocusListener (editorFocusListener);
   }


   /**
    * Unconfigures the editor by removing listeners.
    */
   protected void unconfigureEditor()
   {
      editor.removeFocusListener (editorFocusListener);
      uninstallKeyboardActions();
   }

   //
   // end Sub-Component Management
   //===============================


   //================================
   // begin TextComponentUI Implementation
   //

   /**
    * Tells if the popup is visible or not.
    *
    * @param b  No description provided
    * @return   The editorVisible value
    */
   public boolean isEditorVisible (JBend b)
   {
      return  (editor != null && editor.getParent() == b);
   }


   /**
    * Hides the popup.
    *
    * @param b  The new editorVisible value
    * @param v  The new editorVisible value
    * @return   No description provided
    */
   public boolean setEditorVisible (JBend b, boolean v)
   {
      if (isEditorVisible (b) != v)
      {
         if (v)
         {
            addEditor();
            editor.revalidate();
            editor.repaint();
            editor.requestFocus();
         }
         else
         {
            removeEditor();
            b.requestFocus();
         }
         firePropertyChange (b, "editorVisible", !v, v);
         b.revalidate();
         repaintCurrentValue(); //b.repaint();

         return true;
      }
      return false;
   }


   /**
    * Determines if the JTextComponent is focus traversable. If the JTextComponent is editable
    * this returns false, otherwise it returns true.
    *
    * @param c  No description provided
    * @return   The focusTraversable value
    */
   public boolean isFocusTraversable (JBend c)
   {
      return ! (c.isEnabled() && isEditorVisible (c));
   }

   //
   // end TextComponentUI Implementation
   //==============================


   //=================================
   // begin ComponentUI Implementation

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    * @param c  No description provided
    */
   public void paint (Graphics g, JComponent c)
   {
      if (!isEditorVisible ((JBend) c))
      {
         Rectangle r = rectangleForCurrentValue();
         paintCurrentValueBackground (g, new Rectangle (c.getSize()));
         paintCurrentValue (g, r);
      }
   }


   /**
    * Get the preferredSize attribute of the TextGrabUI object
    *
    * @param c  No description provided
    * @return   The preferredSize value
    */
   public Dimension getPreferredSize (JComponent c)
   {
      Insets insets = getInsets();
      Dimension size = getMinimumSize (c);
      size.width = Math.max (size.width,
         editor.getPreferredSize().width + insets.left + insets.right + 4);
      size.height = Math.max (size.height, editor.getPreferredSize().height + insets.top + insets.bottom + 4);
      return size;
   }


   /**
    * Get the minimumSize attribute of the TextGrabUI object
    *
    * @param c  No description provided
    * @return   The minimumSize value
    */
   public Dimension getMinimumSize (JComponent c)
   {
      Insets insets = getInsets();
      FontMetrics fm = c.getFontMetrics (c.getFont());
      Dimension dim = editor.getMinimumSize();
      Rectangle2D charBounds = fm.getMaxCharBounds (null);
      dim.width = Math.max (fm.stringWidth ("WWW..."), dim.width);
      dim.height = (int) Math.max (charBounds.getHeight(), dim.height);
      dim.width += insets.left + insets.right + 4;
      dim.height += insets.top + insets.bottom + 4;

      return dim;
   }


   /**
    * Get the maximumSize attribute of the TextGrabUI object
    *
    * @param c  No description provided
    * @return   The maximumSize value
    */
   public Dimension getMaximumSize (JComponent c)
   {
      Dimension size = getPreferredSize (c);
      size.width = Short.MAX_VALUE;
      return size;
   }

   //
   // end ComponentUI Implementation
   //===============================


   //======================
   // begin Utility Methods
   //

   /**
    * Returns the area that is reserved for drawing the currently selected item.
    *
    * @return   No description provided
    */
   protected Rectangle rectangleForCurrentValue()
   {
      int width = bend.getWidth();
      int height = bend.getHeight();
      Insets insets = getInsets();
      return new Rectangle (insets.left + 2, insets.top + 2,
         width -  (insets.left + insets.right) - 4,
         height -  (insets.top + insets.bottom) - 4);
   }


   /**
    * Gets the insets from the JTextComponent.
    *
    * @return   The insets value
    */
   protected Insets getInsets()
   {
      return bend.getInsets();
   }

   //
   // end Utility Methods
   //====================


   //===============================
   // begin Painting Utility Methods
   //

   /**
    * Paints the currently selected item.
    *
    * @param g       No description provided
    * @param bounds  No description provided
    */
   public void paintCurrentValue (Graphics g, Rectangle bounds)
   {
      Document doc = editor.getDocument();
      try
      {
         if (doc instanceof AbstractDocument)
         {
             ((AbstractDocument) doc).readLock();
         }

         View rootView = editor.getUI().getRootView (editor);

         /*
          *  Dimension size=bend.getSize();
          *  Rectangle paintArea=new Rectangle();
          *  Insets bendInsets=bend.getInsets();
          *  Insets editorInsets=editor.getInsets();
          *  paintArea.x=bendInsets.left+editorInsets.left;
          *  paintArea.y=bendInsets.top+editorInsets.top;
          *  paintArea.width=size.width-(bendInsets.left+bendInsets.right+editorInsets.left+editorInsets.right);
          *  paintArea.height=size.height-(bendInsets.top+bendInsets.bottom+editorInsets.top+editorInsets.bottom);
          */
         rootView.paint (g, bounds); //paintArea);

      }
      finally
      {
         if (doc instanceof AbstractDocument)
         {
             ((AbstractDocument) doc).readUnlock();
         }
      }
   }


   /**
    * Paints the background of the currently selected item.
    *
    * @param g       No description provided
    * @param bounds  No description provided
    */
   public void paintCurrentValueBackground (Graphics g, Rectangle bounds)
   {
      Color t = g.getColor();
      g.setColor (bend.getBackground());
      g.fillRect (bounds.x, bounds.y, bounds.width, bounds.height);
      g.setColor (t);
   }


   /**
    * Repaint the currently selected item.
    */
   void repaintCurrentValue()
   {
      Rectangle r = rectangleForCurrentValue();
      bend.repaint (r.x, r.y, r.width, r.height);
   }

   //
   // end Painting Utility Methods
   //=============================


   /**
    * Adds keyboard actions to the JTextComponent. Actions on enter and esc are already supplied.
    * Add more actions as you need them.
    */
   protected void installKeyboardActions()
   {
      AbstractAction keyAction =
         new AbstractAction()
         {
            public void actionPerformed (ActionEvent e)
            {
               if ("cancel".equals (e.getActionCommand()))
               {
                  editor.setText (getText());
               }
               else
               {
                  setText (editor.getText());
               }
               setEditorVisible (bend, false);
               repaintCurrentValue();
            }
         };

      bend.registerKeyboardAction (keyAction, "cancel",
         KeyStroke.getKeyStroke (KeyEvent.VK_ESCAPE, 0),
         JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);

      bend.registerKeyboardAction (keyAction, "set",
         KeyStroke.getKeyStroke (KeyEvent.VK_ENTER, 0),
         JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
      if (editor instanceof JTextField)
      {
          ((JTextField) editor).setActionCommand ("set");
          ((JTextField) editor).addActionListener (keyAction);
      }
   }


   /**
    * Removes the keyboard actions that were added by installKeyboardActions().
    */
   protected void uninstallKeyboardActions()
   {
      bend.unregisterKeyboardAction (KeyStroke.getKeyStroke (KeyEvent.VK_ESCAPE, 0));
      bend.unregisterKeyboardAction (KeyStroke.getKeyStroke (KeyEvent.VK_ENTER, 0));
   }

   //
   // end Keyboard Action Management
   //===============================
}

/*
 * $Log: TextGrabUI.java,v $
 * Revision 1.15  2004/11/03 10:17:59  lowende
 * Javadoc warnings removed.
 *
 */
