/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;

import de.uni_paderborn.fujaba.fsa.swing.border.OvalBorder;


/**
 * Layouts the components line by line in such a style that the components will fit into an
 * OvalBorder that may be put around the container.<p>
 *
 * The insets of the OvalBorder that encloses the component directly (i.e. the OvalBorder is
 * the inner border of a CompoundBorder or the OvalBorder is used alone) are ignored.<p>
 *
 * This LayoutManager tries to optimize the horizontal size of the oval by shifting the components
 * vertically.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.13 $
 */
public class OvalLayout implements LayoutManager
{
   /**
    * Creates an OvalLayout with default settings.
    */
   public OvalLayout()
   {
      this (3, 3, 8);
   }


   /**
    * Creates an OvalLayout with the given <code>vgap</code>.
    *
    * @param verticalGap        the vertical spacing between the components
    * @param borderGap          The spacing between the components and the border of the oval
    * @param verticalBorderGap  No description provided
    */
   public OvalLayout (int verticalGap, int borderGap, int verticalBorderGap)
   {
      this.verticalGap = verticalGap;
      this.borderGap = borderGap;
      this.verticalBorderGap = verticalBorderGap;
   }


   /**
    * The vertical spacing between the components
    */
   int verticalGap;

   /**
    * The spacing between the components and the border of the oval
    */
   int borderGap;

   /**
    * More spacing at the top and at the bottom of the oval
    */
   int verticalBorderGap;


   /**
    * ComponentInfo serves as a simple data structure for associating a component with its
    * planned bounds.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.13 $
    */
   private static class ComponentInfo
   {
      /**
       * Creates a new ComponentInfo with the given values
       *
       * @param component  No description provided
       * @param x          No description provided
       * @param y          No description provided
       * @param w          No description provided
       * @param h          No description provided
       */
      ComponentInfo (Component component, int x, int y, int w, int h)
      {
         this (component, new Rectangle (x, y, w, h));
      }


      /**
       * Creates a new ComponentInfo with the given values
       *
       * @param component  No description provided
       * @param bounds     No description provided
       */
      ComponentInfo (Component component, Rectangle bounds)
      {
         this.component = component;
         this.bounds = bounds;
      }


      /**
       * The component the bounds are stored for
       */
      Component component;

      /**
       * The planned bounds for the component
       */
      Rectangle bounds;

      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      int aggregatedWidth;
   }


   /**
    * Container for a layout that has been calculated but not yet applied to the components.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.13 $
    */
   private class TempLayout
   {
      /**
       * Layouts the components in the given container and returns the layout result in form
       * of a ComponentInfo array. Note that the array may contain null elements for components
       * that are not visible. The calculated dimension for the oval can be queried with <code>getOvalDim()</code>
       * . <em>The treeLock of the container must already be obtained by the calling method.
       * </em>
       *
       * @param container  container to create layout for
       */
      TempLayout (Container container)
      {
         this (container, false, 0);
      }


      /**
       * Layouts the components in the given container and returns the layout result in form
       * of a ComponentInfo array. Note that the array may contain null elements for components
       * that are not visible. The calculated dimension for the oval can be queried with <code>getOvalDim()</code>
       * . <em>The treeLock of the container must already be obtained by the calling method.
       * </em>
       *
       * @param container  container to create layout for
       * @param secondTry  is the algorithm alreagy recursing?
       * @param shift      number of pixels to shift all components either up or down in the
       *      layout. This may increase the component size.
       */

      private TempLayout (Container container, boolean secondTry, int shift)
      {
         int componentCount = container.getComponentCount();

         /*
          *  Layouting is done in several passes:
          *
          *  1st Pass:
          *  Count the visible components and create the componentInfo array.
          */
         int visibleComponents = 0;

         for (int i = 0; i < componentCount; i++)
         {
            Component comp = container.getComponent (i);

            if (comp.isVisible())
            {
               visibleComponents++;
            }
         }

         componentInfos = new ComponentInfo[visibleComponents];

         /*
          *  2nd Pass:
          *  - Fill in the componentInfos
          *  - Get preferred sizes of the components
          *  and make the preferred size the actual size
          *  (We cache this way the size information
          *  because the size information is required later
          *  again, but getPreferredSize() can be a slow
          *  operation)
          *  - Calculate total height
          *  - Calculate the maximum width of one component
          */
         int maxWidth = 0;
         int height = borderGap + verticalBorderGap;

         if (shift > 0)
         {
            height += shift;
         }

         int k = 0;

         for (int i = 0; i < componentCount; i++)
         {
            Component comp = container.getComponent (i);

            if (comp.isVisible())
            {
               Dimension preferredSize = comp.getPreferredSize();

               componentInfos[k] = new ComponentInfo (comp, 0, height, preferredSize.width, preferredSize.height);

               maxWidth = Math.max (maxWidth, preferredSize.width);

               height += preferredSize.height;

               if (k != visibleComponents - 1)
               {
                  height += verticalGap;
               }

               k++;
            }
         }

         if (shift < 0)
         {
            height += -shift; // Okay, that's equal to height -= shift ;)
         }

         /*
          *  Guess the dimensions of the oval using
          *  the total height and the maximum width
          */
         ovalDim.width = maxWidth + 4;
         ovalDim.height = height + borderGap + verticalBorderGap;

         /*
          *  3rd Pass:
          *  Check for all components whether they fit into
          *  the guessed borders
          */
         for (int i = 0; i < componentInfos.length; i++)
         {
            int newWidth = isComponentInsideOval (componentInfos[i], ovalDim);

            if (newWidth != -1)
            {
               /*
                *  The oval dimensions have to be increased because
                *  this component does not fit into the current dimensions
                */
               ovalDim.width = newWidth;
            }
         }

         ovalDim.width += borderGap * 2;

         /*
          *  4th Pass:
          *  Calculate for all components the x coordinate
          *  using the now calculated width of the oval.
          *  Add to the coordinates and the dimension the insets
          *  of this container.
          */
         Insets insets = getInsetsWithoutOval (container);

         ovalDim.width += insets.left + insets.right;
         ovalDim.height += insets.top + insets.bottom;

         for (int i = 0; i < componentInfos.length; i++)
         {
            componentInfos[i].bounds.x =  (ovalDim.width - componentInfos[i].bounds.width) / 2 + insets.left;

            componentInfos[i].bounds.y += insets.top;
         }

         /*
          *  5th Pass:
          *  Evaluate the statistic done by isComponentInsideOval()
          *  whether the components could be moved up or down
          *  to enhance appearance and try a layout with that
          *  value.
          */
         int diff = getYDiff();

         if (!secondTry && Math.abs (diff) > 4)
         {
            TempLayout secondLayout = new TempLayout (container, true, diff);

            if (Math.abs (secondLayout.getYDiff()) < Math.abs (diff))
            {
               ovalDim = secondLayout.getOvalDim();
               componentInfos = secondLayout.getComponentInfos();
            }
         }
      }


      /**
       * The calculated dimensions of the oval
       */
      private Dimension ovalDim = new Dimension();


      /**
       * Returns the calculated dimensions of the oval
       *
       * @return   the calculated dimensions of the oval
       */
      Dimension getOvalDim()
      {
         return ovalDim;
      }


      /**
       * The calculated layout for the components
       */
      private ComponentInfo[] componentInfos;


      /**
       * Returns the calculated layout for the components
       *
       * @return   the calculated layout for the components
       */
      ComponentInfo[] getComponentInfos()
      {
         return componentInfos;
      }


      /**
       * The differences between the current and the optimal y coordinates are stored here
       * by <code>isComponentInsideOval()</code>
       */
      private int yDiff;

      /**
       * The number of yDiffs counted
       */
      private int yDiffNumber;


      /**
       * Returns the average yDiff (the differences between the current and the optimal y coordinates)
       * calculated by <code>isComponentInsideOval()</code>.
       *
       * @return   The yDiff value
       */
      public int getYDiff()
      {
         return (int) Math.round ((double) yDiff / (double) yDiffNumber);
      }


      /**
       * Checks whether the given componentInfo is completely contained in the oval specified
       * by ovalSize. If it fits, -1 is returned, otherwise the width of the oval that can
       * contain the component at its current position.
       *
       * @param componentInfo  No description provided
       * @param ovalSize       No description provided
       * @return               The componentInsideOval value
       */

      private int isComponentInsideOval (ComponentInfo componentInfo, Dimension ovalSize)
      {
         /*
          *  The following formula calculates the x position of
          *  the oval border at the y position given by componentInfo.bounds.y.
          *  All calculations in this method are done relative to
          *  the center of the ellipsis, thus some normalizations
          *  are neccessary.
          *  The formula can be easily deduced from the implicit
          *  ellipsis formula:
          *  b�x� + a�y� - a�b� = 0.
          *  The formula below in human readable notation:
          *  x = sqrt( a� (1 - y�/b�))
          */
         int borderPosAtY1
             = (int) Math.ceil (Math.sqrt (
            Math.pow (ovalSize.width / 2.0, 2.0)
            *  (1 - Math.pow (ovalSize.height / 2.0 - componentInfo.bounds.y, 2.0) / Math.pow (ovalSize.height / 2, 2.0))));

         /*
          *  And once again the same for the lower edge of
          *  the component
          */
         int borderPosAtY2
             = (int) Math.ceil (Math.sqrt (
            Math.pow (ovalSize.width / 2.0, 2.0)
            *  (1 - Math.pow (ovalSize.height / 2.0 - componentInfo.bounds.y - componentInfo.bounds.height + 1, 2.0) / Math.pow (ovalSize.height / 2, 2.0))));

         /*
          *  Check whether the component fits inside
          *  the calculated border.
          */
         int result;

         if (Math.ceil (componentInfo.bounds.width / 2.0) > Math.min (borderPosAtY1, borderPosAtY2))
         {
            /*
             *  The component does not fit inside the border.
             *  Calculate the new required width of the ellipsis.
             *  b�x� + a�y� - a�b� = 0
             *  => a�y� - a�b� = - b�x�
             *  => a�(y� - b�) = - b�x�
             *  => a� = b�x� / (b� - y�)
             *  => a = sqrt(b�x� / (b� - y�))
             *  => a = |bx| / sqrt(b� - y�)
             *  We have to use again two values, one for the top of the component
             *  and one for the bottom and use the maximum.
             */
            int newWidth1 = (int)  (Math.abs ( (ovalSize.height / 2.0) * Math.ceil (componentInfo.bounds.width / 2.0))
               / Math.sqrt (Math.pow (ovalSize.height / 2.0, 2.0) - Math.pow (ovalSize.height / 2.0 - componentInfo.bounds.y, 2.0)));

            int newWidth2 = (int)  (Math.abs ( (ovalSize.height / 2.0) * Math.ceil (componentInfo.bounds.width / 2.0))
               / Math.sqrt (Math.pow (ovalSize.height / 2.0, 2.0) - Math.pow (ovalSize.height / 2.0 - componentInfo.bounds.y - componentInfo.bounds.height, 2.0)));

            result = Math.max (newWidth1, newWidth2) * 2;
         }
         else
         {
            result = -1;
         }

         /*
          *  Do some statistics, that might help to improve
          *  the layout: Calculate the position where the
          *  component would have fit into the bounds
          *  b�x� + a�y� - a�b� = 0
          *  => a�y� = a�b� - b�x�
          *  => y� = b� - b�x�/a�
          *  => y� = b� (1 - x�/a�)
          *  => y  = sqrt(b� (1 - x�/a�)
          *  You may also choose to ignore all these formulas
          *  and consider the following code as magic ;)
          */
         int betterYPos = (int) Math.round (Math.sqrt (Math.pow (ovalSize.height / 2.0, 2.0)
            *  (1 - Math.pow ( (componentInfo.bounds.width) / 2.0, 2.0) / Math.pow (ovalSize.width / 2.0, 2.0))));
         /*
          *  Do not take the component in the middle of the
          *  container into account
          */
         if (! ( (componentInfo.bounds.y < ovalSize.height / 2) &&  (componentInfo.bounds.y + componentInfo.bounds.height - 1 > ovalSize.height / 2)) ||  (componentInfos.length % 2 == 0))
         {
            /*
             *  The betterYPos value is relative to the middle of
             *  the ellipsis but always positive. Using the old y position
             *  try to find out whether the betterYPos value should point
             *  to the upper or lower half of the ellipsis and make the
             *  value relative to the top.
             */
            if (componentInfo.bounds.y + componentInfo.bounds.height / 2 > ovalSize.height / 2)
            {
               betterYPos += ovalSize.height / 2 - componentInfo.bounds.height / 2;
            }
            else
            {
               betterYPos = ovalSize.height / 2 - betterYPos - componentInfo.bounds.height / 2;
            }

            yDiff += betterYPos - componentInfo.bounds.y;
            yDiffNumber++;
         }

         return result;
      }
   }


   /**
    * Lay outs the components in the container that they fit into an oval that the container
    * can contain ... (weird sentence)
    *
    * @param container  No description provided
    */

   public void layoutContainer (Container container)
   {
      synchronized (container.getTreeLock())
      {
         TempLayout tempLayout = new TempLayout (container);

         ComponentInfo[] componentInfos = tempLayout.getComponentInfos();

         /*
          *  Write the information to the components
          */
         for (int i = 0; i < componentInfos.length; i++)
         {
            if (componentInfos[i] != null)
            {
               componentInfos[i].component.setBounds (componentInfos[i].bounds);
            }
         }
      }
   }


   /**
    * Returns the minimum size that the container may use. Is currently equal to the preferred
    * layout size.
    *
    * @param container  No description provided
    * @return           No description provided
    */
   public Dimension minimumLayoutSize (Container container)
   {
      return preferredLayoutSize (container);
   }


   /**
    * Returns the preferred size of this layout.
    *
    * @param container  No description provided
    * @return           No description provided
    */

   public Dimension preferredLayoutSize (Container container)
   {
      synchronized (container.getTreeLock())
      {
         TempLayout tempLayout = new TempLayout (container);

         return tempLayout.getOvalDim();
      }
   }


   /**
    * Adds a component to the layout. <em>Not used by this LayoutManager.</em>
    *
    * @param name  The object added.
    * @param comp  The object added.
    */
   public void addLayoutComponent (String name, Component comp) { }


   /**
    * Removes a component from the layout. <em>Not used by this LayoutManager.</em>
    *
    * @param comp  No description provided
    */
   public void removeLayoutComponent (Component comp) { }


   /**
    * Calculates the insets of this container, but ignores the insets of an directly enclosing
    * OvalBorder, if it exists.
    *
    * @param container  the container to calculate the insets for.
    * @return           the insets of the container without the insets of the inner OvalBorder.
    */

   public static Insets getInsetsWithoutOval (Container container)
   {
      Insets insets = container.getInsets();

      if (container instanceof JComponent)
      {
         Border border =  ((JComponent) container).getBorder();

         while (border instanceof CompoundBorder)
         {
            border =  ((CompoundBorder) border).getInsideBorder();
         }

         if (border instanceof OvalBorder)
         {
            Insets borderInsets = border.getBorderInsets (container);
            insets = new Insets (insets.top - borderInsets.top, insets.left - borderInsets.left, insets.bottom - borderInsets.bottom, insets.right - borderInsets.right);
         }
      }

      return insets;
   }


   /**
    * Get the insetsTilBorder attribute of the OvalLayout class
    *
    * @param c           No description provided
    * @param thisBorder  No description provided
    * @param stopBorder  No description provided
    * @param insets      No description provided
    * @return            The insetsTilBorder value
    */
   private static boolean getInsetsTilBorder (JComponent c, Border thisBorder, Border stopBorder, Insets insets)
   {
      if (thisBorder != stopBorder)
      {
         if (thisBorder == null)
         {
            return true;
         }
         else if (thisBorder instanceof CompoundBorder)
         {
            boolean goon = getInsetsTilBorder (c,  ((CompoundBorder) thisBorder).getOutsideBorder(), stopBorder, insets);

            if (goon)
            {
               goon = getInsetsTilBorder (c,  ((CompoundBorder) thisBorder).getInsideBorder(), stopBorder, insets);
            }

            return goon;
         }
         else
         {
            Insets borderInsets = thisBorder.getBorderInsets (c);

            insets.top += borderInsets.top;
            insets.left += borderInsets.left;
            insets.right += borderInsets.right;
            insets.bottom += borderInsets.bottom;

            return true;
         }
      }
      else
      {
         return false;
      }
   }


   /**
    * Calculates the insets of this container til the given oval border.
    *
    * @param c           the component the border belongs to
    * @param stopBorder  No description provided
    * @return            the insets of the component starting from the outer border til the
    *      given border
    */
   public static Insets getInsetsTilBorder (JComponent c, Border stopBorder)
   {
      Insets insets = new Insets (0, 0, 0, 0);

      getInsetsTilBorder (c, c.getBorder(), stopBorder, insets);

      return insets;
   }
}

/*
 * $Log: OvalLayout.java,v $
 * Revision 1.13  2004/11/03 10:17:58  lowende
 * Javadoc warnings removed.
 *
 */
