/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.geom.Point2D;

import javax.swing.*;


/**
 * constraints for DecoratorLayout to add some stuff to a JLine</p> UMLClass: 'LineDecoratorConstraints'
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.10 $
 */
public class LineDecoratorConstraints implements DecoratorLayout.Constraints
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte CENTER = 0;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte LEFT = 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte RIGHT = 2;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte HORIZ_ALIGN = LEFT | RIGHT;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte TOP = 4;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte BOTTOM = 8;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte VERT_ALIGN = TOP | BOTTOM;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte BORDER_ALIGN = HORIZ_ALIGN | VERT_ALIGN;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte TOP_LEFT = TOP | LEFT;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte TOP_RIGHT = TOP | RIGHT;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte BOTTOM_LEFT = BOTTOM | LEFT;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static byte BOTTOM_RIGHT = BOTTOM | RIGHT;


   /**
    * Constructor for class LineDecoratorConstraints
    *
    * @param target       No description provided
    * @param pos          No description provided
    * @param gap          No description provided
    * @param decAlign     No description provided
    * @param targetAlign  No description provided
    */
   public LineDecoratorConstraints (JLine target, double pos, double gap, byte decAlign, byte targetAlign)
   {
      setDecoratorAlignment (decAlign);
      setTargetAlignment (targetAlign);
      setPosition (pos);
      setGap (gap);
      setTarget (target);
   }


   /**
    * Constructor for class LineDecoratorConstraints
    *
    * @param target  No description provided
    * @param pos     No description provided
    * @param gap     No description provided
    */
   public LineDecoratorConstraints (JLine target, double pos, double gap)
   {
      this (target, pos, gap, CENTER, TOP);
   }


   /**
    * UMLAttribute: 'decoratorAlignment : Double'
    */
   private byte decoratorAlignment = CENTER;

   /**
    * UMLAttribute: 'decoratorAlignment : Double'
    */
   private byte targetAlignment = TOP;

   /**
    * UMLAttribute: 'gap : Double'
    */
   private double gap = 0;

   /**
    * UMLAttribute: 'position : Byte'
    */
   private double position = 0.5;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Component target = null;


   /**
    * UMLMethod: 'Read access method for attribute target : Component'
    *
    * @return   The target value
    */
   public Component getTarget()
   {
      return target;
   }


   /**
    * UMLMethod: 'Write access method for attribute target : Component'
    *
    * @param target  The new target value
    * @return        No description provided
    */
   public Component setTarget (Component target)
   {
      if (this.target != target)
      {
         if (! (target instanceof JLine))
         {
            throw new IllegalArgumentException ("Target must be a JLine");
         }

         this.target = target;
      } // if

      return this.target;
   }


   /**
    * Get the location attribute of the LineDecoratorConstraints object
    *
    * @param comp  No description provided
    * @return      The location value
    */
   public Point getLocation (Component comp)
   {
      JLine line = (JLine) getTarget();

      if (line == null)
      {
         return comp.getLocation();
      }

      Point transform;
      if (line.getParent() != null && comp.getParent() != null)
      {
         transform = SwingUtilities.convertPoint (line.getParent(), 0, 0, comp.getParent());
      }
      else
      {
         transform = new Point (0, 0);
      }

      Point2D linePos = line.getPointAt (getPosition());
      double posX = linePos.getX() + transform.x;
      double posY = linePos.getY() + transform.y;

      double angle = line.getAngle (getPosition());
      Point2D n = line.getNormal (getPosition());
      double nx = n.getX();
      double ny = n.getY();

      Dimension compDim = comp.getSize();

      if (gap != 0)
      {
         int sign = 0;
         if ( (targetAlignment & TOP) != 0)
         {
            sign = 1;
         }
         else if ( (targetAlignment & BOTTOM) != 0)
         {
            sign = -1;
         }

         if (sign != 0)
         {
            if (angle > 1.5 * Math.PI || angle <= 0.5 * Math.PI)
            {
               posX -= sign * nx * gap;
               posY -= sign * ny * gap;
            }
            else
            {
               posX += sign * nx * gap;
               posY += sign * ny * gap;
            }
         }
      }

      byte align = decoratorAlignment;

      if ( (decoratorAlignment & HORIZ_ALIGN) == HORIZ_ALIGN)
      {
         //determine best horizontal fit
         if (nx == 0)
         {
            // use center, turn off left- and right-bit
            align &= ~HORIZ_ALIGN;
         }
         else if (angle < Math.PI / 2.05 ||  (angle > Math.PI && angle < 1.45 * Math.PI))
         {
            // use left, turn off right-bit
            align &= ~RIGHT;
         }
         else if ( (angle >= Math.PI / 2.05 && angle <= Math.PI / 1.95)
            ||  (angle >= Math.PI * 1.45 && angle <= Math.PI * 1.55))
         {
            //FIX ME: still wrong
            if ( (targetAlignment & BOTTOM) == BOTTOM)
            {
               // use right, turn off left-bit
               align &= ~LEFT;
            }
            else
            {
               // use left, turn off right-bit
               align &= ~RIGHT;
            }
         }
         else
         {
            // use right, turn off left-bit
            align &= ~LEFT;
         }
      }

      if ( (decoratorAlignment & VERT_ALIGN) == VERT_ALIGN)
      {
         //determine best vertical fit
         if (Math.abs (ny) <= 0.02)
         {
            // use center, turn off top- and bottom-bit
            align &= ~VERT_ALIGN;
         }
         else
         {
            // use bottom, turn off top-bit
            align &= ~TOP;
         }
      }

      double compX;

      double
         compY;

      if ( (align & HORIZ_ALIGN) == CENTER)
      {
         compX = (double) compDim.width / 2;
      }
      else if ( (align & LEFT) != 0)
      {
         compX = 0;
      }
      else
      {
         compX = compDim.width;
      }

      if ( (align & VERT_ALIGN) == CENTER)
      {
         compY = (double) compDim.height / 2;
      }
      else if ( (align & TOP) != 0)
      {
         compY = 0;
      }
      else
      {
         compY = compDim.height;
      }

      return new Point ((int) Math.rint (posX - compX), (int) Math.rint (posY - compY));
   }


   /**
    * UMLMethod: 'Read access method for attribute position : Double'
    *
    * @return   The position value
    */
   public double getPosition()
   {
      return position;
   }


   /**
    * UMLMethod: 'Write access method for attribute position : Double'
    *
    * @param position  The new position value
    * @return          No description provided
    */
   public double setPosition (double position)
   {
      if (this.position != position)
      {
         if (position < 0 || position > 1)
         {
            throw new IllegalArgumentException ("Position must be between 0 and 1");
         }

         this.position = position;
      } // if

      return this.position;
   }


   /**
    * UMLMethod: 'Read access method for attribute decoratorAlignment : Byte'
    *
    * @return   The decoratorAlignment value
    */
   public byte getDecoratorAlignment()
   {
      return decoratorAlignment;
   }


   /**
    * UMLMethod: 'Write access method for attribute decoratorAlignment : Byte'
    *
    * @param decoratorAlignment  The new decoratorAlignment value
    * @return                    No description provided
    */
   public byte setDecoratorAlignment (byte decoratorAlignment)
   {
      if (this.decoratorAlignment != decoratorAlignment)
      {
         if ( (decoratorAlignment & BORDER_ALIGN) != decoratorAlignment)
         {
            throw new IllegalArgumentException ("Decorator alignment invalid");
         }
         this.decoratorAlignment = decoratorAlignment;
      } // if

      return this.decoratorAlignment;
   }


   /**
    * UMLMethod: 'Read access method for attribute targetAlignment : Byte'
    *
    * @return   The targetAlignment value
    */
   public byte getTargetAlignment()
   {
      return targetAlignment;
   }


   /**
    * UMLMethod: 'Write access method for attribute targetAlignment : Byte'
    *
    * @param targetAlignment  The new targetAlignment value
    * @return                 No description provided
    */
   public byte setTargetAlignment (byte targetAlignment)
   {
      if (this.targetAlignment != targetAlignment)
      {
         if (targetAlignment != CENTER &&
            targetAlignment != TOP &&
            targetAlignment != BOTTOM)
         {
            throw new IllegalArgumentException ("Target alignment must be one of CENTER, TOP or BOTTOM");
         }
         this.targetAlignment = targetAlignment;
      } // if

      return this.targetAlignment;
   }


   /**
    * UMLMethod: 'Read access method for attribute gap : Double'
    *
    * @return   The gap value
    */
   public double getGap()
   {
      return gap;
   }


   /**
    * UMLMethod: 'Write access method for attribute gap : Double'
    *
    * @param gap  The new gap value
    * @return     No description provided
    */
   public double setGap (double gap)
   {
      if (this.gap != gap)
      {
         this.gap = gap;
      } // if

      return this.gap;
   }


   /**
    * UMLMethod: '+ removeYou () : Void'
    */
   public void removeYou()
   {
      setTarget (null);
   }
}

/*
 * $Log: LineDecoratorConstraints.java,v $
 * Revision 1.10  2004/11/03 10:17:58  lowende
 * Javadoc warnings removed.
 *
 */
