/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.event.*;

import javax.swing.*;


/**
 * show value with JLabel , edit with JTextField
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.21 $
 */
public class JTextFieldLabel extends EditableTextComponent
    implements ActionListener, FocusListener
{
   /**
    * Default constructor, creates a new JTextFieldLabel with normal string editing.
    */
   public JTextFieldLabel()
   {
      super();
   }


   /**
    * Creates a new JTextFieldLabel.
    *
    * @param numbersOnly  if set to <code>true</code>, this JTextFieldLabel will only support
    *      numbers as inputs. In this case, the methods getNumber() and setNumber() can be
    *      used to obtain the value of the text field. If set to <code>false</code> this JTextFieldLabel
    *      accepts all inputs.
    */
   public JTextFieldLabel (boolean numbersOnly)
   {
      super();

      this.numbersOnly = numbersOnly;
   }


   /**
    * Decides whether this text field label only accepts numbers. If set to true, the methods
    * getNumber() and setNumber() can be used to access the content.
    */
   private boolean numbersOnly;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param component  No description provided
    */
   protected void updateEditor (JComponent component)
   {
      if (component instanceof JTextField)
      {
         JTextField textField = (JTextField) component;
         textField.setText (getText());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param component  No description provided
    */
   protected void enableEditor (JComponent component)
   {
      if (component instanceof JTextField)
      {
         super.enableEditor (component);

         JTextField textField = (JTextField) component;

         textField.setText (getText());
         textField.setFont (getFont());
         textField.setHorizontalAlignment (getHorizontalAlignment());
         textField.addFocusListener (this);
         textField.addActionListener (this);

         if (numbersOnly)
         {
            textField.setInputVerifier (IntegerVerifier.get());
         }
         else
         {
            textField.setInputVerifier (null);
         } // end of else

         if (hasFocus() || getLabel().hasFocus())
         {
            textField.requestFocus();
         }
      } // end of if ()
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param component  No description provided
    */
   protected void disableEditor (JComponent component)
   {
      if (component instanceof JTextField)
      {
         super.disableEditor (component);

         JTextField textField = (JTextField) component;
         textField.removeActionListener (this);
         textField.removeFocusListener (this);

         setText (textField.getText());
         if (textField.hasFocus())
         {
            getLabel().requestFocus();
         }
      }
   }


   /**
    * Get the editorClassName attribute of the JTextFieldLabel object
    *
    * @return   The editorClassName value
    */
   protected String getEditorClassName()
   {
      return JTextField.class.getName();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void updateJLabel()
   {
      getLabel().setText (getText());
   }


   /**
    * Returns the integer representation of the contents of this JTextFieldLabel. Should be
    * only used when this JTextFieldLabel was created in numbersOnly mode.
    *
    * @return   The number value
    */
   public int getNumber()
   {
      try
      {
         return Integer.parseInt (getText());
      }
      catch (NumberFormatException e)
      {
         return 0;
      }
   }


   /**
    * Sets this JTextFieldLabel to the representation of the given number.
    *
    * @param number  the number to set.
    */
   public void setNumber (int number)
   {
      int oldNumber = getNumber();
      setText (Integer.toString (number));
      firePropertyChange ("number", new Integer (oldNumber), new Integer (number));
   }


   /**
    * Sets the visible attribute of the JTextFieldLabel object
    *
    * @param vis  The new visible value
    */
   public void setVisible (boolean vis)
   {
      super.setVisible (vis);
      getLabel().setVisible (vis);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void actionPerformed (ActionEvent e)
   {
      if (isEditing())
      {
         JTextField textField = (JTextField) EditorManager.get().
            getEditor (getClass().getName(), getEditorClassName());

         if (numbersOnly)
         {
            int number = 0;

            try
            {
               number = Integer.parseInt (textField.getText());
            }
            catch (NumberFormatException ex)
            {
            }

            setNumber (number);
         }
         else
         {
            setText (textField.getText());
         }
         setDisplayMode();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void focusGained (FocusEvent e)
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void focusLost (FocusEvent e)
   {
      if (isEditing())
      {
         setDisplayMode();
      }
   }
}

/*
 * $Log: JTextFieldLabel.java,v $
 * Revision 1.21  2004/11/03 10:17:58  lowende
 * Javadoc warnings removed.
 *
 */
