/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.geom.Point2D;
import java.util.*;

import javax.swing.*;
import javax.swing.plaf.ComponentUI;

import de.upb.tools.fca.FEmptyListIterator;


/**
 * Polyline consisting of several small lines </p> Work in progress. Bugs guaranteed!
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.34.2.2 $
 */
public class JPolyLine extends JBendLine
{
   /**
    * Constructor for class JPolyLine
    *
    * @param start  No description provided
    * @param end    No description provided
    */
   public JPolyLine (JBend start, JBend end)
   {
      super (start, end);
      insertInLines (0, new LineSegment (start, end));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void startPointChanged()
   {
      if (sizeOfLines() == 0)
      {
         super.startPointChanged();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void endPointChanged()
   {
      if (sizeOfLines() == 0)
      {
         super.endPointChanged();
      }
   }


   /**
    * <pre>
    *            0..1                      N
    * JPolyLine ----------------------------- LineSegment
    *            polyLine              lines
    * </pre>
    */
   private Vector lines;


   /**
    * please do not change lines directly. Use the bends- or points-methods instead
    *
    * @param pos  No description provided
    * @return     The fromLines value
    */
   public LineSegment getFromLines (int pos)
   {
      if (lines == null)
      {
         return null;
      }

      return (LineSegment) lines.get (pos);
   }


   /**
    * please do not change lines directly. Use the bends- or points-methods instead
    *
    * @param line  No description provided
    * @return      The indexFromLines value
    */
   public int getIndexFromLines (LineSegment line)
   {
      if (lines == null || line == null)
      {
         return -1;
      }

      return lines.indexOf (line);
   }


   /**
    * please do not change lines directly. Use the bends- or points-methods instead
    *
    * @param pos    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   boolean insertInLines (int pos, LineSegment value)
   {
      boolean changed = false;

      if (value != null)
      {
         int size = sizeOfLines();

         if (pos < 0 || pos > size)
         {
            throw new ArrayIndexOutOfBoundsException (pos);
         }

         if (!hasInLines (value))
         {
            if (this.lines == null)
            {
               this.lines = new Vector();
            }

            this.lines.add (pos, value);
            value.setPolyLine (this);

            if (value.getParent() != null)
            {
               value.getParent().remove (value);
            }
            this.add (value);

            value.setUI (ui);
            revalidate();

            changed = true;
         }
      }
      return changed;
   }


   /**
    * please do not change lines directly. Use the bends- or points-methods instead
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInLines (LineSegment value)
   {
      return  ( (this.lines != null) &&
          (value != null) &&
         this.lines.contains (value));
   }


   /**
    * please do not change lines directly. Use the bends- or points-methods instead
    *
    * @return   No description provided
    */
   public final ListIterator iteratorOfLines()
   {
      return iteratorOfLines (0);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line  No description provided
    * @return      No description provided
    */
   public final ListIterator iteratorOfLines (LineSegment line)
   {
      return iteratorOfLines (getIndexFromLines (line));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param start  No description provided
    * @return       No description provided
    */
   public ListIterator iteratorOfLines (int start)
   {
      return  ( (this.lines == null)
         ? FEmptyListIterator.get()
         : this.lines.listIterator (start));
   }


   /**
    * please do not change lines directly. Use the bends- or points-methods instead
    *
    * @return   No description provided
    */
   public int sizeOfLines()
   {
      return  ( (this.lines == null)
         ? 0
         : this.lines.size());
   }


   /**
    * please do not change lines directly. Use the bends- or points-methods instead
    *
    * @param value  No description provided
    * @return       No description provided
    */
   boolean removeFromLines (LineSegment value)
   {
      boolean changed = false;

      if ( (this.lines != null) &&  (value != null))
      {
         changed = this.lines.remove (value);
         this.remove (value);
         if (value.getPolyLine() == this)
         {
            value.setPolyLine (null);
         }
         revalidate();
      }
      return changed;
   }


   /**
    * please do not change lines directly. Use the bends- or points-methods instead
    *
    * @param pos  No description provided
    * @return     No description provided
    */
   boolean removeFromLines (int pos)
   {
      boolean changed = false;

      if ( (this.lines != null) &&  (pos >= 0 && pos < sizeOfLines()))
      {
         LineSegment line = (LineSegment) this.lines.get (pos);
         this.lines.remove (pos);
         if (line != null && line.getPolyLine() == this)
         {
            line.setPolyLine (null);
         }
         changed =  (line != null);
      }
      else
      {
         throw new ArrayIndexOutOfBoundsException (pos);
      }

      return changed;
   }


   /**
    * please do not change lines directly. Use the bends- or points-methods instead </p> Warning:
    * This method leaves the object in an invalid state. Use for destruction only!
    */
   void removeAllFromLines()
   {
      if (lines == null)
      {
         return;
      }

      int size = lines.size();
      for (int i = size - 1; i >= 0; i--)
      {
         removeFromLines (i);
      }
   }


   /**
    * @param d  No description provided
    * @return   the line segment at the given position between 0 and 1
    */
   public LineSegment getLineAt (double d)
   {
      if (d > 1 || d < 0)
      {
         throw new IllegalArgumentException (d + " not in interval [0,1]");
      }

      double seekLength =  (d * getLength());
      double tmpLength = -1;
      LineSegment line = null;
      Iterator lineIter = iteratorOfLines();
      if (lineIter.hasNext())
      {
         line = (LineSegment) lineIter.next();
         tmpLength = line.getLength();
      }
      while (lineIter.hasNext() && tmpLength < seekLength)
      {
         line = (LineSegment) lineIter.next();
         tmpLength += line.getLength();
      }
      return line;
   }


   /**
    * @param d  No description provided
    * @return   the linepoint at the given position between 0 and 1
    */
   public Point2D getPointAt (double d)
   {
      if (d > 1 || d < 0)
      {
         throw new IllegalArgumentException (d + " not in interval [0,1]");
      }

      if (sizeOfLines() == 0)
      {
         return super.getPointAt (d);
      }

      double totalLength = getLength();

      if (totalLength == 0 || d == 0)
      {
         return getStartPoint();
      }
      else if (d == 1)
      {
         return getEndPoint();
      }

      Point2D point = null;
      double seekLength =  (d * totalLength);
      double tmpLength = -1;
      double lineLength = 0;
      LineSegment line = null;
      Iterator lineIter = iteratorOfLines();
      if (lineIter.hasNext())
      {
         line = (LineSegment) lineIter.next();
         tmpLength = lineLength = line.getLength();
      }
      while (lineIter.hasNext() && tmpLength < seekLength)
      {
         line = (LineSegment) lineIter.next();
         lineLength = line.getLength();
         tmpLength += lineLength;
      }

      d =  (seekLength - tmpLength + lineLength) / lineLength;

      if (d == 0)
      {
         point = line.getStartPoint();
      }
      else if (d == 1)
      {
         point = line.getEndPoint();
      }
      else
      {
         point = line.getPointAt (d);
      }

      return new Point2D.Double (point.getX() + getX(), point.getY() + getY());
   }


   /**
    * Get the closestLine attribute of the JPolyLine object
    *
    * @param p  No description provided
    * @return   The closestLine value
    */
   public LineSegment getClosestLine (Point p)
   {
      return getClosestLine (p.x, p.y);
   }


   /**
    * Get the closestLine attribute of the JPolyLine object
    *
    * @param x  No description provided
    * @param y  No description provided
    * @return   The closestLine value
    */
   public LineSegment getClosestLine (int x, int y)
   {
      if (sizeOfLines() == 0)
      {
         return null;
      }

      x -= getX();
      y -= getY();
      LineSegment closestLine = null;
      double minDist = Double.MAX_VALUE;
      Iterator lineIter = iteratorOfLines();
      while (lineIter.hasNext())
      {
         LineSegment tmpLine = (LineSegment) lineIter.next();
         double dist = tmpLine.getDistance (x, y);
         if (dist < minDist)
         {
            minDist = dist;
            closestLine = tmpLine;
         }
      }

      return closestLine;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param x  No description provided
    * @param y  No description provided
    * @return   No description provided
    */
   public boolean contains (int x, int y)
   {
      if (sizeOfLines() == 0)
      {
         return super.contains (x, y);
      }

      Dimension size = getSize();
      if ( (x >= 0) &&  (x < size.width) &&  (y >= 0) &&  (y < size.height))
      {
         Iterator iter = iteratorOfLines();
         Point loc = null;
         while (iter.hasNext())
         {
            LineSegment line = (LineSegment) iter.next();
            loc = line.getLocation (loc);
            if (line.contains (x - loc.x, y - loc.y))
            {
               return true;
            }
         }
      }

      return false;
   }


   /**
    * Get the distance attribute of the JPolyLine object
    *
    * @param x  No description provided
    * @param y  No description provided
    * @return   The distance value
    */
   public double getDistance (int x, int y)
   {
      if (sizeOfLines() == 0)
      {
         return super.getDistance (x, y);
      }

      LineSegment line = getClosestLine (x, y);
      return  (line != null ? line.getDistance (x - getX(), y - getY()) : Double.MAX_VALUE);
   }


   /**
    * @param x  No description provided
    * @param y  No description provided
    * @return   the position of the projection of (x, y) to the line as value between 0 and
    *      1
    */
   public double getPosition (int x, int y)
   {
      if (sizeOfLines() == 0)
      {
         return super.getPosition (x, y);
      }

      LineSegment closestLine = getClosestLine (x, y);

      double tmpLength = 0;
      double totalLength = getLength();
      LineSegment line = null;
      Iterator lineIter = iteratorOfLines();
      if (lineIter.hasNext())
      {
         line = (LineSegment) lineIter.next();
         tmpLength = line.getLength();
      }
      while (lineIter.hasNext() && line != closestLine)
      {
         line = (LineSegment) lineIter.next();
         tmpLength += line.getLength();
      }
      tmpLength -= line.getLength() *  (1 - line.getPosition (x - getX(), y - getY()));

      return  (tmpLength / totalLength);
   }


   /**
    * Get the direction attribute of the JPolyLine object
    *
    * @param d  No description provided
    * @param p  No description provided
    * @return   The direction value
    */
   public Point2D getDirection (double d, Point2D p)
   {
      LineSegment line = getLineAt (d);
      return line.getDirection (p);
   }


   /**
    * Get the normal attribute of the JPolyLine object
    *
    * @param d  No description provided
    * @param p  No description provided
    * @return   The normal value
    */
   public Point2D getNormal (double d, Point2D p)
   {
      LineSegment line = getLineAt (d);
      return line.getNormal (p);
   }


   /**
    * Get the angle attribute of the JPolyLine object
    *
    * @param d  No description provided
    * @return   The angle value
    */
   public double getAngle (double d)
   {
      LineSegment line = getLineAt (d);
      return line.getAngle();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private double totalLength = -1;


   /**
    * Get the length attribute of the JPolyLine object
    *
    * @return   The length value
    */
   public double getLength()
   {
      if (isCoeffsDirty())
      {
         computeCoeffs();
      }

      return totalLength;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void computeCoeffs()
   {
      super.computeCoeffs();
      if (sizeOfLines() == 0)
      {
         totalLength = getStartToEndDistance();
      }
      else
      {
         totalLength = 0;
         Iterator lineIter = iteratorOfLines();
         while (lineIter.hasNext())
         {
            LineSegment line = (LineSegment) lineIter.next();
            totalLength += line.getLength();
         }
      }
   }
   //================================================================================================

   /**
    * Sets the pointAt attribute of the JPolyLine object
    *
    * @param index  The new pointAt value
    * @param x      The new pointAt value
    * @param y      The new pointAt value
    * @return       No description provided
    */
   public boolean setPointAt (int index, int x, int y)
   {
      boolean changed = false;
      int size = sizeOfLines();
      LineSegment line;
      if (index >= 0 && size >= index)
      {
         Point location = getLocation();
         if (index == 0)
         {
            line = getFromLines (0);
            changed = line.setStartPoint (x - location.x, y - location.y);
         }
         else
         {
            line = getFromLines (index - 1);
            changed = line.setEndPoint (x - location.x, y - location.y);
         }
         return changed;
      }
      throw new ArrayIndexOutOfBoundsException (index);
   }


   /**
    * Get the fromPoints attribute of the JPolyLine object
    *
    * @param index  No description provided
    * @return       The fromPoints value
    */
   public Point getFromPoints (int index)
   {
      int size = sizeOfLines();
      if (size == 0 && index >= 0 && index <= 1)
      {
         if (index == 0)
         {
            return getStartPoint();
         }
         else
         {
            return getEndPoint();
         }
      }
      else if (index >= 0 && index < size)
      {
         Point point;
         if (index == 0)
         {
            point = getFromLines (0).getStartPoint();
         }
         else
         {
            point = getFromLines (index - 1).getEndPoint();
         }

         Point location = getLocation();
         point.x += location.x;
         point.y += location.y;

         return point;
      }
      else
      {
         throw new ArrayIndexOutOfBoundsException (index);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param start  No description provided
    * @return       No description provided
    */
   public ListIterator iteratorOfPoints (int start)
   {
      if (sizeOfLines() == 0)
      {
         return super.iteratorOfPoints (start);
      }
      return new PointIterator (iteratorOfLines (start == 0 ? 0 : start - 1), start);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfPoints()
   {
      return sizeOfBends();
   }

   //#######################################################################################################

   /**
    * Sets the startBend attribute of the JPolyLine object
    *
    * @param value  The new startBend value
    * @return       No description provided
    */
   public boolean setStartBend (JBend value)
   {
      boolean changed = super.setStartBend (value);
      if (sizeOfLines() > 0)
      {
         changed = changed | setBendAt (0, value);
      }
      return changed;
   }


   /**
    * Sets the endBend attribute of the JPolyLine object
    *
    * @param value  The new endBend value
    * @return       No description provided
    */
   public boolean setEndBend (JBend value)
   {
      boolean changed = super.setEndBend (value);
      int size = sizeOfLines();
      if (size > 0)
      {
         changed = changed | setBendAt (size, value);
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param bend  No description provided
    * @return      No description provided
    */
   public boolean hasInBends (JBend bend)
   {
      if (bend == null)
      {
         return false;
      }
      else if (lines == null)
      {
         return  (getStartBend() == bend || getEndBend() == bend);
      }

      return super.hasInBends (bend);
   }


   /**
    * Sets the bendAt attribute of the JPolyLine object
    *
    * @param index  The new bendAt value
    * @param b      The new bendAt value
    * @return       No description provided
    */
   public boolean setBendAt (int index, JBend b)
   {
      int size = sizeOfLines();
      LineSegment line;
      boolean changed = false;
      if (size >= index)
      {
         if (index == 0)
         {
            line = getFromLines (0);
            changed = line.setStartBend (b);
            if (getStartBend() != b)
            {
               changed = changed | setStartBend (b);
            }
         }
         else
         {
            line = getFromLines (index - 1);
            changed = line.setEndBend (b);
            if (size == index && getEndBend() != b)
            {
               changed = changed | setEndBend (b);
            }
            else if (size > index)
            {
               line = getFromLines (index);
               changed = changed | line.setStartBend (b);
            }
         }
         return changed;
      }
      throw new ArrayIndexOutOfBoundsException (index);
   }


   /**
    * Get the fromBends attribute of the JPolyLine object
    *
    * @param index  No description provided
    * @return       The fromBends value
    */
   public JBend getFromBends (int index)
   {
      int size = sizeOfBends();
      if (index < 0 || index >= size)
      {
         throw new ArrayIndexOutOfBoundsException (index);
      }

      JBend bend;
      if (index == 0)
      {
         bend = getStartBend();
      }
      else if (index == size - 1)
      {
         bend = getEndBend();
      }
      else
      {
         bend = getFromLines (index - 1).getEndBend();
      }

      return bend;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfBends()
   {
      int lineSize = sizeOfLines();
      return  (lineSize == 0 ? 2 : lineSize + 1);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param start  No description provided
    * @return       No description provided
    */
   public ListIterator iteratorOfBends (int start)
   {
      if (sizeOfLines() == 0)
      {
         return super.iteratorOfBends (start);
      }
      return new BendIterator (iteratorOfLines (start == 0 ? 0 : start - 1), start);
   }


   /**
    * insert the bend between the 2 closest bends
    *
    * @param bend  No description provided
    * @return      No description provided
    */
   public boolean insertInBends (JBend bend)
   {
      int index = -1;
      if (sizeOfBends() == 2)
      {
         index = 1;
      }
      else
      {
         Point point = bend.getPoint();

         if (bend.getParent() != null && getParent() != null)
         {
            point = SwingUtilities.convertPoint (bend.getParent(), point, getParent());
         }

         double pos = getPosition (point);
         if (pos < 0)
         {
            index = 1;
         }
         else if (pos > 1)
         {
            index = sizeOfBends() - 1;
         }
         else
         {
            LineSegment segment = getClosestLine (point);

            index = getIndexFromBends (segment.getEndBend());
         }
      }

      if (index > -1)
      {
         return insertInBends (index, bend);
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param index  No description provided
    * @param bend   No description provided
    * @return       No description provided
    */
   public boolean insertInBends (int index, JBend bend)
   {
      int size = sizeOfBends();
      int lineSize = sizeOfLines();

      if (index < 0 || index > size)
      {
         throw new ArrayIndexOutOfBoundsException (index);
      }

      if (bend != null && !hasInBends (bend))
      {
         LineSegment newLine = new LineSegment();
         if (index < lineSize)
         {
            insertInLines (index, newLine);

            newLine.setStartBend (bend);
            newLine.setEndBend (getFromLines (index + 1).getStartBend());
            if (index == 0)
            {
               setStartBend (bend);
            }
            else
            {
               getFromLines (index - 1).setEndBend (bend);
            }
         }
         else
         {
            insertInLines (index - 1, newLine);

            newLine.setEndBend (bend);

            if (index == size)
            {
               newLine.setStartBend (getFromLines (index - 2).getEndBend());
               setEndBend (bend);
            }
            else
            {
               newLine.setStartBend (getFromLines (index).getStartBend());
               getFromLines (index).setStartBend (bend);
            }
         }

         revalidate();
         return true;
      }
      return false;
   }


   /**
    * @param bend  No description provided
    * @return      the index of bend or -1 if not in this line
    */
   public int getIndexFromBends (JBend bend)
   {
      if (sizeOfLines() == 0)
      {
         if (bend == getStartBend())
         {
            return 0;
         }
         else if (bend == getEndBend())
         {
            return 1;
         }
         else
         {
            return -1;
         }
      }

      return super.getIndexFromBends (bend);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param bend  No description provided
    * @return      No description provided
    */
   public boolean removeFromBends (JBend bend)
   {
      int index = getIndexFromBends (bend);

      if (index > -1)
      {
         return removeFromBends (index);
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param index  No description provided
    * @return       No description provided
    */
   public boolean removeFromBends (int index)
   {
      int size = sizeOfLines();
      if (index < 0 || index > size)
      {
         throw new ArrayIndexOutOfBoundsException (index);
      }

      LineSegment incomingLine =  (index > 0) ? getFromLines (index - 1) : null;
      LineSegment outgoingLine =  (index < size) ? getFromLines (index) : null;
      JBend removeBend = null;
      boolean changed = false;

      if (outgoingLine != null)
      {
         JBend bend = outgoingLine.getEndBend();
         removeBend = outgoingLine.getStartBend();

         changed = removeFromLines (outgoingLine); //delete outgoing line
         outgoingLine.removeYou();

         if (incomingLine != null)
         {
            //connect remaining lines again
            incomingLine.setEndBend (bend);
         }
         else
         { //bend is first bend now

            setStartBend (bend);
         }
      }
      else if (incomingLine != null)
      {
         JBend bend = incomingLine.getStartBend();
         removeBend = incomingLine.getEndBend();

         changed = removeFromLines (incomingLine); //delete incoming line
         incomingLine.removeYou();

         setEndBend (bend);
      }
      changed = changed |  (removeBend != null && removeBend.removeFromLines (this));
      revalidate();
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromBends()
   {
      removeAllFromLines();
      setStartBend (null);
      setEndBend (null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      removeAllFromBends();
   }


   /**
    * Sets the foreground attribute of the JPolyLine object
    *
    * @param c  The new foreground value
    */
   public void setForeground (Color c)
   {
      super.setForeground (c);
      Iterator lineIter = iteratorOfLines();
      while (lineIter.hasNext())
      {
         LineSegment line = (LineSegment) lineIter.next();

         line.setForeground (c);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void revalidate()
   {
      invalidateCoeffs();
      super.revalidate();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    */
   public void paintComponent (Graphics g)
   {
      /*
       *  Overwrite the paintComponent() of JLine.
       *  The real drawing is done by the children
       *  of the object.
       */
   }


   /**
    * Sets the UI of the object. The given UI will be also delegated to the children of the
    * object.
    *
    * @param ui  The UI to use. Must be a subclass of LineUI.
    */
   public void setUI (ComponentUI ui)
   {
      super.setUI (ui);

      Iterator lineIter = iteratorOfLines();
      while (lineIter.hasNext())
      {
         LineSegment line = (LineSegment) lineIter.next();

         line.setUI (ui);
      }
   }


   /**
    * Get the preferredBounds attribute of the JPolyLine object
    *
    * @return   The preferredBounds value
    */
   public Rectangle getPreferredBounds()
   {
      Rectangle result = super.getPreferredBounds();
      Rectangle tmpRect = null;
      int x = getX();
      int y = getY();
      Component[] children = getComponents();

      for (int i = 0; i < children.length; i++)
      {
         Component comp = children[i];

         if (!comp.isVisible())
         {
            continue;
         }

         if (comp instanceof JLine)
         {
            tmpRect =  ((JLine) comp).getPreferredBounds();
         }
         else
         {
            tmpRect = comp.getBounds (tmpRect);
         }

         tmpRect.translate (x, y);

         result.add (tmpRect);
      }

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean adjustingBounds = false;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean adjustBounds()
   {
      adjustingBounds = true;
      boolean changed = false;

      try
      {
         Point oldLoc = getLocation();
         changed = super.adjustBounds();
         Point newLoc = getLocation();

         if (!oldLoc.equals (newLoc))
         {
            Iterator lineIter = iteratorOfLines();
            while (lineIter.hasNext())
            {
               LineSegment line = (LineSegment) lineIter.next();
               if (line.getStartBend() == null || line.getEndBend() == null)
               {
                  continue;
               }

               line.invalidateStartTransform();
               line.invalidateEndTransform();
               line.adjustBounds();
            }
         }
      }
      finally
      {
         adjustingBounds = false;
      }
      return changed;
   }


   /**
    * Sets the adjustingBounds attribute of the JPolyLine object
    *
    * @param adjust  The new adjustingBounds value
    */
   void setAdjustingBounds (boolean adjust)
   {
      adjustingBounds = adjust;
   }


   /**
    * Get the adjustingBounds attribute of the JPolyLine object
    *
    * @return   The adjustingBounds value
    */
   public boolean isAdjustingBounds()
   {
      return adjustingBounds;
   }
   //----------------------------------- inner classes -------------------------------------

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.34.2.2 $
    */
   private static class PointIterator extends BendIterator
   {
      /**
       * Constructor for class PointIterator
       *
       * @param lineIter  No description provided
       */
      public PointIterator (ListIterator lineIter)
      {
         super (lineIter);
      }


      /**
       * Constructor for class PointIterator
       *
       * @param lineIter  No description provided
       * @param start     No description provided
       */
      public PointIterator (ListIterator lineIter, int start)
      {
         super (lineIter, start);
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      public Object next()
      {
         JBend bend = (JBend) super.next();
         if (bend == null)
         {
            throw new NoSuchElementException();
         }
         return bend.getPoint();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      public Object previous()
      {
         JBend bend = (JBend) super.previous();
         if (bend == null)
         {
            throw new NoSuchElementException();
         }
         return bend.getPoint();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.34.2.2 $
    */
   private static class BendIterator implements ListIterator
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private ListIterator lineIterator = null;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private LineSegment line = null;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private boolean forward = true;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private int nextIndex;


      /**
       * Constructor for class BendIterator
       *
       * @param lineIter  No description provided
       */
      public BendIterator (ListIterator lineIter)
      {
         this (lineIter, 0);
      }


      /**
       * Constructor for class BendIterator
       *
       * @param lineIter  No description provided
       * @param start     No description provided
       */
      public BendIterator (ListIterator lineIter, int start)
      {
         this.lineIterator = lineIter;
         nextIndex = start;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      public Object next()
      {
         JBend bend = null;

         if (line == null || nextIndex > 1 ||  (!forward && lineIterator.hasNext()))
         {
            line = (LineSegment) lineIterator.next();
         }

         if (nextIndex == 0)
         {
            bend = line.getStartBend();
         }
         else
         {
            bend = line.getEndBend();
         }

         nextIndex++;
         forward = true;

         return bend;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      public boolean hasNext()
      {
         return  ( (line != null &&  (nextIndex <= 1 || !forward)) || lineIterator.hasNext());
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      public int nextIndex()
      {
         return nextIndex;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      public Object previous()
      {
         JBend bend = null;
         if (nextIndex < 1)
         {
            throw new NoSuchElementException();
         }
         if (nextIndex == 1)
         {
            if (line == null)
            {
               line = (LineSegment) lineIterator.next();
            }
            bend = line.getStartBend();
         }
         else
         {
            line = (LineSegment) lineIterator.previous();

            bend = line.getEndBend();
         }
         nextIndex--;
         forward = false;
         return bend;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      public boolean hasPrevious()
      {
         return nextIndex > 0;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      public int previousIndex()
      {
         return nextIndex - 1;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public void remove()
      {
         throw new UnsupportedOperationException();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param o  No description provided
       */
      public void add (Object o)
      {
         throw new UnsupportedOperationException();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param o  No description provided
       */
      public void set (Object o)
      {
         throw new UnsupportedOperationException();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.34.2.2 $
    */
   public static class LineSegment extends JBendLine
   {
      /**
       * Constructor for class LineSegment
       */
      LineSegment()
      {
         this (null, null);
      }


      /**
       * Constructor for class LineSegment
       *
       * @param start  No description provided
       * @param end    No description provided
       */
      LineSegment (JBend start, JBend end)
      {
         super (start, end);
      }


      /**
       * Get the stroke attribute of the LineSegment object
       *
       * @return   The stroke value
       */
      public Stroke getStroke()
      {
         Stroke result = super.getStroke();
         if (result == null)
         {
            JPolyLine pl = this.polyLine;
            if (pl != null)
            {
               result = pl.getStroke();
            }
         }
         return result;
      }


      /*
       *  public boolean setStartBend (JBend bend)
       *  {
       *  boolean result = super.setStartBend(bend);
       *  if (result && polyLine != null)
       *  {
       *  int index = polyLine.getIndexFromLines(this);
       *  if (bend == null)
       *  {
       *  polyLine.removeFromBends(index);
       *  }
       *  else
       *  {
       *  polyLine.setBendAt(index, bend);
       *  }
       *  }
       *  return result;
       *  }
       *  public boolean setEndBend (JBend bend)
       *  {
       *  boolean result = super.setEndBend(bend);
       *  if (result && polyLine != null)
       *  {
       *  int index = polyLine.getIndexFromLines(this) + 1;
       *  if (bend == null)
       *  {
       *  polyLine.removeFromBends(index);
       *  }
       *  else
       *  {
       *  polyLine.setBendAt(index, bend);
       *  }
       *  }
       *  return result;
       *  }
       */
      /**
       * <pre>
       *              N                      0..1
       * LineSegment ----------------------------- JPolyLine
       *              lines              polyLine
       * </pre>
       */
      private JPolyLine polyLine;


      /**
       * Sets the polyLine attribute of the LineSegment object
       *
       * @param value  The new polyLine value
       * @return       No description provided
       */
      boolean setPolyLine (JPolyLine value)
      {
         if (this.polyLine != value)
         {
            if (this.polyLine != null)
            {
               JPolyLine oldValue = this.polyLine;
               this.polyLine = null;
               oldValue.removeFromLines (this);
            }

            this.polyLine = value;

            if (value != null && !value.hasInLines (this))
            {
               this.polyLine = null;
               throw new RuntimeException ("Cannot add Line to PolyLine in LineSegment.setPolyLine.\n" +
                  "Please use JPolyLine.insertInLines instead");
            }

            return true;
         }

         return false;
      }


      /**
       * Get the polyLine attribute of the LineSegment object
       *
       * @return   The polyLine value
       */
      public JPolyLine getPolyLine()
      {
         return this.polyLine;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      boolean adjustBounds()
      {
         boolean changed = false;

         if (! (getStartBend() == null || getEndBend() == null))
         {
            Rectangle bounds = LineSegment.this.getPreferredBounds();

            if (this.polyLine != null && !this.polyLine.isAdjustingBounds())
            {
               this.polyLine.adjustBounds();

               if (! (bounds.x < 0 || bounds.y < 0))
               {
                  changed = super.adjustBounds();
               }
            }
            else
            {
               changed = super.adjustBounds();
            }
         }
         return changed;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      void invalidateCoeffs()
      {
         super.invalidateCoeffs();
         if (this.polyLine != null)
         {
            this.polyLine.invalidateCoeffs();
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve
       * documentation.
       */
      public void removeYou()
      {
         setPolyLine (null);
         super.removeYou();
      }
   }
}

/*
 * $Log: JPolyLine.java,v $
 * Revision 1.34.2.2  2005/08/23 08:24:52  lowende
 * Removed compile warnings.
 *
 */
