/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;

import javax.swing.*;


/**
 * A JGrab is a JBend that is attached to a JComponent. It has a GrabManager that acts as a
 * association object to the JComponent and manages the updating of the grab position when
 * the Component changes or other grabs are added<p>
 *
 * <h2> Associations </h2> <pre>
 *        N                      0..1
 * JGrab ----------------------------- GrabManager
 *        grabs               manager
 * </pre>
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.20.2.1 $
 * @see       de.uni_paderborn.fujaba.fsa.swing.JBend
 * @see       de.uni_paderborn.fujaba.fsa.swing.JBendLine
 * @see       de.uni_paderborn.fujaba.fsa.swing.GrabManager
 */
public class JGrab extends JBend
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @deprecated
    */
   public final static int TOP = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @deprecated
    */
   public final static int RIGHT = 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @deprecated
    */
   public final static int BOTTOM = 2;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @deprecated
    */
   public final static int LEFT = 3;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @deprecated
    */
   private Direction orientation = Direction.LEFT;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean autoOrientation = true;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private double alignment = 0.5;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean autoAlignment = true;


   /**
    * Constructor for class JGrab
    */
   public JGrab()
   {
      super();
   }


   /**
    * Constructor for class JGrab
    *
    * @param manager  No description provided
    */
   public JGrab (GrabManager manager)
   {
      this();

      setManager (manager);
   }


   /**
    * Constructor for class JGrab
    *
    * @param target  No description provided
    */
   public JGrab (JComponent target)
   {
      this();
      setTarget (target);
   }


   /**
    * Set the orientation of the Grab at its target component. Calls setLayoutOrientation to
    * set the orientation and then calls the GrabManager to revalidate the Grabs.<p>
    *
    * Resets the autoOrientation property if set.
    *
    * @param orientation  one of <pre>JGrab.TOP</pre>, <pre>JGrab.BOTTOM</pre>, <pre>JGrab.LEFT</pre>
    *      or >pre>JGrab.RIGHT</pre>
    * @see                #setLayoutOrientation
    * @see                #getOrientation
    * @see                #setAutoOrientation
    * @see                #setAlignment
    * @see                #TOP
    * @see                #BOTTOM
    * @see                #LEFT
    * @see                #RIGHT
    */
   public void setOrientation (Direction orientation)
   {
      boolean changed = false;

      // orientation should be modified
      // (which means, that someone might have dragged the grab)
      // so autoOrientation must be disabled, even if orientation doesn't change
      if (isAutoOrientation())
      {
         autoOrientation = false;
         changed = true;
      }

      if (orientation != this.orientation)
      {
         setLayoutOrientation (orientation);
         changed = true;
      }

      if (changed)
      {
         GrabManager manager = getManager();
         if (manager != null)
         {
            manager.revalidate();
         }
      }
   }


   /**
    * @return   the Orientation of the Grab
    * @see      #setOrientation
    * @see      #setLayoutOrientation
    * @see      #setAutoOrientation
    * @see      #isAutoOrientation
    * @see      #TOP
    * @see      #BOTTOM
    * @see      #LEFT
    * @see      #RIGHT
    */
   public Direction getOrientation()
   {
      return orientation;
   }


   /**
    * Set the Orientation without notifying the GrabManager or resetting the autoOrientation
    * property
    *
    * @param orientation
    * @see                #setOrientation
    */
   public void setLayoutOrientation (Direction orientation)
   {
      if (this.orientation != orientation)
      {
         Direction old = this.orientation;
         this.orientation = orientation;
         revalidate();
         resetPoint();
         firePropertyChange ("orientation", old, orientation);
      }
   }


   /**
    * set the autoOrientation property<p>
    *
    * if set to true the orientation is determined by the GrabLayouter otherwise the value
    * of <pre>getOrientation()</pre> is used
    *
    * @param auto  The new autoOrientation value
    * @see         #setOrientation
    * @see         #setLayoutOrientation
    * @see         #getOrientation
    * @see         de.uni_paderborn.fujaba.fsa.swing.GrabLayouter
    */
   public void setAutoOrientation (boolean auto)
   {
      if (auto != autoOrientation)
      {
         autoOrientation = auto;
         if (auto && manager != null)
         {
            manager.revalidate();
         }
      }
   }


   /**
    * get the autoOrientation property
    *
    * @return   true if the orientation is determined by the GrabLayouter, false otherwise
    * @see      #setAutoOrientation
    * @see      #setOrientation
    * @see      #setLayoutOrientation
    * @see      #getOrientation
    * @see      de.uni_paderborn.fujaba.fsa.swing.GrabLayouter
    */
   public boolean isAutoOrientation()
   {
      return autoOrientation;
   }


   /**
    * Set the alignment of the Grab at its target component. <p>
    *
    * Calls setLayoutAlignment to set the orientation and then calls the GrabManager to revalidate
    * the Grabs. Resets the autoAlignment property if set.
    *
    * @param align  The new alignment value
    * @see          #setLayoutAlignment
    * @see          #getAlignment
    * @see          #setAutoAlignment
    * @see          #setOrientation
    */
   public void setAlignment (double align)
   {
      if (align > 1)
      {
         align = 1;
      }

      if (align < 0)
      {
         align = 0;
      }

      boolean changed = false;

      // alignment should be modified
      // (which means, that someone might have dragged the grab)
      // so autoAlignment must be disabled, even if alignment doesn't change
      if (isAutoAlignment())
      {
         autoAlignment = false;
         changed = true;
      }

      if (this.alignment != align)
      {
         setLayoutAlignment (align);
         changed = true;
      }

      if (changed)
      {
         GrabManager manager = getManager();
         if (manager != null)
         {
            manager.revalidate();
         }
      }
   }


   /**
    * Set the Alignment without notifying the GrabManager or resetting the autoAlignment property
    *
    * @param align  The new layoutAlignment value
    * @see          #setOrientation
    */
   public void setLayoutAlignment (double align)
   {
      if (this.alignment != align)
      {
         double old = this.alignment;
         this.alignment = align;
         firePropertyChange ("alignment", old, align);
      }
   }


   /**
    * @return   the alignment of the Grab
    * @see      #setAlignment
    * @see      #setLayoutAlignment
    * @see      #setAutoAlignment
    * @see      #isAutoAlignment
    */
   public double getAlignment()
   {
      return this.alignment;
   }


   /**
    * set the autoOrientation property <p>
    *
    * if set to true the orientation is determined by the GrabLayouter otherwise the value
    * of <pre>getOrientation()</pre> is used
    *
    * @param auto  The new autoAlignment value
    * @see         #setOrientation
    * @see         #setLayoutOrientation
    * @see         #getOrientation
    * @see         de.uni_paderborn.fujaba.fsa.swing.GrabLayouter
    */
   public void setAutoAlignment (boolean auto)
   {
      if (auto != autoAlignment)
      {
         autoAlignment = auto;
         if (auto && manager != null)
         {
            manager.revalidate();
         }
      }
   }


   /**
    * get the autoAlignment property
    *
    * @return   true if the alignment is determined by the GrabLayouter, false otherwise
    * @see      #setAutoAlignment
    * @see      #setAlignment
    * @see      #setLayoutAlignment
    * @see      #getAlignment
    * @see      de.uni_paderborn.fujaba.fsa.swing.GrabLayouter
    */
   public boolean isAutoAlignment()
   {
      return autoAlignment;
   }


   /**
    * Get the touchOffset attribute of the JGrab object
    *
    * @return   The touchOffset value
    */
   public int getTouchOffset()
   {
      if (this.ui != null && this.ui instanceof GrabUI)
      {
         return  ((GrabUI) this.ui).getTouchOffset (this);
      }
      return 0;
   }


   /**
    * Get the touchLength attribute of the JGrab object
    *
    * @return   The touchLength value
    */
   public int getTouchLength()
   {
      if (this.ui != null && this.ui instanceof GrabUI)
      {
         return  ((GrabUI) this.ui).getTouchLength (this);
      }
      else
      {
         if (getOrientation() == Direction.LEFT || getOrientation() == Direction.RIGHT)
         {
            return getHeight();
         }
         else
         {
            return getWidth();
         }
      }
   }


   /**
    * Get the alignmentOffset attribute of the JGrab object
    *
    * @return   The alignmentOffset value
    */
   public int getAlignmentOffset()
   {
      if (this.ui != null && this.ui instanceof GrabUI)
      {
         return  ((GrabUI) this.ui).getAlignmentOffset (this);
      }
      return getTouchOffset() + getTouchLength() / 2;
   }


   /**
    * Get the padOffset attribute of the JGrab object
    *
    * @return   The padOffset value
    */
   public int getPadOffset()
   {
      int offset = 0;
      if (this.ui != null && this.ui instanceof GrabUI)
      {
         offset =  ((GrabUI) this.ui).getPadOffset (this);
      }
      return offset;
   }


   /**
    * Does nothing. <p>
    *
    * Positioning is done by <pre>setOrientation</pre> and <pre>setAlignment</pre>
    *
    * @param x
    * @param y
    * @see      #setOrientation
    * @see      #setAlignment
    */
   public void setPoint (int x, int y) { }


   /**
    * Default is the center of that side of the bounds that is defined by <pre>getOrientation()</pre>
    *
    * @return   No description provided
    * @see      #getOrientation
    */
   protected Point calculatePoint()
   {
      if (this.ui != null && this.ui instanceof BendUI)
      {
         return  ((BendUI) this.ui).calculatePoint (this);
      }
      else
      {
         Rectangle bounds = getBounds();
         Point point = new Point();
         if (getOrientation() == Direction.LEFT)
         {
            point.x = bounds.x;
            point.y = bounds.y + bounds.height / 2;
         }
         else if (getOrientation() == Direction.RIGHT)
         {
            point.x = bounds.x + bounds.width - 1;
            point.y = bounds.y + bounds.height / 2;
         }
         else if (getOrientation() == Direction.BOTTOM)
         {
            point.x = bounds.x + bounds.width / 2;
            point.y = bounds.y + bounds.height - 1;
         }
         else
         {
            point.x = bounds.x + bounds.width / 2;
            point.y = bounds.y;
         }
         return point;
      }
   }


   /**
    * Sets the bounds attribute of the JGrab object
    *
    * @param x       The new bounds value
    * @param y       The new bounds value
    * @param width   The new bounds value
    * @param height  The new bounds value
    */
   public void setBounds (int x, int y, int width, int height)
   {
      Rectangle bounds = getBounds();
      super.setBounds (x, y, width, height);
      if (width != bounds.width || height != bounds.height)
      {
         GrabManager manager = getManager();
         if (manager != null)
         {
            manager.revalidate();
         }
      }
   }
   // ------------------------------------------------------------------------

   /**
    * <pre>
    *        N                      0..1
    * JGrab ----------------------------- GrabManager
    *        grabs               manager
    * </pre> <p>
    *
    * The GrabManager is an association object to the target JComponent.<br>
    * It manages events and calls the GrabLayouter to layout the Grab positions when necessary
    */
   private GrabManager manager = null;


   /**
    * Sets the manager attribute of the JGrab object
    *
    * @param value  The new manager value
    * @return       No description provided
    */
   public boolean setManager (GrabManager value)
   {
      boolean changed = false;

      if (this.manager != value)
      {
         GrabManager oldValue = this.manager;
         if (this.manager != null)
         {
            this.manager = null;
            oldValue.removeFromGrabs (this);
         }
         this.manager = value;
         if (value != null)
         {
            value.addToGrabs (this);
         }
         firePropertyChange ("manager", oldValue, value);
         changed = true;
      }
      return changed;
   }


   /**
    * Get the manager attribute of the JGrab object
    *
    * @return   The manager value
    */
   public GrabManager getManager()
   {
      return manager;
   }

   // ------------------------------------------------------------------------

   /**
    * Only a Comfort Operation.<p>
    *
    * The association to the target JComponent is implemented by the GrabManager.<br>
    * So if the target does not have a GrabManager yet, a new one is added. Then this grab
    * is added to the GrabManager.
    *
    * @param target  The JComponent to connect to
    * @see           #getTarget
    * @see           #setManager
    * @see           de.uni_paderborn.fujaba.fsa.swing.GrabManager
    */
   public void setTarget (JComponent target)
   {
      GrabManager manager = getManager();
      if (manager != null && manager.getTarget() == target)
      {
         return;
      }

      if (target == null)
      {
         setManager (null);
      }
      else
      {
         manager = (GrabManager) target.getClientProperty (GrabManager.TARGET_PROPERTY);
         if (manager == null)
         {
            manager = new GrabManager (target);
         }

         setManager (manager);
      }
   }


   /**
    * @return   null if the GrabManager is null, the target of the GrabManager otherwise
    * @see      #setTarget
    * @see      #getManager
    * @see      de.uni_paderborn.fujaba.fsa.swing.GrabManager#getTarget
    */
   public JComponent getTarget()
   {
      GrabManager manager = getManager();

      return  (manager == null) ? null : manager.getTarget();
   }
}

/*
 * $Log: JGrab.java,v $
 * Revision 1.20.2.1  2005/05/25 11:15:49  fklar
 * modified 'setOrientation(Direction)' and 'setAlignement(double)', so autoOrientation and autoAlignement will be deactivated by a call to these methods
 *
 */
