/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.event.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.*;

import javax.swing.*;

import de.uni_paderborn.fujaba.basic.AbstractListIterator;
import de.uni_paderborn.fujaba.fsa.listener.AncestorNotifier;


/**
 * A JLine that can have a JBend at each end.<p>
 *
 * Startpoint and Endpoint are defined by the two Bends if they are attached. Otherwise this
 * does the same as the "classic" JLine.<p>
 *
 * <h2>Associations</h2> <pre>
 *            N                      0..1
 * JBendLine ----------------------------- JBend
 *            outgoingLines     startBend
 *
 *            N                      0..1
 * JBendLine ----------------------------- JBend
 *            incomingLines       endBend
 * </pre>
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.19 $
 * @see       de.uni_paderborn.fujaba.fsa.swing.JLine
 * @see       de.uni_paderborn.fujaba.fsa.swing.JBend
 * @see       de.uni_paderborn.fujaba.fsa.swing.JGrab
 */
public class JBendLine extends JLine implements PropertyChangeListener
{
   /**
    * Constructor for class JBendLine
    */
   public JBendLine()
   {
      this (new Point (0, 0), new Point (0, 0));
   }


   /**
    * @param start  start of line
    * @param end    end of line
    */
   public JBendLine (Point start, Point end)
   {
      this (new JBend (start), new JBend (end));
   }


   /**
    * @param start  the start Bend
    * @param end    the end Bend
    */
   public JBendLine (JBend start, JBend end)
   {
      super();
      setOpaque (false);
      setStartBend (start);
      setEndBend (end);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private AncestorManager ancestorListener = null;


   /**
    * Get the ancestorListener attribute of the JBendLine object
    *
    * @return   The ancestorListener value
    */
   public AncestorManager getAncestorListener()
   {
      if (ancestorListener == null)
      {
         ancestorListener = new AncestorManager();
         this.addHierarchyListener (ancestorListener);
         this.addHierarchyBoundsListener (ancestorListener);
      }
      return ancestorListener;
   }

   // -------------------------------------------------------------------------------

   /**
    * <pre>
    *            N                      0..1
    * JBendLine ----------------------------- JBend
    *            outgoingLines     startBend
    * </pre>
    */
   private JBend startBend;


   /**
    * Set the start bend of the Line.<p>
    *
    * If the Bend has no Parent yet, it is assigned to the parent of the Line. Furthermore
    * if the parent of the Bend is the same as the parent of the Line, it is changed whenever
    * the parent of the Line changes
    *
    * @param value  the new Bend
    * @return       true if the bend has changed, false otherwise
    * @see          #getStartBend
    * @see          #setEndBend
    * @see          #setStartPoint
    */
   public boolean setStartBend (JBend value)
   {
      if (this.startBend != value)
      {
         if (this.startBend != null)
         {
            JBend oldValue = this.startBend;
            this.startBend = null;
            unregisterStartBend (oldValue);
         }
         this.startBend = value;
         if (value != null)
         {
            registerStartBend (value);
            getAncestorListener().invalidate();
            invalidateStartTransform();

            setStartPointFromBend (value.getPoint());
         }

         return true;
      }

      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param bend  No description provided
    */
   protected void registerStartBend (JBend bend)
   {
      bend.addToOutgoingLines (this);

      bend.addPropertyChangeListener ("point", this);

      bend.addHierarchyListener (getAncestorListener());
      bend.addHierarchyBoundsListener (getAncestorListener());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param bend  No description provided
    */
   protected void unregisterStartBend (JBend bend)
   {
      bend.removePropertyChangeListener ("point", this);

      bend.removeHierarchyListener (getAncestorListener());
      bend.removeHierarchyBoundsListener (getAncestorListener());
      bend.removeFromOutgoingLines (this);
   }


   /**
    * Get the startBend attribute of the JBendLine object
    *
    * @return   The startBend value
    */
   public JBend getStartBend()
   {
      return this.startBend;
   }

   // ---------------------------------------------------------------------------------

   /**
    * <pre>
    *            N                   0..1
    * JBendLine -------------------------- JBend
    *            incomingLines    endBend
    * </pre>
    */
   private JBend endBend;


   /**
    * Set the end bend of the Line.<p>
    *
    * If the Bend has no Parent yet, it is assigned to the parent of the Line. Furthermore
    * if the parent of the Bend is the same as the parent of the Line, it is changed whenever
    * the parent of the Line changes
    *
    * @param value  the new Bend
    * @return       true if the bend has changed, false otherwise
    * @see          #getEndBend
    * @see          #setStartBend
    * @see          #setEndPoint
    */
   public boolean setEndBend (JBend value)
   {
      if (this.endBend != value)
      {
         if (this.endBend != null)
         {
            JBend oldValue = this.endBend;
            this.endBend = null;
            unregisterEndBend (oldValue);
         }
         this.endBend = value;
         if (value != null)
         {
            registerEndBend (value);

            getAncestorListener().invalidate();
            invalidateEndTransform();

            setEndPointFromBend (this.endBend.getPoint());
         }

         return true;
      }

      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param bend  No description provided
    */
   protected void registerEndBend (JBend bend)
   {
      bend.addToIncomingLines (this);

      bend.addPropertyChangeListener ("point", this);

      bend.addHierarchyListener (getAncestorListener());
      bend.addHierarchyBoundsListener (getAncestorListener());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param bend  No description provided
    */
   protected void unregisterEndBend (JBend bend)
   {
      bend.removePropertyChangeListener ("point", this);

      bend.removeHierarchyListener (getAncestorListener());
      bend.removeHierarchyBoundsListener (getAncestorListener());
      bend.removeFromIncomingLines (this);
   }


   /**
    * Get the endBend attribute of the JBendLine object
    *
    * @return   The endBend value
    */
   public JBend getEndBend()
   {
      return this.endBend;
   }


   /**
    * Sets the bendAt attribute of the JBendLine object
    *
    * @param index  The new bendAt value
    * @param bend   The new bendAt value
    * @return       No description provided
    */
   public boolean setBendAt (int index, JBend bend)
   {
      boolean changed = false;
      if (index == 0)
      {
         changed = setStartBend (bend);
      }
      else if (index == 1)
      {
         changed = setEndBend (bend);
      }
      else
      {
         throw new ArrayIndexOutOfBoundsException (index);
      }

      return changed;
   }


   /**
    * Get the fromBends attribute of the JBendLine object
    *
    * @param index  No description provided
    * @return       The fromBends value
    */
   public JBend getFromBends (int index)
   {
      if (index == 0)
      {
         return getStartBend();
      }
      else if (index == 1)
      {
         return getEndBend();
      }
      else
      {
         throw new ArrayIndexOutOfBoundsException (index);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final ListIterator iteratorOfBends()
   {
      return iteratorOfBends (0);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param bend  No description provided
    * @return      No description provided
    */
   public final ListIterator iteratorOfBends (JBend bend)
   {
      return iteratorOfBends (getIndexFromBends (bend));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param start  No description provided
    * @return       No description provided
    */
   public ListIterator iteratorOfBends (int start)
   {
      return new BendIterator (start);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfBends()
   {
      return 2;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param bend  No description provided
    * @return      No description provided
    */
   public boolean hasInBends (JBend bend)
   {
      if (bend == null)
      {
         return false;
      }

      return  (getIndexFromBends (bend) > -1);
   }


   /**
    * @param bend  No description provided
    * @return      the index of bend or -1 if not in this line
    */
   public int getIndexFromBends (JBend bend)
   {
      int index = -1;
      Iterator bendIter = iteratorOfBends();
      while (bendIter.hasNext())
      {
         index++;
         if (bend == bendIter.next())
         {
            return index;
         }
      }
      return -1;
   }


   /**
    * Get the nextFromBends attribute of the JBendLine object
    *
    * @param bend  No description provided
    * @return      The nextFromBends value
    */
   public JBend getNextFromBends (JBend bend)
   {
      JBend result = null;
      if (bend != null)
      {
         int index = getIndexFromBends (bend);
         if (index > -1 && index < sizeOfBends() - 1)
         {
            result = getFromBends (index + 1);
         }
      }
      return result;
   }


   /**
    * Get the prevFromBends attribute of the JBendLine object
    *
    * @param bend  No description provided
    * @return      The prevFromBends value
    */
   public JBend getPrevFromBends (JBend bend)
   {
      JBend result = null;
      if (bend != null)
      {
         int index = getIndexFromBends (bend);
         if (index > 0)
         {
            result = getFromBends (index - 1);
         }
      }
      return result;
   }
   // ---------------------------------------------------------------------------------

   /**
    * Get the partner for the bend
    *
    * @param bend  No description provided
    * @return      startBend if bend is endBend, endBend if bend is startBend, null otherwise
    */
   public JBend getOtherBend (JBend bend)
   {
      JBend startBend = getStartBend();
      JBend endBend = getEndBend();
      if (bend == startBend)
      {
         return endBend;
      }

      if (bend == endBend)
      {
         return startBend;
      }

      return null;
   }


   /**
    * Set the value of startPoint.<p>
    *
    * Sets the startPoint value, the position of the startBend and calls the endPoints GrabManager
    * for revalidation.
    *
    * @param x  x-coordinate to assign to startPoint
    * @param y  y-coordinate to assign to startPoint
    * @return   No description provided
    * @see      de.uni_paderborn.fujaba.fsa.swing.JLine#getStartPoint()
    * @see      #setEndPoint
    */
   public boolean setStartPoint (int x, int y)
   {
      Point startPoint = getStartPoint();
      if (x != startPoint.x || y != startPoint.y)
      {
         super.setStartPoint (x, y);

         JBend bend = getStartBend();

         if (bend != null)
         {
            Point transform = getStartTransform();

            bend.setPoint (x - transform.x, y - startTransform.y);
         }
         startPointChanged();
         return true;
      }
      return false;
   }


   /**
    * Sets the startPointFromBend attribute of the JBendLine object
    *
    * @param p  The new startPointFromBend value
    */
   private final void setStartPointFromBend (Point p)
   {
      setStartPointFromBend (p.x, p.y);
   }


   /**
    * Sets the startPointFromBend attribute of the JBendLine object
    *
    * @param x  The new startPointFromBend value
    * @param y  The new startPointFromBend value
    */
   private void setStartPointFromBend (int x, int y)
   {
      Point transform = getStartTransform();

      setStartPoint (x + transform.x, y + transform.y);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void startPointChanged()
   {
      JBend bend = getEndBend();
      if (bend != null && bend instanceof JGrab)
      {
         JGrab grab = (JGrab) bend;
         GrabManager manager = grab.getManager();
         if (manager != null)
         {
            manager.revalidate();
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   Point startTransform = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void invalidateStartTransform()
   {
      startTransform = null;
   }


   /**
    * Get the startTransform attribute of the JBendLine object
    *
    * @return   The startTransform value
    */
   protected Point getStartTransform()
   {
      if (startTransform == null)
      {
         if (startBend == null)
         {
            startTransform = new Point (0, 0);
         }
         else
         {
            Container bendParent = startBend.getParent();
            Container lineParent = this.getParent();

            if (bendParent == null || lineParent == null)
            {
               startTransform = new Point (0, 0);
            }
            else
            {
               startTransform = SwingUtilities.convertPoint (bendParent, 0, 0, lineParent);
            }
         }
      }
      return startTransform;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void calculateStartPoint()
   {
      setStartPointFromBend (getStartBend().getPoint());
   }


   /**
    * Get the startPoint attribute of the JBendLine object
    *
    * @param p  No description provided
    * @return   The startPoint value
    */
   public Point getStartPoint (Point p)
   {
      if (startTransform == null && getStartBend() != null)
      {
         calculateStartPoint();
      }

      return super.getStartPoint (p);
   }


   /**
    * Set the value of endPoint.<p>
    *
    * Sets the endPoint value, the position of the endBend and calls the endPoints GrabManager
    * for revalidation.
    *
    * @param x  x-coordinate to assign to endPoint
    * @param y  y-coordinate to assign to endPoint
    * @return   No description provided
    * @see      de.uni_paderborn.fujaba.fsa.swing.JLine#getEndPoint()
    * @see      #setStartPoint
    */
   public boolean setEndPoint (int x, int y)
   {
      Point endPoint = getEndPoint();
      if (x != endPoint.x || y != endPoint.y)
      {
         super.setEndPoint (x, y);

         JBend bend = getEndBend();

         if (bend != null)
         {
            Point transform = getEndTransform();

            bend.setPoint (x - transform.x, y - endTransform.y);
         }
         endPointChanged();
         return true;
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void endPointChanged()
   {
      JBend bend = getStartBend();
      if (bend != null && bend instanceof JGrab)
      {
         JGrab grab = (JGrab) bend;
         GrabManager manager = grab.getManager();
         if (manager != null)
         {
            manager.revalidate();
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   Point endTransform = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void invalidateEndTransform()
   {
      endTransform = null;
   }


   /**
    * Get the endTransform attribute of the JBendLine object
    *
    * @return   The endTransform value
    */
   protected Point getEndTransform()
   {
      if (endTransform == null)
      {
         if (endBend == null)
         {
            endTransform = new Point (0, 0);
         }
         else
         {
            Container bendParent = endBend.getParent();
            Container lineParent = this.getParent();

            if (bendParent == null || lineParent == null)
            {
               endTransform = new Point (0, 0);
            }
            else
            {
               endTransform = SwingUtilities.convertPoint (bendParent, 0, 0, lineParent);
            }
         }
      }

      return endTransform;
   }


   /**
    * Sets the endPointFromBend attribute of the JBendLine object
    *
    * @param p  The new endPointFromBend value
    */
   private final void setEndPointFromBend (Point p)
   {
      setEndPointFromBend (p.x, p.y);
   }


   /**
    * Sets the endPointFromBend attribute of the JBendLine object
    *
    * @param x  The new endPointFromBend value
    * @param y  The new endPointFromBend value
    */
   private void setEndPointFromBend (int x, int y)
   {
      Point transform = getEndTransform();
      setEndPoint (x + transform.x, y + transform.y);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void calculateEndPoint()
   {
      setEndPointFromBend (getEndBend().getPoint());
   }


   /**
    * Get the endPoint attribute of the JBendLine object
    *
    * @param p  No description provided
    * @return   The endPoint value
    */
   public Point getEndPoint (Point p)
   {
      if (endTransform == null && getEndBend() != null)
      {
         calculateEndPoint();
      }
      return super.getEndPoint (p);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param x  No description provided
    * @param y  No description provided
    * @return   No description provided
    */
   public boolean contains (int x, int y)
   {
      if (super.contains (x, y))
      {
         boolean result = true;
         Point transform;
         int tmpX;
         int
            tmpY;
         if (startBend != null)
         {
            transform = getStartTransform();
            tmpX = x + getX() - transform.x - startBend.getX();
            tmpY = y + getY() - transform.y - startBend.getY();

            result = !startBend.contains (tmpX, tmpY);
         }
         if (result && endBend != null)
         {
            transform = getEndTransform();
            tmpX = x + getX() - transform.x - endBend.getX();
            tmpY = y + getY() - transform.y - endBend.getY();

            result = !endBend.contains (tmpX, tmpY);
         }
         return result;
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      setStartBend (null);
      setEndBend (null);
      if (getClientProperty (AncestorNotifier.KEY) != null && getClientProperty (AncestorNotifier.KEY) instanceof AncestorNotifier)
      {
          ((AncestorNotifier) getClientProperty (AncestorNotifier.KEY)).removeYou();
         putClientProperty (AncestorNotifier.KEY, null);
      }
      super.removeYou();
   }

   // ------------------------- Listener Methods ---------------------------------

   /**
    * The PropertyChangeListener-Method.<p>
    *
    * Called when the parent of the Line changes to set the parent of the grabs to the new
    * parent if their old parent was the old parent of the line. Or when the point of either
    * of the Bends changes to update the Line.
    *
    * @param e  The event
    * @see      java.beans.PropertyChangeListener
    * @see      java.beans.PropertyChangeEvent
    */
   public void propertyChange (PropertyChangeEvent e)
   {
      Point newPoint = (Point) e.getNewValue();

      if (e.getSource() == startBend)
      {
         setStartPointFromBend (newPoint);
      }
      else
      {
         setEndPointFromBend (newPoint);
      }
      //}
   }


   /**
    * Get the closestCommonAncestor attribute of the AncestorManager object
    *
    * @param c1  No description provided
    * @param c2  No description provided
    * @return    The closestCommonAncestor value
    */
   public static Container getClosestCommonAncestor (Container c1, Container c2)
   {
      if (c1 == null || c2 == null)
      {
         return null;
      }

      if (c1 == c2)
      {
         return c1;
      }

      if (SwingUtilities.isDescendingFrom (c2, c1))
      {
         return c1;
      }

      while (! (SwingUtilities.isDescendingFrom (c1, c2) || c2 == null))
      {
         c2 = c2.getParent();
      }
      return c2;
   }

   // --------------------- AncestorManager ------------------------

   /**
    * No comment provided by developer, please add a comment to improve documentation. <<<<
    *
    * @author    $Author: schneider $
    * @version   $Revision: 1.19 $
    */
   private class AncestorManager implements HierarchyListener, HierarchyBoundsListener
   {
      /*
       *  AncestorListener
       */
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private Container closestCommonAncestorStart = null;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private Container closestCommonAncestorEnd = null;


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public void invalidate()
      {
         closestCommonAncestorStart = null;
         closestCommonAncestorEnd = null;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void hierarchyChanged (HierarchyEvent e)
      {
         long flags = e.getChangeFlags();
         if ( (flags & HierarchyEvent.PARENT_CHANGED) != 0)
         {
            Object source = e.getSource();
            Container parent = e.getChangedParent();

            if (e.getChanged().getParent() == null)
            {
               //ancestor removed
               if (closestCommonAncestorStart != null &&
                   (source == getStartBend() || source == JBendLine.this) &&
                  SwingUtilities.isDescendingFrom (parent, closestCommonAncestorStart))
               {
                  closestCommonAncestorStart = null;
               }
               if (closestCommonAncestorEnd != null &&
                   (source == getEndBend() || source == JBendLine.this) &&
                  SwingUtilities.isDescendingFrom (parent, closestCommonAncestorEnd))
               {
                  closestCommonAncestorEnd = null;
               }
            }
            else
            {
               //ancestor added
               /*
                *  closestCommonAncestorStart = JBendLine.getClosestCommonAncestor (JBendLine.this, getStartBend());
                *  invalidateStartTransform();
                *  closestCommonAncestorEnd = JBendLine.getClosestCommonAncestor (JBendLine.this, getEndBend());
                *  invalidateEndTransform();
                */
               if (source == getStartBend())
               {
                  if (closestCommonAncestorStart == null)
                  {
                     closestCommonAncestorStart = JBendLine.getClosestCommonAncestor (JBendLine.this, parent);
                     if (closestCommonAncestorStart != null)
                     {
                        invalidateStartTransform();
                     }
                  }
               }
               else if (source == getEndBend())
               {
                  if (closestCommonAncestorEnd == null)
                  {
                     closestCommonAncestorEnd = JBendLine.getClosestCommonAncestor (JBendLine.this, parent);
                     if (closestCommonAncestorEnd != null)
                     {
                        invalidateEndTransform();
                     }
                  }
               }
               else if (source == JBendLine.this)
               {
                  if (closestCommonAncestorStart == null)
                  {
                     closestCommonAncestorStart = JBendLine.getClosestCommonAncestor (parent, getStartBend());
                     if (closestCommonAncestorStart != null)
                     {
                        invalidateStartTransform();
                     }
                  }
                  if (closestCommonAncestorEnd == null)
                  {
                     closestCommonAncestorStart = JBendLine.getClosestCommonAncestor (parent, getEndBend());
                     if (closestCommonAncestorEnd != null)
                     {
                        invalidateEndTransform();
                     }
                  }
               }
            }
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void ancestorMoved (HierarchyEvent e)
      {
         Object source = e.getSource();
         Component changed = e.getChanged();

         if (changed == source)
         {
            return;
         }

         if (source == getStartBend() || source == JBendLine.this)
         {
            if (closestCommonAncestorStart != null && JBendLine.this.startTransform != null &&
               SwingUtilities.isDescendingFrom (changed, closestCommonAncestorStart))
            {
               invalidateStartTransform();
            }
         }
         if (source == getEndBend() || source == JBendLine.this)
         {
            if (closestCommonAncestorEnd != null && JBendLine.this.endTransform != null &&
               SwingUtilities.isDescendingFrom (changed, closestCommonAncestorEnd))
            {
               invalidateEndTransform();
            }
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void ancestorResized (HierarchyEvent e)
      {
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation. <<<<
    *
    * @author    $Author: schneider $
    * @version   $Revision: 1.19 $
    */
   private class BendIterator extends AbstractListIterator
   {
      /**
       * Constructor for class BendIterator
       *
       * @param start  No description provided
       */
      public BendIterator (int start)
      {
         super (start);
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param index  No description provided
       * @return       No description provided
       */
      protected Object get (int index)
      {
         return JBendLine.this.getFromBends (index);
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      protected int size()
      {
         return JBendLine.this.sizeOfBends();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param o  No description provided
       */
      public void add (Object o)
      {
         throw new UnsupportedOperationException();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public void remove()
      {
         throw new UnsupportedOperationException();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param o  No description provided
       */
      public void set (Object o)
      {
         if (currentIndex() >= 0 && currentIndex() < size())
         {
            JBendLine.this.setBendAt (currentIndex(), (JBend) o);
         }
         else
         {
            throw new NoSuchElementException();
         }
      }
   }
}

/*
 * $Log: JBendLine.java,v $
 * Revision 1.19  2004/10/20 17:49:45  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
