/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.util.*;

import javax.swing.*;
import javax.swing.plaf.PanelUI;

import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashSet;


/**
 * A JBend can be the start- or endPoint of a JBendLine<p>
 *
 * <h2> Associations </h2> <pre>
 *        0..1                      N
 * JBend ----------------------------- JBendLine
 *        startBend     outgoingLines
 *
 *        0..1                      N
 * JBend ----------------------------- JBendLine
 *        endBend       incomingLines
 * </pre>
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.23.2.1 $
 * @see       de.uni_paderborn.fujaba.fsa.swing.JBendLine
 * @see       de.uni_paderborn.fujaba.fsa.swing.JLine
 */
public class JBend extends JPanel
{ //JComponent

   /**
    * Creates a JBend at position (0, 0)
    *
    * @see   #JBend(java.awt.Point)
    */
   public JBend()
   {
      this (new Point (0, 0));
   }


   /**
    * Creates a JBend at the given position
    *
    * @param pos  position of jbend
    * @see        #JBend()
    */
   public JBend (Point pos)
   {
      this (pos, null, null);
   }


   /**
    * Create a Bend at position pos attached to incoming Line and outgoing Line
    *
    * @param pos       the position of the bend
    * @param incoming  the incoming line to attach to
    * @param outgoing  the outgoing line to attach to
    */
   public JBend (Point pos, JBendLine incoming, JBendLine outgoing)
   {
      super();
      BendUI theUI = (BendUI) DefaultBendUI.createUI (this);
      setUI (theUI);
      setSize (getPreferredSize());
      setPoint (pos);
      addToIncomingLines (incoming);
      addToOutgoingLines (outgoing);
   }

   // -----------------------------------------------------------------------

   /**
    * <pre>
    *          0..1                   N
    * JBend ----------------------------- JBendLine
    *          startBend     outgoingLines
    * </pre>
    *
    * @see   #incomingLines
    */
   private Set outgoingLines;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToOutgoingLines (JBendLine value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.outgoingLines == null)
         {
            //todo: FHashSet should be used here to avoid ConcurrentModificationException but iterator.remove() is used!
            this.outgoingLines = new FHashSet();
         }
         changed = this.outgoingLines.add (value);
         if (changed)
         {
            value.setStartBend (this);
            //setVisible (true);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInOutgoingLines (JLine value)
   {
      return  ( (this.outgoingLines != null) &&
          (value != null) &&
         this.outgoingLines.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfOutgoingLines()
   {
      return  ( (this.outgoingLines == null)
         ? FEmptyIterator.get()
         : this.outgoingLines.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfOutgoingLines()
   {
      return  ( (this.outgoingLines == null)
         ? 0
         : this.outgoingLines.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromOutgoingLines (JBendLine value)
   {
      boolean changed = false;

      if ( (this.outgoingLines != null) &&  (value != null))
      {
         changed = this.outgoingLines.remove (value);
         if (changed)
         {
            value.setStartBend (null);
            /*
             *  if (sizeOfIncomingLines() == 0 && sizeOfOutgoingLines() == 0)
             *  {
             *  setVisible (false);
             *  }
             */
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromOutgoingLines()
   {
      JBendLine tmpValue;
      Iterator iter = this.iteratorOfOutgoingLines();

      while (iter.hasNext())
      {
         tmpValue = (JBendLine) iter.next();
         this.removeFromOutgoingLines (tmpValue);
      }
   }

   // -----------------------------------------------------------------------

   /**
    * <pre>
    *          0..1                            N
    * JBend ----------------------------- JBendLine
    *          endBend     incomingLines
    * </pre>
    *
    * @see   #outgoingLines
    */
   private Set incomingLines;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToIncomingLines (JBendLine value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.incomingLines == null)
         {
            this.incomingLines = new FHashSet();
         }
         changed = this.incomingLines.add (value);
         if (changed)
         {
            value.setEndBend (this);
            //setVisible (true);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInIncomingLines (JLine value)
   {
      return  ( (this.incomingLines != null) &&
          (value != null) &&
         this.incomingLines.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfIncomingLines()
   {
      return  ( (this.incomingLines == null)
         ? FEmptyIterator.get()
         : this.incomingLines.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfIncomingLines()
   {
      return  ( (this.incomingLines == null)
         ? 0
         : this.incomingLines.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromIncomingLines (JBendLine value)
   {
      boolean changed = false;

      if ( (this.incomingLines != null) &&  (value != null))
      {
         changed = this.incomingLines.remove (value);
         if (changed)
         {
            value.setEndBend (null);
            /*
             *  if (sizeOfIncomingLines() == 0 && sizeOfOutgoingLines() == 0)
             *  {
             *  setVisible (false);
             *  }
             */
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromIncomingLines()
   {
      JBendLine tmpValue;
      Iterator iter = this.iteratorOfIncomingLines();

      while (iter.hasNext())
      {
         tmpValue = (JBendLine) iter.next();
         this.removeFromIncomingLines (tmpValue);
      }
   }

   // -----------------------------------------------------------------------

   /**
    * Access Method for lines Attribute
    *
    * @param value  No description provided
    * @return       No description provided
    * @see          #hasInIncomingLines
    * @see          #hasInOutgoingLines
    */
   public boolean hasInLines (JLine value)
   {
      return hasInIncomingLines (value) || hasInOutgoingLines (value);
   }


   /**
    * Get the firstFromLines attribute of the JBend object
    *
    * @return   The firstFromLines value
    */
   public JBendLine getFirstFromLines()
   {
      Iterator iter = this.iteratorOfLines();

      if (iter.hasNext())
      {
         return (JBendLine) iter.next();
      }
      else
      {
         return null;
      }
   }


   /**
    * Access Method for lines Attribute
    *
    * @return   No description provided
    * @see      #iteratorOfIncomingLines
    * @see      #iteratorOfOutgoingLines
    */
   public Iterator iteratorOfLines()
   {
      return
         new Iterator()
         {
            private Iterator iter = iteratorOfIncomingLines();
            private boolean outgoing = false;


            public boolean hasNext()
            {
               if (!iter.hasNext() && !outgoing)
               {
                  outgoing = true;
                  iter = iteratorOfOutgoingLines();
               }
               return iter.hasNext();
            }


            public Object next() throws NoSuchElementException
            {
               if (hasNext())
               {
                  return iter.next();
               }
               throw new NoSuchElementException();
            }


            public void remove()
            {
               iter.remove();
            }
         };
   }


   /**
    * Access Method for lines Attribute
    *
    * @return   No description provided
    * @see      #sizeOfIncomingLines
    * @see      #sizeOfOutgoingLines
    */
   public int sizeOfLines()
   {
      return sizeOfIncomingLines() + sizeOfOutgoingLines();
   }


   /**
    * Access Method for lines Attribute
    *
    * @param value  No description provided
    * @return       No description provided
    * @see          #removeFromIncomingLines
    * @see          #removeFromOutgoingLines
    */
   public boolean removeFromLines (JBendLine value)
   {
      return removeFromIncomingLines (value) | removeFromOutgoingLines (value);
   }


   /**
    * Access Method for lines Attribute
    *
    * @see   #removeAllFromIncomingLines
    * @see   #removeAllFromOutgoingLines
    */
   public void removeAllFromLines()
   {
      removeAllFromIncomingLines();
      removeAllFromOutgoingLines();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Point point = null;


   /**
    * Store the value of point in p, or if p is null in a new Point
    *
    * @param p  No description provided
    * @return   Value of point.
    * @see      #getPoint()
    * @see      #setPoint(java.awt.Point)
    * @see      #setPoint(int, int)
    */
   public Point getPoint (Point p)
   {
      if (point == null)
      {
         resetPoint();
      }

      if (p == null)
      {
         return new Point (point);
      }

      p.x = point.x;
      p.y = point.y;

      return p;
   }


   /**
    * Get the value of point Same as getPoint(null)
    *
    * @return   value of point
    * @see      #getPoint(java.awt.Point)
    * @see      #setPoint(java.awt.Point)
    * @see      #setPoint(int, int)
    */
   public final Point getPoint()
   {
      return getPoint (null);
   }


   /**
    * Set the Point where the Line should end to p Fires a PropertyChangeEvent with name "point"
    * if p does not equal the current point<p>
    *
    * Computes the new Location of this JBend and calls <pre>setLocation</pre>
    *
    * @param p  The new point value
    * @see      #setPoint(int, int)
    * @see      #getPoint()
    * @see      #getPoint(java.awt.Point)
    */
   public final void setPoint (Point p)
   {
      setPoint (p.x, p.y);
   }


   /**
    * Set the Point where the Line should end to p.
    * Fires a PropertyChangeEvent with name "point"
    * if p does not equal the current point.<p>
    *
    * Computes the new Location of this JBend and calls <pre>setLocation</pre>
    *
    * @param x  The new point value
    * @param y  The new point value
    * @see      #setPoint(java.awt.Point)
    * @see      #getPoint()
    * @see      #getPoint(java.awt.Point)
    */
   public void setPoint (int x, int y)
   {
      Point oldPoint = getPoint();
      int dx = x - oldPoint.x;
      int dy = y - oldPoint.y;

      if (dx != 0 || dy != 0)
      {
         // fixed: oldPoint is calculated in JBend#calculatePoint()
         //			pos is the current position stored in java.awt.Component
         // sometimes these values differ
         // --> results in setting wrong position
         //		(-1/-1) at creation-time of bend
         //Point pos = getLocation();
         //setLocation (pos.x + dx, pos.y + dy);
         setLocation (oldPoint.x + dx, oldPoint.y + dy);
      }
      //actual setting of point is done in setBounds
   }


   /**
    * Sets the Bounds and calls resetPoint to update the point accordingly
    *
    * @param x       The new bounds value
    * @param y       The new bounds value
    * @param width   The new bounds value
    * @param height  The new bounds value
    * @see           #setPoint(java.awt.Point)
    * @see           #setPoint(int, int)
    * @see           #resetPoint
    */
   public void setBounds (int x, int y, int width, int height)
   {
      super.setBounds (x, y, width, height);
      resetPoint();
   }


   /**
    * Calculates the value of Point from the current Bounds with a call to <pre> calculatePoint() </pre>
    * <p>
    *
    * If the new value differs from the current point, the current point is updated and a PropertyChangeEvent
    * is fired with propertyName "point"
    *
    * @see   #calculatePoint()
    * @see   #setPoint(int, int)
    * @see   #setPoint(java.awt.Point)
    * @see   #setBounds (int, int, int, int)
    * @see   java.beans.PropertyChangeEvent
    */
   protected void resetPoint()
   {
      Point newPoint = calculatePoint();
      Point oldPoint = point;
      if (!newPoint.equals (oldPoint))
      {
         //de.uni_paderborn.fujaba.basic.FujabaDebug.printStackTrace(1, 10);
         point = newPoint;
         firePropertyChange ("point", oldPoint, newPoint);
      }
   }


   /**
    * Calculates the value of Point from the current Bounds Default is the center of the Bounds
    *
    * @return   No description provided
    * @see      #resetPoint()
    */
   protected Point calculatePoint()
   {
      if (this.ui != null && this.ui instanceof BendUI)
      {
         return  ((BendUI) this.ui).calculatePoint (this);
      }
      else
      {
         Rectangle bounds = getBounds();

         return new Point ((int) bounds.getCenterX(), (int) bounds.getCenterY());
      }
   }


   /**
    * Sets the uI attribute of the JBend object
    *
    * @param ui  The new uI value
    */
   public void setUI (BendUI ui)
   {
      Point oldPoint = getPoint();
      super.setUI (ui);
      if (ui != null)
      {
         Point newPoint = calculatePoint();
         if (newPoint.x != oldPoint.x || newPoint.y != oldPoint.y)
         {
            Point pos = getLocation();
            pos.x += newPoint.x - oldPoint.x;
            pos.y += newPoint.y - oldPoint.y;
            setLocation (pos);
         }
      }
   }


   /**
    * Get the uI attribute of the JBend object
    *
    * @return   The uI value
    */
   public PanelUI getUI()
   {
      return (PanelUI) ui;
   }


   /**
    * Get the maximumSize attribute of the JBend object
    *
    * @return   The maximumSize value
    */
   public Dimension getMaximumSize()
   {
      Dimension size = null;
      if (ui != null)
      {
         size = ui.getMaximumSize (this);
      }
      return  (size != null) ? size : super.getMaximumSize();
   }


   /**
    * Get the minimumSize attribute of the JBend object
    *
    * @return   The minimumSize value
    */
   public Dimension getMinimumSize()
   {
      Dimension size = null;
      if (ui != null)
      {
         size = ui.getMinimumSize (this);
      }
      return  (size != null) ? size : super.getMinimumSize();
   }


   /**
    * Get the preferredSize attribute of the JBend object
    *
    * @return   The preferredSize value
    */
   public Dimension getPreferredSize()
   {
      Dimension size = null;
      if (ui != null)
      {
         size = ui.getPreferredSize (this);
      }
      return  (size != null) ? size : super.getPreferredSize();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, Object oldValue, Object newValue)
   {
      super.firePropertyChange (name, oldValue, newValue);
   }
}

/*
 * $Log: JBend.java,v $
 * Revision 1.23.2.1  2005/05/25 11:14:14  fklar
 * fixed location-restoring of bends in a polyline (bug occured in FSAPolyLine#applyProperties)
 *
 */
