/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.geom.GeneralPath;


/**
 * A JBendLine with arrows at the ends. Only the arrow at the ending end is implmented.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.17.2.1 $
 */
public class JArrowedBendLine extends JBendLine
{
   // Arrow Styles.

   /**
    * Plain line endings, no arrow.
    */
   public final static int NONE = 0;
   /**
    * Open arrow, not closed, not filled.
    */
   public final static int OPEN = 1;
   /**
    * Closed arrow, not filled.
    */
   public final static int CLOSED = 2;
   /**
    * Closed and filled arrow.
    */
   public final static int FILLED = 3;

   /**
    * Length of end arrow.
    */
   private double endArrowLength = 14;
   /**
    * Width of end arrow.
    */
   private double endArrowWidth = 5;
   /**
    * Arrow's point "to the left of the line".
    */
   private Point endArrowLeftPoint;
   /**
    * Arrow's point "to the right of the line".
    */
   private Point endArrowRightPoint;
   /**
    * End arrow style, see Arrow Styles.
    */
   private int endArrowStyle = OPEN;


   /**
    * Constructor for class JArrowedBendLine
    */
   public JArrowedBendLine()
   {
      super (new Point (0, 0), new Point (0, 0));
   }


   /**
    * @param start  start of line
    * @param end    end of line
    */
   public JArrowedBendLine (Point start, Point end)
   {
      super (new JBend (start), new JBend (end));
   }


   /**
    * @param start  the start Bend
    * @param end    the end Bend
    */
   public JArrowedBendLine (JBend start, JBend end)
   {
      super (start, end);
   }


   /**
    * Sets the endArrowLength attribute of the JArrowedBendLine object
    *
    * @param length  The new endArrowLength value
    */
   public void setEndArrowLength (int length)
   {
      endArrowLength = length;
   }


   /**
    * Sets the endArrowWidth attribute of the JArrowedBendLine object
    *
    * @param width  The new endArrowWidth value
    */
   public void setEndArrowWidth (int width)
   {
      endArrowWidth = width;
   }


   /**
    * Sets the endArrowStyle attribute of the JArrowedBendLine object
    *
    * @param style  The new endArrowStyle value
    */
   public void setEndArrowStyle (int style)
   {
      endArrowStyle = style;
   }



   /**
    * Get the preferredBounds attribute of the JArrowedBendLine object
    *
    * @return   The preferredBounds value
    */
   public Rectangle getPreferredBounds()
   {
      Rectangle lineBounds = super.getPreferredBounds();
      computeArrows();

      // the arrow probably causes bigger bounds, so calculate them

      int leftmostX
          = Math.min (Math.min (Math.min (lineBounds.x,
         lineBounds.x + lineBounds.width),
         endArrowLeftPoint.x),
         endArrowRightPoint.x);

      int rightmostX
          = Math.max (Math.max (Math.max (lineBounds.x,
         lineBounds.x + lineBounds.width),
         endArrowLeftPoint.x),
         endArrowRightPoint.x);

      int topmostY
          = Math.min (Math.min (Math.min (lineBounds.y,
         lineBounds.y + lineBounds.height),
         endArrowLeftPoint.y),
         endArrowRightPoint.y);

      int bottommostY
          = Math.max (Math.max (Math.max (lineBounds.y,
         lineBounds.y + lineBounds.height),
         endArrowLeftPoint.y),
         endArrowRightPoint.y);

      return new Rectangle (leftmostX, topmostY,
         rightmostX - leftmostX + 1, bottommostY - topmostY + 1);
   }



   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    */
   public void paintComponent (Graphics g)
   {
      // first paint line, then arrow

      super.paintComponent (g);

      computeArrows();

      Point end = getEndPoint();
      Point loc = getLocation();

      int xl = (int)  (endArrowLeftPoint.x - loc.getX());
      int yl = (int)  (endArrowLeftPoint.y - loc.getY());

      int xr = (int)  (endArrowRightPoint.x - loc.getX());
      int yr = (int)  (endArrowRightPoint.y - loc.getY());

      int xe = (int)  (end.getX() - loc.getX());
      int ye = (int)  (end.getY() - loc.getY());

      int[] xs =
         {xl, xe, xr};
      int[] ys =
         {yl, ye, yr};

      GeneralPath head = new GeneralPath (GeneralPath.WIND_EVEN_ODD);
      head.moveTo (xl, yl);
      head.lineTo (xe, ye);
      head.lineTo (xr, yr);

      Color oldColor = g.getColor();
      g.setColor (getForeground());

      switch (endArrowStyle)
      {
         case NONE:
            break;
         case OPEN:
            g.drawPolyline (xs, ys, 3);
            break;
         case CLOSED:
            Color saveArrowColor = g.getColor();
            g.setColor (getBackground());
            g.fillPolygon (xs, ys, 3);
            g.setColor (saveArrowColor);
            g.drawPolygon (xs, ys, 3);
            break;
         case FILLED:
            g.fillPolygon (xs, ys, 3);
            break;
         default:
            // throw an exception?
            break;
      }

      g.setColor (oldColor);
   }



   /**
    * Computes coordinates for the arrows. Start arrow not implemented yet.
    */
   private void computeArrows()
   {
      Point start = getStartPoint();
      Point end = getEndPoint();

      double slope =  (start.getY() - end.getY()) /  (end.getX() - start.getX());
      double alpha = Math.atan (slope);
      double beta = Math.PI / 2 - alpha;
      double gamma = Math.atan (endArrowWidth / endArrowLength);

      double delta = beta - gamma;
      double deltaRight = beta + gamma;
      double arrowSideLength =
         Math.sqrt (endArrowWidth * endArrowWidth + endArrowLength * endArrowLength);

      // need to subtract instead of add if start is more to the right than end
      double slopeCorrect = 1;
      if (end.getX() - start.getX() < 0)
      {
         slopeCorrect = -1;
      }

      endArrowLeftPoint = new Point (
         (int)  (end.getX() - slopeCorrect * Math.sin (delta) * arrowSideLength),
         (int)  (end.getY() + slopeCorrect * Math.cos (delta) * arrowSideLength)
         );

      endArrowRightPoint = new Point (
         (int)  (end.getX() - slopeCorrect * Math.sin (deltaRight) * arrowSideLength),
         (int)  (end.getY() + slopeCorrect * Math.cos (deltaRight) * arrowSideLength)
         );
   }
}

/*
 * $Log: JArrowedBendLine.java,v $
 * Revision 1.17.2.1  2005/08/23 08:24:52  lowende
 * Removed compile warnings.
 *
 */
