/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.geom.Point2D;

import javax.swing.*;
import javax.swing.plaf.ComponentUI;


/**
 * This UI lets a JLine appear as two parrellel lines with a small distance between them. Use
 * createUI() to get an instance of this class.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.10 $
 */
public class DoubleLineUI extends DefaultLineUI
{
   /**
    * This class is a singleton. This is the object of it.
    */
   private static DoubleLineUI ui = null;


   /**
    * Returns an instance of this class for the given JComponent.
    *
    * @param c  the JComponent the LineUI should be returned for.
    * @return   An instance of this class.
    */
   public static ComponentUI createUI (JComponent c)
   {
      if (ui == null)
      {
         ui = new DoubleLineUI();
      }
      return ui;
   }


   /**
    * Constructor for class DoubleLineUI
    */
   protected DoubleLineUI() { }


   /**
    * The distance between the two lines in pixels. Obviously, the value will be rounded while
    * drawing, however, the distance of sloped lines is affected by the value
    */
   private final static double DISTANCE = 2.6;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param c  No description provided
    * @param x  No description provided
    * @param y  No description provided
    * @return   No description provided
    */
   public boolean contains (JComponent c, int x, int y)
   {
      JLine line = (JLine) c;
      return line.getDistance (x + line.getX(), y + line.getY()) < line.getDelta() + DISTANCE / 2.0 + 0.5;
   }


   /**
    * Paints the two parrallel lines for the given JLine.
    *
    * @param g     No description provided
    * @param line  No description provided
    */
   public void paintLine (Graphics g, JLine line)
   {
      Color oldColor = g.getColor();

      Point start = line.getStartPoint();
      Point end = line.getEndPoint();
      Point loc = line.getLocation();

      /*
       *  Draw the parrallel lines by adding to the start
       *  and end coordinates the normal vector of the line
       *  streched by the DISTANCE value.
       *  This is done in both directions, thus two parrallel lines are
       *  drawn.
       */
      Point2D normal = line.getNormal();
      Point2D.Double distance = new Point2D.Double (normal.getX() * DISTANCE / 2.0, normal.getY() * DISTANCE / 2.0);

      g.setColor (line.getForeground());

      g.drawLine ((int)  (start.x - loc.x + distance.x + 0.5), (int)  (start.y - loc.y + distance.y + 0.5), (int)  (end.x - loc.x + distance.x + 0.5), (int)  (end.y - loc.y + distance.y + 0.5));
      g.drawLine ((int)  (start.x - loc.x - distance.x + 0.5), (int)  (start.y - loc.y - distance.y + 0.5), (int)  (end.x - loc.x - distance.x + 0.5), (int)  (end.y - loc.y - distance.y + 0.5));

      g.setColor (oldColor);
   }


   /**
    * Returns the preferred bounds of the given JComponent, using the start and end points
    * of the JComponent.
    *
    * @param line  No description provided
    * @return      a rectangle with the preferred bounds of the given component.
    */
   public Rectangle getPreferredLineBounds (JLine line)
   {
      Point start = line.getStartPoint();
      Point end = line.getEndPoint();
      Point2D normal = line.getNormal();
      Point2D.Double distance = new Point2D.Double (normal.getX() * DISTANCE / 2.0, normal.getY() * DISTANCE / 2.0);
      double delta = line.getDelta();

      return new Rectangle (
         (int)  (Math.min (start.x, end.x) + distance.x + 0.5 - delta),
         (int)  (Math.min (start.y, end.y) + distance.y + 0.5 - delta),
         (int)  (Math.abs (end.x - start.x) + 2.0 * distance.x + 1 + 2.0 * delta),
         (int)  (Math.abs (end.y - start.y) + 2.0 * distance.y + 1 + 2.0 * delta)
         );
   }
}

/*
 * $Log: DoubleLineUI.java,v $
 * Revision 1.10  2004/11/03 10:17:56  lowende
 * Javadoc warnings removed.
 *
 */
