/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa;

import java.awt.*;
import java.lang.ref.WeakReference;
import java.util.*;
import java.util.List;

import javax.swing.*;
import de.uni_paderborn.fujaba.asg.ASGElement;

import de.uni_paderborn.fujaba.fsa.listener.*;
import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.upb.lib.userinterface.EventSource;
import de.upb.tools.fca.FHashSet;
import de.upb.tools.fca.FLinkedList;


/**
 * @author    $Author: cschneid $
 * @version   $Revision: 1.35 $
 */
public class SelectionManager implements EventSource, SelectionListener
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static SelectionManager manager;


   /**
    * Constructor for class SelectionManager
    */
   private SelectionManager()
   {
      selectedComponents = new FLinkedList();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public static SelectionManager get()
   {
      if (manager == null)
      {
         manager = new SelectionManager();
      }
      return manager;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private WeakReference popupSource;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private WeakReference logicPopupSource;


   /**
    * Sets the popupSource attribute of the SelectionManager object
    *
    * @param source       The new popupSource value
    * @param logicSource  The new popupSource value
    */
   public void setPopupSource (Component source, LogicUnparseInterface logicSource)
   {
      setPopupSource (source);
      setLogicPopupSource (logicSource);
      notifyPopupSourceListeners();
   }


   /**
    * Sets the popupSource attribute of the SelectionManager object
    *
    * @param source  The new popupSource value
    */
   public void setPopupSource (Component source)
   {
      this.popupSource =  (source == null ? null : new WeakReference (source));
   }


   /**
    * Sets the logicPopupSource attribute of the SelectionManager object
    *
    * @param logicSource  The new logicPopupSource value
    */
   public void setLogicPopupSource (LogicUnparseInterface logicSource)
   {
      this.logicPopupSource =  (logicSource == null ? null : new WeakReference (logicSource));
   }


   /**
    * Get the logicPopupSource attribute of the SelectionManager object
    *
    * @return   The logicPopupSource value
    */
   public LogicUnparseInterface getLogicPopupSource()
   {
      return (LogicUnparseInterface)  (this.logicPopupSource == null ? null : this.logicPopupSource.get());
   }


   /**
    * Get the popupSource attribute of the SelectionManager object
    *
    * @return   The popupSource value
    */
   public Component getPopupSource()
   {
      return (Component)  (this.popupSource == null ? null : this.popupSource.get());
   }


   /**
    * Returns an iterator of the objects from the environment of the source object. This could
    * be a selection of multiple objects. Used for actions that manipulate more than one object.
    * The first element is the element under popup if popup was called. Implements EventSource
    * interface from UserInterfaceManager.
    *
    * @return   The source value
    */
   public Object getSource()
   {
      LinkedList list = new LinkedList();

      LogicUnparseInterface iface = getLogicPopupSource();
      if (iface != null)
      {
         list.add (iface);
      }

      Iterator iter = iteratorOfSelection();
      while (iter.hasNext())
      {
         FSAObject object = (FSAObject) iter.next();
         if (object != null)
         {
            LogicUnparseInterface incr = object.getLogic();
            if (incr != null)
            {
               if (!list.contains (incr))
               {
                  list.add (incr);
               }
            }
         }
      }

      return list.iterator();
   }


   /**
    * Handed over to the action as the command string for the event. This string allows a "modal"
    * component to specify one of several commands, depending on its state. For example, a
    * single button might toggle between "show details" and "hide details". The source object
    * and the event would be the same in each case, but the command string would identify the
    * intended action. Implements EventSource interface from UserInterfaceManager.
    *
    * @return   The actionCommand value
    */
   public String getActionCommand()
   {
      return null;
   }


   /**
    * Sets the selected attribute of the SelectionManager object
    *
    * @param component  The new selected value
    * @param selected   The new selected value
    */
   public void setSelected (JComponent component, boolean selected)
   {
      setSelected (component, selected, false);
   }


   /**
    * Sets the selected attribute of the SelectionManager object
    *
    * @param component  The new selected value
    * @param selected   The new selected value
    * @param multi      The new selected value
    */
   public void setSelected (JComponent component, boolean selected, boolean multi)
   {
      boolean oldSelected = hasInSelectedComponents (component);

      if (!multi)
      {
         clear();
      }

      if (selected != oldSelected)
      {
         if (selected)
         {
            addToSelectedComponents (component);
         }
         else if (multi)
         {
            removeFromSelectedComponents (component);
         }
      }
      else
      {
         setFocused (component, selected);
      }
   }


   /**
    * Get the selected attribute of the SelectionManager object
    *
    * @param component  No description provided
    * @return           The selected value
    */
   public boolean isSelected (JComponent component)
   {
      return hasInSelectedComponents (component);
   }


   /**
    * set the component focused It is set as the focused Object and the previously focused
    * Object is unfocused. If this Object is not yet selected, it is selected, too.
    *
    * @param component  The new focused value
    * @param focus      The new focused value
    */
   public void setFocused (JComponent component, boolean focus)
   {
      boolean oldFocused =  (getFocusedComponent() == component);

      if (oldFocused != focus)
      {
         if (focus)
         {
            setFocusedComponent (component);
         }
         else
         {
            setFocusedComponent (null);
         }
      }
   }


   /**
    * Get the focused attribute of the SelectionManager object
    *
    * @param component  No description provided
    * @return           The focused value
    */
   public boolean isFocused (JComponent component)
   {
      return  (getFocusedComponent() == component);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JComponent focusedComponent;


   /**
    * Sets the focusedComponent attribute of the SelectionManager object
    *
    * @param component  The new focusedComponent value
    */
   public void setFocusedComponent (JComponent component)
   {
      if (component != focusedComponent)
      {
         JComponent oldFocused = focusedComponent;
         focusedComponent = component;

         if (oldFocused != null)
         {
            FSAObject fsa = FSAObject.getFSAObjectFromJComponent (oldFocused);
            if (fsa != null)
            {
               fsa.setFocused (false);
            }
            SelectionListenerHelper.setFocused (oldFocused, false);
            setPopupSource (null, null);
         }

         if (component != null)
         {
            if (!hasInSelectedComponents (component))
            {
               addToSelectedComponents (component);
            }

            if (component != selectedComponents.getLast())
            {
               selectedComponents.remove (component);
               selectedComponents.add (component);
            }
            FSAObject fsa = FSAObject.getFSAObjectFromJComponent (component);
            if (fsa != null)
            {
               fsa.setFocused (true);
            }
            SelectionListenerHelper.setFocused (component, true);
            setPopupSource (component, FSAObject.getLogicFromJComponent (component));
         }
      }
   }


   /**
    * Get the focusedComponent attribute of the SelectionManager object
    *
    * @return   The focusedComponent value
    */
   public JComponent getFocusedComponent()
   {
      return focusedComponent;
   }


   /**
    * todo: use Vector-like concurrent list here when available
    */
   private FLinkedList selectedComponents;


   /**
    * Access method for an one to n association.
    *
    * @param component  The object added.
    */
   public void addToSelectedComponents (JComponent component)
   {
      if (component != null)
      {
         if (!hasInSelectedComponents (component))
         {
            selectedComponents.add (component);
            FSAObject fsa = FSAObject.getFSAObjectFromJComponent (component);
            if (fsa != null)
            {
               fsa.setSelected (true);
            }
            SelectionListenerHelper.setSelected (component, true);
            SelectionListenerHelper.addSelectionListener (component, this);
         }
         else if (selectedComponents.getLast() != component)
         {
            selectedComponents.remove (component);
            selectedComponents.add (component);
         }
         setFocusedComponent (component);
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param components  The object added.
    */
   public void addToSelectedComponents (JComponent[] components)
   {
      if (components != null)
      {
         for (int i = 0; i < components.length; i++)
         {
            addToSelectedComponents (components[i]);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfSelectedComponents()
   {
      return selectedComponents.iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfSelectedComponentsAsIncrements()
   {
      return iteratorOfSelectionAsIncrements();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    * @return        No description provided
    */
   public boolean hasInSelectedComponents (JComponent object)
   {
      return selectedComponents.contains (object);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfSelectedComponents()
   {
      return selectedComponents.size();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    */
   public void replaceSelectedComponents (JComponent object)
   {
      clear();
      addToSelectedComponents (object);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param objects  No description provided
    */
   public void replaceSelectedComponents (JComponent[] objects)
   {
      clear();
      addToSelectedComponents (objects);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    */
   public void removeFromSelectedComponents (JComponent object)
   {
      if (object != null && hasInSelectedComponents (object))
      {
         SelectionListenerHelper.removeSelectionListener (object, this);
         if (object == getFocusedComponent())
         {
            setFocusedObject (null);
         }
         selectedComponents.remove (object);
         FSAObject fsa = FSAObject.getFSAObjectFromJComponent (object);
         if (fsa != null)
         {
            fsa.setSelected (false);
         }
         SelectionListenerHelper.setSelected (object, false);
      }
   }


   /**
    * deselect all
    */
   public void removeAllFromSelectedComponents()
   {
      for (Iterator it = iteratorOfSelectedComponents(); it.hasNext(); )
      {
         JComponent component = (JComponent) it.next();
         removeFromSelectedComponents (component);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param objects  No description provided
    */
   public void removeFromSelectedComponents (JComponent[] objects)
   {
      if (objects != null)
      {
         for (int i = 0; i < objects.length; i++)
         {
            removeFromSelectedComponents (objects[i]);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    * @return        No description provided
    */
   public JComponent prevComponent (JComponent object)
   {
      int index;
      if (object == null ||  (index = selectedComponents.indexOf (object)) <= 0)
      {
         index = selectedComponents.size();
      }

      if (index > 0)
      {
         JComponent prev = (JComponent) selectedComponents.get (index - 1);
         if (prev != object)
         {
            return prev;
         }
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    * @return        No description provided
    */
   public JComponent nextComponent (JComponent object)
   {
      int index;
      int size = selectedComponents.size();
      if (object == null ||  (index = selectedComponents.indexOf (object)) >= size - 1)
      {
         index = -1;
      }

      if (index < size - 1)
      {
         JComponent next = (JComponent) selectedComponents.get (index + 1);
         if (next != object)
         {
            return next;
         }
      }
      return null;
   }


   /**
    * This method returns the last 'index' selected items.
    *
    * @param length  number of the returned objects.
    * @param obj     If this object is not in the last 'length' selections, it will be returned
    *      as the last selected Object in the resulting array.
    * @return        An array consisting of the last 'length' selected objects.
    */
   public JComponent[] getLastSelectedComponents (JComponent obj, int length)
   {
      int size = selectedComponents.size();

      if (length > size || length == -1)
      {
         length = size;
      }

      List objects = selectedComponents.subList (size - length, size);

      if (obj != null)
      {
         if (objects.contains (obj))
         {
            objects.remove (obj);
         }
         else
         {
            objects.remove (0);
         }
         objects.add (obj);
      }

      return (JComponent[]) objects.toArray (new JComponent[size]);
   }


   /**
    * Get the lastSelectedComponentsOfType attribute of the SelectionManager object
    *
    * @param type    No description provided
    * @param length  No description provided
    * @return        The lastSelectedComponentsOfType value
    */
   public JComponent[] getLastSelectedComponentsOfType (JComponent type, int length)
   {
      Class typeClass = type.getClass();

      return getLastSelectedComponentsOfType (typeClass, length);
   }


   /**
    * Get the lastSelectedComponentsOfType attribute of the SelectionManager object
    *
    * @param typeClass  No description provided
    * @param length     No description provided
    * @return           The lastSelectedComponentsOfType value
    */
   public JComponent[] getLastSelectedComponentsOfType (Class typeClass, int length)
   {
      LinkedList objects = new LinkedList();
      ListIterator iter = selectedComponents.listIterator (selectedComponents.size());
      int size = 0;
      while (iter.hasPrevious() && size < length)
      {
         JComponent temp = (JComponent) iter.previous();
         if (typeClass.isInstance (temp))
         {
            objects.add (0, temp);
            size++;
         }
      }

      return (JComponent[]) objects.toArray (new JComponent[size]);
   }


   /**
    * Get the selectedComponents attribute of the SelectionManager object
    *
    * @return   The selectedComponents value
    */
   public JComponent[] getSelectedComponents()
   {
      int size = selectedComponents.size();
      if (size > 0)
      {
         JComponent[] objects = new JComponent[selectedComponents.size()];

         return (JComponent[]) selectedComponents.toArray (objects);
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void clear()
   {
      while (selectedComponents.size() > 0)
      {
         removeFromSelectedComponents ((JComponent) selectedComponents.getFirst());
      }
   }

   //##############################################################
   //#             Compatibility Methods for FSAObjects           #
   //##############################################################

   /**
    * Sets the selected attribute of the SelectionManager object
    *
    * @param object    The new selected value
    * @param selected  The new selected value
    */
   public void setSelected (FSAObject object, boolean selected)
   {
      setSelected (object, selected, false);
   }


   /**
    * Sets the selected attribute of the SelectionManager object
    *
    * @param object    The new selected value
    * @param selected  The new selected value
    * @param multi     The new selected value
    */
   public void setSelected (FSAObject object, boolean selected, boolean multi)
   {
      setSelected ( (object == null ? (JComponent) null : object.getJComponent()), selected, multi);
   }


   /**
    * set the component focused It is set as the focused Object and the previously focused
    * Object is unfocused. If this Object is not yet selected, it is selected, too.
    *
    * @param object  The new focused value
    * @param focus   The new focused value
    */
   public void setFocused (FSAObject object, boolean focus)
   {
      setFocused ( (object == null ? (JComponent) null : object.getJComponent()), focus);
   }


   /**
    * Sets the focusedObject attribute of the SelectionManager object
    *
    * @param object  The new focusedObject value
    */
   public void setFocusedObject (FSAObject object)
   {
      setFocusedComponent ( (object == null ? null : object.getJComponent()));
   }


   /**
    * Get the focusedObject attribute of the SelectionManager object
    *
    * @return   The focusedObject value
    */
   public FSAObject getFocusedObject()
   {
      return  (focusedComponent == null ? null : FSAObject.getFSAObjectFromJComponent (focusedComponent));
   }


   /**
    * Access method for an one to n association.
    *
    * @param object  The object added.
    */
   public void addToSelection (FSAObject object)
   {
      if (object != null)
      {
         addToSelectedComponents (object.getJComponent());
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param objects  The object added.
    */
   public void addToSelection (FSAObject[] objects)
   {
      if (objects != null)
      {
         for (int i = 0; i < objects.length; i++)
         {
            addToSelection (objects[i]);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfSelection()
   {
      return new JComponentAsFSAIterator (selectedComponents.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfSelectionAsIncrements()
   {
      LinkedList list = new LinkedList();

      Iterator iter = iteratorOfSelection();
      while (iter.hasNext())
      {
         LogicUnparseInterface incr =  ((FSAObject) iter.next()).getLogic();
         if (incr != null)
         {
            list.add (incr);
         }
      }

      return list.iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    * @return        No description provided
    */
   public boolean hasInSelection (FSAObject object)
   {
      return  (object != null && hasInSelectedComponents (object.getJComponent()));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfSelection()
   {
      return sizeOfSelectedComponents();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    */
   public void replaceSelection (FSAObject object)
   {
      replaceSelectedComponents ( (object == null ? null : object.getJComponent()));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param objects  No description provided
    */
   public void replaceSelection (FSAObject[] objects)
   {
      clear();
      addToSelection (objects);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    */
   public void removeFromSelection (FSAObject object)
   {
      if (object != null)
      {
         removeFromSelectedComponents (object.getJComponent());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param objects  No description provided
    */
   public void removeFromSelection (FSAObject[] objects)
   {
      if (objects != null)
      {
         for (int i = 0; i < objects.length; i++)
         {
            if (objects[i] != null)
            {
               removeFromSelectedComponents (objects[i].getJComponent());
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    * @return        No description provided
    */
   public FSAObject prevObject (FSAObject object)
   {
      JComponent tmpResult =  (object == null ? null : object.getJComponent());
      FSAObject result = null;

      do
      {
         tmpResult = prevComponent (tmpResult);
         result = FSAObject.getFSAObjectFromJComponent (tmpResult);
      } while (tmpResult != null && result == null);

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    * @return        No description provided
    */
   public FSAObject nextObject (FSAObject object)
   {
      JComponent tmpResult =  (object == null ? null : object.getJComponent());
      FSAObject result = null;

      do
      {
         tmpResult = nextComponent (tmpResult);
         result = FSAObject.getFSAObjectFromJComponent (tmpResult);
      } while (tmpResult != null && result == null);

      return result;
   }


   /**
    * This method returns the last 'index' selected items.
    *
    * @param length  number of the returned objects.
    * @param obj     If this object is not in the last 'length' selections, it will be returned
    *      as the last selected Object in the resulting array.
    * @return        An array consisting of the last 'length' selected objects.
    */
   public FSAObject[] getLastSelections (FSAObject obj, int length)
   {
      LinkedList objects = new LinkedList();
      FSAObject prev = prevObject (null);

      for (int i = 0; i < length && prev != null; i++, prev = prevObject (prev))
      {
         objects.add (prev);
      }

      if (obj != null)
      {
         if (objects.contains (obj))
         {
            objects.remove (obj);
         }
         else
         {
            objects.remove (0);
         }
         objects.add (obj);
      }
      return (FSAObject[]) objects.toArray (new FSAObject[objects.size()]);
   }


   /**
    * This method returns the last 'index' selected items, which are instances of the class
    * of the type object.
    *
    * @param length  number of the returned objects.
    * @param type    Only objects of this type are returned.
    * @return        An array consisting of the last 'length' selected objects.
    */
   public FSAObject[] getLastSelectionsOfType (FSAObject type, int length)
   {
      Class typeClass = type.getClass();

      return getLastSelectionsOfType (typeClass, length);
   }


   /**
    * Returns all selections of type 'typeClass'.
    *
    * @param typeClass  No description provided
    * @return           The lastSelectionsOfType value
    */
   public FSAObject[] getLastSelectionsOfType (Class typeClass)
   {
      return getLastSelectionsOfType (typeClass, Integer.MAX_VALUE);
   }


   /**
    * Returns 'length' number of selected objects of type 'typeclass'.
    *
    * @param typeClass  No description provided
    * @param length     No description provided
    * @return           The lastSelectionsOfType value
    */
   public FSAObject[] getLastSelectionsOfType (Class typeClass, int length)
   {
      LinkedList objects = new LinkedList();
      ListIterator iter = selectedComponents.listIterator (selectedComponents.size());
      int size = 0;
      while (iter.hasPrevious() && size < length)
      {
         JComponent temp = (JComponent) iter.previous();
         FSAObject object = FSAObject.getFSAObjectFromJComponent (temp);
         if (object != null && typeClass.isInstance (object))
         {
            objects.add (0, object);
            size++;
         }
      }

      return (FSAObject[]) objects.toArray (new FSAObject[size]);
   }


   /**
    * Get the selectedObjects attribute of the SelectionManager object
    *
    * @return   The selectedObjects value
    */
   public FSAObject[] getSelectedObjects()
   {
      int size = selectedComponents.size();
      if (size > 0)
      {
         LinkedList objects = new LinkedList();
         Iterator iter = iteratorOfSelectedComponents();
         while (iter.hasNext())
         {
            JComponent temp = (JComponent) iter.next();
            FSAObject object = FSAObject.getFSAObjectFromJComponent (temp);
            if (object != null)
            {
               objects.add (object);
            }
         }
         if (objects.size() > 0)
         {
            return (FSAObject[]) objects.toArray (new FSAObject[objects.size()]);
         }
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param event  No description provided
    */
   public void selectionChanged (SelectionEvent event)
   {
      JComponent source = (JComponent) event.getSource();

      if (event.isFocusChanged())
      {
         if (getFocusedComponent() == source && !event.isFocused())
         {
            setFocusedComponent (null);
         }
         else if (getFocusedComponent() != source && event.isFocused())
         {
            setFocusedComponent (source);
         }
      }
      if (event.isSelectionChanged())
      {
         if (hasInSelectedComponents (source) && !event.isSelected())
         {
            removeFromSelectedComponents (source);
         }
         else if (!hasInSelectedComponents (source) && event.isSelected())
         {
            addToSelectedComponents (source);
         }
      }
   }


   /**
    * Stores listeners to popupSource
    */
   private Set popupSourceListeners;


   /**
    * Notify all listeners to popupSource
    */
   private void notifyPopupSourceListeners()
   {
      if (popupSourceListeners != null)
      {
         LogicUnparseInterface source = getLogicPopupSource();
         Iterator it = popupSourceListeners.iterator();
         while (it.hasNext())
         {
            PopupSourceListener listener = (PopupSourceListener) it.next();
            try
            {
               listener.popupSourceChanged (source);
            }
            catch (Exception e)
            {
               e.printStackTrace();
            }
         }
      }
   }


   /**
    * Add a listener that receives a method call upon change of the logicPopupSource
    *
    * @param listener  to be added
    */
   public void addToPopupSourceListeners (PopupSourceListener listener)
   {
      if (listener != null)
      {
         if (popupSourceListeners == null)
         {
            popupSourceListeners = new FHashSet();
         }
         popupSourceListeners.add (listener);
      }
   }


   /**
    * Remove a listener for logicPopupSource
    *
    * @param listener  to be removed
    */
   public void removeFromPopupSourceListeners (PopupSourceListener listener)
   {
      if (popupSourceListeners != null)
      {
         popupSourceListeners.remove (listener);
      }
   }


   /**
    * Add all fsaObjects belonging to an element to the selection.
    *
    * @param element  which element to be selected
    */
   public void addToSelection (ASGElement element)
   {
      for (Iterator it = element.iteratorOfFsaObjects(); it.hasNext(); )
      {
         FSAObject fsaObject = (FSAObject) it.next();
         addToSelection (fsaObject);
      }
   }
}

/*
 * $Log: SelectionManager.java,v $
 * Revision 1.35  2005/02/14 14:28:08  cschneid
 * introduced message view, getParentElement() for FElement, code generation and compiler messages in message view
 *
 */
