/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.lang.ref.WeakReference;
import java.util.*;

import javax.swing.*;
import javax.swing.event.AncestorListener;
import org.apache.log4j.Logger;
import org.apache.log4j.Priority;

import de.uni_paderborn.fujaba.asg.*;
import de.uni_paderborn.fujaba.basic.BasicIncrement;
import de.uni_paderborn.fujaba.basic.FujabaPropertyChangeSupport;
import de.uni_paderborn.fujaba.fsa.listener.*;
import de.uni_paderborn.fujaba.fsa.unparse.FSAInterface;
import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.uni_paderborn.fujaba.fsa.update.AbstractUpdater;
import de.uni_paderborn.fujaba.fsa.update.LogicAndFsaUpdater;
import de.uni_paderborn.fujaba.preferences.ColorsPreferences;
import de.upb.lib.userinterface.UserInterfaceManager;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashSet;


/**
 * FSAObject is the Controller in the MVC
 * UMLUnparseInterface-FSAObject-JComponent. It has an association to a
 * JComponent that visualises (a part of) a UMLUnparseInterface. It can update
 * the JComponent through the appropriate get- and set-Methods of the
 * UMLUnparseInterface, which are specified by the propertyName-Attribute
 * Additionally it has several wrapper methods which simply call the according
 * methods of the jComponent. the several listener methods of FSAObject allow
 * "persistent" handling of listeners. This means, that in case the jComponent
 * changes, the listeners are removed from the old component and added to the
 * new one UMLClass: 'FSAObject'
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.98.2.7 $
 */
public abstract class FSAObject extends BasicIncrement implements
   PropertyChangeListener, ParentListener
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (FSAObject.class);

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public final static String JCOMPONENT_CLIENT_PROPERTY = "FSAObject:revJComponent";

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public final static Color COLOR_FOREGROUND = ColorsPreferences.get().DEFAULT_FOREGROUND;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public final static Color COLOR_BACKGROUND = ColorsPreferences.get().DEFAULT_BACKGROUND;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public final static Color COLOR_FOCUSED = ColorsPreferences.get().DEFAULT_FOCUSED;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public final static Color COLOR_SELECTED = ColorsPreferences.get().DEFAULT_SELECTED;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String LOCATION = "location";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String DIMENSION = "dimension";

   /**
    * UMLAttribute: ' propertyName : String'
    */
   private String propertyName = null;

   /**
    * UMLAttribute: ' jComponent : JComponent'
    */
   private final JComponent jComponent;


   /**
    * Constructor for class FSAObject
    */
   public FSAObject()
   {
      this (null, null);
//      jComponent = createJComponent();
//      setFSAObjectFromJComponent (jComponent, this);
//
//      init (null, null, TRANSIENT_PROPERTIES_DEFAULT);
   }


   /**
    * Constructor for class FSAObject
    *
    * @param incr      No description provided
    * @param propName  No description provided
    */
   public FSAObject (LogicUnparseInterface incr, String propName)
   {
      this (incr, propName, null);
//      jComponent = createJComponent();
//      setFSAObjectFromJComponent (jComponent, this);
//
//      init (incr, propName, TRANSIENT_PROPERTIES_DEFAULT);
   }


   /**
    * Constructor for class FSAObject
    *
    * This is the typical constructor for creating a new FSA instance.
    *
    * @param incr      No description provided
    * @param propName  No description provided
    * @param parent    No description provided
    */
   public FSAObject (LogicUnparseInterface incr, String propName,
                     JComponent parent)
   {
      jComponent = createJComponent();
      setFSAObjectFromJComponent (jComponent, this);
      setParentOfJComponent (parent);

      init (incr, propName, TRANSIENT_PROPERTIES_DEFAULT);
   }


   /**
    * Constructor for class FSAObject
    *
    * Use this constructor, if you want transient properties to
    * be disabled right at creation time. Use this constructor to store
    * initial-location of FSAObject. Otherwise transientProperties
    * must be disabled and FSAObject#saveLocation() must be called,
    * if initial-location should be saved.
    *
    * @param incr                 Defines the logic for this FSAObject.
    * @param propName             No description provided
    * @param parent               Parent of this FSAObject.
    * @param transientProperties  Should properties of this FSAObject be transient?
    */
   public FSAObject (LogicUnparseInterface incr, String propName,
                     JComponent parent, boolean transientProperties)
   {
      this.jComponent = createJComponent();
      setFSAObjectFromJComponent (this.jComponent, this);
      setParentOfJComponent (parent);

      init (incr, propName, transientProperties);
   }


   /**
    * Constructor for class FSAObject
    *
    * @param incr          No description provided
    * @param propName      No description provided
    * @param parent        No description provided
    * @param myJComponent  No description provided
    */
   public FSAObject (LogicUnparseInterface incr, String propName,
                     JComponent parent, JComponent myJComponent)
   {
      this.jComponent = retrieveJComponent (myJComponent);
      setFSAObjectFromJComponent (jComponent, this);
      setParentOfJComponent (parent);

      init (incr, propName, TRANSIENT_PROPERTIES_DEFAULT);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param myJComponent  No description provided
    * @return              No description provided
    */
   private JComponent retrieveJComponent (JComponent myJComponent)
   {
      JComponent tmpComp = null;
      if (myJComponent != null)
      {
         if (getFSAObjectFromJComponent (myJComponent) != null)
         {
            throw new IllegalArgumentException ("JComponent already has a FSAObject");
         }
         tmpComp = myJComponent;
      }
      else
      {
         tmpComp = createJComponent();
      }
      return tmpComp;
   }


   /**
    * Initializes this FSAObject instance.
    * Adds a parent-listener, sets property-name and logic of this FSAObject
    * intializes the transient-flag and restores formerly saved properties.
    *
    * @param incr                 No description provided
    * @param propName             No description provided
    * @param transientProperties  Should properties of this FSAObject be transient?
    */
   private void init (LogicUnparseInterface incr, String propName, boolean transientProperties)
   {
      addParentListener (this);
      setPropertyName (propName);

      setLogic (incr);

      this.transientProperties = transientProperties;

      initFSAProperties();
   }


   /**
    * Get the fSAQualifier attribute of the FSAObject object
    *
    * <pre>
    * This function will search this FSAObject objects parents.
    * The first parent which is a ASGDiagram will be returned.
    * If this FSAObject has no ASGDiagram-parent the first ASGElement
    * contained in this FSAObject objects parent-chain will be returned.
    * If none of the former described objects exist, null is returned.
    * </pre>
    *
    * @return   The fSAQualifier value
    */
   public ASGElement getFSAQualifier()
   {
      FSAObject fsa = getParent();
      LogicUnparseInterface diag = null;
      ASGElement firstElement = null;
      while ( (fsa != null) && ! (diag instanceof ASGDiagram))
      {
         diag = fsa.getLogic();
         if ( (firstElement == null) &&  (diag instanceof ASGElement))
         {
            firstElement = (ASGElement) diag;
         }
         fsa = fsa.getParent();
      }
      if (diag instanceof ASGDiagram)
      {
         return (ASGDiagram) diag;
      }
      return firstElement;
   }


   /**
    * Get the ASGInformation for the FSAObject
    *
    * @return   The associated ASGInformation, null if not created yet
    */
   public ASGInformation getASGInformation()
   {
      return getASGInformation (false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PropertyChangeListener asgInformationListener = new ASGInformationListener (this);


   /*
    *  Code in removeYou () should be enough to ensure proper garbage collection of FSAs. But better
    *  to be safe than sorry and so avoiding hard reference to FSAObject here.
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.98.2.7 $ $Date: 2005/11/14 21:55:18 $
    */
   private final static class ASGInformationListener implements PropertyChangeListener
   {
      /**
       *Constructor for class ASGInformationListener
       *
       * @param obj  No description provided
       */
      ASGInformationListener (FSAObject obj)
      {
         this.fsaObject = new WeakReference (obj);
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private final WeakReference fsaObject;


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param evt  No description provided
       */
      public void propertyChange (final PropertyChangeEvent evt)
      {
         final FSAObject obj = (FSAObject) fsaObject.get();
         if (obj != null && !obj.writingASGInformation)
         {
            if (logger.isEnabledFor (Priority.DEBUG))
            {
               logger.debug ("ASGInformationListener has been notified about propertychange of '" + evt.getPropertyName() + "': newValue '" + evt.getNewValue() + "', oldValue '" + evt.getOldValue() + "'");
            }

            // TODO:	apply properties dependent on what has changed...
            //			don't apply properties, that haven't changed
            obj.applyProperties();
         }
      }
   }


   /**
    * Get the ASGInformation for the FSAObject
    *
    * @param create  create the ASGInformation if it does not exist
    * @return        The associated ASGInformation, null if not created yet
    */
   public ASGInformation getASGInformation (boolean create)
   {
      LogicUnparseInterface incr = getLogic();

      if (incr instanceof ASGElement)
      {
         ASGElement asgElement = (ASGElement) incr;
         ASGElement asgParent = getFSAQualifier();
         ASGUnparseInformation unparseInfo = asgElement.getFromUnparseInformations (asgParent);

         if (create && unparseInfo == null)
         {
            unparseInfo = new ASGUnparseInformation();
            asgElement.addToUnparseInformations (asgParent, unparseInfo);
         }

         if (unparseInfo != null)
         {
            ASGInformation info = unparseInfo.getFromASGInformation (getPropertyName());

            if (create && info == null)
            {
               info = new ASGInformation();
               unparseInfo.addToASGInformation (getPropertyName(), info);
            }

            if (info != null)
            {
               info.addPropertyChangeListener (asgInformationListener);
            }
            return info;
         }
      }

      return null;
   }


   /**
    * Get the unparseInformation attribute of the FSAObject object
    *
    * @param property  No description provided
    * @return          The unparseInformation value
    */
   public String getUnparseInformation (String property)
   {
      ASGInformation info = getASGInformation();
      if (info != null)
      {
         return info.getFromInformation (property);
      }
      return null;
   }


   /**
    * Are we currently modifying unparse information?
    */
   boolean writingASGInformation = false;


   /**
    * Store textual information in this FSAObject.
    *
    * @param property  Key under which information is stored.
    * @param value     The String object to be added.
    */
   public void addUnparseInformation (String property, String value)
   {
      writingASGInformation = true;
      try
      {
         getASGInformation (true).addToInformation (property, value);
      }
      finally
      {
         writingASGInformation = false;
      }
   }


   /**
    * Remove information from this FSAObject stored under the specified key.
    *
    * @param property  Key under which information is stored.
    * @return          A boolean value specifying whether information has been removed.
    */
   public boolean removeFromUnparseInformation (String property)
   {
      writingASGInformation = true;
      boolean success = false;

      ASGInformation info = getASGInformation();

      if (info != null)
      {
         success = info.removeKeyFromInformation (property);
      }

      writingASGInformation = false;

      return success;
   }


   /**
    * Remove all information stored in this FSAObject.
    */
   public void removeAllUnparseInformation()
   {
      ASGInformation info = getASGInformation();
      if (info != null)
      {
         writingASGInformation = true;
         try
         {
            ASGUnparseInformation parent = info.getParent();
            info.removeYou();

            if (parent != null && parent.sizeOfASGInformation() == 0)
            {
               parent.removeYou();
            }
         }
         finally
         {
            writingASGInformation = false;
         }
      }
   }


   /**
    * Adds data of a Point object (x and y coordinate) to unparse information.
    *
    * @param property  Key for the point to be added.
    * @param value     The Point object whose data should be stored.
    * @see             FSAObject#removePointFromUnparseInformation(String)
    */
   public void addPointToUnparseInformation (String property, Point value)
   {
      if (value == null)
      {
         return;
      }

      addUnparseInformation (property + "_X", Integer.toString (value.x));
      addUnparseInformation (property + "_Y", Integer.toString (value.y));
   }


   /**
    * Removes data of a point specified by <code>property</code> from
    * unparse information.
    *
    * @param property  The key for the point to be removed.
    * @return          A boolean value specifying whether the point was deleted.
    * 		  The point has been deleted, if at least x or y - position
    *         could be removed from unparse information.
    * @see             FSAObject#addPointToUnparseInformation(String, Point)
    */
   public boolean removePointFromUnparseInformation (String property)
   {
      boolean success = removeFromUnparseInformation (property + "_X");
      success |= removeFromUnparseInformation (property + "_Y");

      return success;
   }


   /**
    * Get the pointFromUnparseInformation attribute of the FSAObject object
    *
    * @param property  No description provided
    * @return          The pointFromUnparseInformation value
    */
   public Point getPointFromUnparseInformation (String property)
   {
      Point p = null;

      String xValue = getUnparseInformation (property + "_X");
      String yValue = getUnparseInformation (property + "_Y");
      if (xValue != null && yValue != null)
      {
         try
         {
            p = new Point (Integer.parseInt (xValue), Integer.parseInt (yValue));
         }
         catch (NumberFormatException e)
         {
            e.printStackTrace();
         }
      }
      return p;
   }


   /**
    * Access method for an one to n association.
    *
    * @param property  The object added.
    * @param value     The object added.
    */
   public void addDimensionToUnparseInformation (String property, Dimension value)
   {
      addUnparseInformation (property + "_Width", Integer.toString (value.width));
      addUnparseInformation (property + "_Height", Integer.toString (value.height));
   }


   /**
    * Get the dimensionFromUnparseInformation attribute of the FSAObject object
    *
    * @param property  No description provided
    * @return          The dimensionFromUnparseInformation value
    */
   public Dimension getDimensionFromUnparseInformation (String property)
   {
      Dimension dim = null;

      String width = getUnparseInformation (property + "_Width");
      String height = getUnparseInformation (property + "_Height");
      if (width != null && height != null)
      {
         try
         {
            dim = new Dimension (Integer.parseInt (width), Integer.parseInt (height));
         }
         catch (NumberFormatException e)
         {
            e.printStackTrace();
         }
      }
      return dim;
   }


   /**
    * Initializes properties stored in qualifying ASGElement and applies
    * them to this FSAObject.
    *
    * Calls 'Template Method' applyProperties() if this FSAObject is
    * restored from ASGElement information. If this FSAObject is a new
    * one (it has no stored properties) a new location will be given to
    * this FSAObject. If properties aren't transient, this initial location
    * will be stored.
    */
   protected void initFSAProperties()
   {
      if (jComponent == null)
      {
         return;
      }

      int iPropertyInfoSize = 0;
      ASGInformation info = getASGInformation();
      if (info != null)
      {
         iPropertyInfoSize = info.sizeOfInformation();
      }

      // try to determine, whether this FSAObject is a new one, or restored...
      // we assume, it is restored, if it doesn't contain
      // any property-information

      // FSAObject is restored
      if (iPropertyInfoSize > 0)
      {
         applyProperties();

         // make sure data will be stored again
         this.setTransientProperties (false);
      }
      // FSAObject seems to be new
      else
      {
         setDefaultProperties();
      }
   }


   /**
    * This methods sets the default properties for new FSAObjects.
    */
   protected void setDefaultProperties()
   {
      if (jComponent.getX() == 0 && jComponent.getY() == 0)
      {
         initLocation();

         // store initial location
         if (!isTransientProperties())
         {
            this.saveLocation();
         }
      }
   }


   /**
    * Set the FSAObject to an initial location.
    * This is a template method that may be overwritten by a child-class.
    *
    * This basic implementation tries to set the FSAObject to
    * the last pointer-position. If no last pointer-position
    * is found, it will be set to a random location.
    */
   protected void initLocation()
   {
      // try to set new FSAObject to last pointer-position
      Point cursorPosition = UserInterfaceManager.get().getLastPointerPosition();
      if (cursorPosition != null)
      {
         jComponent.setLocation (cursorPosition);
         UserInterfaceManager.get().setLastPointerPosition (null);
      }
      // otherwise set new FSAObject to a random position
      else
      {
         // TODO: improve this code, so FSAObject won't be placed
         // under existing objects and at a free position, that is
         // in the current view of the user
         jComponent.setLocation ((int)  (Math.random() * 400), (int)  (Math.random() * 200));
      }
   }


   /**
    * Read formerly saved properties from logic and apply them to the GUI.
    *
    * Objects derived from FSAObject that save additional FSA-properties should
    * override this method, so these properties will be loaded when necessary.
    *
    * <pre>
    * NOTE: If information changes in logic, ASGInformationListener will
    * call 'applyProperties()' to adjust this graphical instance.
    * So don't override 'initFSAProperties()', because this method
    * won't be called when information changes in logic. Instead override
    * 'applyProperties()', so e.g. an 'undo'-call will result in
    * applying undo-properties to child-classes instead of just applying
    * properties to FSAObject.
    * </pre>
    *
    * @see   FSAObject#saveAdditionalFSAProperties()
    */
   protected void applyProperties()
   {
      Point location = getPointFromUnparseInformation (LOCATION);
      if (location != null)
      {
         jComponent.setLocation (location);
      }

      Dimension dimension = getDimensionFromUnparseInformation (DIMENSION);
      if (dimension != null)
      {
         jComponent.setPreferredSize (dimension);
      }
   }


   /**
    * default setting: do not save properties
    */
   public static transient boolean TRANSIENT_PROPERTIES_DEFAULT = true;

   /**
    * This flag indicates, whether properties of this FSAObject should
    * be stored in its qualifying ASGElement. If the flag is set, no
    * property information will be saved. Otherwise information like
    * position, dimension and specific object properties will be saved.
    */
   private transient boolean transientProperties = TRANSIENT_PROPERTIES_DEFAULT;


   /**
    * Get the transientProperties attribute of the FSAObject object
    *
    * @return   The transientProperties value
    */
   public boolean isTransientProperties()
   {
      return transientProperties;
   }


   /**
    * Sets the transientProperties attribute of the FSAObject object.
    * If enabled, properties (such as location)  will be persistent.
    *
    * If transientProperties are enabled, all unparse information will
    * be removed by this call.
    *
    * <pre>
    * Note that properties must be saved by an explicit call to
    * 'saveFSAProperties()' or another finegranular 'save'-call.
    * </pre>
    *
    * @param enabled  The new transientProperties value
    */
   public void setTransientProperties (boolean enabled)
   {
      // only call code, if value changes
      if (this.transientProperties == enabled)
      {
         return;
      }

      this.transientProperties = enabled;

      // remove all unparse information,
      // if this FSAObject is now transient
      if (enabled)
      {
         removeAllUnparseInformation();
      }
   }


   /**
    * Part of 'Template Method'-design pattern for saving FSA-properties.
    *
    * If properties for this FSAObject object are transient,
    * this method removes all unparse information. Otherwise it
    * calls all own primitive save-methods. Then 'saveAdditionalFSAProperties()'
    * is called, which should be implemented by a concrete FSAObject,
    * that wants to save some additional properties.
    *
    * @see   FSAObject#saveAdditionalFSAProperties()
    * @see   FSAObject#applyProperties()
    */
   public void saveFSAProperties()
   {
      if (!isTransientProperties())
      {
         saveLocation();

         // TODO: activate storing dimension of FSAObject, after introducing
         // new attribute 'autoDimension:boolean' (see: JGrab#autoAlignement)
         // and saving dimension everytime a component is resized
         // (see: ComponentBorderListener#mouseReleased(MouseEvent)
         //saveDimension();

         saveAdditionalFSAProperties();

         if (getID().equals (getPropertyName()) || getPropertyName() == null)
         {
            logger.warn ("do not use null propertyName for persistent FSAs - loss of layout may be the result!");
         }
      }
      else
      {
         removeAllUnparseInformation();
      }
   }


   /**
    * Stores location of this FSAObject in associated unparse information.
    */
   public void saveLocation()
   {
      addPointToUnparseInformation (LOCATION, getLocation());
   }


   /**
    * Stores dimension of this FSAObject in associated unparse information.
    */
   public void saveDimension()
   {
      if (getJComponent() != null)
      {
         addDimensionToUnparseInformation (DIMENSION, getJComponent().getSize());
      }
   }


   /**
    * Part of 'Template Method'-design pattern for saving FSA-properties.
    *
    * Should be implemented only by child-classes that have additional
    * properties to be saved. Will be called from within method
    * 'saveFSAProperties()'.
    *
    * Allows fine-granular saving of FSA-properties.
    *
    * @see   FSAObject#saveFSAProperties()
    * @see   FSAObject#applyProperties()
    */
   protected void saveAdditionalFSAProperties()
   {
      // empty implementation!
      // for every additional FSA-property in class FSAObject
      // introduce a new method saveXXX, like 'saveLocation())'
      // and call it from within 'saveFSAProperties()'.
   }


   /**
    * class of the AbstractUpdater that usually fits best the jComponent
    *
    * @return   The defaultUpdaterClass value
    */
   public Class getDefaultUpdaterClass()
   {
      return LogicAndFsaUpdater.class;
   }


   /**
    * creates an instance of the defaultUpdaterClass, configured with umlIncr,
    * propertyName and the default fsaAttrName.
    * </p>
    * Just add it to the updaters to activate it.
    *
    * @return   an instance of the defaultUpdaterClass ready to be used
    */
   public AbstractUpdater createDefaultUpdater()
   {
      Class updaterClass = getDefaultUpdaterClass();
      AbstractUpdater result = null;

      if (updaterClass != null)
      {
         try
         {
            result = (AbstractUpdater) updaterClass.newInstance();
         }
         catch (Exception e)
         {
            throw new RuntimeException ("Exception in " + this
               + ".createDefaultUpdater: " + e.getMessage());
         }

         result.setLogicObject (getLogic());
         result.setLogicAttrName (propertyName);
         result.setFsaAttrName (getDefaultAttrName());

         try
         {
            result.setListenerActive (true);
         }
         catch (Exception e)
         {
            log.error ("Listener not added for " + result);
         }
      }
      return result;
   }


   /**
    * the default name used in createDefaultUpdater for fsaAttrName of the
    * Updater
    *
    * @return   The defaultAttrName value
    */
   public String getDefaultAttrName()
   {
      return null;
   }


   /**
    * The name of the property in the LogicUnparseInterface that is visualized
    * by this Object's JComponent. Defaults to value of getID(). UMLMethod:
    * 'Read access method for attribute propertyName : String'
    *
    * @return   The propertyName value
    */
   public String getPropertyName()
   {
      if (propertyName == null)
      {
         setPropertyName (getID());
      }

      return propertyName;
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   final static Logger logger = Logger.getLogger (FSAObject.class);


   /**
    * The Name of the property in the LogicUnparseInterface The appropriate
    * get- and set-methods for this property are derived from the UMLIncr and
    * are used for the set/getPropertyValue methods and the current
    * propertyValue is cached for use by getPropertyValue ()
    *
    * @param propertyName  The new propertyName value
    * @return              No description provided
    * @see                 #getPropertyName
    */
   public String setPropertyName (String propertyName)
   {
      if (propertyName == null)
      {
         if (logger.isEnabledFor (Priority.WARN) && !isTransientProperties())
         {
            logger.warn ("do not use null propertyName for FSAs - loss of layout may be the result!");
         }
         propertyName = getID();
      }

      if ( (this.propertyName == null)
         || !this.propertyName.equals (propertyName))
      {
         FSAInterface iface = getFSAInterface();
         if (iface != null)
         {
            iface.removeFromFsaObjects (this);

         } // if

         this.propertyName = propertyName;

         if (iface != null)
         {
            iface.addToFsaObjects (this);
         } // if

      } // if

      return this.propertyName;
   }


   /**
    * Get the parentID attribute of the FSAObject object
    *
    * @return   The parentID value
    */
   public String getParentID()
   {
      return getID (getParent());
   }


   /**
    * Get the iD attribute of the FSAObject class
    *
    * @param object  No description provided
    * @return        The iD value
    */
   public static String getID (FSAObject object)
   {
      String id = "@__null";

      if (object != null)
      {
         id = object.getID();
      }
      return id;
   }


   /**
    * Get the qualifiedName attribute of the FSAObject object
    *
    * @return   The qualifiedName value
    */
   public String getQualifiedName()
   {
      //FIXME: for debug purposes some more work than necessary
      String name = getQualifiedName (getParent(), getPropertyName());
      if (!name.equals (currentQualifiedName))
      {
         currentQualifiedName = name;
      }
      return name;
   }


   /**
    * Get the qualifiedName attribute of the FSAObject class
    *
    * @param parent  No description provided
    * @param name    No description provided
    * @return        The qualifiedName value
    */
   public static String getQualifiedName (FSAObject parent, String name)
   {
      return getID (parent) + "." + name;
   }


   /**
    * creates a standard JComponent for this FSAObject UMLMethod: '+
    * createJComponent () : Void'
    *
    * @return   No description provided
    */
   protected abstract JComponent createJComponent();


   /**
    * UMLMethod: '+ getJComponent () : Void'
    *
    * @return   The jComponent value
    */
   public JComponent getJComponent()
   {
      return jComponent;
   }


   /**
    * Get the jComponent attribute of the FSAObject class
    *
    * @param object  No description provided
    * @return        The jComponent value
    */
   public static JComponent getJComponent (FSAObject object)
   {
      if (object == null)
      {
         return null;
      }

      return object.getJComponent();
   }


   /**
    * Get the Object in the JComponent hierarchy where the JComponent for this
    * Object should be connected to If the JComponent of this instance is
    * non-null the parent of that JComponent is returned. Otherwise the value
    * last set by setParentOfJComonent() is returned.
    *
    * @return   The parentOfJComponent value
    * @see      #setParentOfJComponent
    */
   public Container getParentOfJComponent()
   {
      return  (jComponent == null ? null : jComponent.getParent());
   }


   /**
    * Set the Object in the JComponent hierarchy that the JComponent for this
    * Object should be connected to If the JComponent is non-null, it is
    * connected to that parent. Otherwise it is stored for later use by
    * setJComponent()
    *
    * @param comp  The new parentOfJComponent value
    * @see         #getParentOfJComponent
    */
   public void setParentOfJComponent (Container comp)
   {
      Container parent = jComponent.getParent();

      if (parent != null && parent != comp)
      {
         parent.remove (jComponent);
      }

      if (comp != null && parent != comp)
      {
         comp.add (jComponent);
      }
   }


   /**
    * Get the logicFromJComponent attribute of the FSAObject class
    *
    * @param comp  No description provided
    * @return      The logicFromJComponent value
    */
   public static LogicUnparseInterface getLogicFromJComponent (JComponent comp)
   {
      FSAObject fsa = getFSAObjectFromJComponent (comp);
      if (fsa != null)
      {
         return fsa.getLogic();
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param comp  No description provided
    * @return      No description provided
    */
   public static FSAObject findFSAObjectFromJComponent (JComponent comp)
   {
      FSAObject result = null;
      while (comp != null && result == null)
      {
         result = getFSAObjectFromJComponent (comp);
         if (result == null)
         {
            Container parent = comp.getParent();
            if (parent instanceof JComponent)
            {
               comp = (JComponent) parent;
            }
            else
            {
               comp = null;
            }
         }
      }
      return result;
   }


   /**
    * Get the fSAObjectFromJComponent attribute of the FSAObject class
    *
    * @param comp  No description provided
    * @return      The fSAObjectFromJComponent value
    */
   public static FSAObject getFSAObjectFromJComponent (JComponent comp)
   {
      if (comp != null)
      {
         return (FSAObject) comp.getClientProperty (JCOMPONENT_CLIENT_PROPERTY);
      }
      return null;
   }


   /**
    * Sets the fSAObjectFromJComponent attribute of the FSAObject class
    *
    * @param jComp    The new fSAObjectFromJComponent value
    * @param fsaComp  The new fSAObjectFromJComponent value
    */
   protected static void setFSAObjectFromJComponent (JComponent jComp,
                                                     FSAObject fsaComp)
   {
      if (jComp != null)
      {
         jComp.putClientProperty (JCOMPONENT_CLIENT_PROPERTY, fsaComp);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param property  No description provided
    * @return          No description provided
    */
   public FSAObject findChildForProperty (String property)
   {
      FSAObject obj = this;
      FSAObject result = null;

      while (result == null && obj != null)
      {
         String objProp = obj.getPropertyName();
         if ( (property == null && objProp == null)
            ||  (property != null && property.equals (objProp)))
         {
            result = obj;
         }
         else
         {
            obj = obj.getParent();
         }
      }
      return result;
   }


   /**
    * @return   the highest FSALayeredPane in the hierarchy
    */
   public FSALayeredPane getDiagramRoot()
   {
      FSAObject current = this;
      FSAObject parent = getParent();

      while (parent != null)
      {
         current = parent;
         parent = current.getParent();
      }

      if (current instanceof FSALayeredPane)
      {
         return (FSALayeredPane) current;
      }
      else
      {
         return null;
      }
   }


   /**
    * get the parent of this FSAObject.
    * <p/>
    * <p/>
    * There is no setParent-Method, because the hierarchy of FSAObjects is
    * based on the hierarchy of the underlying JComponents. A path from a
    * FSAObject to its parent may spread over more than one JComponent - making
    * a setParent-Method semantically ambiguous.
    *
    * @return   the parent of this FSAObject
    */
   public FSAContainer getParent()
   {
      Component parentComponent = getParentOfJComponent();
      FSAObject parent = null;

      while (parentComponent != null && parent == null)
      {
         if (parentComponent instanceof JComponent)
         {
            Object parentCandidate =  ((JComponent) parentComponent)
               .getClientProperty (JCOMPONENT_CLIENT_PROPERTY);

            if (parentCandidate != null
               && parentCandidate instanceof FSAObject)
            {
               if (parentCandidate instanceof FSAContainer)
               {
                  parent = (FSAObject) parentCandidate;
               }
               else
               {
                  throw new RuntimeException ("FSAObject "
                     + parentCandidate
                     + " should not have children, as it is no FSAContainer!");
               }
            }
         }
         parentComponent = parentComponent.getParent();
      }

      //FIX ME: working with FSAObjects instead of FSAContainers for backward
      // compatibility
      return (FSAContainer) parent;
   }


   /**
    * Get the preferredSize attribute of the FSAObject object
    *
    * @return   The preferredSize value
    */
   public Dimension getPreferredSize()
   {
      if (jComponent == null)
      {
         return null;
      }

      return jComponent.getPreferredSize();
   }


   /**
    * Sets the preferredSize attribute of the FSAObject object
    *
    * @param newDimension  The new preferredSize value
    */
   public void setPreferredSize (Dimension newDimension)
   {
      if (getJComponent() != null)
      {
         getJComponent().setPreferredSize (newDimension);
      } // end of if ()
   }


   /**
    * Get the minimumSize attribute of the FSAObject object
    *
    * @return   The minimumSize value
    */
   public Dimension getMinimumSize()
   {
      if (jComponent == null)
      {
         return null;
      }

      return jComponent.getMinimumSize();
   }


   /**
    * Get the maximumSize attribute of the FSAObject object
    *
    * @return   The maximumSize value
    */
   public Dimension getMaximumSize()
   {
      if (jComponent == null)
      {
         return null;
      }

      return jComponent.getMaximumSize();
   }


   /**
    * Set this object selected Equals to setSelected (selected, false)
    *
    * @param selected  The new selected value
    */
   public void setSelected (boolean selected)
   {
      firePropertyChange ("selected", !selected, selected);
   }


   /**
    * @return   true, if this object is selected
    * @see      #setSelected (boolean)
    */
   public boolean isSelected()
   {
      return SelectionManager.get().hasInSelectedComponents (getJComponent());
   }


   /**
    * set this Object focused If this Object is selectable by means of the
    * SelectionManager (SelectionManager.get().isSelectable(this) returns
    * true), it is set as the focused Object in the SelectionManager and the
    * previously focused Object is unfocused. If this Object is not yet
    * selected, it is selected, too.
    *
    * @param focus  The new focused value
    * @see          de.uni_paderborn.fujaba.fsa.SelectionManager
    * @see          #setSelected (boolean)
    * @see          #isFocused ()
    */
   public void setFocused (boolean focus)
   {
      firePropertyChange ("focused", !focus, focus);
   }


   /**
    * @return   true, if this object is focused
    * @see      #setFocused (boolean)
    */
   public boolean isFocused()
   {
      return  (SelectionManager.get().getFocusedComponent() == getJComponent());
   }


   /**
    * <pre>
    * <p/>
    * <p/>
    * <p/>
    *               0..1   swingAdapter   0..1 ----------------
    *    FSAObject ----------------------------| propertyName | FSAInterface
    *               fsaObjects    fsaInterface ----------------
    * <p/>
    * <p/>
    * <p/>
    * </pre>
    */
   private FSAInterface fsaInterface;


   /**
    * @param elem  The new fSAInterface value
    */
   public void setFSAInterface (FSAInterface elem)
   {
      if (this.fsaInterface != elem)
      {
         FSAInterface oldFSAInterface = this.fsaInterface;
         // newPartner
         if (this.fsaInterface != null)
         {
            // inform old partner
            this.fsaInterface = null;

            oldFSAInterface.removeFromFsaObjects (this);
         } // if

         this.fsaInterface = elem;

         if (elem != null)
         {
            // inform new partner
            elem.addToFsaObjects (this);
         } // if

      } // if

   }


   /**
    * @return   The fSAInterface value
    */
   public FSAInterface getFSAInterface()
   {
      return this.fsaInterface;
   }


   /**
    * <pre>
    * <p/>
    * <p/>
    * <p/>
    *               0..1   swingAdapter   0..1 ----------------
    *    FSAObject ----------------------------| propertyName | LogicUnparseInterface
    *               fsaObjects         logic ----------------
    * <p/>
    * <p/>
    * <p/>
    * </pre>
    *
    * @param elem  The new logic value
    */
   public void setLogic (LogicUnparseInterface elem)
   {
      if (getLogic() != elem)
      {
         FSAInterface iface = null;
         if (elem != null)
         {
            iface = elem.getFSAInterface();
            if (iface == null)
            {
               throw new UnsupportedOperationException ("The element does not support the FSA interface");
            }
         }
         setFSAInterface (iface);
      } // if
   }


   /**
    * @return   The logic value
    */
   public LogicUnparseInterface getLogic()
   {
      if (this.fsaInterface == null)
      {
         return null;
      }

      return this.fsaInterface.getLogic();
   }


   /**
    * Get the logic attribute of the FSAObject object
    *
    * @param recursive  No description provided
    * @return           The logic value
    *
    * FIXME: recursive flag isn't used in implementation
    */
   public LogicUnparseInterface getLogic (boolean recursive)
   {
      FSAContainer container = null;
      LogicUnparseInterface result = getLogic();

      if (result == null)
      {
         container = getParent();
         if (container != null)
         {
            result = container.getLogic();
         } // end of if ()
      } // end of if ()
      return result;
   }


   /**
    * <pre>
    * <p/>
    * <p/>
    * <p/>
    *               0..1                N
    *    FSAObject ----------------------- AbstractUpdater
    *               fsaComponent     updater
    * <p/>
    * <p/>
    * <p/>
    * </pre>
    */
   private FHashSet updater;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToUpdater (AbstractUpdater value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.updater == null)
         {
            this.updater = new FHashSet();
         }
         try
         {
            changed = this.updater.add (value);
            if (changed)
            {
               value.setFsaObject (this);
            }
         }
         catch (Exception e)
         {
            log.error ("Problem during initialization of updater. Removing updater: "
               + value);
            e.printStackTrace();
            this.updater.remove (value);
            //		 value.removeYou();
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInUpdater (AbstractUpdater value)
   {
      return  ( (this.updater != null) &&  (value != null) && this.updater
         .contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfUpdater()
   {
      return  ( (this.updater == null) ? FEmptyIterator.get() : this.updater
         .iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public int sizeOfUpdater()
   {
      return  ( (this.updater == null) ? 0 : this.updater.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromUpdater (AbstractUpdater value)
   {
      boolean changed = false;

      if ( (this.updater != null) &&  (value != null))
      {
         changed = this.updater.remove (value);
         if (changed)
         {
            value.setFsaObject (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public void removeAllFromUpdater()
   {
      AbstractUpdater tmpValue;
      Iterator iter = this.iteratorOfUpdater();

      while (iter.hasNext())
      {
         tmpValue = (AbstractUpdater) iter.next();
         this.removeFromUpdater (tmpValue);
      }
   }


   /**
    * UMLMethod: '+ removeYou () : Void'
    */
   public void removeYou()
   {
      if (asgInformationListener != null)
      {
         ASGInformation info = getASGInformation (false);
         if (info != null)
         {
            info.removePropertyChangeListener (asgInformationListener);
         }
      }
      setParentOfJComponent (null);
      removeAllFromUpdater();
      setFSAInterface (null);
      setSelected (false);
      super.removeYou();
   }


   /**
    * @return   true if this is a delegated component
    */
   public boolean isDelegated()
   {
      return false;
   }

   //
   // begin parent listener section
   //

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private transient ParentNotifier parentNotifier = null;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private transient FSAContainer currentParent = null;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private transient String currentQualifiedName = null;


   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addParentListener (ParentListener listener)
   {
      if (listener != null)
      {
         if (parentNotifier == null)
         {
            parentNotifier = new ParentNotifier (this);
         }
         parentNotifier.addListener (listener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param listener  No description provided
    */
   public void removeParentListener (ParentListener listener)
   {
      if (listener != null && parentNotifier != null)
      {
         parentNotifier.removeListener (listener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param evt  No description provided
    */
   public void parentChanged (ParentEvent evt)
   {
      FSAContainer newParent = evt.getNewParent();

      String oldName = currentQualifiedName;
      currentQualifiedName = null;

      if (currentParent != newParent)
      {
         if (fsaInterface != null)
         {
            fsaInterface.updateKeyInFsaObjects (oldName, this);
         }
         currentParent = newParent;
      }
   }

   //
   // end parent listener section
   //

   //
   // begin PCL section
   //

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private transient FujabaPropertyChangeSupport propertyChangeSupport = null;


   /**
    * add propertychanngelistener for events on this fsaObject and its
    * jComponent
    * </p>
    * The events of the jComponent will have it as source, the events from the
    * fsaObject will have the fsaObject itself.
    * </p>
    * A FujabaPropertyChangeSupport is used to organize listeners and to avoid
    * redundant listener registrations
    *
    * @param listener  The object added.
    */
   public void addPropertyChangeListener (PropertyChangeListener listener)
   {
      if (propertyChangeSupport == null)
      {
         propertyChangeSupport = new FujabaPropertyChangeSupport (this);
      }

      if (jComponent != null)
      {
         Iterator iter = propertyChangeSupport.keysOfChildren();
         while (iter.hasNext())
         {
            String key = (String) iter.next();
            jComponent.removePropertyChangeListener (key, this);
         }
         if (propertyChangeSupport.sizeOfListeners() == 0)
         {
            jComponent.addPropertyChangeListener (this);
         }
      }
      propertyChangeSupport.addToListeners (listener);
   }


   /**
    * Access method for an one to n association.
    *
    * @param propertyName  The object added.
    * @param listener      The object added.
    */
   public void addPropertyChangeListener (String propertyName,
                                          PropertyChangeListener listener)
   {
      if (propertyChangeSupport == null)
      {
         propertyChangeSupport = new FujabaPropertyChangeSupport (this);
      }

      if (jComponent != null)
      {
         if (!propertyChangeSupport.hasListeners (propertyName))
         {
            jComponent.addPropertyChangeListener (propertyName, this);
         }
      }
      propertyChangeSupport.addPropertyChangeListener (propertyName, listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param listener  No description provided
    */
   public void removePropertyChangeListener (PropertyChangeListener listener)
   {
      if (propertyChangeSupport != null)
      {
         boolean globalListener = propertyChangeSupport.hasInListeners (listener);

         propertyChangeSupport.removePropertyChangeListener (listener);
         if (jComponent != null)
         {
            if (globalListener
               && propertyChangeSupport.sizeOfListeners() == 0)
            {
               jComponent.removePropertyChangeListener (this);
               Iterator iter = propertyChangeSupport.entriesOfChildren();
               while (iter.hasNext())
               {
                  Map.Entry entry = (Map.Entry) iter.next();
                  if ( ((FujabaPropertyChangeSupport) entry.getValue())
                     .sizeOfListeners() > 0)
                  {
                     jComponent.addPropertyChangeListener ((String) entry.getKey(), this);
                  }
               }
            }
            else
            {
               Iterator iter = propertyChangeSupport.entriesOfChildren();
               while (iter.hasNext())
               {
                  Map.Entry entry = (Map.Entry) iter.next();
                  if ( ((FujabaPropertyChangeSupport) entry.getValue())
                     .sizeOfListeners() == 0)
                  {
                     jComponent.removePropertyChangeListener ((String) entry.getKey(), this);
                  }
               }
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param propertyName  No description provided
    * @param listener      No description provided
    */
   public void removePropertyChangeListener (String propertyName,
                                             PropertyChangeListener listener)
   {
      if (propertyChangeSupport != null)
      {
         propertyChangeSupport.removePropertyChangeListener (propertyName,
            listener);

         if (jComponent != null)
         {
            if (!propertyChangeSupport.hasListeners (propertyName))
            {
               jComponent.removePropertyChangeListener (propertyName, this);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfPropertyChangeListeners()
   {
      return  (propertyChangeSupport == null ? FEmptyIterator.get()
         : propertyChangeSupport.iteratorOfListeners());
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param property  No description provided
    * @return          No description provided
    */
   public Iterator iteratorOfPropertyChangeListeners (String property)
   {
      return  (propertyChangeSupport == null ? FEmptyIterator.get()
         : propertyChangeSupport.iteratorOfListeners (property));
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param e  No description provided
    */
   protected void firePropertyChange (PropertyChangeEvent e)
   {
      if (propertyChangeSupport != null)
      {
         propertyChangeSupport.firePropertyChange (e);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, Object oldValue,
                                      Object newValue)
   {
      if (oldValue == newValue || propertyChangeSupport == null)
      {
         return;
      }

      propertyChangeSupport.firePropertyChange (name, oldValue, newValue);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, boolean oldValue,
                                      boolean newValue)
   {
      if (oldValue == newValue || propertyChangeSupport == null)
      {
         return;
      }

      firePropertyChange (name, Boolean.valueOf (oldValue), Boolean.valueOf (newValue));
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, int oldValue, int newValue)
   {
      if (oldValue == newValue)
      {
         return;
      }
      firePropertyChange (name, new Integer (oldValue), new Integer (newValue));
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, double oldValue,
                                      double newValue)
   {
      if (oldValue == newValue || propertyChangeSupport == null)
      {
         return;
      }
      firePropertyChange (name, new Double (oldValue), new Double (newValue));
   }


   /**
    * forward event to the FujabaPropertyChangeSupport
    *
    * @param event  No description provided
    */
   public void propertyChange (PropertyChangeEvent event)
   {
      if (event.getSource() == jComponent && propertyChangeSupport != null)
      {
         propertyChangeSupport.firePropertyChange (event);
      }
   }

   //
   // end PCL section
   //

   //
   // ----------------------------------- JComponent Listener Section
   // --------------------------------------------
   //

   //
   // begin JComponent AncestorListener section
   //

   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addAncestorListener (AncestorListener listener)
   {
      jComponent.addAncestorListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param listener  No description provided
    */
   public void removeAncestorListener (AncestorListener listener)
   {
      jComponent.removeAncestorListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfAncestorListeners()
   {
      EventListener[] listeners = jComponent.getListeners (AncestorListener.class);
      return Arrays.asList (listeners).iterator();
   }

   //
   // end JComponent AncestorListenr section
   //

   //
   // begin JComponent VetoableChangeListener section
   //

   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addVetoableChangeListener (VetoableChangeListener listener)
   {
      jComponent.addVetoableChangeListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param listener  No description provided
    */
   public void removeVetoableChangeListener (VetoableChangeListener listener)
   {
      jComponent.removeVetoableChangeListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfVetoableChangeListeners()
   {
      EventListener[] listeners = jComponent.getListeners (VetoableChangeListener.class);
      return Arrays.asList (listeners).iterator();
   }

   //
   // end JComponent VetoableChangeListener section
   //

   //
   // begin JComponent ComponentListener section
   //

   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addComponentListener (ComponentListener listener)
   {
      jComponent.addComponentListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param listener  No description provided
    */
   public void removeComponentListener (ComponentListener listener)
   {
      jComponent.removeComponentListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfComponentListeners()
   {
      EventListener[] listeners = jComponent.getListeners (ComponentListener.class);
      return Arrays.asList (listeners).iterator();
   }

   //
   // end JComponent ComponentListener section
   //

   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addFocusListener (FocusListener listener)
   {
      jComponent.addFocusListener (listener);
   }


   /**
    * No cRomment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param listener  No description provided
    */
   public void removeFocusListener (FocusListener listener)
   {
      jComponent.removeFocusListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfFocusListeners()
   {
      EventListener[] listeners = jComponent.getListeners (FocusListener.class);
      return Arrays.asList (listeners).iterator();
   }

   //
   // end JComponent FocusListener section
   //

   //
   // begin JComponent InputMethodListener section
   //

   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addInputMethodListener (InputMethodListener listener)
   {
      jComponent.addInputMethodListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param listener  No description provided
    */
   public void removeInputMethodListener (InputMethodListener listener)
   {
      jComponent.removeInputMethodListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfInputMethodListeners()
   {
      EventListener[] listeners = jComponent.getListeners (InputMethodListener.class);
      return Arrays.asList (listeners).iterator();
   }

   //
   // end JComponent InputMethodListener section
   //

   //
   // begin JComponent InputMethodListener section
   //

   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addKeyListener (KeyListener listener)
   {
      jComponent.addKeyListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param listener  No description provided
    */
   public void removeKeyListener (KeyListener listener)
   {
      jComponent.removeKeyListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfKeyListeners()
   {
      EventListener[] listeners = jComponent.getListeners (KeyListener.class);
      return Arrays.asList (listeners).iterator();
   }

   //
   // end JComponent KeyListener section
   //

   //
   // begin JComponent MouseListener section
   //
   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addMouseListener (MouseListener listener)
   {
      jComponent.addMouseListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param listener  No description provided
    */
   public void removeMouseListener (MouseListener listener)
   {
      jComponent.removeMouseListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfMouseListeners()
   {
      EventListener[] listeners = jComponent.getListeners (MouseListener.class);
      return Arrays.asList (listeners).iterator();
   }

   //
   // end JComponent MouseListener section
   //

   //
   // begin JComponent MouseMotionListener section
   //
   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addMouseMotionListener (MouseMotionListener listener)
   {
      jComponent.addMouseMotionListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param listener  No description provided
    */
   public void removeMouseMotionListener (MouseMotionListener listener)
   {
      jComponent.removeMouseMotionListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfMouseMotionListeners()
   {
      EventListener[] listeners = jComponent.getListeners (MouseMotionListener.class);
      return Arrays.asList (listeners).iterator();
   }

   //
   // end JComponent MouseMotionListener section
   //

   //
   // ------------------------------------ end JComponent Listeners section
   // ----------------------------------------
   //

   //
   // ------------------------------------ JComponent Adapter section
   // ----------------------------------------------
   //
   /**
    * Sets the visible attribute of the FSAObject object
    *
    * @param vis  The new visible value
    */
   public void setVisible (boolean vis)
   {
      this.jComponent.setVisible (vis);
   }


   /**
    * Get the visible attribute of the FSAObject object
    *
    * @return   The visible value
    */
   public boolean isVisible()
   {
      return  (this.jComponent != null && this.jComponent.isVisible());
   }


   /**
    * Sets the foreground attribute of the FSAObject object
    *
    * @param color  The new foreground value
    */
   public void setForeground (Color color)
   {
      this.jComponent.setForeground (color);
      jComponent.putClientProperty (ForegroundHighlighter.OLDCOLOR_CLIENTPROPERTY, null);
   }


   /**
    * Get the foreground attribute of the FSAObject object
    *
    * @return   The foreground value
    */
   public Color getForeground()
   {
      if (jComponent == null)
      {
         return null;
      }

      return jComponent.getForeground();
   }


   /**
    * Sets the background attribute of the FSAObject object
    *
    * @param color  The new background value
    */
   public void setBackground (Color color)
   {
      this.jComponent.setBackground (color);
   }


   /**
    * Get the background attribute of the FSAObject object
    *
    * @return   The background value
    */
   public Color getBackground()
   {
      if (jComponent == null)
      {
         return null;
      }

      return this.jComponent.getBackground();
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public void invalidate()
   {
      this.jComponent.invalidate();
   }


   /**
    * Sets the Location of the JComponent to location
    *
    * @param location  The new location value
    */
   public final void setLocation (Point location)
   {
      if (location != null)
      {
         setLocation (location.x, location.y);
      }
   }


   /**
    * Sets the Location of the JComponent to x,y
    *
    * @param x  The new location value
    * @param y  The new location value
    */
   public void setLocation (int x, int y)
   {
      Point oldLocation = getLocation();
      if (oldLocation.x != x || oldLocation.y != y)
      {
         getJComponent().setLocation (x, y);
      }
   }


   /**
    * @return   the Location of the JComponent or null if the JComponent is null
    */
   public Point getLocation()
   {
      if (jComponent == null)
      {
         return null;
      }

      return jComponent.getLocation();
   }


   /**
    * @return   the size of the JComponent or null if the JComponent is null
    */
   public Dimension getSize()
   {
      if (jComponent == null)
      {
         return null;
      }

      return jComponent.getSize();
   }


   /**
    * Get the location attribute of the FSAObject object
    *
    * @param point  No description provided
    * @return       The location value
    */
   public Point getLocation (Point point)
   {
      if (jComponent == null)
      {
         return null;
      }

      return jComponent.getLocation (point);
   }


   /**
    * Drags the JComponent to an absolute position,
    * removes the transient-flag and saves the new location.
    *
    * @param newX  No description provided
    * @param newY  No description provided
    */
   public void dragTo (int newX, int newY)
   {
      setLocation (newX, newY);

      // activate persistent storage of location
      setTransientProperties (false);

      // store new location
      saveLocation();
   }


   /**
    * Drags the JComponent by an offset of x/y,
    * removes the transient-flag and saves the new location.
    *
    * @param deltaX  No description provided
    * @param deltaY  No description provided
    */
   public void dragByDelta (int deltaX, int deltaY)
   {
      Point location = getJComponent().getLocation();

      setLocation (location.x + deltaX, location.y + deltaY);

      // activate persistent storage of location
      setTransientProperties (false);

      // store new location
      saveLocation();
   }


   /**
    * Sets the JComponent's opaque property
    *
    * @param opaque  The new opaque value
    */
   public void setOpaque (boolean opaque)
   {
      getJComponent().setOpaque (opaque);
   }

   //
   // ---------------------------------------- end JComponent adapter section
   // ---------------------------------------
   //

   /**
    * Sets the font attribute of the FSAObject object
    *
    * @param newFont  The new font value
    */
   public void setFont (Font newFont)
   {
      getJComponent().setFont (newFont);
   }


   /**
    * Get the font attribute of the FSAObject object
    *
    * @return   The font value
    */
   public Font getFont()
   {
      if (jComponent == null)
      {
         return null;
      }

      return jComponent.getFont();
   }


   /**
    * set the italic flag for this component.
    *
    * @param italic  the flag.
    */
   public void setItalic (boolean italic)
   {
      Font theFont = getFont();

      int style = theFont.getStyle();

      if (theFont.isItalic() != italic)
      {
         style +=  ( (italic) ? Font.ITALIC : -Font.ITALIC);
         theFont = theFont.deriveFont (style);
         setFont (theFont);
      } // end of if ()
   }


   /**
    * returns true if the component has set to italic.
    *
    * @return   true if the component is italic.
    */
   public boolean isItalic()
   {
      Font theFont = getFont();
      return theFont.isItalic();
   }


   /**
    * set the italic flag for this component.
    *
    * @param bold  The new bold value
    */
   public void setBold (boolean bold)
   {
      Font theFont = getFont();
      int style = theFont.getStyle();

      if (theFont.isBold() != bold)
      {
         style +=  ( (bold) ? Font.BOLD : -Font.BOLD);
         theFont = theFont.deriveFont (style);
         setFont (theFont);
      } // end of if ()
   }


   /**
    * returns true if the component has set to bold.
    *
    * @return   true if the component is bold.
    */
   public boolean isBold()
   {
      Font theFont = getFont();

      return theFont.isBold();
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private boolean underlined;


   /**
    * Get the underlined attribute of the FSAObject object
    *
    * @return   The underlined value
    */
   public boolean isUnderlined()
   {
      return underlined;
   }


   /**
    * Sets the underlined attribute of the FSAObject object
    *
    * @param newUnderlined  The new underlined value
    */
   public void setUnderlined (boolean newUnderlined)
   {
      if (underlined != newUnderlined)
      {
         underlined = newUnderlined;
      } // end of if ()
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      return getClass().getName() + "@"
         + Integer.toHexString (this.hashCode()) + "{" + jComponent + "}";
   }
}

/*
 * $Log: FSAObject.java,v $
 * Revision 1.98.2.7  2005/11/14 21:55:18  lowende
 * The collapsed property of classes is now loaded correctly again
 * and(!) the preferences options for collapsing classes works for new (especially parsed) classes now.
 *
 */
