/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.coobra.util;

import java.io.File;
import java.util.Enumeration;
import java.util.Hashtable;

import javax.swing.filechooser.FileFilter;


/**
 * A convenience implementation of FileFilter that filters out all files except for those type
 * extensions that it knows about. Extensions are of the type ".foo", which is typically found
 * on Windows and Unix boxes, but not on Macinthosh. Case is ignored. Example - create a new
 * filter that filerts out all files but gif and jpg image files: JFileChooser chooser = new
 * JFileChooser(); ExampleFileFilter filter = new ExampleFileFilter( new String{"gif", "jpg"},
 * "JPEG & GIF Images") chooser.addChoosableFileFilter(filter); chooser.showOpenDialog(this);
 *
 * @author    Jeff Dinkins
 * @version   1.12 12/03/01
 */
public class ExtensionFileFilter extends FileFilter
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Hashtable filters = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String description = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String fullDescription = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean useExtensionsInDescription = true;


   /**
    * Creates a file filter. If no filters are added, then all files are accepted.
    *
    * @see   #addExtension
    */
   public ExtensionFileFilter()
   {
      this.filters = new Hashtable();
   }


   /**
    * Creates a file filter that accepts files with the given extension. Example: new ExampleFileFilter("jpg");
    *
    * @param extension  No description provided
    * @see              #addExtension
    */
   public ExtensionFileFilter (String extension)
   {
      this (extension, null);
   }


   /**
    * Creates a file filter that accepts the given file type. Example: new ExampleFileFilter("jpg",
    * "JPEG Image Images"); Note that the "." before the extension is not needed. If provided,
    * it will be ignored.
    *
    * @param extension    No description provided
    * @param description  No description provided
    * @see                #addExtension
    */
   public ExtensionFileFilter (String extension, String description)
   {
      this();
      if (extension != null)
      {
         addExtension (extension);
      }
      if (description != null)
      {
         setDescription (description);
      }
   }


   /**
    * Creates a file filter from the given string array. Example: new ExampleFileFilter(String
    * {"gif", "jpg"}); Note that the "." before the extension is not needed adn will be ignored.
    *
    * @param filters  No description provided
    * @see            #addExtension
    */
   public ExtensionFileFilter (String[] filters)
   {
      this (filters, null);
   }


   /**
    * Creates a file filter from the given string array and description. Example: new ExampleFileFilter(String
    * {"gif", "jpg"}, "Gif and JPG Images"); Note that the "." before the extension is not
    * needed and will be ignored.
    *
    * @param filters      No description provided
    * @param description  No description provided
    * @see                #addExtension
    */
   public ExtensionFileFilter (String[] filters, String description)
   {
      this();
      for (int i = 0; i < filters.length; i++)
      {
         // add filters one by one
         addExtension (filters[i]);
      }
      if (description != null)
      {
         setDescription (description);
      }
   }


   /**
    * Return true if this file should be shown in the directory pane, false if it shouldn't.
    * Files that begin with "." are ignored.
    *
    * @param f  No description provided
    * @return   No description provided
    * @see      #getExtension
    * @see      javax.swing.filechooser.FileFilter#accept
    */
   public boolean accept (File f)
   {
      if (f != null)
      {
         if (f.isDirectory())
         {
            return true;
         }
         String extension = getExtension (f);
         if (extension != null && filters.get (extension) != null)
         {
            return true;
         }
         ;
      }
      return false;
   }


   /**
    * Return the extension portion of the file's name .
    *
    * @param f  No description provided
    * @return   The extension value
    * @see      #getExtension
    * @see      javax.swing.filechooser.FileFilter#accept
    */
   public String getExtension (File f)
   {
      if (f != null)
      {
         String filename = f.getName();
         int i = filename.lastIndexOf ('.');
         if (i > 0 && i < filename.length() - 1)
         {
            String extension = filename.substring (i + 1).toLowerCase();
            if ("gz".equals (extension))
            {
               i = filename.lastIndexOf ('.', i - 1);
               if (i > 0)
               {
                  extension = filename.substring (i + 1).toLowerCase();
               }
            }
            return extension;
         }
         ;
      }
      return null;
   }


   /**
    * Adds a filetype "dot" extension to filter against. For example: the following code will
    * create a filter that filters out all files except those that end in ".jpg" and ".tif":
    * ExampleFileFilter filter = new ExampleFileFilter(); filter.addExtension("jpg"); filter.addExtension("tif");
    * Note that the "." before the extension is not needed and will be ignored.
    *
    * @param extension  The object added.
    */
   public void addExtension (String extension)
   {
      if (filters == null)
      {
         filters = new Hashtable (5);
      }
      filters.put (extension.toLowerCase(), this);
      fullDescription = null;
   }


   /**
    * Returns the human readable description of this filter. For example: "JPEG and GIF Image
    * Files (*.jpg, *.gif)"
    *
    * @return   The description value
    * @see      #setDescription
    * @see      #setExtensionListInDescription
    * @see      #isExtensionListInDescription
    * @see      javax.swing.filechooser.FileFilter#getDescription
    */
   public String getDescription()
   {
      if (fullDescription == null)
      {
         if (description == null || isExtensionListInDescription())
         {
            fullDescription = description == null ? "(" : description + " (";
            // build the description from the extension list
            Enumeration extensions = filters.keys();
            if (extensions != null)
            {
               fullDescription += "." + (String) extensions.nextElement();
               while (extensions.hasMoreElements())
               {
                  fullDescription += ", ." + (String) extensions.nextElement();
               }
            }
            fullDescription += ")";
         }
         else
         {
            fullDescription = description;
         }
      }
      return fullDescription;
   }


   /**
    * Sets the human readable description of this filter. For example: filter.setDescription("Gif
    * and JPG Images");
    *
    * @param description  The new description value
    * @see                #setDescription
    * @see                #setExtensionListInDescription
    * @see                #isExtensionListInDescription
    */
   public void setDescription (String description)
   {
      this.description = description;
      fullDescription = null;
   }


   /**
    * Determines whether the extension list (.jpg, .gif, etc) should show up in the human readable
    * description. Only relevent if a description was provided in the constructor or using
    * setDescription();
    *
    * @param b  The new extensionListInDescription value
    * @see      #getDescription
    * @see      #setDescription
    * @see      #isExtensionListInDescription
    */
   public void setExtensionListInDescription (boolean b)
   {
      useExtensionsInDescription = b;
      fullDescription = null;
   }


   /**
    * Returns whether the extension list (.jpg, .gif, etc) should show up in the human readable
    * description. Only relevent if a description was provided in the constructor or using
    * setDescription();
    *
    * @return   The extensionListInDescription value
    * @see      #getDescription
    * @see      #setDescription
    * @see      #setExtensionListInDescription
    */
   public boolean isExtensionListInDescription()
   {
      return useExtensionsInDescription;
   }
}

/*
 * $Log: ExtensionFileFilter.java,v $
 * Revision 1.7  2004/10/20 17:49:40  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
