/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.coobra.actions;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Vector;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import de.uni_paderborn.fujaba.app.FrameMain;


/**
 * @author    $Author: lowende $
 * @version   $Revision: 1.7.2.2 $
 */
public class CheckoutVersionChooser extends JDialog
{
   /**
    * data shown in the list
    */
   Vector data;
   /**
    * list that shows the versions
    */
   JList list;
   /**
    * indicates if free selections are allowed
    */
   boolean allowFreeSelection = false;

   /**
    * listener to ensure correct selection
    */
   private ListSelectionListener listSelectionListener =
      new ListSelectionListener()
      {
         boolean inValueChanged = false;


         /**
          * Called whenever the value of the selection changes.
          *
          * @param e  the event that characterizes the change.
          */
         public synchronized void valueChanged (ListSelectionEvent e)
         {
            if (!inValueChanged && !allowFreeSelection)
            {
               try
               {
                  inValueChanged = true;
                  int[] selection = list.getSelectedIndices();
                  if (selection != null && selection.length > 0)
                  {
                     int largestSelectedIndex = selection[selection.length - 1];
                     if (data.size() > 1 && largestSelectedIndex > 0)
                     {
                        list.setSelectionInterval (1, largestSelectedIndex);
                     }
                     else
                     {
                        list.setSelectionInterval (0, 0);
                     }
                  }
                  else
                  {
                     list.setSelectionInterval (0, 0); //none
                  }
               }
               finally
               {
                  inValueChanged = false;
               }
            }
         }
      };


   /**
    * Creates a modal dialog to choose a version for checkout
    *
    * @throws HeadlessException  if GraphicsEnvironment.isHeadless() returns true
    */
   public CheckoutVersionChooser() throws HeadlessException
   {
      super (FrameMain.get(), true);
      setup();
      pack();
   }


   /**
    * set up the components
    */
   private void setup()
   {
      setTitle ("Choose version");
      setDefaultCloseOperation (WindowConstants.DO_NOTHING_ON_CLOSE);
      Container contentPane = getContentPane();
      contentPane.setLayout (new BorderLayout());
      JPanel bottom = new JPanel();
      contentPane.add (bottom, BorderLayout.SOUTH);
      data = new Vector();
      list = new JList (data);
      list.setSelectionMode (ListSelectionModel.SINGLE_INTERVAL_SELECTION);
      list.addListSelectionListener (listSelectionListener);
      contentPane.add (new JScrollPane (list), BorderLayout.CENTER);
      JButton okButton = new JButton ("OK");
      bottom.setLayout (new FlowLayout (FlowLayout.CENTER));
      bottom.add (okButton);
      getRootPane().setDefaultButton (okButton);
      okButton.addActionListener (
         new ActionListener()
         {
            /**
             * Invoked when the button was clicked.
             *
             * @param e  event
             */
            public void actionPerformed (ActionEvent e)
            {
               dispose();
            }
         });
   }


   /**
    * additionally centers the dialog
    *
    * @see   JDialog#pack()
    */
   public void pack()
   {
      super.pack();

      Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
      Dimension frameSize = this.getSize();
      if (frameSize.height > screenSize.height)
      {
         frameSize.height = screenSize.height;
      }
      if (frameSize.width > screenSize.width)
      {
         frameSize.width = screenSize.width;
      }
      this.setLocation ( (screenSize.width - frameSize.width) / 2,  (screenSize.height - frameSize.height) / 2);
   }


   /**
    * @param versions  sorted collection of versions
    */
   public void chooseFromVersions (Collection versions)
   {
      data.clear();
      data.add ("none (clear repository on next checkin)");
      data.addAll (versions);
      list.updateUI();
      if (data.size() > 1)
      {
         list.setSelectionInterval (1, data.size() - 1);
      }
      else
      {
         list.setSelectionInterval (0, 0); //none
      }
      pack();
      show(); //modal
   }


   /**
    * @return   the latest chosen version or null if none
    */
   public Object getLargestChosenVersion()
   {
      int[] selection = list.getSelectedIndices();
      if (selection != null && selection.length > 0)
      {
         int largestSelectedIndex = selection[selection.length - 1];
         if (largestSelectedIndex > 0)
         {
            return data.get (largestSelectedIndex);
         }
         else
         {
            return null;
         }
      }
      else
      {
         return null;
      }
   }


   /**
    * @return   the latest chosen version or null if none
    */
   public Object getSmallestChosenVersion()
   {
      int[] selection = list.getSelectedIndices();
      if (selection != null && selection.length > 0)
      {
         int smallestSelectedIndex = selection[0];
         if (smallestSelectedIndex > 1)
         {
            return data.get (smallestSelectedIndex - 1);
         }
         else
         {
            return null;
         }
      }
      else
      {
         return null;
      }
   }


   /**
    * Get the allowFreeSelection attribute of the CheckoutVersionChooser object
    *
    * @return   The allowFreeSelection value
    */
   public boolean isAllowFreeSelection()
   {
      return allowFreeSelection;
   }


   /**
    * Sets the allowFreeSelection attribute of the CheckoutVersionChooser object
    *
    * @param allowFreeSelection  The new allowFreeSelection value
    */
   public void setAllowFreeSelection (boolean allowFreeSelection)
   {
      this.allowFreeSelection = allowFreeSelection;
   }
}

/*
 * $Log: CheckoutVersionChooser.java,v $
 * Revision 1.7.2.2  2006/03/13 13:12:04  lowende
 * Compile warnings removed.
 *
 */
