/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.coobra;

import java.awt.event.ActionEvent;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Iterator;

import javax.swing.AbstractAction;
import javax.swing.SwingUtilities;

import org.apache.log4j.Logger;

import de.tu_bs.coobra.LocalRepository;
import de.tu_bs.coobra.ObjectChange;
import de.tu_bs.coobra.ObjectChangeCause;
import de.tu_bs.coobra.ObjectChangeStringCause;
import de.tu_bs.coobra.errors.ErrorHandlerInterface;
import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.UnifiedClassLoader;
import de.uni_paderborn.fujaba.metamodel.FProject;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.uni_paderborn.fujaba.uml.UMLProject;
import de.upb.lib.userinterface.ActionExecutionListener;
import de.upb.lib.userinterface.UserInterfaceManager;



/**
 * The change manager provides access to the CoObRA LocalRepository for the current Virtual
 * Machine. (Fujaba uses only one repository at a time)
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.32.2.7 $
 */
public class FujabaChangeManager
{
   /**
    * log4j logging
    */
   final static transient Logger log = Logger.getLogger (FujabaChangeManager.class);

   /**
    * the LocalRepository that is used by this VMs CoObRAPersistentObjects
    */
   private static LocalRepository vMRepository;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static ErrorHandlerInterface errorHandler =
      new FujabaErrorHandler();


   /**
    * Get the vMRepository attribute of the CoObRAPersistentObject class
    *
    * @return   The VMRepository
    */
   public static LocalRepository getVMRepository()
   {
      if (vMRepository == null && GeneralPreferences.get().isRepositoryActivated())
      {
         setVMRepository (new LocalRepository());
      }
      return vMRepository;
   }


   /**
    * Changes the repository for CoObRAPersistentObjects in this Virtual Machine. <br>
    * Use with care!
    *
    * @param vMRepository  the new repository for all newly created CoObRAPersistentObjects
    */
   public static void setVMRepository (LocalRepository vMRepository)
   {
      if (vMRepository != null)
      {
         vMRepository.setClassLoader (UnifiedClassLoader.get());
         LocalRepository.setErrorHandler (errorHandler);
      }
      FujabaChangeManager.vMRepository = vMRepository;
   }


   /**
    * Stores the ActionExecutionListener for pushing a cause on user action
    */
   private static ActionExecutionListener ael;


   /**
    * Adds a listener to the UserInterfaceManager to push a cause on the begin of every Action
    * tyken by the user.
    */
   public static void addActionCauseListener()
   {
      //Listener for Cause push/pop in Multiplex Action

      if (ael == null)
      {
         ael =
            new ActionExecutionListener()
            {
               /**
                * This method will be called before actionPerformed() is executed.
                *
                * @param e  The action-defined event.
                */
               public void preActionNotify (ActionEvent e)
               {
                  ObjectChangeCause cause;
                  //Clear cause stack
                  while ( (cause = ObjectChange.peekCause()) != null)
                  {
                     ObjectChange.popCause (cause);
                  }

                  //Set new cause
                  ObjectChange.pushCause (new ObjectChangeStringCause ("Action: " + e.getActionCommand()));
               }


               /**
                * This method will be called after actionPerformed was executed.
                *
                * @param e  The action-defined event.
                */
               public void postActionNotify (ActionEvent e)
               {
                  //remove any cause (pretty bad!)
                  while (ObjectChange.peekCause() != null)
                  {
                     ObjectChange.popCause (ObjectChange.peekCause());
                  }

                  //this does not belong here - please move to an own ActionExecutionListener if still needed
                  //not nice to refer those action commands here but works
                  //if ( (!"openProject".equals (e.getActionCommand())) &&
                  //    (!"saveProject".equals (e.getActionCommand())))
                  //{
                  //   UMLProject.get().setSaved (false);
                  //}
               }
            };
      }
      UserInterfaceManager.get().addExecutionListener (ael);
   }


   /**
    * Helper method to determine if a field may contain multiple items
    *
    * @param cls        Class in which the field resides
    * @param fieldName  name of the field (e.g. "position")
    * @return           true when field is a collection (has an addTo-Method)
    */
   public static boolean isCollectionField (Class cls, String fieldName)
   {
      if (cls != null && fieldName != null && fieldName.length() >= 1)
      {
         //look for an addTo-Method
         String suffix = fieldName.substring (0, 1).toUpperCase() + fieldName.substring (1);
         String addName = "addTo" + suffix;
         Method addMethod = null;
         Method[] methods = cls.getMethods();
         for (int i = 0; i < methods.length; ++i)
         {
            Method method = methods[i];
            if (method.getName().equals (addName))
            {
               Class[] paramTypes = method.getParameterTypes();
               if (paramTypes.length == 1)
               {
                  addMethod = method;
                  break;
               }
            }
         }
         return addMethod != null;
      }
      else
      {
         return false;
      }
   }


   /**
    * Get the uMLProject attribute of the FujabaChangeManager class
    *
    * @return   The uMLProject value
    */
   public static UMLProject getUMLProject()
   {
      UMLProject project = null;
      LocalRepository vmRepository = getVMRepository();
      if (vmRepository != null)
      {
         if (vmRepository.getRootObject() instanceof UMLProject)
         {
            return (UMLProject) vmRepository.getRootObject();
         }
         Iterator it = vmRepository.iteratorOfKnownObjects();
         while (it.hasNext())
         {
            Object obj = it.next();
            if (obj instanceof FProject)
            {
               project = (UMLProject) obj;
               break;
            }
         }
         vmRepository.setRootObject (project);
      }

      return project;
   }


   /**
    * @return   true when an undo or redo operation is in progress (in current thread)
    */
   public static boolean isInUndoRedo()
   {
      return ObjectChange.peekCause() != null && ObjectChange.peekCause().isUndoRedo();
   }


   /**
    * update the enabled state of undo- and redo-action
    */
   public static void updateUndoRedoActions()
   {
      final LocalRepository repository = getVMRepository();
      final AbstractAction undoAction = UserInterfaceManager.get().getFromActions ("coobra.undo");
      final AbstractAction redoAction = UserInterfaceManager.get().getFromActions ("coobra.redo");
      if (undoAction != null && redoAction != null)
      {
         SwingUtilities.invokeLater (
            new Runnable()
            {
               /**
                * @see   Thread#run()
                */
               public void run()
               {
                  if (repository != null)
                  {
                     undoAction.setEnabled (repository.isUndoPossible());
                     redoAction.setEnabled (repository.isRedoPossible());
                  }
                  else
                  {
                     undoAction.setEnabled (false);
                     redoAction.setEnabled (false);
                  }
               }
            });
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param runnable  No description provided
    */
   public static void executeTransient (Runnable runnable)
   {
      boolean transientMode = ASGElement.isInTransientMode();
      try
      {
         HashSet oldTransient = new HashSet (ASGElement.getTransientElements());
         ASGElement.setInTransientMode (true);

         runnable.run();

         HashSet newTransient = new HashSet (ASGElement.getTransientElements());
         newTransient.removeAll (oldTransient);
         Iterator iter = newTransient.iterator();
         while (iter.hasNext())
         {
            ASGElement element = (ASGElement) iter.next();
            element.removeYou();
            iter.remove();
         }
      }
      finally
      {
         ASGElement.setInTransientMode (transientMode);
      }
   }
}

/*
 * $Log: FujabaChangeManager.java,v $
 * Revision 1.32.2.7  2006/03/13 13:12:04  lowende
 * Compile warnings removed.
 *
 */
