/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen.digester.rules;

import java.beans.*;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;
import org.xml.sax.Attributes;

import de.uni_paderborn.fujaba.codegen.digester.XDigester;
import de.uni_paderborn.fujaba.codegen.digester.XRule;


/**
 * Rule implementation that sets properties on the object at the top of the stack, based on
 * attributes with corresponding names.
 *
 * @author    Craig McClanahan
 * @version   $Revision: 1.7 $ $Date: 2004/10/29 13:46:53 $
 */
public class XSetPropertiesRule extends XRule
{
   /**
    * Default constructor sets only the the associated Digester.
    *
    * @param digester  The digester with which this rule is associated
    */
   public XSetPropertiesRule (XDigester digester)
   {
      super (digester);
   }


   /**
    * Process the beginning of this element.
    *
    * @param attributes  The attribute list of this element
    * @throws Exception  Exception description not provided
    */
   public void begin (Attributes attributes) throws Exception
   {

      // Build a set of attribute names and corresponding values
      HashMap values = new HashMap();
      for (int i = 0; i < attributes.getLength(); i++)
      {
         String name = attributes.getQName (i);
         String value = attributes.getValue (i);
         values.put (name, value);
      }

      // Populate the corresponding properties of the top object
      Object top = digester.peek();
      if (digester.getDebug() >= 1)
      {
         digester.log ("Set " + top.getClass().getName() + " properties");
      }
      populate (top, values);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param target  No description provided
    * @param values  No description provided
    */
   protected static void populate (Object target, Map values)
   {
      Map propertyDescriptors = getProperties (target);
      Iterator valueIter = values.entrySet().iterator();
      while (valueIter.hasNext())
      {
         Map.Entry entry = (Map.Entry) valueIter.next();
         String property = (String) entry.getKey();
         String valueStr = (String) entry.getValue();

         PropertyDescriptor descriptor = (PropertyDescriptor) propertyDescriptors.get (property);
         setValue (target, valueStr, descriptor);
      }
   }


   /**
    * Sets the value attribute of the XSetPropertiesRule class
    *
    * @param target      The new value value
    * @param valueStr    The new value value
    * @param descriptor  The new value value
    */
   protected static void setValue (Object target, String valueStr, PropertyDescriptor descriptor)
   {
      PropertyEditor editor = getEditor (descriptor);
      if (editor != null)
      {
         editor.setAsText (valueStr);
         Object value = editor.getValue();
         Method writeMethod = descriptor.getWriteMethod();
         try
         {
            writeMethod.invoke (target, new Object[]{value});
         }
         catch (IllegalArgumentException e)
         {
         }
         catch (IllegalAccessException e)
         {
         }
         catch (InvocationTargetException e)
         {
         }
      }
   }


   /**
    * Get the editor attribute of the XSetPropertiesRule class
    *
    * @param descriptor  No description provided
    * @return            The editor value
    */
   protected static PropertyEditor getEditor (PropertyDescriptor descriptor)
   {
      PropertyEditor editor = null;
      Class customEditorClass = descriptor.getPropertyEditorClass();
      if (customEditorClass != null)
      {
         try
         {
            editor = (PropertyEditor) customEditorClass.newInstance();
         }
         catch (InstantiationException e)
         {
         }
         catch (IllegalAccessException e)
         {
         }
      }

      if (editor == null)
      {
         editor = PropertyEditorManager.findEditor (descriptor.getPropertyType());
      }

      return editor;
   }


   /**
    * Get the properties attribute of the XSetPropertiesRule class
    *
    * @param target  No description provided
    * @return        The properties value
    */
   protected static Map getProperties (Object target)
   {
      Class targetClass = target.getClass();
      HashMap propertyDescriptors = new HashMap();
      try
      {
         BeanInfo beanInfo = Introspector.getBeanInfo (targetClass);
         PropertyDescriptor[] tmpDescriptors = beanInfo.getPropertyDescriptors();
         for (int i = 0; i < tmpDescriptors.length; i++)
         {
            propertyDescriptors.put (tmpDescriptors[i].getName(), tmpDescriptors[i]);
         }
      }
      catch (IntrospectionException e)
      {
      }
      return propertyDescriptors;
   }
}

/*
 * $Log: XSetPropertiesRule.java,v $
 * Revision 1.7  2004/10/29 13:46:53  lowende
 * Some improvements for parsed code
 *
 */
