/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen.digester.rules;

import java.lang.reflect.Method;
import org.xml.sax.Attributes;

import de.uni_paderborn.fujaba.codegen.digester.XDigester;
import de.uni_paderborn.fujaba.codegen.digester.XRule;
import de.uni_paderborn.tools.fsa.FSAUtility;


/**
 * Rule implementation that calls a method on the top (parent) object, passing arguments collected
 * from subsequent <code>CallParamRule</code> rules or from the body of this element.
 *
 * @author    Craig McClanahan
 * @version   $Revision: 1.8 $ $Date: 2004/10/29 13:46:53 $
 */
public class XCallMethodRule extends XRule
{
   /**
    * Construct a "call method" rule with the specified method name. The parameter types (if
    * any) default to java.lang.String.
    *
    * @param digester    The associated Digester
    * @param methodName  Method name of the parent method to call
    * @param paramCount  The number of parameters to collect, or zero for a single argument
    *      from the body of this element.
    */
   public XCallMethodRule (XDigester digester, String methodName,
                           int paramCount)
   {
      this (digester, methodName, paramCount, (Class[]) null);
   }


   /**
    * Construct a "call method" rule with the specified method name.
    *
    * @param digester    The associated Digester
    * @param methodName  Method name of the parent method to call
    * @param paramCount  The number of parameters to collect, or zero for a single argument
    *      from the body of ths element
    * @param paramTypes  The Java class names of the arguments (if you wish to use a primitive
    *      type, specify the corresonding Java wrapper class instead, such as <code>java.lang.Boolean</code>
    *      for a <code>boolean</code> parameter)
    */
   public XCallMethodRule (XDigester digester, String methodName,
                           int paramCount, String paramTypes[])
   {

      super (digester);
      this.methodName = methodName;
      this.paramCount = paramCount;
      if (paramTypes == null)
      {
         this.paramTypes = new Class[paramCount];
         for (int i = 0; i < this.paramTypes.length; i++)
         {
            this.paramTypes[i] = "abc".getClass();
         }
      }
      else
      {
         this.paramTypes = new Class[paramTypes.length];
         for (int i = 0; i < this.paramTypes.length; i++)
         {
            try
            {
               this.paramTypes[i] = digester.getClassLoader().loadClass (paramTypes[i]);
            }
            catch (ClassNotFoundException e)
            {
               this.paramTypes[i] = null; // Will cause NPE later

            }
         }
      }

   }


   /**
    * Construct a "call method" rule with the specified method name.
    *
    * @param digester    The associated Digester
    * @param methodName  Method name of the parent method to call
    * @param paramCount  The number of parameters to collect, or zero for a single argument
    *      from the body of ths element
    * @param paramTypes  The Java classes that represent the parameter types of the method
    *      arguments (if you wish to use a primitive type, specify the corresonding Java wrapper
    *      class instead, such as <code>java.lang.Boolean.TYPE</code> for a <code>boolean</code>
    *      parameter)
    */
   public XCallMethodRule (XDigester digester, String methodName,
                           int paramCount, Class paramTypes[])
   {

      super (digester);
      this.methodName = methodName;
      this.paramCount = paramCount;
      if (paramTypes == null)
      {
         this.paramTypes = new Class[paramCount];
         for (int i = 0; i < this.paramTypes.length; i++)
         {
            this.paramTypes[i] = "abc".getClass();
         }
      }
      else
      {
         this.paramTypes = new Class[paramTypes.length];
         for (int i = 0; i < this.paramTypes.length; i++)
         {
            this.paramTypes[i] = paramTypes[i];
         }
      }

   }


   // ----------------------------------------------------- Instance Variables


   /**
    * The body text collected from this element.
    */
   protected String bodyText = null;

   /**
    * The method name to call on the parent object.
    */
   protected String methodName = null;

   /**
    * The number of parameters to collect from <code>MethodParam</code> rules. If this value
    * is zero, a single parameter will be collected from the body of this element.
    */
   protected int paramCount = 0;

   /**
    * The parameter types of the parameters to be collected.
    */
   protected Class paramTypes[] = null;


   // --------------------------------------------------------- Public Methods


   /**
    * Process the start of this element.
    *
    * @param attributes  The attribute list for this element
    * @throws Exception  Exception description not provided
    */
   public void begin (Attributes attributes) throws Exception
   {

      // Push an array to capture the parameter values if necessary
      if (paramCount > 0)
      {
         String parameters[] = new String[paramCount];
         for (int i = 0; i < parameters.length; i++)
         {
            parameters[i] = null;
         }
         digester.push (parameters);
      }

   }


   /**
    * Process the body text of this element.
    *
    * @param bodyText    The body text of this element
    * @throws Exception  Exception description not provided
    */
   public void body (String bodyText) throws Exception
   {

      if (paramCount == 0)
      {
         this.bodyText = bodyText;
      }

   }


   /**
    * Process the end of this element.
    *
    * @throws Exception  Exception description not provided
    */
   public void end() throws Exception
   {

      // Retrieve or construct the parameter values array
      String parameters[] = null;
      if (paramCount > 0)
      {
         parameters = (String[]) digester.pop();
      }
      else
      {
         parameters = new String[1];
         parameters[0] = bodyText;
         if (paramTypes.length == 0)
         {
            paramTypes = new Class[1];
            paramTypes[0] = "abc".getClass();
         }
      }

      // Construct the parameter values array we will need
      Object paramValues[] = new Object[paramTypes.length];

      for (int i = 0; i < this.paramTypes.length; i++)
      {
         if (digester.getDebug() >= 1)
         {
            digester.log (this + ".parameters[" + i + "]=" + parameters[i] + ".");
            digester.log (this + ".paramTypes[" + i + "]=" + this.paramTypes[i] + ".");
         }

         paramValues[i] =
            FSAUtility.convert (parameters[i], this.paramTypes[i]);
      }

      // Invoke the required method on the top object
      Object top = digester.peek();
      if (digester.getDebug() >= 1)
      {
         StringBuffer sb = new StringBuffer ("Call ");
         sb.append (top.getClass().getName());
         sb.append (".");
         sb.append (methodName);
         sb.append ("(");
         for (int i = 0; i < paramValues.length; i++)
         {
            if (i > 0)
            {
               sb.append (",");
            }
            if (paramValues[i] == null)
            {
               sb.append ("null");
            }
            else
            {
               sb.append (paramValues[i].toString());
            }
            sb.append ("/");
            if (paramTypes[i] == null)
            {
               sb.append ("null");
            }
            else
            {
               sb.append (paramTypes[i].getName());
            }
         }
         sb.append (")");
         digester.log (sb.toString());
      }
      Method method = top.getClass().getMethod (methodName, paramTypes);
      method.invoke (top, paramValues);

   }


   /**
    * Clean up after parsing is complete.
    *
    * @throws Exception  Exception description not provided
    */
   public void finish() throws Exception
   {

      bodyText = null;
      methodName = null;
      paramCount = 0;
      paramTypes = null;

   }

}

/*
 * $Log: XCallMethodRule.java,v $
 * Revision 1.8  2004/10/29 13:46:53  lowende
 * Some improvements for parsed code
 *
 */
