/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.*;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.preferences.JavaPreferences;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.fca.FQueue;
import de.upb.tools.fca.FTreeSet;


/**
 * Class UMLStoryPatternHandler
 *
 * @author    $Author: l3_g5 $
 * @version   $Revision: 1.86.2.4 $
 */
public class UMLStoryPatternOOHandler extends OOGenStrategyHandler
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLStoryPatternOOHandler.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected transient HashMap boundObjects = new HashMap();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient HashMap fixedLinks = new HashMap();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected transient HashMap isomorphicBinding = new HashMap();

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeSet modifiedObjects = new TreeSet();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeSet modifiedSetObjects = new TreeSet();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeSet modifiedLinks = new TreeSet();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeSet modifiedSetLinks = new TreeSet();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeSet modifiedMultiLinkObjects = new TreeSet();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeSet modifiedMultiLinkLinks = new TreeSet();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeSet multiLinkEntries = new TreeSet();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeSet atPostVariables = new TreeSet();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeSet constraints = new TreeSet();

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient TreeSet typecastObjects = new TreeSet();

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient LinkedList availableLinks = new LinkedList();

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient boolean modifiedSetItems = false;

   /**
    * Variable name used for the type cast of an object
    */
   public final static String TEMPORARY_TYPE_CAST_VAR_NAME =
      JavaPreferences.get().getInternalVariablePrefix() + "tmpTypeCastObject";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TEMPORARY_ASSIGN_VAR_PREFIX =
      JavaPreferences.get().getInternalVariablePrefix() + "Assign";


   /**
    * Default Constructor
    */
   public UMLStoryPatternOOHandler()
   {
      super();
   }


   /**
    * Get the responsible attribute of the UMLStoryPatternOOHandler object
    *
    * @param incr  No description provided
    * @return      The responsible value
    */
   public boolean isResponsible (FElement incr)
   {
      return  (incr instanceof UMLStoryPattern);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final boolean needToken()
   {
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param prevToken   No description provided
    * @param param       No description provided
    * @param asgElement  No description provided
    * @return            No description provided
    */
   public OOGenToken generateSourceCode (FElement asgElement, OOGenToken prevToken, Object param[])
   {
      UMLStoryPattern thePattern = (UMLStoryPattern) asgElement;

      if (log.isDebugEnabled())
      {
         log.debug (this + ".generateSourceCode(thePattern=" + thePattern + ")");
      }
      // Fix Me: Ensure that all items in the diagram has an instanceOf
      //         association and the type names are the same.
      try
      {
         if (thePattern.getType() != UMLStoryPattern.STORYPATTERN)
         {
            throw new RuntimeException ("Can't generate Java code for a storyboard");
         }

         OOGenStrategyClient client = (OOGenStrategyClient) getClientOfChain();

         OOGenToken top = CodeGenStrategy.checkTokenList (thePattern, prevToken, thePattern.getFirstOOGenToken(), thePattern.getLastOOGenToken());
         OOGenToken lastToken = thePattern.getLastOOGenToken();

         // delete all tokens
         top.deleteToSection (null, lastToken);

         String section = top.insertTopAndBottomToken ("StoryDiagram");
         top = top.getTopToken (section);
         top.appendStatement (generateCode (null, STORY_DIAGRAM_TOP, null));

         OOGenToken bottom = top.getBottomToken (section);
         bottom.appendStatement (generateCode (null, STORY_DIAGRAM_BOTTOM, null));

         // fill the hash table with objects and the DList with links
         top = findBoundObjectsAndAvailableLink (thePattern, top);
         modifiedSetItems =  (modifiedSetObjects.size() > 0 || modifiedSetLinks.size() > 0);

         // first, check if the object is already bound (!= null)
         // second, ensure the type cast if needed, and
         // last, add pre conditions for bound objects
         Iterator itemsIter = boundObjects.values().iterator();
         while (itemsIter.hasNext())
         {
            UMLObject object = (UMLObject) itemsIter.next();
            top = generateCodeForAttrValuePairs (top, object);
         }

         top = generateCodeForLinks (availableLinks, top, thePattern);

         // Fix Me: I generate the code for the constraint here because
         //         diagsIterator is much more work to find out the earliest position
         //         in the code where all objects that are referenced in a
         //         constraint are bound.
         //         Later some one can do in the other way.

         // constraints must not be generated into LinkSearchToSet section
         // Not very nice solution
         if (top.getNext() != null)
         {
            if ("LinkSearchToSetBottom".equals (top.getNext().getSectionName()) ||
               "LinkCheckSetBottom".equals (top.getNext().getSectionName()))
            {
               top = top.getNext();
            }
         }

         top = generateCodeForAtPostOrConstraints (top, constraints);

         // generate code for modified items (delete, create)
         // first all deleted objects then all deleted links
         // second all created objects then all created links
         top = generateCodeForModifiedItems (thePattern, top);

         // Generate code for collabstats.  If the masterCollabStat exists
         // then walk through the list of the "real" collabStats, the substats
         // of masterCollabStat.
         UMLCollabStat masterCollabStat = thePattern.getRevMasterCollabStat();

         // insert the collab stat into the master collab stat again when diagsIterator has lost its father
         // todo: remove this method after a while when bug with vanishing collab stats is fixed
         Iterator itemsIterator = thePattern.iteratorOfElements();
         while (itemsIterator.hasNext())
         {
            FElement item = (FElement) itemsIterator.next();
            if (item instanceof UMLCollabStat)
            {
               UMLCollabStat collabStat = (UMLCollabStat) item;
               if (collabStat.getFatherStat() == null && collabStat != masterCollabStat)
               {
                  if (masterCollabStat != null)
                  {
                     collabStat.setFatherStat (masterCollabStat);
                  }
                  else
                  {
                     log.error ("found collaboration statement in pattern without a master collab stat!");
                  }
               }
            }
         }

         if (masterCollabStat != null)
         {
            // When the collabstats are saved, their order seems to be messed, so reorder them.
            thePattern.renumberCollabStats();

            Enumeration enumeration = masterCollabStat.elementsOfSubStats();
            while (enumeration.hasMoreElements())
            {
               UMLCollabStat aCollabStat = (UMLCollabStat) enumeration.nextElement();
               top = client.generateSourceCodeFor (aCollabStat, top, null);
            }
         }

         if ( (thePattern.getRevStoryPattern().isForEach()))
         {
            if (thePattern.getRevStoryPattern().getFlowActivity().getForEachSeq() != null)
            {
               // Insert ForEach Flow
               section = top.insertTopAndBottomToken ("ForEachFlow");
               top = top.getTopToken (section);
               bottom = top.getBottomToken (section);

               section = top.newSection ("MethodBody_ActivityDiagram_FlowControl", bottom);
               top = top.getNextOrAppend (section, bottom);
               top.appendStatement (client.methodFlowForEachStart());
               top = thePattern.getRevStoryPattern().getFlowActivity().getForEachSeq().generateSourceCode (top, bottom);
               section = top.newSection ("MethodBody_ActivityDiagram_FlowControl", bottom);
               top = top.getNextOrAppend (section, bottom);
               top.appendStatement (client.methodFlowForEachEnd());
            }
         }
         else
         {
            top.appendStatement (client.setInternalVariables (modifiedSetItems));
         }
      }
      finally
      {
         // ensure that there are no references to the items
         boundObjects.clear();
         fixedLinks.clear();
         isomorphicBinding.clear();
         modifiedObjects.clear();
         modifiedSetObjects.clear();
         modifiedLinks.clear();
         modifiedSetLinks.clear();
         atPostVariables.clear();
         constraints.clear();
         thePattern.setMayBeConstraints (null);
         availableLinks.clear();
         modifiedMultiLinkLinks.clear();
         modifiedMultiLinkObjects.clear();
         multiLinkEntries.clear();
         typecastObjects.clear();
      }

      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected HashMap negApplTokens = new HashMap();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param availableLinks  No description provided
    * @param top             No description provided
    * @param thePattern      No description provided
    * @return                No description provided
    */
   protected OOGenToken generateCodeForLinks (LinkedList availableLinks, OOGenToken top, UMLStoryPattern thePattern)
   {
      boolean postpone = true;
      OOGenStrategyClient client = (OOGenStrategyClient) getClientOfChain();

      Iterator linkIter = availableLinks.iterator();
      while (linkIter.hasNext())
      {
         Traversable currentLink = (Traversable) linkIter.next();
         OOGenToken lastToken = null;

         if (currentLink instanceof UMLMultiLink ||
             (postpone && !postponeLinkCodeGeneration ((UMLLink) currentLink, boundObjects, isomorphicBinding)))
         {
            linkIter.remove();
            if (currentLink instanceof UMLMultiLink)
            {
               // CREATE CODE FOR MULTILINK
               top = generateCodeForMultiLink (thePattern, top, (UMLMultiLink) currentLink);
            }
            else
            {
               // CREATE CODE FOR NORMAL LINK
               UMLLink umlLink = (UMLLink) currentLink;
               final boolean negative = createNegativeLinkSearch (umlLink, boundObjects, isomorphicBinding);

               if (umlLink.getSource().isNegative() && this.boundObjects.containsValue (umlLink.getSource()))
               {
                  OOGenToken[] tmpTokens = (OOGenToken[]) this.negApplTokens.get (umlLink.getSource());
                  lastToken = top;
                  top = tmpTokens[0];

                  top.removeFromStatement (top.getLastOfStatement());
               }
               else if (umlLink.getTarget().isNegative() && this.boundObjects.containsValue (umlLink.getTarget()))
               {
                  OOGenToken[] tmpTokens = (OOGenToken[]) this.negApplTokens.get (umlLink.getTarget());
                  lastToken = top;
                  top = tmpTokens[0];

                  top.removeFromStatement (top.getLastOfStatement());
               }
               else
               {
                  lastToken = null;
               }

               OOGenToken[] tokens = generateCodeForLink (thePattern, top, umlLink);

               if (lastToken != null)
               {
                  top = lastToken;
                  tokens[0].appendStatement (client.setInternalVariables (false));
               }
               else if (negative)
               {
                  top = tokens[1];

                  tokens[0].appendStatement (client.setInternalVariables (false));

                  if (umlLink.getSource().isNegative())
                  {
                     this.negApplTokens.put (umlLink.getSource(), tokens);
                  }
                  else if (umlLink.getTarget().isNegative())
                  {
                     this.negApplTokens.put (umlLink.getTarget(), tokens);
                  }
               }
               else
               {
                  top = tokens[0];
               }
            }
            linkIter = availableLinks.iterator();
         }
         if (postpone && !linkIter.hasNext() && availableLinks.size() > 0)
         {
            postpone = false;
            linkIter = availableLinks.iterator();
         }
      }

      return top;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlLink             No description provided
    * @param boundObjects        No description provided
    * @param isomorphicBindings  No description provided
    * @return                    No description provided
    */
   protected boolean createNegativeLinkSearch (UMLLink umlLink, HashMap boundObjects, HashMap isomorphicBindings)
   {
      return  ( (umlLink.getSource().isNegative() || umlLink.getTarget().isNegative()) &&
          ( (umlLink.getPriority (boundObjects, isomorphicBindings) % UMLLink.P_OPTIONAL != UMLLink.P_CHECK &&
         umlLink.getPriority (boundObjects, isomorphicBindings) % UMLLink.P_OPTIONAL != UMLLink.P_CHECK_TO_MANY)));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlLink             No description provided
    * @param boundObjects        No description provided
    * @param isomorphicBindings  No description provided
    * @return                    No description provided
    */
   protected boolean postponeLinkCodeGeneration (UMLLink umlLink, HashMap boundObjects, HashMap isomorphicBindings)
   {
      UMLObject source = umlLink.getSource();
      UMLObject target = umlLink.getTarget();

      if (umlLink.getType() == UMLLink.NEGATIVE)
      {
         if (boundObjects.get (source.getID()) != source || boundObjects.get (target.getID()) != target)
         {
            return true;
         }
      }
      if (createNegativeLinkSearch (umlLink, boundObjects, isomorphicBindings))
      {
         if (source.isNegative())
         {
            Iterator links = source.iteratorOfRevSource();
            while (links.hasNext())
            {
               UMLLink tmpLink = (UMLLink) links.next();
               if (isNavigable (tmpLink, source))
               {
                  UMLObject tmpTarget = tmpLink.getTarget();
                  if (boundObjects.get (tmpTarget.getID()) != tmpTarget)
                  {
                     return true;
                  }
               }
            }
            links = source.iteratorOfRevTarget();
            while (links.hasNext())
            {
               UMLLink tmpLink = (UMLLink) links.next();
               if (isNavigable (tmpLink, source))
               {
                  UMLObject tmpSource = tmpLink.getSource();
                  if (boundObjects.get (tmpSource.getID()) != tmpSource)
                  {
                     return true;
                  }
               }
            }
         }
         else if (target.isNegative())
         {
            Iterator links = target.iteratorOfRevSource();
            while (links.hasNext())
            {
               UMLLink tmpLink = (UMLLink) links.next();
               if (isNavigable (tmpLink, target))
               {
                  UMLObject tmpTarget = tmpLink.getTarget();
                  if (boundObjects.get (tmpTarget.getID()) != tmpTarget)
                  {
                     return true;
                  }
               }
            }
            links = target.iteratorOfRevTarget();
            while (links.hasNext())
            {
               UMLLink tmpLink = (UMLLink) links.next();
               if (isNavigable (tmpLink, target))
               {
                  UMLObject tmpSource = tmpLink.getSource();
                  if (boundObjects.get (tmpSource.getID()) != tmpSource)
                  {
                     return true;
                  }
               }
            }
         }
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param topToken    No description provided
    * @param thePattern  No description provided
    * @return            No description provided
    */
   private OOGenToken[] insertNegativeBindingsBlock (OOGenToken topToken, UMLStoryPattern thePattern)
   {
      String section = topToken.insertTopAndBottomToken ("NegativeApplicationCondition");
      topToken = topToken.getTopToken (section);
      OOGenToken bottomToken = topToken.getBottomToken (section);

      topToken.appendStatement (OO.lineComment ("check negative application condition"));
      topToken.appendStatement (generateCode (thePattern, OOGenStrategyHandler.STORY_DIAGRAM_TOP, null));
      bottomToken.appendStatement (generateCode (thePattern, OOGenStrategyHandler.STORY_DIAGRAM_BOTTOM, null));
      bottomToken.appendStatement (OO.assignStat (OOVariable.FUJABA_SUCCESS, OO.not (OOVariable.FUJABA_SUCCESS)));
      bottomToken.appendStatement (OO.ensure (OOVariable.FUJABA_SUCCESS));
      bottomToken.appendStatement (OO.assignStat (OOVariable.FUJABA_SUCCESS, OOIdentifierExpr.FALSE_IDENTIFIER));
      bottomToken.appendStatement (OO.lineComment ("end of negative application condition"));
      bottomToken.appendStatement (OO.emptyLine());

      return new OOGenToken[]{topToken, bottomToken};
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param thePattern  No description provided
    * @param top         No description provided
    * @param umlLink     No description provided
    * @return            No description provided
    */
   protected OOGenToken[] generateCodeForLink (UMLStoryPattern thePattern, OOGenToken top, UMLLink umlLink)
   {
      OOGenStrategyClient client = (OOGenStrategyClient) getClientOfChain();
      OOGenToken continueToken = null;

      if (fixedLinks.get (umlLink.getID()) != umlLink)
      {
         fixedLinks.put (umlLink.getID(), umlLink);

         if (createNegativeLinkSearch (umlLink, boundObjects, isomorphicBinding))
         {
            OOGenToken[] tokens = insertNegativeBindingsBlock (top, thePattern);
            top = tokens[0];
            continueToken = tokens[1];
         }

         // create code for the link
         top = client.generateSourceCodeFor (umlLink, top,
            new Object[]{
            boundObjects,
            new Integer (UMLLink.SEARCH),
            Boolean.valueOf (false),
            Boolean.valueOf (thePattern.getRevStoryPattern().isForEach()),
            isomorphicBinding});

         if (umlLink.getPriority (boundObjects, isomorphicBinding) % UMLLink.P_OPTIONAL != UMLLink.P_CHECK &&
            umlLink.getPriority (boundObjects, isomorphicBinding) % UMLLink.P_OPTIONAL != UMLLink.P_CHECK_TO_MANY)
         {
            UMLObject newBoundObject = umlLink.getUnboundObject (boundObjects);
            top = generateCodeForAttrValuePairs (top, newBoundObject);

            // insert new object in boundObjects
            top = markObjectAsBound (thePattern, top, newBoundObject);

            if (newBoundObject.isSet())
            {
               // add TreeSet variable to the Activity Diagram
               UMLObject set = new UMLObject();
               set.setObjectName (newBoundObject.getObjectName());
               set.setObjectType ("TreeSet");
               thePattern.addToVariables (set);
            }
            else
            {
               thePattern.addToVariables (newBoundObject);
            }

            // insert new links and revalue the old ones
            addAvailableLinks (thePattern, newBoundObject);
         }
      }

      if (continueToken == null)
      {
         continueToken = top;
      }
      return new OOGenToken[]{top, continueToken};
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param thePattern  No description provided
    * @param top         No description provided
    * @param multiLink   No description provided
    * @return            No description provided
    */
   protected OOGenToken generateCodeForMultiLink (UMLStoryPattern thePattern, OOGenToken top, UMLMultiLink multiLink)
   {
      OOGenStrategyClient client = (OOGenStrategyClient) getClientOfChain();

      UMLObject sourceObject;
      UMLObject targetObject;

      //sourceBound is true if source is bound
      boolean sourceBound = false;
      //targetBound is true if target is bound
      boolean targetBound = false;

      boolean sourceHasCreateModifier;
      boolean targetHasCreateModifier;

      int priority = multiLink.getPriority (boundObjects, isomorphicBinding);

      switch (priority)
      {
         // P_MULTILINK_FIRST
         case UMLMultiLink.P_MULTILINK_FIRST:
            top = client.generateSourceCodeFor (multiLink, top,
               new Object[]{
               boundObjects,
               availableLinks,
               fixedLinks,
               new Integer (UMLMultiLink.MULTILINK_SEARCH),
               Boolean.valueOf (thePattern.getRevStoryPattern().isForEach())});

            targetObject = multiLink.getTargetObject();
            top = generateCodeForAttrValuePairs (top, targetObject);
            top = markObjectAsBound (thePattern, top, targetObject);
            thePattern.addToVariables (targetObject);
            addAvailableLinks (thePattern, targetObject);
            break;
         // P_MULTILINK_LAST !not implement
         case UMLMultiLink.P_MULTILINK_LAST:
            top = client.generateSourceCodeFor (multiLink, top,
               new Object[]{
               boundObjects,
               availableLinks,
               fixedLinks,
               new Integer (UMLMultiLink.MULTILINK_SEARCH),
               Boolean.valueOf (thePattern.getRevStoryPattern().isForEach())});

            sourceObject = multiLink.getTargetObject();
            top = generateCodeForAttrValuePairs (top, sourceObject);
            top = markObjectAsBound (thePattern, top, sourceObject);
            thePattern.addToVariables (sourceObject);
            addAvailableLinks (thePattern, sourceObject);
            break;
         // P_MULTILINK_BOUND_TO_UNBOUND
         case UMLMultiLink.P_MULTILINK_BOUND_TO_UNBOUND:
            targetObject = multiLink.getTargetObject();
            targetHasCreateModifier = targetObject.getModifier() == UMLObject.CREATE;

            top = client.generateSourceCodeFor (multiLink, top,
               new Object[]{
               boundObjects,
               availableLinks,
               fixedLinks,
               new Integer (UMLMultiLink.MULTILINK_SEARCH),
               Boolean.valueOf (thePattern.getRevStoryPattern().isForEach())});

            if (!targetHasCreateModifier)
            {
               top = generateCodeForAttrValuePairs (top, targetObject);
               top = markObjectAsBound (thePattern, top, targetObject);
               thePattern.addToVariables (targetObject);
               addAvailableLinks (thePattern, targetObject);
            }
            break;
         // P_MULTILINK_UNBOUND_TO_BOUND
         case UMLMultiLink.P_MULTILINK_UNBOUND_TO_BOUND:
            sourceObject = multiLink.getSourceObject();
            sourceHasCreateModifier = sourceObject.getModifier() == UMLObject.CREATE;

            top = client.generateSourceCodeFor (multiLink, top,
               new Object[]{
               boundObjects,
               availableLinks,
               fixedLinks,
               new Integer (UMLMultiLink.MULTILINK_SEARCH),
               Boolean.valueOf (thePattern.getRevStoryPattern().isForEach())});

            if (!sourceHasCreateModifier)
            {
               top = generateCodeForAttrValuePairs (top, sourceObject);
               top = markObjectAsBound (thePattern, top, sourceObject);
               thePattern.addToVariables (sourceObject);
               addAvailableLinks (thePattern, sourceObject);
            }
            break;
         // P_MULTILINK_CHECK
         case UMLMultiLink.P_MULTILINK_CHECK:
            top = client.generateSourceCodeFor (multiLink, top,
               new Object[]{
               boundObjects,
               availableLinks,
               fixedLinks,
               new Integer (UMLMultiLink.MULTILINK_SEARCH),
               Boolean.valueOf (thePattern.getRevStoryPattern().isForEach())});
            break;
         // P_MULTILINK_ENTRY
         case UMLMultiLink.P_MULTILINK_ENTRY:
            top = client.generateSourceCodeFor (multiLink, top,
               new Object[]{
               boundObjects,
               availableLinks,
               fixedLinks,
               new Integer (UMLMultiLink.MULTILINK_SEARCH),
               Boolean.valueOf (thePattern.getRevStoryPattern().isForEach())});

            sourceObject = multiLink.getSourceObject();
            targetObject = multiLink.getTargetObject();
            sourceBound = boundObjects.get (sourceObject.getID()) != null;
            targetBound = boundObjects.get (targetObject.getID()) != null;
            sourceHasCreateModifier = sourceObject.getModifier() == UMLObject.CREATE;
            targetHasCreateModifier = targetObject.getModifier() == UMLObject.CREATE;

            UMLObject boundObject = null;
            if (!sourceBound && !sourceHasCreateModifier &&  (sourceObject.getType() == UMLObject.NORM))
            {
               boundObject = sourceObject;
            }
            else if (!targetBound && !targetHasCreateModifier &&  (targetObject.getType() == UMLObject.NORM))
            {
               boundObject = targetObject;
            }

            multiLink.setPath (true);
            insertToAvailableLinks (multiLink);

            while ( (multiLink.getNextMultiLink() != null) &&
                (multiLink.getNextMultiLink().getType() != UMLMultiLink.LAST))
            {
               multiLink = multiLink.getNextMultiLink();
               multiLink.setPath (true);
               insertToAvailableLinks (multiLink);
            }

            if (boundObject != null)
            {
               top = generateCodeForAttrValuePairs (top, boundObject);
               top = markObjectAsBound (thePattern, top, boundObject);
               thePattern.addToVariables (boundObject);
               addAvailableLinks (thePattern, boundObject);
            }
            break;
         // P_MULTILINK_ENTRY_OPTIONAL
         case UMLMultiLink.P_MULTILINK_ENTRY_OPTIONAL:
            top = client.generateSourceCodeFor (multiLink, top,
               new Object[]{
               boundObjects,
               availableLinks,
               fixedLinks,
               new Integer (UMLMultiLink.MULTILINK_SEARCH),
               Boolean.valueOf (thePattern.getRevStoryPattern().isForEach())});

            sourceObject = multiLink.getSourceObject();
            targetObject = multiLink.getTargetObject();
            sourceBound = boundObjects.get (sourceObject.getID()) != null;
            targetBound = boundObjects.get (targetObject.getID()) != null;
            sourceHasCreateModifier = sourceObject.getModifier() == UMLObject.CREATE;
            targetHasCreateModifier = targetObject.getModifier() == UMLObject.CREATE;

            boundObject = null;
            if (!sourceBound && !sourceHasCreateModifier &&  ( (sourceObject.getType() == UMLObject.OPTIONAL) ||  (sourceObject.getType() == UMLObject.SET)))
            {
               boundObject = sourceObject;
            }
            else if (!targetBound && !targetHasCreateModifier &&  ( (targetObject.getType() == UMLObject.OPTIONAL) ||  (targetObject.getType() == UMLObject.SET)))
            {
               boundObject = targetObject;
            }

            multiLink.setPath (true);
            insertToAvailableLinks (multiLink);

            while ( (multiLink.getNextMultiLink() != null) &&
                (multiLink.getNextMultiLink().getType() != UMLMultiLink.LAST))
            {
               multiLink = multiLink.getNextMultiLink();
               multiLink.setBindOptionalAndSet (true);
               multiLink.setPath (true);
               insertToAvailableLinks (multiLink);
            }

            if (boundObject != null)
            {
               top = generateCodeForAttrValuePairs (top, boundObject);
               top = markObjectAsBound (thePattern, top, boundObject);
               thePattern.addToVariables (boundObject);

               if (boundObject.getType() == UMLObject.SET)
               {
                  UMLObject set = new UMLObject();
                  set.setObjectName (boundObject.getObjectName());
                  set.setObjectType ("TreeSet");
                  thePattern.addToVariables (set);
               }
               else
               {
                  addAvailableLinks (thePattern, boundObject);
               }
            }
            break;
         // P_MULTILINK_PATH
         // traversed along the multilink path and generate code for each multilink
         default:
            top = client.generateSourceCodeFor (multiLink, top,
               new Object[]{
               boundObjects,
               availableLinks,
               fixedLinks,
               new Integer (UMLMultiLink.MULTILINK_SEARCH),
               Boolean.valueOf (thePattern.getRevStoryPattern().isForEach())});

            targetObject = multiLink.getTargetObject();
            targetBound = boundObjects.get (targetObject.getID()) != null;
            targetHasCreateModifier = targetObject.getModifier() == UMLObject.CREATE;

            int type = targetObject.getType();

            if (!targetBound && !targetHasCreateModifier &&
                ( (type == UMLObject.NORM) ||  ( (type != UMLObject.NORM) &&  (multiLink.getBindOptionalAndSet() == true))))
            {
               top = generateCodeForAttrValuePairs (top, targetObject);
               top = markObjectAsBound (thePattern, top, targetObject);
               addAvailableLinks (thePattern, targetObject);
            }

            if (!targetBound && !targetHasCreateModifier &&  (type == UMLObject.SET) &&  (multiLink.getBindOptionalAndSet() == true))
            {
               UMLObject set = new UMLObject();
               set.setObjectName (targetObject.getObjectName());
               set.setObjectType ("TreeSet");
               thePattern.addToVariables (set);
            }
            else if (!targetBound && !targetHasCreateModifier &&
                ( (type == UMLObject.NORM) ||  ( (type == UMLObject.OPTIONAL) &&  (multiLink.getBindOptionalAndSet() == true))))
            {
               thePattern.addToVariables (targetObject);
            }

            if (multiLink.isEntry() &&  (multiLink.getBindOptionalAndSet() == false))
            {
               multiLink.setBindOptionalAndSet (true);
               multiLink.setPath (false);
               insertToAvailableLinks (multiLink);
            }
            break;
      }

      return top;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param thePattern  No description provided
    * @param prevToken   No description provided
    * @param object      No description provided
    * @return            No description provided
    */
   protected OOGenToken markObjectAsBound (UMLStoryPattern thePattern,
                                           OOGenToken prevToken,
                                           UMLObject object)
   {
      FTreeSet tmpContainer;

      boundObjects.put (object.getID(), object);
      UMLClass objectType = object.getInstanceOf();

      if (objectType == null)
      {
         throw new RuntimeException ("can't mark UMLObject '" + object + "' as bound: object-type is null");
      }

      String typeName = objectType.getFullClassName();
      String objectName = object.getObjectName();

      FTreeSet isomorphicBindings = getIsomorphicBindings (thePattern, typeName, objectName);
      if (isomorphicBindings != null)
      {
         Iterator iter = isomorphicBindings.iterator();

         if (iter.hasNext())
         {
            if (!object.isSet())
            {
               UMLObject tmpObject;
               int count = 0;
               OOInfixExprLeft leftExpression = null;
               OOExpression currentExpression = null;
               OOExpression tmpExpression = null;

               while (iter.hasNext())
               {
                  tmpObject = (UMLObject) iter.next();

                  if (tmpObject.getType() != UMLObject.NEGATIVE && tmpObject != object)
                  {
                     currentExpression = (OOExpression) generateCode (tmpObject,
                        ISOMORPHIC_BINDING_BODY,
                        new Object[]{
                        objectName,
                        tmpObject.getObjectName(),
                        Boolean.valueOf (tmpObject.isOptional()),
                        Boolean.valueOf (tmpObject.isSet())});
                     if (count == 0)
                     {
                        tmpExpression = currentExpression;
                     }
                     else if (count == 1)
                     {
                        leftExpression = OO.infixOp (tmpExpression, OOInfixOp.AND_OP, currentExpression);
                     }
                     else
                     {
                        leftExpression.append (OOInfixOp.AND_OP, currentExpression);
                     }
                     count++;
                  }
               }
               if (leftExpression != null || currentExpression != null)
               {
                  prevToken = prevToken.insertNewToken ("IsomorphicBinding");
                  appendOOStatement (prevToken,
                     object,
                     ISOMORPHIC_BINDING,
                     new Object[]{
                     objectName,
                     Boolean.valueOf (object.isOptional()),
                      (leftExpression != null ?
                     leftExpression :
                     currentExpression)});
               }
            }
            else
            {
               prevToken = prevToken.insertNewToken ("IsomorphicBindingSet");
               UMLObject tmpObject;
               while (iter.hasNext())
               {
                  tmpObject = (UMLObject) iter.next();

                  if (tmpObject.getType() != UMLObject.NEGATIVE && tmpObject != object)
                  {
                     // both objects must be bound.
                     if (boundObjects.get (tmpObject.getID()) != null)
                     {
                        appendOOStatement (prevToken,
                           tmpObject,
                           ISOMORPHIC_BINDING_SET,
                           new Object[]{
                           objectName,
                           tmpObject.getObjectName(),
                           Boolean.valueOf (tmpObject.isSet())});
                     }
                  }
               }
            }
         }
         // debug message for Albert
         if (log.isDebugEnabled())
         {
            log.debug ("generate code for isomorphic binding");
         }
      }
      else
      {
         tmpContainer = new FTreeSet();
         isomorphicBinding.put (typeName, tmpContainer);
      }

      addBindings (isomorphicBinding, object, objectType);

      return prevToken;
   } // markObjectAsBound


   /**
    * Get the isomorphicBindings attribute of the UMLStoryPatternOOHandler object
    *
    * @param thePattern  No description provided
    * @param typeName    No description provided
    * @param objectName  No description provided
    * @return            The isomorphicBindings value
    */
   public FTreeSet getIsomorphicBindings (UMLStoryPattern thePattern, String typeName, String objectName)
   {
      FTreeSet isomorphicBindings = null;

      FTreeSet tmpContainer = (FTreeSet) isomorphicBinding.get (typeName);
      if (tmpContainer != null)
      {
         isomorphicBindings = thePattern.filterIsomorphicBindings (objectName, tmpContainer);
      }
      return isomorphicBindings;
   }


   /**
    * Access method for a To N-association.
    *
    * @param isomorphicBinding  The object added.
    * @param object             The object added.
    * @param objectType         The object added.
    */
   protected void addBindings (Map isomorphicBinding, UMLObject object, UMLClass objectType)
   {
      addBindings (isomorphicBinding, object, objectType, true, true);
   }


   /**
    * Access method for a To N-association.
    *
    * @param isomorphicBinding  The object added.
    * @param object             The object added.
    * @param objectType         The object added.
    * @param addSuperClasses    The object added.
    * @param addSubClasses      The object added.
    */
   protected void addBindings (Map isomorphicBinding, UMLObject object, UMLClass objectType,
                               boolean addSuperClasses, boolean addSubClasses)
   {
      if (objectType != null)
      {
         String typeName = objectType.getFullClassName();
         Set tmpContainer = (Set) isomorphicBinding.get (typeName);
         if (tmpContainer == null)
         {
            tmpContainer = new FTreeSet();
            isomorphicBinding.put (typeName, tmpContainer);
         }
         tmpContainer.add (object);

         if (addSuperClasses)
         {
            Iterator superClasses = objectType.iteratorOfRevSubclass();
            while (superClasses.hasNext())
            {
               addBindings (isomorphicBinding, object,  ((UMLGeneralization) superClasses.next()).getSuperclass(),
                  true, false);
            }
         }
         if (addSubClasses)
         {
            Iterator subClasses = objectType.iteratorOfRevSuperclass();
            while (subClasses.hasNext())
            {
               addBindings (isomorphicBinding, object,  ((UMLGeneralization) subClasses.next()).getSubclass(),
                  false, true);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param thePattern  No description provided
    * @param prevToken   No description provided
    * @return            No description provided
    */
   public OOGenToken findBoundObjectsAndAvailableLink (UMLStoryPattern thePattern, OOGenToken prevToken)
   {
      UMLLink umlLink;

      if (log.isDebugEnabled())
      {
         log.debug (toString() + ":entering findBoundObjectsAndAvailableLinks.");
      }

      boundObjects.clear();
      availableLinks.clear();
      isomorphicBinding.clear();
      modifiedObjects.clear();
      modifiedSetObjects.clear();
      modifiedLinks.clear();
      modifiedSetLinks.clear();
      constraints.clear();
      thePattern.setMayBeConstraints (null);

      atPostVariables.clear();
      fixedLinks.clear();

      // look for constraints
      StringBuffer mayBeConstBuf = new StringBuffer();

      Iterator constIter = thePattern.iteratorOfConstraints();
      while (constIter.hasNext())
      {
         UMLConstraint umlConstraint = (UMLConstraint) constIter.next();
         String constText = umlConstraint.getText();

         if (log.isDebugEnabled())
         {
            log.debug (umlConstraint.toString() + ":");
         }

         if (!constText.trim().toLowerCase().startsWith ("maybe"))
         {
            if (log.isDebugEnabled())
            {
               log.debug ("\tadding to constraints");
            }
            constraints.add (umlConstraint);
         }
         else
         {
            if (log.isDebugEnabled())
            {
               log.debug ("\tanalyzing maybe constraint");
            }

            StringBuffer rawConstText = new StringBuffer (umlConstraint.getText().trim().substring (5).trim());
            StringBuffer constTextBuf = new StringBuffer();
            // delete whitespaces
            // now, the format is name=name, nothing else is excepted
            int i = 0;
            int len = rawConstText.length();
            char ch;
            boolean error = false;
            for (i = 0; i < len && !error; i++)
            {
               ch = rawConstText.charAt (i);
               // check if there are only '==' and no '='
               if ( (ch == '=') &&  (i > 0) &&  (i < len - 1) && ! ( (rawConstText.charAt (i - 1) == '=') ||  (rawConstText.charAt (i + 1) == '=')))
               {
                  error = true;
               }
               if (ch != ' ' && ch != '\t')
               {
                  constTextBuf.append (ch);
               }
            }
            // concat all maybe constraints
            if (constTextBuf.length() > 0 && !error)
            {
               mayBeConstBuf.append (constTextBuf).append (";");
            }
         }
      }

      // convert stringbuffer to string because search is not provided by a stringbuffer
      if (mayBeConstBuf.length() > 0)
      {
         thePattern.setMayBeConstraints (mayBeConstBuf.toString());
      }

      // find all bound objects
      Iterator itemsIter = thePattern.iteratorOfElements();
      while (itemsIter.hasNext())
      {
         Object element = itemsIter.next();
         if (element instanceof FDiagramItem)
         {
            FDiagramItem umlDiagramItem = (FDiagramItem) element;

            if (log.isDebugEnabled())
            {
               log.debug (umlDiagramItem.toString() + ":");
            }

            if (umlDiagramItem instanceof UMLObject)
            {
               if (log.isDebugEnabled())
               {
                  log.debug ("\tis object.");
               }

               UMLObject umlObject = (UMLObject) umlDiagramItem;
               if (umlObject.getModifier() != UMLStoryPattern.NONE)
               {
                  if (log.isDebugEnabled())
                  {
                     log.debug ("\thas modifier.");
                  }

                  if (umlObject.isSet())
                  {
                     if (log.isDebugEnabled())
                     {
                        log.debug ("\tis set. Added to modified set objects.");
                     }

                     modifiedSetObjects.add (umlObject);
                  }
                  else
                  {
                     if (log.isDebugEnabled())
                     {
                        log.debug ("\tis not set. Added to modified objects.");
                     }

                     modifiedObjects.add (umlObject);
                  }
                  if (umlObject.getModifier() == UMLStoryPattern.CREATE)
                  {
                     if (log.isDebugEnabled())
                     {
                        log.debug ("\tmodifier is create.");
                     }

                     thePattern.addToVariables (umlObject);
                  }
               }
               else
               {
                  if (log.isDebugEnabled())
                  {
                     log.debug ("\tmodifier is none.");
                  }

                  if (!umlObject.isBound())
                  {
                     if (log.isDebugEnabled())
                     {
                        log.debug ("\tobject isn't bound.");
                     }

                     if (umlObject.isSet())
                     {
                        if (log.isDebugEnabled())
                        {
                           log.debug ("\tobject is a set.");
                        }

                        UMLClass objectType = umlObject.getInstanceOf();
                        if (objectType != null)
                        {
                           addBindings (isomorphicBinding, umlObject, objectType);
                        }
                        thePattern.addToVariables (new UMLObject (umlObject.getObjectName(), "TreeSet"));
                     }
                     else
                     {
                        if (log.isDebugEnabled())
                        {
                           log.debug ("\tobject is not a set.");
                        }

                        thePattern.addToVariables (umlObject);
                     }
                  }
               }
               if (umlObject.isBound())
               {
                  if (log.isDebugEnabled())
                  {
                     log.debug ("\tobject is bound.");
                  }
                  prevToken = generateCodeForBoundObjectChecks (prevToken, umlObject);

                  prevToken = markObjectAsBound (thePattern, prevToken, (UMLObject) umlDiagramItem);
                  if (umlObject.isCheckTypeCast())
                  {
                     if (log.isDebugEnabled())
                     {
                        log.debug ("\tcheck type cast.");
                     }

                     thePattern.addToVariables (umlObject);
                  }
               }
            }
            if (umlDiagramItem instanceof UMLLink)
            {
               if (log.isDebugEnabled())
               {
                  log.debug ("\tis link.");
               }

               umlLink = (UMLLink) umlDiagramItem;

               if (umlLink.getAbsoluteModifier() != UMLStoryPattern.NONE)
               {
                  if (log.isDebugEnabled())
                  {
                     log.debug ("\thas absolute modifier.");
                  }

                  if (umlLink.getTarget().isSet() || umlLink.getSource().isSet())
                  {
                     if (log.isDebugEnabled())
                     {
                        log.debug ("\ttarget or source is set.");
                     }

                     modifiedSetLinks.add (umlLink);
                  }
                  else
                  {
                     if (log.isDebugEnabled())
                     {
                        log.debug ("\ttarget and source aren't sets.");
                     }

                     modifiedLinks.add (umlLink);
                  }
               }
            }
         }
      } // while

      // fill list with available links
      itemsIter = boundObjects.values().iterator();
      while (itemsIter.hasNext())
      {
         UMLObject umlObject = (UMLObject) itemsIter.next();

         if (log.isDebugEnabled())
         {
            log.debug (umlObject.toString() + ":adding to available links.");
         }

         addAvailableLinks (thePattern, umlObject);
      }

      if (log.isDebugEnabled())
      {
         log.debug (toString() + ":exiting findBoundObjectsAndAvailableLinks.");
      }

      return prevToken;
   } // findBoundObjectsAndAvailableLink


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param thePattern  No description provided
    * @param prevToken   No description provided
    * @return            No description provided
    */
   protected OOGenToken generateCodeForModifiedItems (UMLStoryPattern thePattern, OOGenToken prevToken)
   {
      Iterator iter;
      UMLObject object;
      UMLLink link;
      UMLMultiLink multiLink;
      int modifier;

      for (modifier = UMLStoryPattern.DELETE; modifier <= UMLStoryPattern.CREATE; modifier++)
      {
         iter = modifiedObjects.iterator();
         while (iter.hasNext())
         {
            object = (UMLObject) iter.next();
            if (object.getModifier() == modifier)
            {
               { //if (! (modifiedMultiLinkObjects.contains (object) &&  (modifier == UMLStoryPattern.CREATE)))

                  prevToken = getClientOfChain().generateSourceCodeFor (object,
                     prevToken,
                     new Object[]{
                     Boolean.valueOf (modifiedSetItems)
                     });

                  if (modifier == UMLStoryPattern.CREATE)
                  {
                     prevToken = generateCodeForAttrValuePairs (prevToken, object);
                  }
               }
            }
         }

         if (modifier == UMLStoryPattern.CREATE)
         {
            // generate code for muiltilinks
            // TODO multiLinkEntries?
            iter = multiLinkEntries.iterator();
            while (iter.hasNext())
            {
               multiLink = (UMLMultiLink) iter.next();
               while (multiLink != null)
               {
                  prevToken = getClientOfChain().generateSourceCodeFor (multiLink, prevToken,
                     new Object[]{
                     boundObjects,
                     availableLinks,
                     fixedLinks,
                     new Integer (UMLMultiLink.MULTILINK_CREATE),
                     Boolean.valueOf (false)});
                  if (multiLink.getSourceObject().getModifier() == UMLObject.CREATE)
                  {
                     prevToken = generateCodeForAttrValuePairs (prevToken, multiLink.getSourceObject());
                  }
                  if (multiLink.getTargetObject().getModifier() == UMLObject.CREATE)
                  {
                     prevToken = generateCodeForAttrValuePairs (prevToken, multiLink.getTargetObject());
                  }
                  multiLink = multiLink.getNextMultiLink();
               }
            }

            // generate code for @post variables
            prevToken = generateCodeForAtPostOrConstraints (prevToken, atPostVariables);
         }

         iter = modifiedSetObjects.iterator();
         while (iter.hasNext())
         {
            object = (UMLObject) iter.next();
            if (object.getModifier() == modifier)
            {
               prevToken = getClientOfChain().generateSourceCodeFor (object,
                  prevToken,
                  new Object[]{Boolean.valueOf (modifiedSetItems)});
            }
         }

         iter = modifiedLinks.iterator();
         while (iter.hasNext())
         {
            link = (UMLLink) iter.next();
            if (! (modifiedMultiLinkLinks.contains (link) &&  (modifier == UMLStoryPattern.CREATE)))
            {
               if (link.getAbsoluteModifier() == modifier)
               {
                  prevToken = getClientOfChain().generateSourceCodeFor (link, prevToken,
                     new Object[]{
                     boundObjects,
                     new Integer (UMLLink.MODIFY),
                     Boolean.valueOf (modifiedSetItems),
                     Boolean.valueOf (thePattern.getRevStoryPattern().isForEach()),
                     isomorphicBinding});
               }
            }
         }

         iter = modifiedSetLinks.iterator();
         while (iter.hasNext())
         {
            link = (UMLLink) iter.next();
            if (link.getAbsoluteModifier() == modifier)
            {
               prevToken = getClientOfChain().generateSourceCodeFor (link, prevToken,
                  new Object[]{
                  boundObjects,
                  new Integer (UMLLink.MODIFY),
                  Boolean.valueOf (modifiedSetItems),
                  Boolean.valueOf (thePattern.getRevStoryPattern().isForEach()),
                  isomorphicBinding});
            }
         }
      }

      return prevToken;
   } // generateCodeForModifiedItems


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param prevToken  No description provided
    * @param symTable   No description provided
    * @return           No description provided
    */
   protected OOGenToken generateCodeForAtPostOrConstraints (OOGenToken prevToken, TreeSet symTable)
   {
      Iterator iter = symTable.iterator();
      while (iter.hasNext())
      {
         UMLDiagramItem item = (UMLDiagramItem) iter.next();

         if (item instanceof FConstraint)
         {
            prevToken = getClientOfChain().generateSourceCodeFor (item, prevToken, null);
         }
         else
         {
            prevToken = getClientOfChain().generateSourceCodeFor (item,
               prevToken,
               new Object[]{
               Boolean.valueOf (modifiedSetItems)
               });
         }
      }

      return prevToken;
   } // generateCodeForAtPostOrConstraints


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param multilink  No description provided
    */
   protected void updateMultilink (UMLMultiLink multilink)
   {
      if ( (multilink != null) &&  (fixedLinks.get (multilink.getID()) == null))
      {
         boolean prevPathValue = multilink.isPath();
         boolean prevOptionalSetValue = multilink.getBindOptionalAndSet();

         multilink.setPath (false);
         multilink.setBindOptionalAndSet (false);
         int priority = multilink.getPriority (boundObjects, isomorphicBinding);

         if (priority < UMLMultiLink.P_MULTILINK_ENTRY)
         {
            int oldPos = availableLinks.indexOf (multilink);
            while (oldPos > -1)
            {
               if (oldPos > -1)
               {
                  availableLinks.remove (oldPos);
               }
               oldPos = availableLinks.indexOf (multilink);
            }

            // insert the object at a specified position
            int insertPos = 0;

            Traversable currentLink;
            ListIterator iter = availableLinks.listIterator();
            while (iter.hasNext())
            {
               currentLink = (Traversable) iter.next();
               if (currentLink.getPriority (boundObjects, isomorphicBinding) > priority)
               {
                  break;
               }
               insertPos++;
            }
            availableLinks.add (insertPos, multilink);
         }
         else
         {
            multilink.setPath (prevPathValue);
            multilink.setBindOptionalAndSet (prevOptionalSetValue);
         }
      }
      else
      {
         // remove fix multilinks in the list availableLinks
         int oldPos = availableLinks.indexOf (multilink);
         while (oldPos > -1)
         {
            if (oldPos > -1)
            {
               availableLinks.remove (oldPos);
            }
            oldPos = availableLinks.indexOf (multilink);
         }
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param thePattern  The object added.
    * @param object      The object added.
    */
   protected void addAvailableLinks (UMLStoryPattern thePattern, UMLObject object)
   {
      Enumeration linksEnum;
      UMLLink tmpLink;
      int insertPos;
      int
         priority;

      if (log.isDebugEnabled())
      {
         log.debug (toString() + ":entering addAvailableLinks.");
      }

      linksEnum = object.allLinksEnumeration();
      while (linksEnum.hasMoreElements())
      {
         tmpLink = (UMLLink) linksEnum.nextElement();

         if (log.isDebugEnabled())
         {
            log.debug (tmpLink.toString());
         }

         if (fixedLinks.get (tmpLink.getID()) == null && isNavigable (tmpLink, object))
         {
            if (log.isDebugEnabled())
            {
               log.debug ("\tfixedLinks.get(tmpLink.getID()) == null)");
            }

            priority = tmpLink.getPriority (boundObjects, isomorphicBinding);

            // an available links is a link that can be traversed
            // and is not yet in the list
            if (priority != UMLLink.P_NONE)
            {
               if (log.isDebugEnabled())
               {
                  log.debug ("\tpriority  != UMLLink.P_NONE");
               }

               int oldPos = availableLinks.indexOf (tmpLink);
               if (oldPos > -1)
               {
                  availableLinks.remove (oldPos);
                  if (log.isDebugEnabled())
                  {
                     log.debug ("\tremoved from available links");
                  }

               }
               // insert the object at a specified position
               insertPos = 0;
               Traversable currentLink;
               ListIterator iter = availableLinks.listIterator();
               while (iter.hasNext())
               {
                  currentLink = (Traversable) iter.next();

                  if (log.isDebugEnabled())
                  {
                     log.debug ("\tcurrentlink = " + currentLink.toString());
                  }

                  if (currentLink.getPriority (boundObjects, isomorphicBinding) > priority)
                  {
                     if (log.isDebugEnabled())
                     {
                        log.debug ("\t\tcurrentlink.getPriority(boundObjects) > priority --> break");
                     }
                     break;
                  }
                  insertPos++;
               }

               if (log.isDebugEnabled())
               {
                  log.debug ("\tadding " + tmpLink.toString() + " at pos " + insertPos);
               }

               availableLinks.add (insertPos, tmpLink);

            }
         }

         // update priority of multilinks
         if (isNavigable (tmpLink, object) && tmpLink.getPriority (boundObjects, isomorphicBinding) != UMLLink.P_NONE)
         {
            updateMultilink (tmpLink.getRevSourceLink());
            updateMultilink (tmpLink.getRevTargetLink());
         }
      }

      // connect MultiLinks
      UMLMultiLink tmpMultiLink;

      FQueue queue = new FQueue();
      Iterator iteratorOfMultiLinks = object.iteratorOfRevContainerObject();
      if (iteratorOfMultiLinks.hasNext())
      {
         while (iteratorOfMultiLinks.hasNext())
         {
            // delete invalidate connections to other multilinks
            tmpMultiLink = (UMLMultiLink) iteratorOfMultiLinks.next();
            tmpMultiLink.setNextMultiLink (null);
            tmpMultiLink.setPreviousMultiLink (null);

            queue.put (tmpMultiLink);
         }
         thePattern.connectMultiLinks (queue);
      }

      iteratorOfMultiLinks = object.iteratorOfRevContainerObject();
      while (iteratorOfMultiLinks.hasNext())
      {
         tmpMultiLink = (UMLMultiLink) iteratorOfMultiLinks.next();

         // insert modified links and object connected with multilinks into the list
         int sourceObjectModifier =  (tmpMultiLink.getSourceObject().getModifier());
         int targetObjectModifier =  (tmpMultiLink.getTargetObject().getModifier());

         if (sourceObjectModifier == UMLObject.CREATE)
         {
            modifiedMultiLinkObjects.add (tmpMultiLink.getSourceObject());
            modifiedMultiLinkLinks.add (tmpMultiLink.getSourceLink());
         }
         if (targetObjectModifier == UMLObject.CREATE)
         {
            modifiedMultiLinkObjects.add (tmpMultiLink.getTargetObject());
            modifiedMultiLinkLinks.add (tmpMultiLink.getTargetLink());
         }

         int sourceLinkModifier =  (tmpMultiLink.getSourceLink().getModifier());
         int targetLinkModifier =  (tmpMultiLink.getTargetLink().getModifier());

         if (sourceLinkModifier == UMLLink.CREATE)
         {
            modifiedMultiLinkLinks.add (tmpMultiLink.getSourceLink());
         }
         if (targetLinkModifier == UMLLink.CREATE)
         {
            modifiedMultiLinkLinks.add (tmpMultiLink.getTargetLink());
         }

         tmpMultiLink.update();
         if (fixedLinks.get (tmpMultiLink.getID()) == null)
         {
            priority = tmpMultiLink.getPriority (boundObjects, isomorphicBinding);

            // insert multilink entries into the list
            if ( (priority == UMLMultiLink.P_MULTILINK_ENTRY) ||  (priority == UMLMultiLink.P_MULTILINK_FIRST) ||  (priority == UMLMultiLink.P_MULTILINK_LAST))
            {
               multiLinkEntries.add (tmpMultiLink);
            }

            if (priority != UMLMultiLink.P_NONE)
            {
               int oldPos = availableLinks.indexOf (tmpMultiLink);
               if (oldPos > -1)
               {
                  availableLinks.remove (oldPos);
               }

               // insert the object at a specified position
               insertPos = 0;
               Traversable currentLink;
               ListIterator iter = availableLinks.listIterator();
               while (iter.hasNext())
               {
                  currentLink = (Traversable) iter.next();
                  if (priority < currentLink.getPriority (boundObjects, isomorphicBinding))
                  {
                     break;
                  }
                  insertPos++;
               }
               availableLinks.add (insertPos, tmpMultiLink);
            }
         }
      }
   } // addAvailableLinks


   /**
    * Get the navigable attribute of the UMLStoryPatternOOHandler object
    *
    * @param link    No description provided
    * @param source  No description provided
    * @return        The navigable value
    */
   protected boolean isNavigable (UMLLink link, UMLObject source)
   {
      return link.isNavigable (source);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param prevToken  No description provided
    * @param object     No description provided
    * @return           No description provided
    */
   protected OOGenToken generateCodeForAttrValuePairs (OOGenToken prevToken, UMLObject object)
   {
      Iterator attrExprIter = object.iteratorOfAttrs();
      UMLAttrExprPair tmpAttr;

      // Post-Attribute-Constraints
      while (attrExprIter.hasNext())
      {
         tmpAttr = (UMLAttrExprPair) attrExprIter.next();
         if (tmpAttr.getQualifier() == UMLAttrExprPair.POST)
         {
            atPostVariables.add (tmpAttr);
         }
      }

      // Pre-Attribute-Constraints
      prevToken = generateCodeForAttrValuePairs (object, prevToken);

      return prevToken;
   } // generateCodeForAttrValuePairs


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object     No description provided
    * @param prevToken  No description provided
    * @return           No description provided
    */
   protected OOGenToken generateCodeForAttrValuePairs (UMLObject object, OOGenToken prevToken)
   {
      Vector aept = new Vector();
      Vector aepf = new Vector();
      Iterator attrExprIter = object.iteratorOfAttrs();
      while (attrExprIter.hasNext())
      {
         UMLAttrExprPair umlAttrExprPair = (UMLAttrExprPair) attrExprIter.next();

         if (umlAttrExprPair.getInstanceOf() == null)
         {
            log.error ("Code for " + umlAttrExprPair + " not generated as Attribute for assertion was removed!");
         }

         if (umlAttrExprPair.getQualifier() != UMLAttrExprPair.POST && umlAttrExprPair.getInstanceOf() != null)
         {
            OOExpression constr = (OOExpression) generateCode (umlAttrExprPair, UML_ATTR_EXPR_PAIR,
               new Object[]{Boolean.valueOf (umlAttrExprPair.getRevAttrs().isSet())});

            if (umlAttrExprPair.isAssertInUnitTest() && object.isBound())
            {
               aept.addElement (constr);
            }
            else
            {
               aepf.addElement (constr);
            }
         }
      }

      OOExpression[] umlAttrExprPairs = OOExpression.toArray (aept);
      if (umlAttrExprPairs.length > 0)
      {
         prevToken = prevToken.insertNewToken ("PreconditionCheck");
         appendOOStatement (prevToken,
            object,
            PRECONDITION_CHECK,
            new Object[]{
            object.getObjectName(),
            object.getObjectType(),
            Boolean.valueOf (object.isBound()),
            Boolean.valueOf (object.isOptional()),
            Boolean.valueOf (object.isSet()),
            Boolean.valueOf (object.isNegative()),
            Boolean.valueOf (true),
            umlAttrExprPairs});
      }

      umlAttrExprPairs = OOExpression.toArray (aepf);
      if (umlAttrExprPairs.length > 0)
      {
         if (aept.size() == 0)
         {
            prevToken = prevToken.insertNewToken ("PreconditionCheck");
         }
         appendOOStatement (prevToken,
            object,
            PRECONDITION_CHECK,
            new Object[]{
            object.getObjectName(),
            object.getObjectType(),
            Boolean.valueOf (object.isBound()),
            Boolean.valueOf (object.isOptional()),
            Boolean.valueOf (object.isSet()),
            Boolean.valueOf (object.isNegative()),
            Boolean.valueOf (false),
            umlAttrExprPairs});
      }

      if (log.isDebugEnabled())
      {
         log.error ("generateCodeForAttrValuePairs(" + object + ")=" + aept + aepf);
      }
      return prevToken;
   } // generateCodeForAttrValuePairs


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param prevToken  No description provided
    * @param object     No description provided
    * @return           No description provided
    */
   protected OOGenToken generateCodeForBoundObjectChecks (OOGenToken prevToken, UMLObject object)
   {
      OOGenStrategyClient client = (OOGenStrategyClient) getClientOfChain();

      // ensure that the object is already bound in cases where objectName != "this" or "super"
      if (!object.getObjectName().equals ("this") && !object.getObjectName().equals ("super"))
      {
         prevToken = prevToken.insertNewToken ("BoundObjectsCheck");
         if (object.isCheckTypeCast() ||  (object.getTypeCastSource() != null && !"".equals (object.getTypeCastSource().trim())))
         {
            appendOOStatement (prevToken, OO.varDecl ("Object", TEMPORARY_TYPE_CAST_VAR_NAME, new OOStringExpr (object.getTypeCastSource())));
//            instanceof already ensures not null
//            appendOOStatement (prevToken,
//               client.constraintTextWithComment (OO.notNullExpr (TEMPORARY_TYPE_CAST_VAR_NAME),
//               "check object " + object.getObjectName() + " is really bound", object.isAssertInUnitTest()));
            OOExpression typeCondition = new OOObjectOfTypeExpr (TEMPORARY_TYPE_CAST_VAR_NAME,
               object.getObjectType());
            if (object.isOptional())
            {
               OOExpression nullCondition = OO.isNullExpr (TEMPORARY_TYPE_CAST_VAR_NAME);
               typeCondition = OO.infixOp (nullCondition, OOInfixOp.OR_OP, typeCondition);
            }
            appendOOStatement (prevToken, client.constraintTextWithComment (typeCondition,
               "ensure correct type and really bound", object.isAssertInUnitTest()));
            appendOOStatement (prevToken, object, TYPE_CAST, null);
            typecastObjects.add (object);
         }
         else
         {
            if (!object.isOptional())
            {
               appendOOStatement (prevToken, client.constraintTextWithComment (OO.notNullExpr
                   (object.getObjectName()), "check object " + object.getObjectName() + " is really bound", object.isAssertInUnitTest()));
            }
         }
      }

      return prevToken;
   } // generateCodeForBoundObjectChecks


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param link  No description provided
    */
   protected void insertToAvailableLinks (Traversable link)
   {
      // insert the object at a specified position
      int insertPos = 0;
      int priority = link.getPriority (boundObjects, isomorphicBinding);
      Traversable currentLink;
      ListIterator iter = availableLinks.listIterator();
      while (iter.hasNext())
      {
         currentLink = (Traversable) iter.next();
         if (priority < currentLink.getPriority (boundObjects, isomorphicBinding))
         {
            break;
         }
         insertPos++;
      }
      availableLinks.add (insertPos, link);
   } // insertToAvailableLinks


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "UMLStoryPatternOOHandler[]";
   }
}

/*
 * $Log: UMLStoryPatternOOHandler.java,v $
 * Revision 1.86.2.4  2005/11/18 10:57:48  l3_g5
 * attribute assertions for multilinks work
 *
 */
