/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.*;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.Utility;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.sdm.*;


/**
 * Class UMLStatechartOOHandler
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.38.2.3 $
 */
public class UMLStatechartOOHandler extends OOGenStrategyHandler
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLStatechartOOHandler.class);


   /**
    * Default Constructor
    */
   public UMLStatechartOOHandler()
   {
      super();
   }


   /**
    * Get the responsible attribute of the UMLStatechartOOHandler object
    *
    * @param incr  No description provided
    * @return      The responsible value
    */
   public boolean isResponsible (FElement incr)
   {
      //return  (incr instanceof UMLStatechart);
      return  (incr instanceof UMLActivityDiagram &&  ((UMLActivityDiagram) incr).isStatechart());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final boolean needToken()
   {
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static int methodNumber;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String reactiveObjectName;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FProject project;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLActivityDiagram activityDiagram;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr       No description provided
    * @param prevToken  No description provided
    * @param param      No description provided
    * @return           No description provided
    */
   public OOGenToken generateSourceCode (
                                         FElement incr,
                                         OOGenToken prevToken,
                                         Object param[])
   {
      activityDiagram = (UMLActivityDiagram) incr;
      if (log.isDebugEnabled())
      {
         log.debug (
            this
            + ".generateSourceCode(statechart = "
            + activityDiagram
            + ","
            + prevToken
            + ")");
      }
      handleModelElements (activityDiagram, false);
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diagram  No description provided
    */
   public void removeGeneratedElements (UMLActivityDiagram diagram)
   {
      handleModelElements (diagram, true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diagram  No description provided
    * @param remove   No description provided
    */
   protected void handleModelElements (UMLActivityDiagram diagram, boolean remove)
   {
      activityDiagram = diagram;
      reactiveObjectName = "myFReactive";
      project = UMLProject.get();

      OOGenStrategyClient client = (OOGenStrategyClient) getClientOfChain();
      OOGenVisitor oOGenVisitor = client.getCurrentOOGenVisitor();

      // This counter is used to distinguish the generated methods for the
      // actions and guards. It should not be possible to generate
      // different methods with the same name.
      methodNumber = 0;

      //I think this method is doing nothing.
      //activityDiagram.searchForBoundStoryObjects (activityDiagram);

      boolean createdFEvent = false;
      boolean createdFReactive = false;
      boolean createdFReactiveAttr = false;
      FClass fEventClass = null;
      FClass fReactiveClass = null;
      FAttr myFReactiveAttr = null;

      ASGElement.setInTransientMode (true);
      try
      {
         fEventClass = project.findFReference (FEvent.class.getName(), false, false);
         if (fEventClass == null)
         {
            fEventClass = project.findFReference (FEvent.class.getName(), true, true);
            createdFEvent = true;
         }
         project.getFTypeList().addToTypes (fEventClass);

         fReactiveClass = project.findFReference (FReactive.class.getName(), false, false);
         if (fReactiveClass == null)
         {
            fReactiveClass = project.findFReference (FReactive.class.getName(), true, true);
            createdFReactive = true;
         }
         project.getFTypeList().addToTypes (fReactiveClass);

         FClass classOfReactiveObject =
            activityDiagram.getStartActivity().getRevStartOfStateChart();
         if (classOfReactiveObject == null
         /*
          *  old projects
          */
            )
         {
            classOfReactiveObject = activityDiagram.getStartActivity().getSpecClass();
         }
         /*
          *  FIXME: Is it okay now to not import the package manually?
          *  classOfReactiveObject.getFile().addToImportedPackages (
          *  UMLProject.get().getNewFromPackages ("de.upb.tools.sdm"));
          */
         // create private attribute myFReactive
         myFReactiveAttr = classOfReactiveObject.getFromFAttrs ("myFReactive");
         if (myFReactiveAttr == null)
         {
            myFReactiveAttr =
               new UMLAttr (
               FDeclaration.PRIVATE,
               FReactive.class.getName(),
               "myFReactive",
               "null",
               false,
               false,
               (UMLClass) classOfReactiveObject);
            myFReactiveAttr.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);

            getClientOfChain().generateSourceCodeFor (myFReactiveAttr, null, null);

            createdFReactiveAttr = true;
         }

         FMethod constructor = createConstructor (classOfReactiveObject);
         handleMethod (activityDiagram, classOfReactiveObject, constructor, false, remove);
         constructor.removeYou();

         Set methods = new TreeSet();
         Set complexStateMethods = new TreeSet();

         methods.add (createWaitForResultForEvent());
         methods.add (createNotifyMe());
         methods.add (createAlwaysTrue());

         methods.add (createInitStatechart (oOGenVisitor, methods, complexStateMethods));

         Iterator methodIter = methods.iterator();
         while (methodIter.hasNext())
         {
            FMethod method = (FMethod) methodIter.next();
            handleMethod (activityDiagram, classOfReactiveObject, method, false, remove);
         }
         methodIter = complexStateMethods.iterator();
         while (methodIter.hasNext())
         {
            FMethod method = (FMethod) methodIter.next();
            handleMethod (activityDiagram, classOfReactiveObject, method, true, remove);
         }
      }
      finally
      {
         ASGElement.setInTransientMode (false);
         if (createdFEvent)
         {
            fEventClass.removeYou();
         }
         if (createdFReactive)
         {
            fReactiveClass.removeYou();
         }
         if (createdFReactiveAttr ||
             (remove && myFReactiveAttr != null &&
             (myFReactiveAttr.isGenerated() ||
            myFReactiveAttr.getDisplayLevel() <= FDeclaration.CODE_DISPLAY_LEVEL)))
         {
            myFReactiveAttr.removeYou();
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diagram        No description provided
    * @param target         No description provided
    * @param umlMethod      No description provided
    * @param complexMethod  No description provided
    * @param remove         No description provided
    */
   private void handleMethod (UMLActivityDiagram diagram, FClass target, FMethod umlMethod,
                              boolean complexMethod, boolean remove)
   {
      boolean old = false;
      FMethod existingMethod = target.getFromFMethods (umlMethod.getFullMethodName());
      if (existingMethod != null)
      {
         //FIXME: Always removing is a violation of the premise that the code generation
         //doesn't change the model, however it removes the need of an explicit user action
         if (existingMethod.isGenerated() || existingMethod.getDisplayLevel() <= FDeclaration.CODE_DISPLAY_LEVEL)
         {
            if (complexMethod)
            {
               isolateStory (existingMethod);
            }
            existingMethod.removeYou();
         }
         else
         {
            log.error ("Won't delete " + existingMethod.getFParent().getFullClassName() + "." +
               existingMethod.getName() + " because it seems to have been created manually");
            old = true;
         }
      }
      if (!remove)
      {
         if (!old)
         {
            getClientOfChain().generateSourceCodeFor (umlMethod, null, null);
         }
         else
         {
            getClientOfChain().generateSourceCodeFor (existingMethod, null, null);
            //mark this method to exclude it from normal method code generation by
            //associating it with its accessed element
            StatechartMethodRef.createAssociation (existingMethod, diagram);
         }
      }
      if (complexMethod)
      {
         isolateStory (umlMethod);
      }
      umlMethod.removeYou();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param complexStateMethod  No description provided
    */
   private void isolateStory (FMethod complexStateMethod)
   {
      UMLActivityDiagram diagram = (UMLActivityDiagram) complexStateMethod.getFStoryDiagram();
      if (diagram != null)
      {
         Iterator elements = diagram.iteratorOfElements();
         while (elements.hasNext())
         {
            FElement element = (FElement) elements.next();
            if (element instanceof UMLStoryActivity)
            {
               UMLStoryActivity story = (UMLStoryActivity) element;

               diagram.removeFromElements (story);
               story.removeAllFromEntry();
               story.removeAllFromExit();
               story.deleteTokens();
               story.getStoryPattern().deleteTokens();
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param oOGenVisitor         No description provided
    * @param methods              No description provided
    * @param complexStateMethods  No description provided
    * @return                     No description provided
    */
   private FMethod createInitStatechart (OOGenVisitor oOGenVisitor, Set methods, Set complexStateMethods)
   {
      OOCallMethodExpr tmpCallMethodExpr;
      OOVariable tmpObjectName;
      OOMethod tmpMethod;

      // initialize myFReactive
      // The myFReactive init stuff must be the first one in initBuf
      // because all other statements depend on it.
      OOGenToken initToken = new OOGenToken();
      initToken.setSectionName ("UMLStatechart_Method");
      OOGenToken bodyToken = new OOGenToken();
      bodyToken.setSectionName ("UMLStatechart_Method");

      OOVariable myFReactiveObjectVar = OO.variable (reactiveObjectName);
      OONewObjectExpr tmpExpression =
         OO.newObject (project.getFTypeList().getFromFTypes (FReactive.class.getName()), null);
      OOStatement tmpAssignment =
         OO.assignStat (myFReactiveObjectVar, tmpExpression);

      tmpObjectName = OO.variable (reactiveObjectName);
      tmpMethod = OO.method ("Handler", OOMethodType.SET_METHOD);
      OOExpression tmpParameters = OOIdentifierExpr.THIS_IDENTIFIER;
      tmpCallMethodExpr = OO.call (tmpObjectName, tmpMethod, tmpParameters);

      initToken.addToStatement (tmpAssignment);
      initToken.addToStatement (new OOExprStatement (tmpCallMethodExpr));

      tmpObjectName = OO.variable (reactiveObjectName);
      tmpMethod = OO.method ("Current", OOMethodType.SET_METHOD);
      OOStringExpr tmpParameters2 =
         new OOStringExpr (activityDiagram.getName());
      tmpCallMethodExpr = OO.call (tmpObjectName, tmpMethod, tmpParameters2);
      bodyToken.addToStatement (new OOExprStatement (tmpCallMethodExpr));

      // Generate the state that contains all the rest. This master state
      // has no actions.
      tmpMethod = OO.method ("makeComplexState");
      tmpParameters2 =
         new OOStringExpr (
         "\""
         + activityDiagram.getName()
         + "\", "
         + OOIdentifierExpr.NULL_IDENTIFIER.getSourceCode (
         oOGenVisitor)
         + ", "
         + OOIdentifierExpr.NULL_IDENTIFIER.getSourceCode (
         oOGenVisitor)
         + ", "
         + OOIdentifierExpr.NULL_IDENTIFIER.getSourceCode (
         oOGenVisitor));
      tmpCallMethodExpr = OO.call (tmpObjectName, tmpMethod, tmpParameters2);
      tmpAssignment =
         OO.assignStat (
         "FComplexState " + activityDiagram.getName(),
         tmpCallMethodExpr);
      initToken.addToStatement (tmpAssignment);

      generateSourceCodeForStateChart (initToken, bodyToken, methods, complexStateMethods);

      // now some code to make the statemachine alive
      tmpObjectName = OO.variable (reactiveObjectName);
      tmpMethod = OO.method ("enqueueEvent");
      OOStringExpr tmpParameterForNewFEventObject =
         new OOStringExpr ("FReactive.INIT_EVENT_PREFIX + \"A\"");
      OONewObjectExpr newFEventObject =
         OO.newObject (
         project.getFTypeList().getFromFTypes (FEvent.class.getName()),
         tmpParameterForNewFEventObject);
      tmpParameters2 =
         new OOStringExpr (
         newFEventObject.getSourceCode (oOGenVisitor)
         + ", FReactive.INTERNAL_EVENT");
      tmpCallMethodExpr = OO.call (tmpObjectName, tmpMethod, tmpParameters2);
      bodyToken.addToStatement (new OOExprStatement (tmpCallMethodExpr));

      if (! (activityDiagram instanceof UMLStatechart) ||  ((UMLStatechart) activityDiagram).isSpawningOwnThread())
      {
         tmpMethod = OO.method ("start");
         tmpCallMethodExpr = OO.call (tmpObjectName, tmpMethod);
         bodyToken.addToStatement (new OOExprStatement (tmpCallMethodExpr));
      }

      // create method initStateChart
      FMethod initStatechartMethod = new UMLMethod ("initStatechart");
      initStatechartMethod.setGenerated (true);
      initStatechartMethod.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
      initStatechartMethod.setResultType (UMLProject.get().getFromBaseTypes (FBaseTypes.VOID));
      initStatechartMethod.setMethodBody (initToken.getSourceCode() + bodyToken.getSourceCode(), true);

      return initStatechartMethod;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private FMethod createAlwaysTrue()
   {
      // create method alwaysTrue
      FMethod alwaysTrue = new UMLMethod ("alwaysTrue");
      alwaysTrue.setGenerated (true);
      alwaysTrue.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
      alwaysTrue.setResultType (UMLProject.get().getFromFBaseTypes (FBaseTypes.BOOLEAN));
      OOGenToken token = new OOGenToken();
      token.setSectionName ("UMLStatechart_Method");

      OOReturnStatement returnTrue = new OOReturnStatement (OOIdentifierExpr.TRUE_IDENTIFIER);

      token.addToStatement (returnTrue);
      alwaysTrue.setMethodBody (token.getSourceCode(), true);

      return alwaysTrue;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private FMethod createNotifyMe()
   {
      // create method notifyMe
      FMethod notifyMe = new UMLMethod ("notifyMe");
      notifyMe.setGenerated (true);
      notifyMe.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
      notifyMe.setResultType (
         UMLProject.get().getFromBaseTypes (FBaseTypes.VOID));
      notifyMe.setUmlSynchronized (true);
      OOGenToken token = new OOGenToken();
      token.setSectionName ("UMLStatechart_Method");

      OOMethod tmpMethod = OO.method ("notify");
      OOCallMethodExpr callNotify = OO.call (tmpMethod);

      token.addToStatement (new OOExprStatement (callNotify));
      notifyMe.setMethodBody (token.getSourceCode(), true);

      return notifyMe;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private FMethod createWaitForResultForEvent()
   {
      // create method waitForResultForEvent
      FMethod waitForResultForEvent = new UMLMethod ("waitForResultForEvent");
      waitForResultForEvent.setGenerated (true);
      waitForResultForEvent.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
      waitForResultForEvent.setResultType (
         UMLProject.get().getFromFBaseTypes (FBaseTypes.VOID));
      waitForResultForEvent.setUmlSynchronized (true);
      FParam param = new UMLParam ("event", FEvent.class.getName());
      waitForResultForEvent.addToParam (param);
      OOGenToken token = new OOGenToken();
      token.setSectionName ("UMLStatechart_Method");

      OOVariable tmpObjectName = OO.variable ("event");
      OOMethod tmpMethod = OO.method ("hasResult", OOMethodType.GET_METHOD);
      OOCallMethodExpr tmpCallMethodExpr = OO.call (tmpObjectName, tmpMethod);
      OOPrefixExpr tmpCondition =
         new OOPrefixExpr (OOPrefixOp.NOT_OP, tmpCallMethodExpr);
      OOWhileStatement whileInWaitForResultForEvent =
         OO.whileStat (tmpCondition);

      tmpMethod = OO.method ("wait");
      OOCallMethodExpr callWait = OO.call (tmpMethod);

      OOExceptionExpr tmpExceptionType =
         OOExceptionExpr.INTERRUPTED_EXCEPTION;
      OOVariable tmpExceptionName = OO.variable ("e");
      OOCatchStatement tmpCatchStatement =
         new OOCatchStatement (tmpExceptionType, tmpExceptionName);

      token.addToStatement (whileInWaitForResultForEvent);
      token.addToStatement (new OOStartBlockStatement());
      token.addToStatement (new OOTryStatement());
      token.addToStatement (new OOStartBlockStatement());
      token.addToStatement (new OOExprStatement (callWait));
      //token.addToStatement(new OOExprStatement(new OOStringExpr(oOGenVisitor.END_OF_STATEMENT)));
      token.addToStatement (OO.endBlock());
      token.addToStatement (tmpCatchStatement);
      token.addToStatement (new OOStartBlockStatement());
      token.addToStatement (OO.endBlock());
      token.addToStatement (OO.endBlock());

      waitForResultForEvent.setMethodBody (token.getSourceCode(), true);

      return waitForResultForEvent;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param classOfReactiveObject  No description provided
    * @return                       No description provided
    */
   private FMethod createConstructor (FClass classOfReactiveObject)
   {
      // create constructor
      FMethod constructorOfReactiveObject = new UMLMethod (classOfReactiveObject.getName());
      constructorOfReactiveObject.setResultType (project.getFromFBaseTypes (FBaseTypes.CONSTRUCTOR));
      constructorOfReactiveObject.setGenerated (true);
      constructorOfReactiveObject.setComment (
         new UMLCommentary ("Creates an Object with the given arguments."));
      constructorOfReactiveObject.setDisplayLevel (
         FDeclaration.CODE_DISPLAY_LEVEL);

      OOMethod initStatechart = OO.method ("initStatechart");
      OOCallMethodExpr tmpCallMethodExpr = OO.call (initStatechart);
      OOExprStatement tmpExprStatement =
         new OOExprStatement (tmpCallMethodExpr);
      OOGenToken token = new OOGenToken();
      token.setSectionName ("UMLStatechart_Method");
      token.addToStatement (tmpExprStatement);

      constructorOfReactiveObject.setMethodBody (token.getSourceCode(), true);

      return constructorOfReactiveObject;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param initToken            No description provided
    * @param bodyToken            No description provided
    * @param methods              No description provided
    * @param complexStateMethods  No description provided
    */
   public void generateSourceCodeForStateChart (
                                                OOGenToken initToken,
                                                OOGenToken bodyToken, Set methods, Set complexStateMethods)
   {
      Iterator iter = activityDiagram.iteratorOfElements();
      while (iter.hasNext())
      {
         Object element = iter.next();
         if (element instanceof FDiagramItem)
         {
            UMLDiagramItem item = (UMLDiagramItem) element;
            String itemName = item.getName();

            if (item instanceof UMLStartActivity)
            {
               handleStartActivity (bodyToken, item);
            }
            else if (item instanceof UMLComplexState)
            {
               handleComplexState (initToken, bodyToken, item, itemName, methods, complexStateMethods);
            } //if (item instanceof UMLComplexState)
            else if (item instanceof UMLStopActivity)
            {
               handleStopActivity (initToken);
            }
            else if (item instanceof UMLTransition)
            {
               handleTransition (bodyToken, item, methods);
            } //if (item instanceof UMLTransition)
         }
      } //while
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param bodyToken  No description provided
    * @param item       No description provided
    * @param methods    No description provided
    */
   private void handleTransition (OOGenToken bodyToken, FDiagramItem item, Set methods)
   {
      UMLTransition transition = (UMLTransition) item;
      UMLActivity revEntry = transition.getRevEntry();
      UMLActivity revExit = transition.getRevExit();
      FMethod guardMethod = null;

      // setInitial is handled above
      if (! (revExit instanceof UMLStartActivity))
      {
         String eventName = transition.getEvent();

         // No event meens implicit event. This is somehow doubled,
         // because this case is also handled in
         // FReactive::makeTransition. But we need the eventName for
         // generating the correct guardName. Maybe it should be
         // handled at one place only, maybe ...
         if (Utility.isNullOrEmpty (eventName))
         {
            eventName = revExit.getName() + FReactive.READY_EVENT_POSTFIX;
         }

         if (eventName.startsWith (FTimer.AFTER))
         {
            // Handle an after-transition.
            int id = methodNumber++;

            // create method setTimeoutForAfterEvent
            String timeout = eventName.substring (FTimer.AFTER.length() + 1);
            OOStringExpr tmpParameters = new OOStringExpr (timeout);
            OOCallMethodExpr tmpCallMethodExpr =
               OO.call (
               revExit.getName(),
               "setTimeoutForAfterEvent",
               tmpParameters);
            bodyToken.addToStatement (
               new OOExprStatement (tmpCallMethodExpr));

            // create method setIdOfAfterTransition
            tmpParameters = new OOStringExpr (new Integer (id).toString());
            tmpCallMethodExpr =
               OO.call (
               revExit.getName(),
               "setIdOfAfterTransition",
               tmpParameters);
            bodyToken.addToStatement (
               new OOExprStatement (tmpCallMethodExpr));

            eventName = FTimer.AFTER + id;
            FMethod afterMethod = new UMLMethod (eventName);
            afterMethod.setGenerated (true);
            afterMethod.setResultType (
               UMLProject.get().getFromFBaseTypes (FBaseTypes.VOID));
            afterMethod.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
            OOGenToken afterToken = new OOGenToken();
            afterToken.setSectionName ("UMLStatechart_Method");

            OOVariable tmpVariable = OO.variable ("FEvent event");
            tmpParameters = new OOStringExpr ("\"" + eventName + "\"");
            OONewObjectExpr tmpExpression =
               OO.newObject (
               project.getFTypeList().getFromFTypes (FEvent.class.getName()),
               tmpParameters);
            OOStatement tmpAssignment =
               OO.assignStat (tmpVariable, tmpExpression);

            tmpParameters =
               new OOStringExpr ("event, FReactive.INTERNAL_EVENT");
            OOCallMethodExpr enqueueEventCall =
               OO.call (reactiveObjectName, "enqueueEvent", tmpParameters);

            OOCallMethodExpr notifyMeCall =
               OO.call (reactiveObjectName, "notifyMe");

            afterToken.addToStatement (tmpAssignment);
            afterToken.addToStatement (
               new OOExprStatement (enqueueEventCall));
            afterToken.addToStatement (new OOExprStatement (notifyMeCall));
            afterMethod.setMethodBody (afterToken.getSourceCode(), true);

            methods.add (afterMethod);
         }
         else if (eventName.startsWith (FPeriodicTimer.WHEN))
         {
            // Handle a WHEN transition.
            int id = methodNumber++;

            // Retrieve the condition for when signal
            int start = eventName.indexOf ("[");
            int end = eventName.indexOf ("]");
            String condition = eventName.substring (start + 1, end).trim();

            // Retrieve the timeout for when signal
            String timeout = eventName.substring (end + 1).trim();
            if (timeout.equals (""))
            {
               timeout = "100";
            }

            // create method setTimeoutForWhenEvent
            OOStringExpr tmpParameters = new OOStringExpr (timeout);
            OOCallMethodExpr tmpCallMethodExpr =
               OO.call (
               revExit.getName(),
               "setTimeoutForWhenEvent",
               tmpParameters);
            bodyToken.addToStatement (
               new OOExprStatement (tmpCallMethodExpr));

            // create method setIdOfWhenTransition
            tmpParameters = new OOStringExpr (new Integer (id).toString());
            tmpCallMethodExpr =
               OO.call (
               revExit.getName(),
               "setIdOfWhenTransition",
               tmpParameters);
            bodyToken.addToStatement (
               new OOExprStatement (tmpCallMethodExpr));

            eventName = FPeriodicTimer.WHEN + id;
            FMethod whenMethod = new UMLMethod (eventName);
            whenMethod.setGenerated (true);
            whenMethod.setResultType (
               UMLProject.get().getFromFBaseTypes (FBaseTypes.VOID));
            whenMethod.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
            OOGenToken afterToken = new OOGenToken();
            afterToken.setSectionName ("UMLStatechart_Method");

            OOVariable tmpVariable = OO.variable ("FEvent event");
            tmpParameters = new OOStringExpr ("\"" + eventName + "\"");
            OONewObjectExpr tmpExpression =
               OO.newObject (
               project.getFTypeList().getFromFTypes (FEvent.class.getName()),
               tmpParameters);
            OOStatement tmpAssignment =
               OO.assignStat (tmpVariable, tmpExpression);

            tmpParameters =
               new OOStringExpr ("event, FReactive.INTERNAL_EVENT");
            OOCallMethodExpr enqueueEventCall =
               OO.call (reactiveObjectName, "enqueueEvent", tmpParameters);

            OOCallMethodExpr notifyMeCall =
               OO.call (reactiveObjectName, "notifyMe");

            afterToken.addToStatement (tmpAssignment);
            afterToken.addToStatement (
               new OOExprStatement (enqueueEventCall));
            afterToken.addToStatement (new OOExprStatement (notifyMeCall));
            whenMethod.setMethodBody (afterToken.getSourceCode(), true);

            methods.add (whenMethod);

            // when conditions are realized as guards
            if (condition != null && condition.length() != 0)
            {
               // replace old guard by when condition
               UMLTransitionGuard guard = new UMLTransitionGuard();
               guard.setType (UMLTransitionGuard.BOOL);
               guard.setBoolExpr (condition);
               guard.setRevGuard (transition);
            }
         }

         // keep only first word
         int sepIndex = eventName.indexOf (" ");
         int braIndex = eventName.indexOf ("(");

         // get the first position of " " or "("
         if ( (sepIndex == -1) ||  (braIndex != -1 && braIndex < sepIndex))
         {
            sepIndex = braIndex;
         }
         if (sepIndex != -1)
         {
            eventName = eventName.substring (0, sepIndex);
         }

         // make guard method for transition
         UMLTransitionGuard guard = transition.getGuard();
         String guardName = null;

         final int guardType = UMLTransitionGuard.getGuardType (transition);
         if (guardType == UMLTransitionGuard.BOOL)
         {
            guardName =
               "guard"
               + methodNumber++
               + "For"
               + Utility.upFirstChar (eventName)
               + "From"
               + Utility.upFirstChar (revExit.getName())
               + "To"
               + Utility.upFirstChar (revEntry.getName());

            guardMethod = new UMLMethod (guardName);
            guardMethod.setGenerated (true);
            guardMethod.setResultType (
               UMLProject.get().getFromFBaseTypes (FBaseTypes.BOOLEAN));
            guardMethod.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
            OOGenToken guardToken = new OOGenToken();
            guardToken.setSectionName ("UMLStatechart_Method");
            FClass clazz =
               activityDiagram.findActivityDiagramOfMasterState()
               .getStartActivity()
               .getRevStartOfStateChart();

            // FEvent parameters have been replaced by
            // the parameters of corresponding event
            // guardMethod.addToParam (new UMLParam (eventName, FEvent.class.getName()));
            OOReturnStatement tmpReturnStatement =
               new OOReturnStatement (guard.getBoolExpr());
            guardToken.addToStatement (tmpReturnStatement);

            // copy parameters from signal method
            FMethod signalMethod =
               clazz.getFromFMethodsByShortName (eventName);
            if ( (signalMethod != null) &&  (signalMethod.isSignal()))
            {
               // copy signature from signal method
               Iterator params = signalMethod.iteratorOfParam();
               while (params.hasNext())
               {
                  FParam param = (FParam) params.next();
                  guardMethod.addToParam (
                     new UMLParam (
                     param.getName(),
                     (UMLType) param.getFParamType()));
               }
            }
            guardName = Utility.quote (guardName);
            guardMethod.setMethodBody (guardToken.getSourceCode(), true);

            methods.add (guardMethod);
         }
         else if (guardType == UMLTransitionGuard.ELSE)
         {
            guardName = "FReactive.ELSE_GUARD";
         }

         // make action method for transition
         String actionName = null;
         if (! (Utility.isNullOrEmpty (transition.getAction())))
         {
            actionName =
               "action"
               + methodNumber++
               + "For"
               + Utility.upFirstChar (eventName)
               + "From"
               + Utility.upFirstChar (revExit.getName())
               + "To"
               + Utility.upFirstChar (revEntry.getName());

            FMethod actionMethod = new UMLMethod (actionName);
            actionMethod.setGenerated (true);
            actionMethod.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
            OOGenToken actionToken = new OOGenToken();
            actionToken.setSectionName ("UMLStatechart_Method");
            FClass clazz =
               activityDiagram.findActivityDiagramOfMasterState()
               .getStartActivity()
               .getRevStartOfStateChart();

            // FIXME: Methods with same name and different signature
            // not handled.
            FMethod signalMethod =
               clazz.getFromFMethodsByShortName (eventName);
            if ( (signalMethod != null) &&  (signalMethod.isSignal()))
            {
               // copy signature from signal method
               actionMethod.setResultType (signalMethod.getFResultType());
               Iterator params = signalMethod.iteratorOfParam();
               while (params.hasNext())
               {
                  FParam param = (FParam) params.next();
                  actionMethod.addToParam (
                     new UMLParam (
                     param.getName(),
                     (UMLType) param.getFParamType()));
               }

               OOStringExpr tmpStringExpr =
                  new OOStringExpr (transition.getAction());
               actionToken.addToStatement (
                  new OOExprStatement (tmpStringExpr));
               actionMethod.setMethodBody (actionToken.getSourceCode(), true);
            }
            else
            {
               actionMethod.setResultType (
                  UMLProject.get().getFromBaseTypes (FBaseTypes.VOID));
               OOStringExpr tmpStringExpr =
                  new OOStringExpr (transition.getAction());
               actionToken.addToStatement (
                  new OOExprStatement (tmpStringExpr));
               actionMethod.setMethodBody (actionToken.getSourceCode(), true);
            }

            methods.add (actionMethod);
         }

         String entryStateName = revEntry.getName();

         // transition ends at UMLStartActivity, check for history start activity
         if (revEntry instanceof UMLStartActivity)
         {
            UMLStartActivity startActivity = (UMLStartActivity) revEntry;
            entryStateName =
                ((UMLActivityDiagram) startActivity.getFirstFromDiagrams())
               .getRevContains()
               .getName();

            // set the history kind of the ComplexState
            if (startActivity.getHistoryKind()
               == FComplexState.HISTORY_DEEP)
            {
               OOStringExpr tmpParameters =
                  new OOStringExpr ("FComplexState.HISTORY_DEEP");
               OOCallMethodExpr setHistoryCall =
                  OO.call (
                  entryStateName,
                  "setHistoryKind",
                  tmpParameters);
               bodyToken.addToStatement (
                  new OOExprStatement (setHistoryCall));
               //bodyBuf.append ("\n" + entryStateName + ".setHistoryKind (FComplexState.HISTORY_DEEP);\n");
            }
            else if (
               startActivity.getHistoryKind()
               == FComplexState.HISTORY_SHALLOW)
            {
               OOStringExpr tmpParameters =
                  new OOStringExpr ("FComplexState.HISTORY_SHALLOW");
               OOCallMethodExpr setHistoryCall =
                  OO.call (
                  entryStateName,
                  "setHistoryKind",
                  tmpParameters);
               bodyToken.addToStatement (
                  new OOExprStatement (setHistoryCall));
               //bodyBuf.append ("\n" + entryStateName + ".setHistoryKind (FComplexState.HISTORY_SHALLOW);\n");
            }
         }
         if (revEntry instanceof UMLStopActivity)
         {
            entryStateName = "stopStateChart";
         }

         OOStringExpr tmpParameters =
            new OOStringExpr (
            revExit.getName()
            + ", "
            + entryStateName
            + ", "
            + Utility.quote (eventName)
            + ", "
            + guardName
            + ", "
            + Utility.quote (actionName));
         OOCallMethodExpr makeTransitionCall =
            OO.call (reactiveObjectName, "makeTransition", tmpParameters);
         bodyToken.addToStatement (new OOExprStatement (makeTransitionCall));
      } //if (!(revExit instanceof UMLStartActivity))
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param initToken  No description provided
    */
   private void handleStopActivity (OOGenToken initToken)
   {
      OOVariable tmpVariable = OO.variable ("FState stopStateChart");
      OOStringExpr tmpParameters =
         new OOStringExpr ("\"" + "stopStateChart\", null, null, null");
      OOCallMethodExpr makeStateCall =
         OO.call (reactiveObjectName, "makeState", tmpParameters);
      OOStatement tmpAssignment = OO.assignStat (tmpVariable, makeStateCall);
      initToken.addToStatement (tmpAssignment);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param initToken            No description provided
    * @param bodyToken            No description provided
    * @param item                 No description provided
    * @param itemName             No description provided
    * @param methods              No description provided
    * @param complexStateMethods  No description provided
    */
   private void handleComplexState (
                                    OOGenToken initToken,
                                    OOGenToken bodyToken,
                                    FDiagramItem item,
                                    String itemName,
                                    Set methods,
                                    Set complexStateMethods)
   {

      UMLComplexState state = (UMLComplexState) item;
      UMLStoryActivity story = state.getStory();

      String entryActionName = null;
      String doActionName = null;
      String exitActionName = null;

      if (story != null)
      {
         // it's a state with a story diagram
         doActionName =
            "doAction" + methodNumber++ + Utility.upFirstChar (itemName);
         FMethod doActionMethod = new UMLMethod (doActionName);
         doActionMethod.setGenerated (true);
         doActionMethod.setResultType (
            UMLProject.get().getFromBaseTypes (FBaseTypes.VOID));
         doActionMethod.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);

         UMLActivityDiagram activityDiag =
            new UMLActivityDiagram (doActionName, (UMLProject) project);
         activityDiag.addToElements (story);
         UMLStartActivity start = new UMLStartActivity ((UMLMethod) doActionMethod);
         activityDiag.addToElements (start);
         activityDiag.addToElements (new UMLTransition
             (start, story, new UMLTransitionGuard()));
         UMLStopActivity stop = new UMLStopActivity();
         activityDiag.addToElements (stop);
         activityDiag.addToElements (new UMLTransition
             (story, stop, new UMLTransitionGuard()));

         /*
          *  getClientOfChain().generateSourceCodeFor (doActionMethod, null, null);
          *  activityDiag.removeFromElements (story);
          *  story.removeAllFromEntry();
          *  story.removeAllFromExit();
          *  story.deleteTokens();
          *  story.getStoryPattern().deleteTokens();
          *  doActionMethod.removeYou();
          */
         complexStateMethods.add (doActionMethod);
      }
      // it's a state with some statechart diagramms
      // handle concurrent states

      String capitalizedItemName = Utility.upFirstChar (itemName);

      String entryActionBody = state.getEntryAction();
      String doActionBody = state.getDoAction();
      String exitActionBody = state.getExitAction();

      if (! (Utility.isNullOrEmpty (entryActionBody)))
      {
         entryActionName =
            "entryAction"
            + methodNumber++
            + capitalizedItemName;
         methods.add (generateActionMethodForState (entryActionName, entryActionBody));
      }

      if (! (Utility.isNullOrEmpty (doActionBody)))
      {
         doActionName = "doAction"
            + methodNumber++
            + capitalizedItemName;
         methods.add (generateActionMethodForState (doActionName, doActionBody));
      }

      if (! (Utility.isNullOrEmpty (exitActionBody)))
      {
         exitActionName = "exitAction"
            + methodNumber++
            + capitalizedItemName;
         methods.add (generateActionMethodForState (exitActionName, exitActionBody));
      }

      Iterator iterContains = state.iteratorOfContains();
      String stateType = "State";
      while (iterContains.hasNext())
      {
         UMLActivityDiagram diag = (UMLActivityDiagram) iterContains.next();
         if (diag.belongsToStateWithSubStates())
         {
            //diag.generateJavaCodeForStateChart (initToken, bodyToken);
            UMLActivityDiagram oldActivityDiagram = activityDiagram;
            activityDiagram = diag;
            this.generateSourceCodeForStateChart (initToken, bodyToken, methods, complexStateMethods);
            activityDiagram = oldActivityDiagram;
            stateType = "ComplexState";
         }
      } //while (iterContains.hasNext ())

      OOStringExpr tmpParameters =
         new OOStringExpr (
         "\""
         + itemName
         + "\", "
         + Utility.quote (entryActionName)
         + ", "
         + Utility.quote (doActionName)
         + ", "
         + Utility.quote (exitActionName));
      OOCallMethodExpr tmpCallMethodExpr =
         OO.call (reactiveObjectName, "make" + stateType, tmpParameters);
      OOStatement tmpAssignment =
         OO.assignStat ("F" + stateType + " " + itemName, tmpCallMethodExpr);
      initToken.addToStatement (tmpAssignment);

      Iterator iter = state.iteratorOfDeferredEvents();
      while (iter.hasNext())
      {
         String deferredEvent = (String) iter.next();

         tmpParameters = new OOStringExpr (Utility.quote (deferredEvent));
         tmpCallMethodExpr =
            OO.call (itemName, "addToDeferredEvents", tmpParameters);
         bodyToken.addToStatement (new OOExprStatement (tmpCallMethodExpr));
      }

      String masterName = retrieveMasterName();
      tmpParameters = new OOStringExpr (itemName);
      tmpCallMethodExpr =
         OO.call (masterName, "addToSubStates", tmpParameters);
      bodyToken.addToStatement (new OOExprStatement (tmpCallMethodExpr));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param bodyToken  No description provided
    * @param item       No description provided
    */
   private void handleStartActivity (OOGenToken bodyToken, FDiagramItem item)
   {
      UMLStartActivity startActivity = (UMLStartActivity) item;
      Iterator iterOfExit = startActivity.iteratorOfExit();
      if (iterOfExit.hasNext())
      {
         UMLTransition transition = (UMLTransition) iterOfExit.next();

         // set initial state
         String masterName = retrieveMasterName();
         OOStringExpr tmpParameter =
            new OOStringExpr (transition.getRevEntry().getName());
         OOCallMethodExpr tmpCallMethodExpr =
            OO.call (masterName, "addToInitial", tmpParameter);
         bodyToken.addToStatement (new OOExprStatement (tmpCallMethodExpr));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private String retrieveMasterName()
   {
      String masterName;
      if (activityDiagram.getRevContains() == null)
      {
         // current instance is activity diagram for statechart
         masterName = activityDiagram.getName();
      }
      else
      {
         // current instance is an activity diagram contained in a complexstate
         masterName = activityDiagram.getRevContains().getName();
      }
      return masterName;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name  No description provided
    * @param body  No description provided
    * @return      No description provided
    */
   private FMethod generateActionMethodForState (String name, String body)
   {
      FMethod method = new UMLMethod (name);
      method.setGenerated (true);
      method.setResultType (
         UMLProject.get().getFromFBaseTypes (FBaseTypes.VOID));
      method.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);

      OOGenToken token = new OOGenToken();
      token.addToStatement (new OOExprStatement (new OOStringExpr (body)));
      method.setMethodBody (token.getSourceCode(), true);

      return method;
   } // generateActionMethodForState


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "UMLStatechartOOHandler[]";
   }
}

/*
 * $Log: UMLStatechartOOHandler.java,v $
 * Revision 1.38.2.3  2006/06/07 09:13:31  creckord
 * - UMLTransitionGuard can be null instead of UMLTransitionGuard.NONE
 * - old "repair assoc" code removed (access methods are not kept, so repairing them is no longer needed)
 * - loop bends for assocs are removed when loop is removed
 *
 */
