/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.*;

import org.apache.log4j.Logger;
import org.apache.log4j.Priority;

import de.uni_paderborn.fujaba.basic.SourceCodeFactory;
import de.uni_paderborn.fujaba.codegen.AccessorOOHandler.MethodDescription;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.fca.FHashMap;
import de.upb.tools.fca.IteratorConcatenation;


/**
 * @author    Carsten Reckord
 * @version   $Revision: 1.9.2.5 $
 */
public abstract class UMLRoleOOHandlerStrategy
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLRoleOOHandlerStrategy.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLRoleOOHandler handler;


   /**
    * Constructor for class UMLRoleOOHandlerStrategy
    */
   public UMLRoleOOHandlerStrategy()
   {
      setupTemplateParameterListeners();
   }


   /**
    * Constructor for class UMLRoleOOHandlerStrategy
    *
    * @param handler  No description provided
    */
   public UMLRoleOOHandlerStrategy (UMLRoleOOHandler handler)
   {
      this();
      setHandler (handler);
   }


   /**
    * Sets the handler attribute of the UMLRoleOOHandlerStrategy object
    *
    * @param handler  The new handler value
    */
   protected void setHandler (UMLRoleOOHandler handler)
   {
      this.handler = handler;
   }


   /**
    * Get the handler attribute of the UMLRoleOOHandlerStrategy object
    *
    * @return   The handler value
    */
   public UMLRoleOOHandler getHandler()
   {
      return this.handler;
   }


   /**
    * Get the responsible attribute of the UMLRoleOOHandlerStrategy object
    *
    * @param umlRole      No description provided
    * @param partnerRole  No description provided
    * @return             The responsible value
    */
   public abstract boolean isResponsible (FRole umlRole, FRole partnerRole);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole             No description provided
    * @param partnerRole         No description provided
    * @param attr                No description provided
    * @param templateParameters  No description provided
    * @param accessMethods       No description provided
    */
   protected abstract void generateReadAccessMethods (FRole umlRole, FRole partnerRole, FAttr attr,
                                                      FHashMap templateParameters, Set accessMethods);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole             No description provided
    * @param partnerRole         No description provided
    * @param attr                No description provided
    * @param templateParameters  No description provided
    * @param accessMethods       No description provided
    */
   protected abstract void generateWriteAccessMethods (FRole umlRole, FRole partnerRole, FAttr attr,
                                                       FHashMap templateParameters, Set accessMethods);


   /**
    * Get the attrType attribute of the UMLRoleOOHandlerStrategy object
    *
    * @param umlRole      No description provided
    * @param partnerRole  No description provided
    * @return             The attrType value
    */
   protected abstract String getAttrType (FRole umlRole, FRole partnerRole);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole      No description provided
    * @param partnerRole  No description provided
    * @return             No description provided
    */
   protected abstract String createKeyChangePreHook (FRole umlRole, FRole partnerRole);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole      No description provided
    * @param partnerRole  No description provided
    * @return             No description provided
    */
   protected abstract String createKeyChangePostHook (FRole umlRole, FRole partnerRole);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void setupTemplateParameterListeners() { }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole             No description provided
    * @param readAccessMethods   No description provided
    * @param writeAccessMethods  No description provided
    * @return                    No description provided
    */
   public FAttr generateRole (FRole umlRole,
                              Set readAccessMethods,
                              Set writeAccessMethods)
   {
      FRole partnerRole = umlRole.getFPartnerRole();
      if (umlRole.getFTarget() == null || partnerRole.getFTarget() == null)
      {
         if (log.isEnabledFor (Priority.ERROR))
         {
            log.error ("role '" + umlRole + "' is not assigned to a class object");
         }
      }
      provideImport (umlRole.getFTarget(), partnerRole.getFTarget());

      FAttr attr = createAttr (umlRole, partnerRole,
         getAttrType (umlRole, partnerRole));
      FHashMap templateParameters = setupTemplateParameters (umlRole, partnerRole);

      if (readAccessMethods != null)
      {
         generateReadAccessMethods (umlRole, partnerRole, attr, templateParameters, readAccessMethods);
      }
      if (writeAccessMethods != null)
      {
         generateWriteAccessMethods (umlRole, partnerRole, attr, templateParameters, writeAccessMethods);
      }

      return attr;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole      No description provided
    * @param partnerRole  No description provided
    * @return             No description provided
    */
   protected FHashMap setupTemplateParameters (FRole umlRole, FRole partnerRole)
   {
      FHashMap templateParameters = new FHashMap();

      templateParameters.put ("$FIELDNAME$", partnerRole.getAttrName());
      templateParameters.put ("$VALUETYPE$", partnerRole.getFTarget().getFRealType().getName());
      templateParameters.put ("$METHOD_SUFFIX$",
         SourceCodeFactory.upFirstChar (partnerRole.getAttrName()).toString());
      templateParameters.put ("$ASSOC$", umlRole.getFAssoc().getName());

      templateParameters.put ("$CHANGE_PREHOOK$", "");
      templateParameters.put ("$CHANGE_POSTHOOK$", "");

      FType qualifierType = getExternalQualifierType (partnerRole);
      if (qualifierType != null)
      {
         templateParameters.put ("$PARTNER_KEY$", "partnerKey, ");
         if (qualifierType instanceof FClass)
         {
            provideImport (umlRole.getFTarget(), (FClass) qualifierType);
         }
      }
      return templateParameters;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole      No description provided
    * @param partnerRole  No description provided
    * @return             No description provided
    */
   protected FHashMap setupKeyChangeParameters (FRole umlRole, FRole partnerRole)
   {
      UMLRoleOOHandlerStrategy partnerStrategy = findResponsible (partnerRole);

      FHashMap templateParameters = setupTemplateParameters (umlRole, partnerRole);
      FHashMap partnerParameters = partnerStrategy.setupTemplateParameters (partnerRole, umlRole);
      String keyString = (String) partnerParameters.get ("$KEYTYPE$");

      templateParameters.put ("$PARTNERMETHOD_SUFFIX$", partnerParameters.get ("$METHOD_SUFFIX$"));
      templateParameters.put ("$KEYTYPE$", keyString);

      return templateParameters;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole      No description provided
    * @param partnerRole  No description provided
    * @param attrType     No description provided
    * @return             No description provided
    */
   protected FAttr createAttr (FRole umlRole, FRole partnerRole, String attrType)
   {
      //FIXME:???target -> partnerRole/umlRole?
      FAttr attr = new UMLAttr (false);

      attr.setName (partnerRole.getAttrName());
      attr.setStatic (false);
      attr.setFinal (false);
      attr.setVisibility (FDeclaration.PRIVATE);
      attr.setInitialValue (null);
      FType tmpType = provideReference (attrType);
      attr.setAttrType (tmpType);

      //No parent, so no get-/set-Accessors are created
      attr.setVisibility (partnerRole.getUmlVisibility());
      attr.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
      attr.setGenerated (true);
      attr.setTransient (umlRole.getFAssoc().isRolesTransient());

      attr.setComment (createAssocComment (umlRole, partnerRole, attr));
      //attr.setImplementingAssocRole (partnerRole);

      getHandler().addToGeneratedIncrs (attr);

      return attr;
   }


   /**
    * @param umlRole
    * @param partnerRole
    * @param attr
    * @return             No description provided
    */
   protected FCommentary createAssocComment (FRole umlRole, FRole partnerRole, FAttr attr)
   {
      SourceCodeFactory factory = UMLProject.getSourceCodeFactory();

      FClass umlTarget = umlRole.getFTarget();
      FClass partnerTarget = partnerRole.getFTarget();

      String umlTargetName =
          (umlTarget.getName() != null) ? umlTarget.getName().trim() : "";
      String partnerTargetName =
          (partnerTarget.getName() != null) ? partnerTarget.getName().trim() : "";

      String umlRoleName =
          (umlRole.getAttrName() != null) ? umlRole.getAttrName().trim() : "";
      String partnerRoleName =
          (partnerRole.getAttrName() != null) ? partnerRole.getAttrName().trim() : "";

      String umlRoleCard = umlRole.getFCard().getCardString();
      String partnerRoleCard = partnerRole.getFCard().getCardString();

      String umlQualifier =
          (umlRole.getFQualifier() != null)
         ? umlRole.getFQualifier().getName()
         : "";
      String partnerQualifier =
          (partnerRole.getFQualifier() != null)
         ? partnerRole.getFQualifier().getName()
         : "";

      String assocName =
          (umlRole.getFAssoc() != null) ? umlRole.getFAssoc().getName() : "";

      String constraintText = "";

      // Create one string from constraints.
      Iterator iter = umlRole.getFAssoc().iteratorOfConstraints();
      while (iter.hasNext())
      {
         FConstraint tmpConst = (FConstraint) iter.next();
         String constraint = tmpConst.getText();

         if (constraint != null)
         {
            constraintText = constraintText.concat (constraint).concat (", ");
         }
      }

      if (constraintText.length() > 0)
      {
         // Delete the last comma.
         constraintText = constraintText.substring (0, constraintText.length() - 1);
      }

      String commentText = factory.assocCommentary (umlTargetName, umlRoleName, umlRoleCard, assocName,
         constraintText, partnerTargetName, partnerRoleName,
         partnerRoleCard, umlQualifier, partnerQualifier,
         umlRole.getAdornment() == FRole.REFERENCE,
         partnerRole.getAdornment() == FRole.REFERENCE,
         umlRole.getAdornment() == FRole.AGGREGATION ||
         umlRole.getAdornment() == FRole.COMPOSITION,
         partnerRole.getAdornment() == FRole.AGGREGATION ||
         partnerRole.getAdornment() == FRole.COMPOSITION);
      FCommentary comment = new UMLCommentary (false);
      comment.setText (commentText);

      getHandler().addToGeneratedIncrs (comment);

      return comment;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole             No description provided
    * @param partnerRole         No description provided
    * @param attr                No description provided
    * @param methodType          No description provided
    * @param blockName           No description provided
    * @param templateParameters  No description provided
    * @param returnType          No description provided
    * @param methodParameters    No description provided
    * @return                    No description provided
    */
   protected MethodDescription createAccessMethod (FRole umlRole, FRole partnerRole, FAttr attr,
                                                   int methodType, String blockName, FHashMap templateParameters,
                                                   FType returnType, FParam[] methodParameters)
   {
      SourceCodeFactory factory = UMLProject.getSourceCodeFactory();

      String methodName = factory.getAccessMethodName (attr.getName(), methodType);

      // Create new set method.
      FMethod umlMethod = new UMLMethod (false);

      umlMethod.setName (methodName);
      umlMethod.setStatic (attr.isStatic());
      umlMethod.setResultType (returnType);
      umlMethod.setVisibility (attr.getVisibility());
      umlMethod.setAccessedAttribute (attr);
      umlMethod.setGenerated (true);

      if (! (returnType instanceof UMLBaseTypes))
      {
         umlMethod.setResultTypeIsPointer (true);
      }

      for (int i = 0; methodParameters != null && i < methodParameters.length; i++)
      {
         umlMethod.addToParam (methodParameters[i]);
      }
      return new MethodDescription (umlMethod, blockName, templateParameters);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name  No description provided
    * @param type  No description provided
    * @return      No description provided
    */
   protected FParam createParam (String name, FType type)
   {
      FParam param = new UMLParam (false);
      param.setName (name);
      param.setParamType (type);
      param.setGenerated (true);
      param.setPointer (! (type instanceof FBaseTypes));

      return param;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole             No description provided
    * @param partnerRole         No description provided
    * @param attr                No description provided
    * @param templateParameters  No description provided
    * @param blockName           No description provided
    * @param methodType          No description provided
    * @return                    No description provided
    */
   protected MethodDescription createInsertMethod (FRole umlRole, FRole partnerRole, FAttr attr,
                                                   FHashMap templateParameters,
                                                   String blockName, int methodType)
   {
      return createInsertMethod (umlRole, partnerRole, attr, templateParameters, blockName, methodType,
         "value", partnerRole.getFTarget().getFRealType());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole             No description provided
    * @param partnerRole         No description provided
    * @param attr                No description provided
    * @param templateParameters  No description provided
    * @param blockName           No description provided
    * @param methodType          No description provided
    * @param paramName           No description provided
    * @param paramType           No description provided
    * @return                    No description provided
    */
   protected MethodDescription createInsertMethod (FRole umlRole, FRole partnerRole, FAttr attr,
                                                   FHashMap templateParameters,
                                                   String blockName, int methodType,
                                                   String paramName, FType paramType)
   {
      String insert = getPartnerAccessMethod (umlRole, partnerRole, SourceCodeFactory.ADD_METHOD);
      templateParameters.put ("$INSERT$", insert);
      String remove = getPartnerAccessMethod (umlRole, partnerRole, SourceCodeFactory.REMOVE_METHOD);
      templateParameters.put ("$REMOVE$", remove);

      FType returnType = UMLProject.get().getFromBaseTypes (UMLBaseTypes.BOOLEAN);

      // Add method parameter.
      FParam[] methodParams;

      FParam valueParam = createParam (paramName, paramType);

      FType qualifierType = getExternalQualifierType (partnerRole);
      if (qualifierType != null)
      {
         FParam qualifierParam = createParam ("partnerKey", qualifierType);
         methodParams = new FParam[]{qualifierParam, valueParam};
      }
      else
      {
         methodParams = new FParam[]{valueParam};
      }

      return createAccessMethod (umlRole, partnerRole, attr, methodType, blockName,
         templateParameters, returnType, methodParams);
   }


   /**
    * @param umlRole
    * @return         The cardinality value
    */
   public static int getCardinality (FRole umlRole)
   {
      int cardinality = 1; //Assume to-1 cardinality if nothing is given

      FCardinality umlCardinality = umlRole.getFCard();
      if (umlCardinality != null)
      {
         cardinality = umlCardinality.getUpperBound();
      }
      return cardinality;
   }


   /**
    * Get the qualifiedCardinality attribute of the UMLRoleOOHandlerStrategy class
    *
    * @param umlRole  No description provided
    * @return         The qualifiedCardinality value
    */
   public static int getQualifiedCardinality (FRole umlRole)
   {
      if (umlRole.getFQualifier() != null)
      {
         return Integer.MAX_VALUE;
      }
      return getCardinality (umlRole);
   }


   /**
    * @param umlRole
    * @return         The ordered value
    */
   protected static boolean isOrdered (FRole umlRole)
   {
      Iterator iter = umlRole.iteratorOfConstraints();
      iter = new IteratorConcatenation (umlRole.getFAssoc().iteratorOfConstraints(), iter);
      while (iter.hasNext())
      {
         UMLConstraint tmpConst = (UMLConstraint) iter.next();
         String constraint = tmpConst.getText();

         if (constraint != null)
         {
            constraint = constraint.toLowerCase();
            if (constraint.equals ("ordered"))
            {
               return true;
            }
         }
      }
      return false;
   }


   /**
    * @param umlRole
    * @return         The sorted value
    */
   protected static boolean isSorted (FRole umlRole)
   {
      Iterator iter = umlRole.iteratorOfConstraints();
      iter = new IteratorConcatenation (umlRole.getFAssoc().iteratorOfConstraints(), iter);
      while (iter.hasNext())
      {
         UMLConstraint tmpConst = (UMLConstraint) iter.next();
         String constraint = tmpConst.getText();

         if (constraint != null)
         {
            constraint = constraint.toLowerCase();
            if (constraint.equals ("sorted"))
            {
               return true;
            }
         }
      }
      return false;
   }


   /**
    * Get the bound attribute of the FRoleOOHandlerStrategy class
    *
    * @param umlRole  No description provided
    * @return         The bound value
    */
   protected static boolean isBound (FRole umlRole)
   {
      FStereotype stereotype = UMLStereotypeManager.get().getFromStereotypes (
         UMLStereotypeManager.JAVA_BEAN_COMPONENT);
      boolean bound = umlRole.getFTarget().hasInStereotypes (stereotype);
      return bound;
   }


   /**
    * Get the toMany attribute of the UMLRoleOOHandlerStrategy class
    *
    * @param umlRole      No description provided
    * @param partnerRole  No description provided
    * @return             The toMany value
    */
   protected static boolean isToMany (FRole umlRole, FRole partnerRole)
   {
      FQualifier qualifier = umlRole.getFQualifier();

      if (qualifier == null)
      {
         return  (getCardinality (partnerRole) > 1);
      }
      return true;
   }


   /**
    * Get the externalQualifierType attribute of the UMLRoleOOHandlerStrategy class
    *
    * @param umlRole  No description provided
    * @return         The externalQualifierType value
    */
   protected static FType getExternalQualifierType (FRole umlRole)
   {
      FQualifier qualifier = umlRole.getFQualifier();

      if (qualifier != null && qualifier.isExternalQualifier())
      {
         return qualifier.getFType();
      }
      return null;
   }


   /**
    * Get the partnerAccessMethod attribute of the UMLRoleOOHandlerStrategy object
    *
    * @param umlRole      No description provided
    * @param partnerRole  No description provided
    * @param methodType   No description provided
    * @return             The partnerAccessMethod value
    */
   protected String getPartnerAccessMethod (FRole umlRole, FRole partnerRole, int methodType)
   {
      boolean remove =  (methodType == SourceCodeFactory.REMOVE_METHOD);
      boolean partnerQualified = !remove &&  (getExternalQualifierType (partnerRole) != null);

      return getPartnerAccessMethod (umlRole, partnerRole, methodType, false, partnerQualified);
   }


   /**
    * Get the partnerAccessMethod attribute of the UMLRoleOOHandlerStrategy object
    *
    * @param umlRole           No description provided
    * @param partnerRole       No description provided
    * @param methodType        No description provided
    * @param qualified         No description provided
    * @param partnerQualified  No description provided
    * @return                  The partnerAccessMethod value
    */
   protected String getPartnerAccessMethod (FRole umlRole, FRole partnerRole, int methodType,
                                            boolean qualified, boolean partnerQualified)
   {
      int upperBound =  (isToMany (partnerRole, umlRole) ? Integer.MAX_VALUE : 1);

      // Get source code factory.
      SourceCodeFactory factory = UMLProject.getSourceCodeFactory();
      return factory.getAssocAccessMethodLine (methodType, umlRole.getAttrName(),
         upperBound, qualified, partnerQualified);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param target  No description provided
    * @param pack    No description provided
    */
   protected void provideImport (FClass target, FPackage pack)
   {
      FPackage targetPack = target.findFPackage();
      if (targetPack != pack)
      {
         ImportFragment importFrag =
            UMLFileOOHandler.findImportFragment (getHandler().getClientOfChain().getCurrentVisitor());

         if (importFrag != null)
         {
            importFrag.addPackage (pack);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param target  No description provided
    * @param clazz   No description provided
    */
   protected void provideImport (FClass target, FClass clazz)
   {
      FPackage targetPack = target.findFPackage();
      FPackage importPack = clazz.getFDeclaredInPackage();
      if (targetPack != importPack)
      {
         ImportFragment importFrag =
            UMLFileOOHandler.findImportFragment (getHandler().getClientOfChain().getCurrentVisitor());

         if (importFrag != null)
         {
            importFrag.addClass (clazz);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param className  No description provided
    * @return           No description provided
    */
   protected FClass provideReference (String className)
   {
      FClass result =
         UMLProject.get().findReference (className,
         false, false);
      if (result == null)
      {
         result =
            UMLProject.get().findReference (className,
            true, false);
         processGeneratedClass (result);
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param typeName  No description provided
    * @return          No description provided
    */
   protected FType provideType (String typeName)
   {

      FType result =
         UMLProject.get().findType (null, typeName,
         false, false);
      if (result == null)
      {
         result =
            provideReference (typeName);
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param packName  No description provided
    * @return          No description provided
    */
   protected FPackage providePackage (String packName)
   {
      FPackage pack = UMLProject.get().findPackage (packName, false, false);
      if (pack == null)
      {
         pack = UMLProject.get().findPackage (packName, true, false);
         processGeneratedPackage (pack);
      }
      return pack;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param clazz  No description provided
    */
   private void processGeneratedClass (FClass clazz)
   {
      if (clazz.getFDeclaredInClass() != null)
      {
         FClass tmpClass = clazz.getFDeclaredInClass();
         if (!tmpClass.isCoobraPersistent())
         {
            processGeneratedClass (tmpClass);
         }
      }
      else if (clazz.getFDeclaredInPackage() != null)
      {
         FPackage pack = clazz.getFDeclaredInPackage();
         processGeneratedPackage (pack);
      }
      getHandler().addToGeneratedIncrs (clazz);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pack  No description provided
    */
   private void processGeneratedPackage (FPackage pack)
   {
      UMLRoleOOHandler handler = getHandler();
      boolean generated =  (!pack.isCoobraPersistent() || pack.isGenerated());

      if (!generated)
      {
         int size = 0;
         Iterator iter = pack.iteratorOfDeclares();
         while (iter.hasNext())
         {
            FClass clazz = (FClass) iter.next();
            if (!handler.hasInGeneratedIncrs (clazz))
            {
               size++;
            }
         }
         if (size == 0)
         {
            iter = pack.iteratorOfPackages();
            while (iter.hasNext())
            {
               FPackage tmpPack = (FPackage) iter.next();
               if (!handler.hasInGeneratedIncrs (tmpPack))
               {
                  size++;
               }
            }
         }
         generated =  (size == 0);
      }
      if (generated)
      {
         handler.addToGeneratedIncrs (pack);
         if (pack.getFParent() != null)
         {
            processGeneratedPackage (pack.getFParent());
         }
      }
   }


   /**
    * <pre>
    *                           0..1   next   0..1
    * UMLRoleOOHandlerStrategy -------------------- UMLRoleOOHandlerStrategy
    *                           previous      next
    * </pre>
    */
   private UMLRoleOOHandlerStrategy previous;


   /**
    * Sets the previous attribute of the UMLRoleOOHandlerStrategy object
    *
    * @param value  The new previous value
    * @return       No description provided
    */
   protected boolean setPrevious (UMLRoleOOHandlerStrategy value)
   {
      boolean changed = false;
      if (this.previous != value)
      {
         if (this.previous != null)
         {
            UMLRoleOOHandlerStrategy oldValue = this.previous;
            this.previous = null;
            oldValue.setNext (null);
         }
         this.previous = value;
         if (value != null)
         {
            value.setNext (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * Get the previous attribute of the UMLRoleOOHandlerStrategy object
    *
    * @return   The previous value
    */
   public UMLRoleOOHandlerStrategy getPrevious()
   {
      return this.previous;
   }


   /**
    * <pre>
    *                           0..1   next   0..1
    * UMLRoleOOHandlerStrategy -------------------- UMLRoleOOHandlerStrategy
    *                           next      previous
    * </pre>
    */
   private UMLRoleOOHandlerStrategy next;


   /**
    * Sets the next attribute of the UMLRoleOOHandlerStrategy object
    *
    * @param value  The new next value
    * @return       No description provided
    */
   public boolean setNext (UMLRoleOOHandlerStrategy value)
   {
      boolean changed = false;
      if (this.next != value)
      {
         if (this.next != null)
         {
            UMLRoleOOHandlerStrategy oldValue = this.next;
            this.next = null;
            oldValue.setPrevious (null);
         }
         this.next = value;
         if (value != null)
         {
            value.setPrevious (this);
            initHandler (value);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param strategy  No description provided
    */
   protected void initHandler (UMLRoleOOHandlerStrategy strategy)
   {
      UMLRoleOOHandler handler = getHandler();
      while (strategy != null)
      {
         strategy.setHandler (handler);
         strategy = strategy.getNext();
      }
   }


   /**
    * Get the next attribute of the UMLRoleOOHandlerStrategy object
    *
    * @return   The next value
    */
   public UMLRoleOOHandlerStrategy getNext()
   {
      return this.next;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole  No description provided
    * @return         No description provided
    */
   protected UMLRoleOOHandlerStrategy findResponsible (FRole umlRole)
   {
      return getHandler().findResponsible (umlRole);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private LinkedHashSet templateParameterListeners;


   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addToTemplateParameterListeners (TemplateParameterListener listener)
   {
      if (listener != null)
      {
         if (templateParameterListeners == null)
         {
            templateParameterListeners = new LinkedHashSet();
         }
         templateParameterListeners.add (listener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param listener  No description provided
    */
   public void removeFromTemplateParameterListeners (TemplateParameterListener listener)
   {
      if (listener != null)
      {
         if (templateParameterListeners != null)
         {
            templateParameterListeners.remove (listener);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element             No description provided
    * @param attr                No description provided
    * @param blockName           No description provided
    * @param templateParameters  No description provided
    */
   protected void changeTemplateParameters (FElement element,
                                            FAttr attr,
                                            String blockName,
                                            FHashMap templateParameters)
   {
      if (templateParameterListeners != null)
      {
         Iterator iter = templateParameterListeners.iterator();
         while (iter.hasNext())
         {
            TemplateParameterListener listener = (TemplateParameterListener) iter.next();
            listener.setupTemplateParameters (getHandler(), element, attr, blockName, templateParameters);
         }
      }
   }
}

/*
 * $Log: UMLRoleOOHandlerStrategy.java,v $
 * Revision 1.9.2.5  2006/02/13 20:45:33  fklar
 * reviewed log-messages (more details and right usage of 'log.isEnabledFor(Priority.xxx)')
 *
 */
