/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.Set;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.metamodel.FAssoc;
import de.uni_paderborn.fujaba.metamodel.FAttr;
import de.uni_paderborn.fujaba.metamodel.FClass;
import de.uni_paderborn.fujaba.metamodel.FDeclaration;
import de.uni_paderborn.fujaba.metamodel.FElement;
import de.uni_paderborn.fujaba.metamodel.FRole;
import de.upb.tools.fca.FHashMap;


/**
 * Class UMLRoleOOHandler
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.8.2.3 $
 */
public class UMLRoleOOHandler
    extends AccessorOOHandler
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLRoleOOHandler.class);


   /**
    * Default Constructor
    */
   public UMLRoleOOHandler() { }


   /**
    * Get the responsible attribute of the FAttrExprPairOOHandler object
    *
    * @param incr  No description provided
    * @return      The responsible value
    */
   public boolean isResponsible (FElement incr)
   {
      return  (incr instanceof FRole);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr       No description provided
    * @param prevToken  No description provided
    * @param param      No description provided
    * @return           No description provided
    */
   public OOGenToken generateSourceCode (FElement incr,
                                         OOGenToken prevToken,
                                         Object param[])
   {
      FRole umlRole = (FRole) incr;
//      FClass target = umlRole.getFTarget();
      FAssoc umlAssoc = umlRole.getFAssoc();

      if (log.isDebugEnabled())
      {
         log.debug (this + ".generateSourceCode(umlRole=" + umlRole + ")");
      }
      checkSanity (umlRole);

//      FStereotype interfaceType =
//         UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.INTERFACE);

      if (! (umlRole.getAdornment() == FRole.REFERENCE ||
         umlAssoc.isVirtualPath() ||
//         target.hasInStereotypes (interfaceType) ||
         umlRole.isParsed()))
      {
         return super.generateSourceCode (incr, prevToken, param);
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole  No description provided
    */
   private void checkSanity (FRole umlRole)
   {
      FClass target = umlRole.getFTarget();
      FAssoc umlAssoc = umlRole.getFAssoc();
      if (umlAssoc == null)
      {
         umlRole.removeYou();
         return;
      }
      if (target == null)
      {
         umlRole.removeYou();
         umlAssoc.removeYou();
         return;
      }
      FRole partnerRole = umlRole.getFPartnerRole();
      if (partnerRole == null)
      {
         umlRole.removeYou();
         umlAssoc.removeYou();
         return;
      }
      FClass partnerTarget = partnerRole.getFTarget();
      if (partnerTarget == null)
      {
         umlRole.removeYou();
         partnerRole.removeYou();
         umlAssoc.removeYou();
         return;
      }
   }


   /**
    * @param umlRole
    * @param partnerRole
    * @param attr         No description provided
    */
   protected void generateAttr (FRole umlRole, FRole partnerRole, FAttr attr)
   {
      FClass target = umlRole.getFTarget();
      FAttr existingAttr = target.getFromFAttrs (attr.getName());

      //FIXME: In the long run, legacy element handling should be removed...
      if (existingAttr != null &&  (existingAttr.isGenerated() ||
         existingAttr.getDisplayLevel() <= FDeclaration.CODE_DISPLAY_LEVEL))
      {
         //FIXME: This is a violation of the premise that the code generation
         //doesn't change the model
         existingAttr.removeYou();
         existingAttr = null;
      }

      if (existingAttr == null)
      {
         attr.setParent( umlRole.getFTarget() );
         attr.setImplementingAssocRole (partnerRole);
         //FIXME: Using the FAttrHandler would be better, but it's hard to decide there,
         //which Attrs need access methods.
         generateAttrCode (attr);
         attr.setParent( null );
      }
      else
      {
         if (log.isInfoEnabled())
         {
            log.info ("Existing attr: " + existingAttr.getName());
         }
         //FIXME: Using the FAttrHandler would be better, but it's hard to decide there,
         //which Attrs need access methods.
         generateAttrCode (existingAttr);
      }
   }


   /**
    * <pre>
    *                           0..1       strategyChain       0..1
    * UMLRoleOOHandlerStrategy <------------------------------------ UMLRoleOOHandler
    *                           strategyChain      uMLRoleOOHandler
    * </pre>
    */
   private UMLRoleOOHandlerStrategy strategyChain;


   /**
    * Sets the strategyChain attribute of the UMLRoleOOHandler object
    *
    * @param value  The new strategyChain value
    * @return       No description provided
    */
   protected boolean setStrategyChain (UMLRoleOOHandlerStrategy value)
   {
      boolean changed = false;
      if ( (this.strategyChain == null && value != null) ||
          (this.strategyChain != null && !this.strategyChain.equals (value)))
      {
         this.strategyChain = value;
         value.setHandler (this);
         value.initHandler (value);
         changed = true;
      }
      return changed;
   }


   /**
    * Get the strategyChain attribute of the UMLRoleOOHandler object
    *
    * @return   The strategyChain value
    */
   public UMLRoleOOHandlerStrategy getStrategyChain()
   {
      return this.strategyChain;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param strategy  No description provided
    */
   public void appendStrategy (UMLRoleOOHandlerStrategy strategy)
   {
      if (strategy != null)
      {
         if (this.strategyChain == null)
         {
            setStrategyChain (strategy);
         }
         else
         {
            UMLRoleOOHandlerStrategy tmpStrategy = this.strategyChain;
            while (tmpStrategy.getNext() != null)
            {
               tmpStrategy = tmpStrategy.getNext();
            }
            tmpStrategy.setNext (strategy);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlRole  No description provided
    * @return         No description provided
    */
   public UMLRoleOOHandlerStrategy findResponsible (FRole umlRole)
   {
      FRole partnerRole = umlRole.getFPartnerRole();
      UMLRoleOOHandlerStrategy strategy = this.strategyChain;

      while (strategy != null)
      {
         if (strategy.isResponsible (umlRole, partnerRole))
         {
            return strategy;
         }
         strategy = strategy.getNext();
      }
      throw new IllegalStateException ("No HandlerStrategy for this role");
   }


   /**
    * Get the assocTemplateName attribute of the UMLRoleOOHandler class
    *
    * @return   The assocTemplateName value
    */
   public static String getAssocTemplateName()
   {
      CodeGenVisitor visitor = CodeGenFactory.get().getCurrentOOVisitor();
      if (visitor instanceof OOGenVisitor)
      {
         return  ((OOGenVisitor) visitor).getAssocTemplateName();
      }
      // return default name
      return "Templates/AssocTemplateFCA.tpl";
   }


   /**
    * Get the referenceTemplateName attribute of the UMLRoleOOHandler class
    *
    * @return   The referenceTemplateName value
    */
   public static String getReferenceTemplateName()
   {
      CodeGenVisitor visitor = CodeGenFactory.get().getCurrentOOVisitor();
      if (visitor instanceof OOGenVisitor)
      {
         return  ((OOGenVisitor) visitor).getReferenceTemplateName();
      }

      // return default name
      return "Templates/ReferenceTemplate.tpl";
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "UMLRoleOOHandler[]";
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.codegen.AccessorOOHandler#getTemplateFileName(de.uni_paderborn.fujaba.asg.ASGElement)
    */
   /**
    * Get the templateFileName attribute of the UMLRoleOOHandler object
    *
    * @param element  No description provided
    * @return         The templateFileName value
    */
   protected String getTemplateFileName (FElement element)
   {
      FRole umlRole = (FRole) element;
      FRole partnerRole = umlRole.getFPartnerRole();

      if (partnerRole.getAdornment() == FRole.REFERENCE)
      {
         return getReferenceTemplateName();
      }
      return getAssocTemplateName();
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.codegen.AccessorOOHandler#createAccessorDescriptions(de.uni_paderborn.fujaba.asg.ASGElement, java.util.Set)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element        No description provided
    * @param accessMethods  No description provided
    * @return               No description provided
    */
   protected FAttr createAccessorDescriptions (FElement element, Set accessMethods)
   {
      FRole umlRole = (FRole) element;
      FRole partnerRole = umlRole.getFPartnerRole();

      UMLRoleOOHandlerStrategy strategy = findResponsible (umlRole);
      FAttr attr = strategy.generateRole (umlRole, accessMethods, accessMethods);

      generateAttr (umlRole, partnerRole, attr);

      return attr;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.codegen.AccessorOOHandler#getTarget(de.uni_paderborn.fujaba.asg.ASGElement, de.uni_paderborn.fujaba.uml.FAttr)
    */
   /**
    * Get the target attribute of the UMLRoleOOHandler object
    *
    * @param element  No description provided
    * @param attr     No description provided
    * @return         The target value
    */
   public FClass getTarget (FElement element, FAttr attr)
   {
      FRole umlRole = (FRole) element;
      return umlRole.getFTarget();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element             No description provided
    * @param attr                No description provided
    * @param blockName           No description provided
    * @param templateParameters  No description provided
    */
   protected void changeTemplateParameters (FElement element,
                                            FAttr attr,
                                            String blockName,
                                            FHashMap templateParameters)
   {
      super.changeTemplateParameters (element, attr, blockName, templateParameters);
      FRole umlRole = (FRole) element;
      UMLRoleOOHandlerStrategy strategy = findResponsible (umlRole);
      strategy.changeTemplateParameters (element, attr, blockName, templateParameters);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element  No description provided
    */
   public void removeOldAccessors (FElement element)
   {
      if (element != null)
      {
         FRole umlRole = (FRole) element;
         checkSanity (umlRole);
         super.removeOldAccessors (umlRole);
         FAttr attr = umlRole.getFPartnerRole().getFAssociatedAttribute();
         if (attr == null)
         {
            FClass target = umlRole.getFTarget();
            String attrName = umlRole.getFPartnerRole().getAttrName();
            attr = target.getFromFAttrs (attrName);
         }
         if (attr != null)
         {
            attr.removeYou();
         }
      }
   }
}

/*
 * $Log: UMLRoleOOHandler.java,v $
 * Revision 1.8.2.3  2006/03/30 15:51:33  lowende
 * Removed compile warnings.
 *
 * Revision 1.8.2.2  2006/03/28 11:15:46  cschneid
 * fixed accessor generation for interfaces, jnlp shortcut
 *
 * Revision 1.8.2.1  2005/09/30 18:56:54  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
