/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.*;
import org.apache.log4j.Logger;
import de.uni_paderborn.fujaba.basic.RuntimeExceptionWithContext;

import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.*;


/**
 * Class UMLLinkOOHandler
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.49.2.3 $
 */
public class UMLLinkOOHandler extends OOGenStrategyHandler
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLLinkOOHandler.class);


   /**
    * Default Constructor
    */
   public UMLLinkOOHandler() { }


   /**
    * Get the responsible attribute of the UMLLinkOOHandler object
    *
    * @param incr  No description provided
    * @return      The responsible value
    */
   public boolean isResponsible (FElement incr)
   {
      return  (incr.getClass().equals (UMLLink.class));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final boolean needToken()
   {
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr       No description provided
    * @param prevToken  No description provided
    * @param param      No description provided
    * @return           No description provided
    */
   public OOGenToken generateSourceCode (FElement incr,
                                         OOGenToken prevToken,
                                         Object param[])
   {
      UMLLink theUMLLink = (UMLLink) incr;

      if (log.isDebugEnabled())
      {
         log.debug (this + ".generateSourceCode(theUMLLink=" + theUMLLink + ")");
      }

      HashMap boundObjects;
      HashMap isomorphicBindings;
      int mode;
      boolean modifiedSetItems;
      boolean forEach;

      // parse params
      try
      {
         boundObjects = (HashMap) param[0];
         mode =  ((Integer) param[1]).intValue();
         modifiedSetItems =  ((Boolean) param[2]).booleanValue();
         forEach =  ((Boolean) param[3]).booleanValue();
         isomorphicBindings = (HashMap) param[4];
      }
      catch (Exception exception)
      {
         throw new IllegalArgumentException ("param=" + Arrays.asList (param));
      }

      // the sourceObject is definitivly bound and the targetObject is only
      // bound if the link is a check link
      UMLObject sourceObject =  (boundObjects.get (theUMLLink.getSource().getID()) == theUMLLink.getSource()) ? theUMLLink.getSource() : theUMLLink.getTarget();
      UMLObject targetObject =  (theUMLLink.getSource() == sourceObject) ? theUMLLink.getTarget() : theUMLLink.getSource();

      if (!theUMLLink.accessable (targetObject))
      {
         UMLObject tmpObject = sourceObject;
         sourceObject = targetObject;
         targetObject = tmpObject;
      }

      // handle qualified-Link
      String trimmedRange = theUMLLink.getRange();
      if (trimmedRange != null)
      {
         trimmedRange = trimmedRange.trim();
      }
      else
      {
         trimmedRange = "";
      }

      boolean sourceQualifierKey = false;
      if ( (theUMLLink.getCorrespondingRole (sourceObject).getQualifier() != null) &&
          (trimmedRange.length() > 0))
      {
         sourceQualifierKey = true;
      }

      boolean targetQualifierKey = false;
      if ( (theUMLLink.getCorrespondingRole (targetObject).getQualifier() != null) &&
          (trimmedRange.length() > 0))
      {
         targetQualifierKey = true;
      }

      if (mode == UMLLink.SEARCH)
      {
         int priority = theUMLLink.getPriority (boundObjects, isomorphicBindings);

         // create code for the link
         switch (priority % UMLLink.P_OPTIONAL)
         {
            case UMLLink.P_CHECK:
               prevToken = prevToken.insertNewToken ("LinkCheck");
               if (theUMLLink.isReference() &&
                   (sourceObject.isNegative() || sourceObject.isOptional()))
               {
                  throw new RuntimeExceptionWithContext ("Error: Source object of reference link is negative or optional. " + theUMLLink.getName(), theUMLLink);
               }

               if (theUMLLink.isReference()
                  && theUMLLink.toOneAccess (theUMLLink.getCorrespondingRole (targetObject)))
               {
                  prevToken.appendStatement (generateCode (theUMLLink,
                     LINK_CHECK_TO_ONE,
                     new Object[]{
                     sourceObject.getObjectName(),
                     sourceObject.getObjectType(),
                     theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                     Boolean.valueOf (sourceObject.isOptional()),
                     Boolean.valueOf (sourceObject.isSet()),
                     Boolean.valueOf (sourceObject.isBound()),
                     targetObject.getObjectName(),
                     targetObject.getInstanceOf(),
                     targetObject.getObjectType(),
                     theUMLLink.getCorrespondingRole (sourceObject).getAttrName(),
                     Boolean.valueOf (targetObject.isOptional()),
                     Boolean.valueOf (targetObject.isNegative()),
                     Boolean.valueOf ( (boundObjects.get (targetObject.getID()) != null)),
                     Boolean.valueOf (sourceQualifierKey),
                     Boolean.valueOf (targetQualifierKey),
                     trimmedRange,
                     theUMLLink.getCorrespondingRole (targetObject).getTarget().getName(),
                     Boolean.valueOf (theUMLLink.getType() == UMLLink.NEGATIVE),
                     Boolean.valueOf (theUMLLink.getType() == UMLLink.OPTIONAL),
                     generateCodeForAttrValuePairs (targetObject)}
                     ));
               }
               else
               {
                  if (theUMLLink.checkNegativeError (sourceObject, targetObject, UMLLink.P_CHECK))
                  {
                     throw new RuntimeExceptionWithContext ("Negative check error generating Java code for link " + theUMLLink.getName()
                        + " between " + sourceObject.getObjectName()
                        + " and " + targetObject.getObjectName(), theUMLLink);
                  }
                  else
                  {

                     boolean bound = boundObjects.containsKey (targetObject.getID());

                     // Optimization only if it is a One-To-Many check link
                     if (!theUMLLink.isReference() &&
                        theUMLLink.toOneAccess (theUMLLink.getCorrespondingRole (sourceObject)) &&
                        !theUMLLink.toOneAccess (theUMLLink.getCorrespondingRole (targetObject)) &&
                        ! ( (targetObject.isNegative() && !bound) || targetObject.isOptional() || sourceQualifierKey || !bound))
                     {
                        UMLObject tmpObject = sourceObject;
                        sourceObject = targetObject;
                        targetObject = tmpObject;

                        boolean tmpBoolean = sourceQualifierKey;
                        sourceQualifierKey = targetQualifierKey;
                        targetQualifierKey = tmpBoolean;
                     }

                     boolean negativeNode =  ( (theUMLLink.getType() == UMLLink.NEGATIVE) || targetObject.isNegative()) && sourceObject.isSet();
                     if ( (targetObject.isSet() || sourceObject.isSet()) && !negativeNode)
                     {
                        if (targetObject.isSet() && sourceObject.isSet())
                        {
                           prevToken.appendStatement (generateCode (theUMLLink,
                              LINK_CHECK_BETWEEN_SETS,
                              new Object[]{
                              theUMLLink.getCorrespondingRole (sourceObject).getAttrName(),
                              theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                              sourceObject.getObjectName(),
                              sourceObject.getObjectType(),
                              targetObject.getObjectName(),
                              targetObject.getObjectType(),
                              Boolean.valueOf (theUMLLink.getTotality()),
                              Boolean.valueOf (theUMLLink.getType() == UMLLink.NEGATIVE)}));
                        }

                        else
                        {
                           if ( (targetObject.isSet() && targetObject.isBound()) ||
                               (sourceObject.isSet() && sourceObject.isBound()))
                           {
                              if (!prevToken.getNext().getSectionName().equals ("LinkCheckSetBottom"))
                              {
                                 prevToken.insertTopAndBottomToken ("LinkCheckSet");
                                 prevToken = prevToken.getTopToken ("LinkCheckSet");
                                 OOGenToken bottomToken = prevToken.getBottomToken ("LinkCheckSet");
                                 prevToken.appendStatement (generateCode (theUMLLink,
                                    LINK_CREATE_SET_TOP,
                                    new Object[]{
                                    targetObject.isSet() ? targetObject.getObjectName() : sourceObject.getObjectName(),
                                    targetObject.isSet() ? targetObject.getInstanceOf() : sourceObject.getInstanceOf(),
                                    targetObject.isSet() ? targetObject.getObjectType() : sourceObject.getObjectType(),
                                    }
                                    ));
                                 LinkedList statements = new LinkedList();
                                 OOStatement.add (statements, OO.endBlock ("while"));
                                 bottomToken.appendStatement (statements);
                              }
                           }

                           prevToken.appendStatement (generateCode (theUMLLink,
                              LINK_CHECK_TO_SET,
                              new Object[]{
                              theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                              Boolean.valueOf (theUMLLink.toOneAccess (theUMLLink.getCorrespondingRole (targetObject))),
                              sourceObject.getObjectName(),
                              sourceObject.getObjectType(),
                              targetObject.getObjectName(),
                              targetObject.getObjectType(),
                              Boolean.valueOf (targetObject.isSet()),
                              Boolean.valueOf (theUMLLink.getType() == UMLLink.NEGATIVE),
                              Boolean.valueOf (targetObject.isSet() ? targetObject.isNegative() : sourceObject.isNegative())}));
                        }
                     }
                     else if (theUMLLink.toOneAccess (theUMLLink.getCorrespondingRole (targetObject)))
                     {
                        // 1 to 1 link check
                        prevToken.appendStatement (generateCode (theUMLLink,
                           LINK_CHECK_TO_ONE,
                           new Object[]{
                           sourceObject.getObjectName(),
                           sourceObject.getObjectType(),
                           theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                           Boolean.valueOf (sourceObject.isOptional()),
                           Boolean.valueOf (sourceObject.isSet()),
                           Boolean.valueOf (sourceObject.isBound()),
                           targetObject.getObjectName(),
                           targetObject.getInstanceOf(),
                           targetObject.getObjectType(),
                           theUMLLink.getCorrespondingRole (sourceObject).getAttrName(),
                           Boolean.valueOf (targetObject.isOptional()),
                           Boolean.valueOf (targetObject.isNegative()),
                           Boolean.valueOf ( (boundObjects.get (targetObject.getID()) != null)),
                           Boolean.valueOf (sourceQualifierKey),
                           Boolean.valueOf (targetQualifierKey),
                           trimmedRange,
                           theUMLLink.getCorrespondingRole (targetObject).getTarget().getName(),
                           Boolean.valueOf (theUMLLink.getType() == UMLLink.NEGATIVE),
                           Boolean.valueOf (theUMLLink.getType() == UMLLink.OPTIONAL),
                           generateCodeForAttrValuePairs (targetObject)}));
                     }
                     else
                     {
                        // 1 to n link check
                        prevToken.appendStatement (generateCode (theUMLLink,
                           LINK_CHECK_TO_MANY,
                           new Object[]{
                           sourceObject,
                           targetObject,
                           Boolean.valueOf ( (boundObjects.get (targetObject.getID()) != null)),
                           Boolean.valueOf (sourceQualifierKey),
                           Boolean.valueOf (targetQualifierKey),
                           trimmedRange,
                           generateCodeForAttrValuePairs (targetObject)}));
                     }
                  }
               }
               break;
            case UMLLink.P_CHECK_TO_MANY:
               if (theUMLLink.checkNegativeError (sourceObject, targetObject, UMLLink.P_CHECK_TO_MANY))
               {
                  throw new RuntimeExceptionWithContext ("Negative check error generating Java code for link " + theUMLLink.getName()
                     + " between " + sourceObject.getObjectName()
                     + " and " + targetObject.getObjectName(), theUMLLink);
               }

               // Bound SourceSet
               boolean targetBound = boundObjects.containsKey (targetObject.getID());

               prevToken = prevToken.insertNewToken ("LinkCheckToMany");

               prevToken.appendStatement (generateCode (theUMLLink,
                  LINK_CHECK_TO_MANY,
                  new Object[]{
                  sourceObject,
                  targetObject,
                  Boolean.valueOf (targetBound),
                  Boolean.valueOf (sourceQualifierKey),
                  Boolean.valueOf (targetQualifierKey),
                  trimmedRange,
                  generateCodeForAttrValuePairs (targetObject)
                  }
                  ));
               break;
            case UMLLink.P_TO_ONE:
               prevToken = prevToken.insertNewToken ("LinkSearchToOne");

               if (theUMLLink.checkNegativeError (sourceObject, targetObject, UMLLink.P_TO_ONE) ||
                  sourceObject.isOptional())
               {
                  throw new RuntimeExceptionWithContext ("Second search error generating Java code for link " + theUMLLink.getName(), theUMLLink);
               }

               prevToken.appendStatement (generateCode (theUMLLink,
                  LINK_SEARCH_TO_ONE,
                  new Object[]{
                  sourceObject.getObjectName(),
                  sourceObject.getInstanceOf(),
                  sourceObject.getObjectType(),
                  theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                  targetObject.getObjectName(),
                  targetObject.getInstanceOf(),
                  targetObject.getObjectType(),
                  theUMLLink.getCorrespondingRole (sourceObject).getAttrName(),
                  theUMLLink.getCorrespondingRole (targetObject).getTarget().getName(),
                  Boolean.valueOf (targetObject.isOptional() || theUMLLink.getType() == UMLLink.OPTIONAL),
                  Boolean.valueOf (sourceQualifierKey),
                  Boolean.valueOf (targetQualifierKey),
                  trimmedRange}));

               if (theUMLLink.isAssertInUnitTest())
               {
                  OOVariable var = OO.variable (targetObject.getObjectName() + "Found");
                  prevToken.appendStatement (OO.varDecl (UMLProject.get().getFromBaseTypes (UMLBaseTypes.BOOLEAN), var, OOIdentifierExpr.TRUE_IDENTIFIER));
                  OOGenToken bottom = prevToken.findSection ("StoryDiagramBottom");
                  bottom.appendStatement (OO.assertStmt (var, "Could not find object for " + targetObject.getObjectName() + "."));
               }

               break;
            case UMLLink.P_TO_MANY:
               if (!targetObject.isSet())
               {
                  String section = prevToken.insertTopAndBottomToken ("LinkSearchToMany");
                  prevToken = prevToken.getTopToken (section);
                  OOGenToken bottomToken = prevToken.getBottomToken (section);

                  if (sourceObject.isNegative()
                     || sourceObject.isOptional()
                     || theUMLLink.getType() == UMLLink.NEGATIVE)
                  {
                     throw new RuntimeExceptionWithContext (
                        "Syntax error: source of link is negative, optional, or link is negative: " + theUMLLink.getName(), theUMLLink);
                  }

                  prevToken.appendStatement (generateCode (theUMLLink, LINK_SEARCH_TO_MANY_TOP,
                     new Object[]{
                     sourceObject.getObjectName(),
                     sourceObject.getInstanceOf(),
                     sourceObject.getObjectType(),
                     theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                     targetObject.getObjectName(),
                     targetObject.getInstanceOf(),
                     targetObject.getObjectType(),
                     theUMLLink.getCorrespondingRole (sourceObject).getAttrName(),
                     theUMLLink.getCorrespondingRole (targetObject).getTarget().getName(),
                     Boolean.valueOf (targetObject.isOptional() || theUMLLink.getType() == UMLLink.OPTIONAL),
                     Boolean.valueOf (targetObject.isSet()),
                     Boolean.valueOf (forEach),
                     Boolean.valueOf (sourceQualifierKey),
                     Boolean.valueOf (targetQualifierKey),
                     trimmedRange}));

                  if (theUMLLink.isAssertInUnitTest())
                  {
                     OOVariable var = OO.variable (targetObject.getObjectName() + "Count");
                     bottomToken.appendStatement (
                        OO.varDecl (
                        UMLProject.get().getFromBaseTypes (UMLBaseTypes.INTEGER),
                        var,
                        OO.infixOp (var, OOInfixOp.ADD_OP, new OOStringExpr ("1"))));
                     OOGenToken bottom = prevToken.findSection ("StoryDiagramBottom");
                     bottom.appendStatement (
                        OO.assertStmt (
                        OO.infixOp (var, OOInfixOp.GREATER_OP, new OOStringExpr ("0")),
                        "Could not find object for " + targetObject.getObjectName() + "."));
                  }

                  bottomToken.appendStatement (
                     generateCode (
                     theUMLLink,
                     LINK_SEARCH_TO_MANY_BOTTOM,
                     new Object[]{
                     sourceObject.getObjectName(),
                     theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                     targetObject.getObjectName(),
                     Boolean.valueOf (targetObject.isOptional() || theUMLLink.getType() == UMLLink.OPTIONAL),
                     Boolean.valueOf (targetObject.isSet()),
                     Boolean.valueOf (forEach)}));
               }
               else
               {
                  if ("LinkSearchToSetTop".equals (prevToken.getPrev().getSectionName()))
                  {
                     prevToken = prevToken.getNext();
                  }

                  String section = prevToken.insertTopAndBottomToken ("LinkSearchToSet");
                  prevToken = prevToken.getTopToken (section);
                  OOGenToken bottomToken = prevToken.getBottomToken (section);

                  if (sourceObject.isNegative()
                     || sourceObject.isOptional()
                     || theUMLLink.getType() == UMLLink.NEGATIVE)
                  {
                     throw new RuntimeExceptionWithContext (
                        "Syntax error: source of link is negative, optional, or link is negative: " + theUMLLink.getName(), theUMLLink);
                  }

                  prevToken.appendStatement (generateCode (theUMLLink, LINK_SEARCH_TO_SET_TOP, null));

                  if (theUMLLink.isAssertInUnitTest())
                  {
                     OOVariable var = OO.variable (targetObject.getObjectName() + "Count");
                     bottomToken.appendStatement (
                        OO.varDecl (
                        UMLBaseTypes.INTEGER,
                        var,
                        OO.infixOp (var, OOInfixOp.ADD_OP, new OOStringExpr ("1"))));
                     OOGenToken bottom = prevToken.findSection ("StoryDiagramBottom");
                     bottom.appendStatement (
                        OO.assertStmt (
                        OO.infixOp (var, OOInfixOp.GREATER_OP, new OOStringExpr ("0")),
                        "Could not find object for " + targetObject.getObjectName() + "."));
                  }

                  bottomToken.appendStatement (generateCode (theUMLLink, LINK_SEARCH_TO_SET_BOTTOM, null));
               }
               break;
            default:
               throw new RuntimeExceptionWithContext ("Error generating Java code for a link. Priority = " + theUMLLink.getPriority (boundObjects, isomorphicBindings), theUMLLink);
         }
         // debug message for Albert
         if (log.isDebugEnabled())
         {
            log.debug ("search link : priority = " + theUMLLink.getPriority (boundObjects, isomorphicBindings) +
               " name = '" + theUMLLink.getName() +
               "' between " + theUMLLink.getSource().getObjectName() + " and " +
               theUMLLink.getTarget().getObjectName());
         }
      }
      else
      { // mode == execute

         prevToken = prevToken.insertNewToken ("LinkModify");

         if (sourceObject.isNegative() || targetObject.isNegative())
         {
            if (log.isDebugEnabled())
            {
               log.debug ("Execute error generating Java code for a modified link.");
            }
         }

         // exchange source and target in case of directed associations where the source object is referenced to.
         FRole srcRole = theUMLLink.getCorrespondingRole (sourceObject);
         if (srcRole.getAdornment() == FRole.REFERENCE)
         {
            UMLObject tmpObject = sourceObject;
            sourceObject = targetObject;
            targetObject = tmpObject;
         }

         modifiedSetItems =  (modifiedSetItems) ?  (targetObject.isSet() || sourceObject.isSet()) : true;
         if (theUMLLink.getAbsoluteModifier() == UMLLink.CREATE)
         {
            if (targetObject.isSet() || sourceObject.isSet())
            {
               if (!prevToken.getNext().getSectionName().equals ("LinkModifySetBottom"))
               {
                  prevToken.insertTopAndBottomToken ("LinkModifySet");
                  prevToken = prevToken.getTopToken ("LinkModifySet");
                  OOGenToken bottomToken = prevToken.getBottomToken ("LinkModifySet");
                  prevToken.appendStatement (generateCode (theUMLLink,
                     LINK_CREATE_SET_TOP,
                     new Object[]{
                     targetObject.isSet() ? targetObject.getObjectName() : sourceObject.getObjectName(),
                     targetObject.isSet() ? targetObject.getInstanceOf() : sourceObject.getInstanceOf(),
                     targetObject.isSet() ? targetObject.getObjectType() : sourceObject.getObjectType(),
                     }
                     ));
                  LinkedList statements = new LinkedList();
                  OOStatement.add (statements, OO.endBlock ("while"));
                  bottomToken.appendStatement (statements);
               }

               FQualifier sourceQual = theUMLLink.getCorrespondingRole (sourceObject).getQualifier();
               boolean qualifiedAnyKey =  (sourceQual != null && sourceQual.isExternalQualifier());

               prevToken.appendStatement (generateCode (theUMLLink,
                  LINK_CREATE_SET,
                  new Object[]{
                  theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                  Boolean.valueOf (theUMLLink.toOneAccess (theUMLLink.getCorrespondingRole (targetObject))),
                  sourceObject.getObjectName(),
                  sourceObject.getObjectType(),
                  targetObject.getObjectName(),
                  targetObject.getObjectType(),
                  Boolean.valueOf (targetObject.isSet()),
                  Boolean.valueOf (qualifiedAnyKey),
                  trimmedRange
                  }
                  ));
            }
            else
            {
               // In case of a qualified assoc with any key, ensure that the source object
               // is an instance of the corresponding class. Otherwise, the write access
               // method is used in a wrong case.

               FQualifier targetQual = theUMLLink.getCorrespondingRole (targetObject).getQualifier();
               if (targetQual != null && targetQual.isExternalQualifier())
               {
                  UMLObject tmpObject = sourceObject;
                  sourceObject = targetObject;
                  targetObject = tmpObject;
               }

               FQualifier sourceQual = theUMLLink.getCorrespondingRole (sourceObject).getQualifier();
               boolean qualifiedAnyKey =  (sourceQual != null && sourceQual.isExternalQualifier());

               prevToken.appendStatement (generateCode (theUMLLink,
                  LINK_CREATE,
                  new Object[]{
                  theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                  Boolean.valueOf (theUMLLink.toOneAccess (theUMLLink.getCorrespondingRole (targetObject))),
                  sourceObject.getObjectName(),
                  Boolean.valueOf (sourceObject.isOptionalAtPost()),
                  targetObject.getObjectName(),
                  Boolean.valueOf (targetObject.isOptionalAtPost()),
                  Boolean.valueOf (qualifiedAnyKey),
                  trimmedRange
                  }
                  ));

            }
            // create code for the new link
            // debug message for Albert

            if (log.isDebugEnabled())
            {
               log.debug ("Create new link : name = '" + theUMLLink.getName() + "'" +
                  " between " + theUMLLink.getSource().getObjectName() + " and " +
                  theUMLLink.getTarget().getObjectName());
            }

         }
         else
         {
            if ( (sourceObject.getModifier() != UMLObject.DELETE || !theUMLLink.isNavigable (sourceObject)) &&
                (targetObject.getModifier() != UMLObject.DELETE || !theUMLLink.isNavigable (targetObject)))
            { // delete the link only if no connected object is deleted

               if (targetObject.isSet() || sourceObject.isSet())
               {
                  if (!prevToken.getNext().getSectionName().equals ("LinkModifySetBottom"))
                  {
                     prevToken.insertTopAndBottomToken ("LinkModifySet");
                     prevToken = prevToken.getTopToken ("LinkModifySet");
                     OOGenToken bottomToken = prevToken.getBottomToken ("LinkModifySet");
                     prevToken.appendStatement (generateCode (theUMLLink,
                        LINK_CREATE_SET_TOP,
                        new Object[]{
                        targetObject.isSet() ? targetObject.getObjectName() : sourceObject.getObjectName(),
                        targetObject.isSet() ? targetObject.getInstanceOf() : sourceObject.getInstanceOf(),
                        targetObject.isSet() ? targetObject.getObjectType() : sourceObject.getObjectType(),
                        }
                        ));
                     LinkedList statements = new LinkedList();
                     OOStatement.add (statements, OO.endBlock ("while"));
                     bottomToken.appendStatement (statements);
                  }

                  prevToken.appendStatement (generateCode (theUMLLink,
                     LINK_DELETE_SET,
                     new Object[]{
                     theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                     Boolean.valueOf (theUMLLink.toOneAccess (theUMLLink.getCorrespondingRole (targetObject))),
                     sourceObject.getObjectName(),
                     sourceObject.getObjectType(),
                     targetObject.getObjectName(),
                     targetObject.getObjectType(),
                     Boolean.valueOf (targetObject.isSet())
                     }
                     ));
               }
               else
               {
                  prevToken.appendStatement (generateCode (theUMLLink,
                     LINK_DELETE,
                     new Object[]{
                     theUMLLink.getCorrespondingRole (targetObject).getAttrName(),
                     Boolean.valueOf (theUMLLink.toOneAccess (theUMLLink.getCorrespondingRole (targetObject))),
                     sourceObject.getObjectName(),
                     Boolean.valueOf (sourceObject.isOptional()),
                     targetObject.getObjectName(),
                     Boolean.valueOf (targetObject.isOptional())
                     }
                     ));
               }

               // create code to delete the link
               // debug message for Albert

               if (log.isDebugEnabled())
               {
                  log.debug ("Delete link : name = '" + theUMLLink.getName() + "'" +
                     " between " + theUMLLink.getSource().getObjectName() + " and " +
                     theUMLLink.getTarget().getObjectName());
               }

            }
         }
      }
      return prevToken;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    * @return        No description provided
    */
   private OOExpression[] generateCodeForAttrValuePairs (UMLObject object)
   {
      OOExpression expression = null;

      Vector aep = new Vector();

      Iterator attrExprIter = object.iteratorOfAttrs();
      while (attrExprIter.hasNext())
      {
         UMLAttrExprPair tmpAttr = (UMLAttrExprPair) attrExprIter.next();

         if (tmpAttr.getInstanceOf() == null)
         {
            log.error ("Code for " + tmpAttr + " not generated as Attribute for assertion was removed.");
         }

         if (tmpAttr.getQualifier() != UMLAttrExprPair.POST && tmpAttr.getInstanceOf() != null)
         {
            expression = (OOExpression) generateCode (tmpAttr, UML_ATTR_EXPR_PAIR, new Object[]{Boolean.FALSE});

            aep.add (expression);
         }
      }

      return OOExpression.toArray (aep);
   } // generateCodeForAttrValuePairs


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "UMLLinkOOHandler[]";
   }
}

/*
 * $Log: UMLLinkOOHandler.java,v $
 * Revision 1.49.2.3  2005/07/13 12:17:27  creckord
 * Fix to last Checkin
 *
 */
