/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.*;

import de.uni_paderborn.fujaba.basic.RuntimeExceptionWithContext;
import de.uni_paderborn.fujaba.metamodel.*;


/**
 * @author    <a href="mailto:creckord@upb.de">Carsten Reckord</a>
 * @version   $Revision: 1.10.2.3 $
 */
public class UMLClassBodyMethodsStrategy extends UMLClassBodyStrategy
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlClass  No description provided
    */
   public void createContents (FClass umlClass)
   {
      SortedSet methods = getMethods (umlClass);
      Iterator iter = methods.iterator();
      CodeGenException exception = new CodeGenException();
      while (iter.hasNext())
      {
         FMethod method = (FMethod) iter.next();
         if (isResponsible (method))
         {
            try
            {
               createMethod (method);
            }
            catch (RuntimeException e)
            {
               String newMessage = "error generating code for method '" + method.getFullMethodName() + "': " + e.getMessage();
               if (e instanceof RuntimeExceptionWithContext)
               {
                  RuntimeExceptionWithContext eWC = (RuntimeExceptionWithContext) e;
                  // creating new exception with more detailed error-message
                  Throwable cause = eWC.getCause() != null ? eWC.getCause() : eWC;
                  e = new RuntimeExceptionWithContext (newMessage, cause, eWC.getContext());
               }
               else
               {
                  // creating new exception with more detailed error-message and context
                  e = new RuntimeExceptionWithContext (newMessage, e, method);
               }
               exception.addToExceptions (e);
            }
         }
      }
      if (exception.sizeOfExceptions() > 0)
      {
         throw exception;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param method  No description provided
    */
   protected void createMethod (FMethod method)
   {
      getClassHandler().generateFMethod (method);
   }


   /**
    * Get the responsible attribute of the UMLClassBodyMethodsStrategy object
    *
    * @param method  No description provided
    * @return        The responsible value
    */
   protected boolean isResponsible (FMethod method)
   {
      if (method == null || method.getFResultType() == null)
      {
         return false;
      }
      else
      {
         return  (!method.isParsed() &&
             (checkFlag (method.isStatic(), createStatic)) &&
            !FBaseTypes.CONSTRUCTOR.equals (method.getFResultType().getName()) &&
             (method.getFAccessedAttribute() == null && !GeneratedMethodRef.hasGeneratedMethodRef (method)));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int createStatic = DEFAULT;


   /**
    * Get the createStatic attribute of the UMLClassBodyMethodsStrategy object
    *
    * @return   The createStatic value
    */
   public boolean isCreateStatic()
   {
      return this.createStatic == REQUIRE;
   }


   /**
    * Sets the createStatic attribute of the UMLClassBodyMethodsStrategy object
    *
    * @param createStatic  The new createStatic value
    */
   public void setCreateStatic (boolean createStatic)
   {
      setCreateStatic (createStatic ? REQUIRE : PROHIBIT);
   }


   /**
    * Sets the createStatic attribute of the UMLClassBodyMethodsStrategy object
    *
    * @param createStatic  The new createStatic value
    */
   public void setCreateStatic (int createStatic)
   {
      if (createStatic == DEFAULT || createStatic == REQUIRE || createStatic == PROHIBIT)
      {
         this.createStatic = createStatic;
      }
      else
      {
         throw new IllegalArgumentException();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void resetCreateStatic()
   {
      setCreateStatic (DEFAULT);
   }


   /**
    * Get the createStatic attribute of the UMLClassBodyMethodsStrategy object
    *
    * @return   The createStatic value
    */
   public int getCreateStatic()
   {
      return this.createStatic;
   }


   /**
    * Get the methods attribute of the UMLClassBodyMethodsStrategy object
    *
    * @param umlClass  No description provided
    * @return          The methods value
    */
   protected SortedSet getMethods (FClass umlClass)
   {
      // create sorted set of attributes
      TreeSet methodsSet = new TreeSet (new MethodComparator());
      Iterator methodsIter = umlClass.iteratorOfMethods();
      while (methodsIter.hasNext())
      {
         methodsSet.add (methodsIter.next());
      }
      return methodsSet;
   }
}

/*
 * $Log: UMLClassBodyMethodsStrategy.java,v $
 * Revision 1.10.2.3  2006/07/10 08:05:35  cschneid
 * sysouts removed, exceptions printed (instead of ignoring completely), caption changed to 'Fujaba 4'
 *
 */
