/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.*;

import org.apache.log4j.Logger;
import org.apache.log4j.Priority;

import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.sequencer.*;
import de.uni_paderborn.fujaba.uml.*;


/**
 * Class UMLActivityDiagramOOHandler
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.44.2.5 $
 */
public class UMLActivityDiagramOOHandler
    extends OOGenStrategyHandler
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLActivityDiagramOOHandler.class);


   /**
    * Default Constructor
    */
   public UMLActivityDiagramOOHandler()
   {
      super();
   }


   /**
    * Get the responsible attribute of the UMLActivityDiagramOOHandler object
    *
    * @param incr  No description provided
    * @return      The responsible value
    */
   public boolean isResponsible (FElement incr)
   {
      return  ( (incr instanceof UMLActivityDiagram) && ! (incr instanceof UMLStatechart));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final boolean needToken()
   {
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr       No description provided
    * @param prevToken  No description provided
    * @param param      No description provided
    * @return           No description provided
    */
   public OOGenToken generateSourceCode (FElement incr,
                                         OOGenToken prevToken,
                                         Object param[])
   {
      UMLActivityDiagram theDiagram = (UMLActivityDiagram) incr;

      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".generateSourceCode(theDiagram=" + theDiagram + "," + prevToken + ")");
         }
      }

      try
      {
         Sequencer sequencer = Sequencer.get();
         theDiagram.createFlowAssociations();

         FlowActivity startActivity = theDiagram.getStartActivity().getFlowActivity();

         Seq seq = sequencer.exploreCFG (startActivity);

         OOGenToken current = CodeGenStrategy.checkTokenList (theDiagram, prevToken,
            theDiagram.getFirstOOGenToken(), theDiagram.getLastOOGenToken());
         OOGenToken lastToken = theDiagram.getLastOOGenToken();

         // remove all token in the list to the last token
         OOGenToken nextToken = current.getNext();
         while ( (nextToken != null) &&  (nextToken != lastToken))
         {
            nextToken.removeYouFromList();
            nextToken.removeYou();
            nextToken = current.getNext();
         }

         current = current.insertNewToken ("MethodBeginBody");
         current.appendStatement (OO.beginMethodBody());

         String section = current.newSection ("MethodBody_ActivityDiagram_Variables", lastToken);
         current = current.getNextOrAppend (section, lastToken);

         OOGenToken variables = current;

         current = current.getNextOrAppend (section, lastToken);

         // set VariableTokenAndIndent for all Stories
         FElement tmpItem;
         boolean hasAStory = false;
         Iterator iter = theDiagram.iteratorOfElements();
         while (iter.hasNext())
         {
            tmpItem = (FElement) iter.next();

            if (tmpItem instanceof UMLStoryActivity)
            {
               hasAStory = true;
               break;
            }
         }

         section = current.newSection ("MethodBody_ActivityDiagram_Head", lastToken);
         current = current.getNextOrAppend (section, lastToken);

         if (hasAStory)
         {
            current.appendStatement (new OOStartMethodBodyStatement());
         }

         section = current.newSection ("MethodBody_ActivityDiagram_Body", lastToken);
         current = current.getNextOrAppend (section, lastToken);

         current = seq.generateSourceCode (current, lastToken);

         current = current.insertNewToken ("MethodEndBody");
         current.appendStatement (OO.endMethodBody());

         // delete superfluous tokens in the list
         current.deleteToSection (null, lastToken);

         // fill the variable token
         variables.appendStatement (generateVariableDeclaration (theDiagram));
      }
      catch (RuntimeException runEx)
      {
         if (runEx instanceof SDMParseException)
         {
            if (log.isEnabledFor (Priority.ERROR))
            {
               FElement context =  ((SDMParseException) runEx).getContext();
               String contextName = null;
               String contextType = null;
               if (context != null)
               {
                  contextType = context.getClass().getName();
                  contextName = context.getName();
               }
               log.error ("CFG-parse error during code generation for '" + theDiagram.getFullName() + "', context: " + contextType + ">>" + contextName);
               log.error ("  -> " + runEx.getMessage());
            }
         }
         else
         {
            if (log.isEnabledFor (Priority.WARN))
            {
               log.warn ("Error during code generation for '" + theDiagram.getFullName() + "'");
               log.warn ("Trying to continue ...");
            }
         }
         throw runEx;
      }
      finally
      {
         theDiagram.removeFlowAssociations();
      }

      return null;
   }


   /**
    * This method generates the variable declaration for the variables in the toDeclaredVariables
    * TreeSet.
    *
    * @param theDiagram  No description provided
    * @return            an array of OOLocalVarDeclStatements
    */
   public OOStatement[] generateVariableDeclaration (UMLActivityDiagram theDiagram)
   {
      LinkedList declaredVar = new LinkedList();

      // first, try to collect all used variables
      OOGenToken currentToken = theDiagram.getFirstOOGenToken();
      while (currentToken.getPrev() != null)
      {
         currentToken = currentToken.getPrev();
      }

      while (currentToken != null)
      {
         // ignore declarations in MethodBody_ActivityDiagram_Body section
         if (!currentToken.getSectionName().equals ("MethodBody_ActivityDiagram_Head"))
         {
            Iterator iter = currentToken.iteratorOfStatement();
            while (iter.hasNext())
            {
               OOStatement tmpOOStatement = (OOStatement) iter.next();

               if (tmpOOStatement instanceof OOLocalVarDeclStatement)
               {
                  OOLocalVarDeclStatement tmpOOLocalVarDeclStatement = (OOLocalVarDeclStatement) tmpOOStatement;
                  OOLocalVarDeclStatement newOOLocalVarDeclStatement = OO.varDecl (tmpOOLocalVarDeclStatement);

//                  if (! (newOOLocalVarDeclStatement.getInitExpr() instanceof OOIdentifierExpr))
//                  {
                  newOOLocalVarDeclStatement.setInitExpr (getInitDefault (newOOLocalVarDeclStatement));
//                  }

                  boolean found = false;
                  Iterator tmpIter = declaredVar.iterator();
                  while (tmpIter.hasNext() && !found)
                  {
                     OOLocalVarDeclStatement tmpVar = (OOLocalVarDeclStatement) tmpIter.next();

                     if (tmpVar.getObjectName().equals (newOOLocalVarDeclStatement.getObjectName()) &&
                        tmpVar.getVarType().equals (newOOLocalVarDeclStatement.getVarType()))
                     {
                        found = true;
                     }
                  }

                  if (!found)
                  {
                     declaredVar.add (newOOLocalVarDeclStatement);
                  }
                  tmpOOLocalVarDeclStatement.setDeclared (true);
               }
            }
         }

         currentToken = currentToken.getNext();
      }

      // sort declaredVar
      Collections.sort (declaredVar, new OOLocalVarDeclStatementComparator());

      if (declaredVar.size() > 0)
      {
         // method has internal variable
         declaredVar.add (new OOEmptyLineStatement());
      }

      return OOStatement.toArray (declaredVar);
   } // generateVariableDeclaration


   /**
    * Get the initDefault attribute of the UMLActivityDiagramOOHandler object
    *
    * @param local  No description provided
    * @return       The initDefault value
    */
   public OOExpression getInitDefault (OOLocalVarDeclStatement local)
   {
      if (local.getVarType() instanceof OOType)
      {
         OOType type = (OOType) local.getVarType();
         FType umlType = type.getUmlType();
         if (umlType != null)
         {
            String name = umlType.getName();
            if ( (umlType instanceof FBaseTypes) &&  (name != null) &&  (name.endsWith ("Array"))
               || umlType instanceof FArray)
            {
               return new OOStringExpr ("{}");
            }
            else if (FBaseTypes.INTEGER.equals (name) ||
               FBaseTypes.BYTE.equals (name) ||
               FBaseTypes.SHORT_INTEGER.equals (name) ||
               FBaseTypes.LONG_INTEGER.equals (name) ||
               FBaseTypes.CHARACTER.equals (name))
            {
               return new OOStringExpr ("0");
            }
            else if (FBaseTypes.BOOLEAN.equals (name))
            {
               return OOIdentifierExpr.FALSE_IDENTIFIER;
            }
            else if (FBaseTypes.FLOAT.equals (name) ||
               FBaseTypes.DOUBLE.equals (name))
            {
               return new OOStringExpr ("0.0f");
            }
         }
      }
      return OOIdentifierExpr.NULL_IDENTIFIER;
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "UMLActivityDiagramOOHandler[]";
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: fklar $
    * @version   $Revision: 1.44.2.5 $
    */
   private static class OOLocalVarDeclStatementComparator implements Comparator
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param left   No description provided
       * @param right  No description provided
       * @return       No description provided
       */
      public int compare (Object left, Object right)
      {
         OOLocalVarDeclStatement leftLocalVarDeclStatement = (OOLocalVarDeclStatement) left;
         OOLocalVarDeclStatement rightLocalVarDeclStatement = (OOLocalVarDeclStatement) right;

         int result = 0;

         // OOContainerType first
         if (leftLocalVarDeclStatement.getVarType() instanceof OOContainerType &&
            rightLocalVarDeclStatement.getVarType() instanceof OOType)
         {
            return -1;
         }

         if (leftLocalVarDeclStatement.getVarType() instanceof OOType &&
            rightLocalVarDeclStatement.getVarType() instanceof OOContainerType)
         {
            return 1;
         }

         if (leftLocalVarDeclStatement.getVarType() instanceof OOContainerType)
         {
            // both var type are OOContainerType
            result = leftLocalVarDeclStatement.getVarType().compareTo (rightLocalVarDeclStatement.getVarType());

            if (result != 0)
            {
               // sort by type
               return result;
            }

         }
         else
         {
            // both var type are OOType
            result = leftLocalVarDeclStatement.getVarType().compareTo (rightLocalVarDeclStatement.getVarType());

            if (result != 0)
            {
               // sort by type
               return result;
            }
         }

         // sort by objectName
         return leftLocalVarDeclStatement.getObjectName().compareTo (rightLocalVarDeclStatement.getObjectName());
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param other  No description provided
       * @return       No description provided
       */
      public boolean equals (Object other)
      {
         return  (compare (OOLocalVarDeclStatementComparator.this, other) == 0);
      }
   }

}

/*
 * $Log: UMLActivityDiagramOOHandler.java,v $
 * Revision 1.44.2.5  2006/02/13 20:45:33  fklar
 * reviewed log-messages (more details and right usage of 'log.isEnabledFor(Priority.xxx)')
 *
 */
