/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.Iterator;

import de.uni_paderborn.fujaba.metamodel.*;


/**
 * Class JavaGenVisitor
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.38 $
 */
public class JavaGenVisitor
    extends OOGenVisitor
{
   /**
    * Default Constructor
    *
    * @param extender  No description provided
    */
   public JavaGenVisitor (String extender)
   {
      super (extender);
   }


   /**
    * Constructor for class JavaGenVisitor
    */
   public JavaGenVisitor()
   {
      this ("java");
   }

   //
   // begin assoc template section
   //

   /**
    * Get the referenceTemplateName attribute of the JavaGenVisitor object
    *
    * @return   The referenceTemplateName value
    */
   public String getReferenceTemplateName()
   {
      return "Templates/ReferenceTemplate.tpl";
   }


   /**
    * Get the assocTemplateName attribute of the JavaGenVisitor object
    *
    * @return   The assocTemplateName value
    */
   public String getAssocTemplateName()
   {
      return "Templates/AssocTemplateFCA.tpl";
   }

   //
   // end assoc template section
   //

   /**
    * get the full qualified containername for type <code>containerType</code>
    *
    * @param containerType  the <code>containerType</code>
    * @param bound          No description provided
    * @return               the full qualified containername
    */
   public String getContainerName (OOCollectionEnum containerType, boolean bound)
   {
      if (OOCollectionEnum.ITERATOR.equals (containerType))
      {
         return "java.util.Iterator";
      }
      if (OOCollectionEnum.LIST_ITERATOR.equals (containerType))
      {
         return "java.util.ListIterator";
      }
      if (OOCollectionEnum.KEY_VALUE_PAIR.equals (containerType))
      {
         return "java.util.Map$Entry";
      }

      StringBuffer containerName = new StringBuffer();

      containerName.append ("de.upb.tools.fca.F");

      if (bound)
      {
         containerName.append ("Prop");
      }

      if (OOCollectionEnum.LIST.equals (containerType))
      {
         containerName.append ("LinkedList");
      }
      else if (OOCollectionEnum.HASHTABLE.equals (containerType))
      {
         containerName.append ("HashSet");
      }
      else if (OOCollectionEnum.SET.equals (containerType))
      {
         containerName.append ("HashSet");
      }
      else if (OOCollectionEnum.MAP.equals (containerType))
      {
         containerName.append ("HashMap");
      }
      else if (OOCollectionEnum.MULTI_MAP.equals (containerType))
      {
         containerName.append ("DuplicatedHashMap");
      }
      else if (OOCollectionEnum.TREESET.equals (containerType))
      {
         containerName.append ("TreeSet");
      }
      else if (OOCollectionEnum.SORTED_MAP.equals (containerType))
      {
         containerName.append ("TreeMap");
      }
      else if (OOCollectionEnum.SORTED_MULTI_MAP.equals (containerType))
      {
         containerName.append ("DuplicatedTreeMap");
      }

      return containerName.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void initProgLangTypes()
   {
      addToProgLangTypes (FBaseTypes.INITIALIZER, "");
      addToProgLangTypes (FBaseTypes.BOOLEAN, "boolean");
      addToProgLangTypes (FBaseTypes.CHARACTER, "char");
      addToProgLangTypes (FBaseTypes.STRING, FBaseTypes.STRING);
      addToProgLangTypes (FBaseTypes.INTEGER, "int");
      addToProgLangTypes (FBaseTypes.BYTE, "byte");
      addToProgLangTypes (FBaseTypes.SHORT_INTEGER, "short");
      addToProgLangTypes (FBaseTypes.LONG_INTEGER, "long");
      addToProgLangTypes (FBaseTypes.FLOAT, "float");
      addToProgLangTypes (FBaseTypes.DOUBLE, "double");
      addToProgLangTypes (FBaseTypes.VOID, "void");
      addToProgLangTypes (FBaseTypes.CONSTRUCTOR, "");
      addToProgLangTypes ("Constructor", "");

      addToProgLangTypes ("BooleanArray", "boolean[]");
      addToProgLangTypes ("CharacterArray", "char[]");
      addToProgLangTypes ("StringArray", "String[]");
      addToProgLangTypes ("IntegerArray", "int[]");
      addToProgLangTypes ("ByteArray", "byte[]");
      addToProgLangTypes ("ShortIntegerArray", "short[]");
      addToProgLangTypes ("LongIntegerArray", "long[]");
      addToProgLangTypes ("FloatArray", "float[]");
      addToProgLangTypes ("DoubleArray", "double[]");
      addToProgLangTypes ("ObjectArray", "Object[]");

      // fix me
      addToProgLangTypes ("FWIterator", "Iterator");
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void initVisibilityStrings()
   {
      visibilityStrings = new String[]{
         "private",
         "public",
         "protected",
         "/*package*/",
         "/*user*/"
         };
   }

   //
   // begin visit OOStatement section
   //

   /**
    * Get the sourceCode attribute of the JavaGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public final String getSourceCode (OOThrowStatement statement)
   {
      StringBuffer result = new StringBuffer();

      result.append (createIndentString());
      result.append ("throw new ");
      result.append (statement.getMyException().getSourceCode (this));
      result.append (" () ;");

      return result.toString();
   }

   //
   // end visit OOStatement section
   //

   //
   // begin visit OOExpression section
   //

   /**
    * Get the sourceCode attribute of the JavaGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOObjectOfTypeExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append (getVarName (expression.getObjectName()));
      result.append (" instanceof ");
      result.append (getOOTypeAsString (expression.getTypeName()));

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the JavaGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOClassOfVariableExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append (getVarName (expression.getObjectName()));
      result.append (".getClass()");

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the JavaGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorOfExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append (getVarName (expression.getObjectName()));
      result.append (".iterator ()");

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the JavaGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorGetExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append (getVarName (expression.getObjectName()));
      result.append (".next ()");

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the JavaGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorHasNextExpr expression)
   {
      StringBuffer result = new StringBuffer();

      result.append (getVarName (expression.getIterName()));
      result.append (".hasNext ()");

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the JavaGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorGotoNextExpr expression)
   {
      return null;
   }


   /**
    * Get the sourceCode attribute of the JavaGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public String getSourceCode (OONewArrayExpr expression)
   {
      StringBuffer result = new StringBuffer();

      Iterator iter = expression.iteratorOfParameter();

      result.append ("new ");
      result.append (getOOTypeAsString (expression.getElementType()));
      result.append ("[");
      if (!iter.hasNext())
      {
         result.append (expression.getSizeExpr().getSourceCode (this));
      }
      result.append ("] ");

      if (iter.hasNext())
      {
         result.append ("{");

         while (iter.hasNext())
         {
            OOExpression tmpParameter = (OOExpression) iter.next();
            result.append (tmpParameter.getSourceCode (this));
            if (iter.hasNext())
            {
               result.append (", ");
            }
         }

         result.append ("}");
      }

      return result.toString();
   }


   //
   // end visit OOExpression section
   //

   //
   // generate section
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void generateClassEndBlock()
   {
      getCurrentBuffer().append (END_BLOCK);
      getCurrentBuffer().append ("\n");

      decIndentNr();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void generateClassStartBlock()
   {
      getCurrentBuffer().append (START_BLOCK);
      getCurrentBuffer().append ("\n");

      incIndentNr();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param thePackage  No description provided
    * @return            No description provided
    */
   public String generateImportPackage (FPackage thePackage)
   {
      String packageName = thePackage.getFullPackageName();

      if (packageName.length() > 0)
      {
         StringBuffer result = new StringBuffer();
         result.append ("import ").append (packageName).append (".*;\n");
         return result.toString();
      }
      return "";
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    * @return          No description provided
    */
   public String generateImportClass (FClass theClass)
   {
      StringBuffer result = new StringBuffer();
      result.append ("import ").append (theClass.getFullClassName()).append (";\n");
      return result.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theFile  No description provided
    */
   public void generatePackageBegin (FFile theFile)
   {
      FPackage thePackage = theFile.getFPackage();
      String packageName =  (thePackage != null ? thePackage.getFullPackageName() : "");

      if (packageName.length() > 0)
      {
         getCurrentBuffer().append ("package ");
         getCurrentBuffer().append (packageName);
         getCurrentBuffer().append (";\n\n");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theFile  No description provided
    */
   public void generatePackageEnd (FFile theFile) { }

   //
   // create section
   //

   //
   // append section
   //

   //
   // get identifier, keywords, literals etc.
   //

   /**
    * Get the attrDeclName attribute of the JavaGenVisitor object
    *
    * @param theAttr  No description provided
    * @return         The attrDeclName value
    */
   public String getAttrDeclName (FAttr theAttr)
   {
      return theAttr.getName();
   }


   /**
    * Get the methodDeclName attribute of the JavaGenVisitor object
    *
    * @param theMethod  No description provided
    * @return           The methodDeclName value
    */
   public String getMethodDeclName (FMethod theMethod)
   {
      return theMethod.getName();
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "JavaGenVisitor[]";
   }
}

/*
 * $Log: JavaGenVisitor.java,v $
 * Revision 1.38  2004/11/30 16:24:44  creckord
 * - removed UpdateImportOfFileHandler.
 * Imports required by our model implementation are now handled
 * during codegen without changes to the model.
 *
 * - changed qualifier in UMLTypelist to use the full qualified name of classes
 *
 */
