/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.io.File;
import java.util.Iterator;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.metamodel.FAttr;
import de.uni_paderborn.fujaba.metamodel.FClass;
import de.uni_paderborn.fujaba.metamodel.FFile;
import de.uni_paderborn.fujaba.metamodel.FMethod;
import de.uni_paderborn.fujaba.metamodel.FPackage;
import de.uni_paderborn.fujaba.metamodel.FParam;
import de.uni_paderborn.fujaba.metamodel.FRole;
import de.uni_paderborn.fujaba.metamodel.FType;
import de.uni_paderborn.fujaba.uml.UMLActivityDiagram;
import de.uni_paderborn.fujaba.uml.UMLAttrExprPair;
import de.uni_paderborn.fujaba.uml.UMLObject;
import de.uni_paderborn.fujaba.uml.UMLProject;
import de.uni_paderborn.fujaba.uml.UMLStartActivity;
import de.uni_paderborn.fujaba.uml.UMLStoryActivity;


/**
 * @author    <a href="mailto:creckord@upb.de">Carsten Reckord</a>
 * @version   $Revision: 1.1 $
 */
public class CreateStandardImportsOOFunction extends CreateImportsOOFunction
{
   /**
    *Constructor for class CreateStandardImportsOOFunction
    */
   public CreateStandardImportsOOFunction()
   {
      super (NORM);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlFile     No description provided
    * @param importFrag  No description provided
    */
   protected void generateImports (FFile umlFile, ImportFragment importFrag)
   {
      FPackage sdmPackage;
      FPackage utilPackage;
      FPackage fcaPackage;
      FClass patternClass;

      boolean transientMode = ASGElement.isInTransientMode();
      try
      {
         ASGElement.setInTransientMode (true);

         sdmPackage = UMLProject.get().getNewFromPackages ("de.upb.tools.sdm");
         utilPackage = UMLProject.get().getNewFromPackages ("java.util");
         fcaPackage = UMLProject.get().getNewFromPackages ("de.upb.tools.fca");

         patternClass = UMLProject.get().getOrNewFromReferences ("java.util.regex.Pattern");
      }
      finally
      {
         ASGElement.setInTransientMode (transientMode);
      }

      boolean sdmAndUtilAndFCAPackagesAdded =  (umlFile.hasInImportedPackages (sdmPackage) &&
         umlFile.hasInImportedPackages (utilPackage) && umlFile.hasInImportedPackages (fcaPackage));

      boolean patternClassAdded = umlFile.hasInImportedClasses (patternClass);

      Iterator iter = umlFile.iteratorOfContains();
      while (iter.hasNext())
      {
         FClass umlClass = (FClass) iter.next();
         FPackage umlPackage = umlClass.findFPackage();

         // add attr types to import
         Iterator attrIter = umlClass.iteratorOfAttrs();
         while (attrIter.hasNext())
         {
            FAttr umlAttr = (FAttr) attrIter.next();
            checkAndAddPackageOfTypeToImport (umlAttr.getFAttrType(), umlPackage, umlFile, importFrag);
         }

         // add inherited classes to import
         Iterator superClassIter = umlClass.iteratorOfSuperClasses();
         while (superClassIter.hasNext())
         {
            FClass superClass = (FClass) superClassIter.next();
            checkAndAddPackageOfTypeToImport (superClass, umlPackage, umlFile, importFrag);
         }

         // and now the methods
         Iterator methodIter = umlClass.iteratorOfMethods();
         while (methodIter.hasNext())
         {
            FMethod umlMethod = (FMethod) methodIter.next();

            // check the return type
            checkAndAddPackageOfTypeToImport (umlMethod.getFResultType(), umlPackage, umlFile, importFrag);

            // check the parameter types
            Iterator paramIter = umlMethod.iteratorOfParam();
            while (paramIter.hasNext())
            {
               FParam umlParam = (FParam) paramIter.next();
               checkAndAddPackageOfTypeToImport (umlParam.getFParamType(), umlPackage, umlFile, importFrag);
            }

            Iterator throwIter = umlMethod.iteratorOfThrowsTypes();
            while (throwIter.hasNext())
            {
               FType tmpThrow = (FType) throwIter.next();
               checkAndAddPackageOfTypeToImport (tmpThrow, umlPackage, umlFile, importFrag);
            }

            if (umlMethod.getFRevSpec() != null)
            {
               UMLActivityDiagram diagram =  ((UMLStartActivity) umlMethod.getFRevSpec()).getActivityDiagram();
               if (diagram != null)
               {
                  Iterator itemsIter = diagram.iteratorOfElements();
                  while (!sdmAndUtilAndFCAPackagesAdded && itemsIter.hasNext())
                  {
                     Object tmpItem = itemsIter.next();

                     if (tmpItem instanceof UMLStoryActivity)
                     {
                        if (!umlFile.hasInImportedPackages (sdmPackage))
                        {
                           importFrag.addPackage (sdmPackage);
                        }
                        if (!umlFile.hasInImportedPackages (utilPackage))
                        {
                           importFrag.addPackage (utilPackage);
                        }
                        if (!umlFile.hasInImportedPackages (fcaPackage))
                        {
                           importFrag.addPackage (fcaPackage);
                        }

                        sdmAndUtilAndFCAPackagesAdded = true;
                     }
                  }

                  Iterator objIter = diagram.iteratorOfObjects();
                  while (objIter.hasNext())
                  {
                     UMLObject umlObject = (UMLObject) objIter.next();

                     // add imports of object's package
                     FClass instanceOfClass = umlObject.getInstanceOf();
                     checkAndAddPackageOfTypeToImport (instanceOfClass, umlPackage, umlFile, importFrag);

                     // add java.util.regex.Pattern if regular expression is used in UMLAttrExprPair
                     Iterator attrExprPairIter = umlObject.iteratorOfAttrs();
                     while (!patternClassAdded && attrExprPairIter.hasNext())
                     {
                        UMLAttrExprPair attrExprPair = (UMLAttrExprPair) attrExprPairIter.next();
                        if (attrExprPair.getOperation() == UMLAttrExprPair.REG_EXPRESSION)
                        {
                           if (!umlFile.hasInImportedClasses (patternClass))
                           {
                              importFrag.addClass (patternClass);

                              patternClassAdded = true;
                           }
                        }
                     }
                  }
               }
            }
         }

         //and now the assocs
         Iterator roleIter = umlClass.iteratorOfRoles();
         while (roleIter.hasNext())
         {
            FRole role = (FRole) roleIter.next();
            FRole partnerRole = role.getFPartnerRole();
            if (partnerRole != null &&
               role.getAdornment() != FRole.REFERENCE)
            {
               FClass partnerClass = partnerRole.getFTarget();
               checkAndAddPackageOfTypeToImport (partnerClass, umlPackage, umlFile, importFrag);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theType     No description provided
    * @param myPackage   No description provided
    * @param theFile     No description provided
    * @param importFrag  No description provided
    */
   protected void checkAndAddPackageOfTypeToImport (FType theType,
                                                    FPackage myPackage,
                                                    FFile theFile, ImportFragment importFrag)
   {
      if (theType != null && theType instanceof FClass)
      {
         FClass theClass = (FClass) theType;
         FPackage tmpPackage = theClass.findFPackage();
         if (tmpPackage != null && tmpPackage != myPackage)
         { // type from other package

            if (! (theFile.hasInImportedClasses (theClass) ||  // not yet in imported files
            theFile.hasInImportedPackages (tmpPackage)))
            { // not yet in imported packages

               String fullPackName = tmpPackage.getPackagePath();
               if (fullPackName != null && !fullPackName.equals ("."))
               {
                  // add valid package
                  if ( ("de" + File.separator + "upb" + File.separator + "tools" + File.separator + "fca").equals (fullPackName))
                  {
                     importFrag.addPackage (tmpPackage);
                  }
                  else
                  {
                     importFrag.addClass (theClass);
                  }
               }
            }
         }
      }
   }

}

/*
 * $Log: CreateStandardImportsOOFunction.java,v $
 * Revision 1.1  2004/11/30 16:24:43  creckord
 * - removed UpdateImportOfFileHandler.
 * Imports required by our model implementation are now handled
 * during codegen without changes to the model.
 *
 * - changed qualifier in UMLTypelist to use the full qualified name of classes
 *
 */
