/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.io.*;
import org.apache.log4j.Logger;
import org.xml.sax.SAXException;

import de.uni_paderborn.fujaba.codegen.digester.XDigester;
import de.uni_paderborn.fujaba.codegen.digester.rules.*;
import de.uni_paderborn.lib.classloader.UPBClassLoader;


/**
 * Class CodeGenTargetParser
 *
 * @author    $Author: schneider $
 * @author    $Author: schneider $
 * @version   $Revision: 1.18 $
 */
public class CodeGenTargetParser
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (CodeGenTargetParser.class);

   /**
    * the Singleton instance
    */
   private static CodeGenTargetParser theInstance = null;


   /**
    * Singleton has private default constructor
    */
   private CodeGenTargetParser()
   {
      super();
   }


   /**
    * get the Singleton instance
    *
    * @return   the Singleton instance
    */
   public static CodeGenTargetParser get()
   {
      if (theInstance == null)
      {
         theInstance = new CodeGenTargetParser();
      }

      return theInstance;
   }


   /**
    * @param filename
    * @deprecated      will be replaced by parseFile
    */
   public void parse (String filename)
   {
      parseFile (filename);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param filename  No description provided
    * @param loader    No description provided
    */
   public void parse (String filename, ClassLoader loader)
   {
      parseFile (filename, loader);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param filename  No description provided
    * @return          No description provided
    */
   public boolean parseFile (String filename)
   {
      return parseFile (filename, UPBClassLoader.get());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param filename  No description provided
    * @param loader    No description provided
    * @return          No description provided
    */
   public boolean parseFile (String filename, ClassLoader loader)
   {
      CodeGenTargetParserPair pair = null;

      try
      {
         pair = parsePair (filename, loader);
      }
      catch (Exception exception)
      {
         return false;
      }

      if (pair != null && pair.isValid())
      {
         CodeGenFactory.get().addToCodeGenTarget (pair.getTarget());
         CodeGenFactory.get().addToCodeGenStrategy (pair.getStrategy());

         return true;
      }
      else
      {
         return false;
      }
   }


   /**
    * try to parse filename.. return targeentry and strategy as pair if successful , else return
    * null
    *
    * @param filename  name of targetfile
    * @return          No description provided
    */
   protected CodeGenTargetParserPair parsePair (String filename)
   {
      return parsePair (filename, UPBClassLoader.get());
   }


   /**
    * try to parse filename with loader as ClassLoader . return targeentry and strategy as
    * pair if successful , else return null
    *
    * @param filename  name of targetfile
    * @param loader    the classloader
    * @return          No description provided
    */
   protected CodeGenTargetParserPair parsePair (String filename, ClassLoader loader)
   {
      InputStream fis;

      fis = loader.getResourceAsStream (filename);

      if (fis == null)
      {
         return null;
      }

      BufferedInputStream bis = new BufferedInputStream (fis);

      CodeGenTargetParserPair result = new CodeGenTargetParserPair();

      XDigester digester = createDigester (result);
      digester.setClassLoader (loader);

      try
      {
         digester.push (result);
         digester.parse (bis);
         bis.close();
      }
      catch (IOException e)
      {
         // log.error(e);
         log.error ("IOException:" + e);
         e.printStackTrace();

         return null;
      }
      catch (SAXException e)
      {
         //	log.error(e);
         log.error ("SAXException:" + e);
         e.printStackTrace();

         return null;
      }

      digester.pop();

      return result;
   }


   /**
    * UMLAttribute : 'debugLevel : int '
    */
   private int debugLevel = 0;


   /**
    * Get the value of debugLevel.
    *
    * @return   Value of debugLevel.
    */
   public int getDebugLevel()
   {
      return this.debugLevel;
   }


   /**
    * Set the value of debugLevel.
    *
    * @param debugLevel  Value to assign to debugLevel.
    */
   public void setDebugLevel (int debugLevel)
   {
      if (this.debugLevel != debugLevel)
      {
         this.debugLevel = debugLevel;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pair  No description provided
    * @return      No description provided
    */
   private XDigester createDigester (CodeGenTargetParserPair pair)
   {
      XDigester digester = new XDigester();

      digester.push (pair);
      digester.setDebug (getDebugLevel());

      digester.setValidating (false);

      //
      // add rules
      //

      // create target entries
      digester.addObjectCreate ("*/codegentarget",
         "de.uni_paderborn.fujaba.codegen.CodeGenTarget");
      digester.addSetNext ("*/codegentarget",
         "setTarget",
         "de.uni_paderborn.fujaba.codegen.CodeGenTarget");

      digester.addObjectCreate ("*/codegentargetentry",
         "de.uni_paderborn.fujaba.codegen.CodeGenTargetEntry");
      digester.addSetNext ("*/codegentargetentry",
         "addToCodeGenTargetEntry", "de.uni_paderborn.fujaba.codegen.CodeGenTargetEntry");

      digester.addRule ("*/strategyName",
         new FUICallMethodRule (digester,
         "setStrategyName", 0));
      digester.addRule ("*/visitorName",
         new FUICallMethodRule (digester,
         "setVisitorName", 0));

      // create strategy
      digester.addObjectCreate ("*/codegenstrategy",
         "de.uni_paderborn.fujaba.codegen.CodeGenStrategy",
         "name");

      digester.addSetNext ("*/codegenstrategy",
         "setStrategy",
         "de.uni_paderborn.fujaba.codegen.CodeGenStrategy");

      // set name
      digester.addRule ("*/name",
         new FUICallMethodRule (digester,
         "setName", 0));

      digester.addRule ("*/fullName",
         new FUICallMethodRule (digester,
         "setFullName", 0));

      // create visitor
      digester.addObjectCreate ("*/visitor",
         "de.uni_paderborn.fujaba.codegen.CodeGenVisitor",
         "name");
      digester.addSetNext ("*/visitor",
         "addToCodeGenVisitor",
         "de.uni_paderborn.fujaba.codegen.CodeGenVisitor");

      // create some objects
      digester.addObjectCreate ("*/object",
         "java.lang.Object",
         "name");

      // add object to parent
      digester.addRule ("*/object",
         new XAddToRule (digester, "method", "name"));

      // set mode
      digester.addRule ("*/object/mode",
         new FUICallMethodRule (digester,
         "setMode", 0,
         new String[]{
         "int"
         }
         ));

      // set continueChain
      digester.addRule ("*/object/continueChain",
         new FUICallMethodRule (digester,
         "setContinueChain", 0,
         new String[]{
         "boolean"
         }
         ));

      digester.addRule ("*/property",
         new XSetPropertyRule (digester, "name", "value"));
      digester.addRule ("*/properties",
         new XSetPropertiesRule (digester));

      return digester;
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "CodeGenTargetParser[]";
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param args  No description provided
    */
   public static void main (String args[])
   {
      CodeGenTargetParser parser = CodeGenTargetParser.get();

      parser.parse ("de/uni_paderborn/fujaba/codegen/javatarget.xml");

      System.exit (0);
   }

}

/*
 * $Log: CodeGenTargetParser.java,v $
 * Revision 1.18  2004/10/20 17:49:30  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
