/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.io.File;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Iterator;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.upb.tools.fca.*;


/**
 * Class CodeGenStrategy <h2> Associations </h2> <pre>
 *                -------- 0..1       hasStrategies       0..1
 * CodeGenFactory | name |------------------------------------- CodeGenStrategy
 *                -------- codeGenFactory      codeGenStrategy
 *
 *                 -------- 0..1        hasVisitors        0..1
 * CodeGenStrategy | name |------------------------------------- CodeGenVisitor
 *                 -------- codeGenStrategy      codeGenVisitor
 *
 *                  0..1                        0..1
 * CodeGenStrategy ---------------------------------- CodeGenFactory
 *                  currentStrategy   currentFactory
 *
 *                  0..1                          0..1
 * CodeGenStrategy ------------------------------------ CodeGenVisitor
 *                  currentStrategy     currentVisitor
 * </pre>
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.45.2.1 $
 */
public abstract class CodeGenStrategy
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (CodeGenStrategy.class);


   /**
    * Default Constructor
    */
   public CodeGenStrategy()
   {
      // add end of chain handler as last element of the handler chain
      EndOfChainOOHandler endOfChainOOHandler = new EndOfChainOOHandler();
      addToHandlerChain (endOfChainOOHandler);

      // add end of chain function as last element of the function chain
      EndOfChainOOFunction endOfChainFunction = new EndOfChainOOFunction();
      setFunction (endOfChainFunction);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theProject  No description provided
    */
   public abstract void generateFProject (FProject theProject);


   /**
    * Generates sourcecode for a given <code>FElement</code> without storing it in a file.
    *
    * @param element  The model element for which source code will be generated.
    * @return         the generated sourcecode
    */
   public final StringBuffer generateFElement (FElement element)
   {
      //if (log.isInfoEnabled()) log.info (this + ".generateFElement(" + element + ")");
      return generateFElement (element, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element  No description provided
    * @return         No description provided
    */
   public boolean handlerIncrNeedsToken (FElement element)
   {
      CodeGenStrategyHandler nextHandler = null;
      boolean result = false;
      boolean done = false;

      Iterator iter = this.iteratorOfHandlerChain();
      while (!done && iter.hasNext())
      {
         nextHandler = (CodeGenStrategyHandler) iter.next();

         if (nextHandler.isResponsible (element))
         {
            result = nextHandler.incrNeedsToken (element);
            done = true;
         }
      }

      return result;
   }


   /**
    * Generates sourcecode for a given <code>FElement</code> with the possibility of saving
    * it to a file.
    *
    * @param element  The model element for which source code will be generated.
    * @param save     Defines whether the generated sourcecode has to be saved or not.
    * @return         the generated sourcecode
    */
   public final StringBuffer generateFElement (FElement element, boolean save)
   {
      boolean needsToken = handlerIncrNeedsToken (element);
      OOGenToken token = null;
      StringBuffer buf = null;
      CodeGenVisitor visitor = getCurrentVisitor();

      // init code buffer
      visitor.initBuffer();

      if (needsToken)
      {
         token = new OOGenToken();
      }

      generateSourceCodeFor (element, token, new Object[]{Boolean.valueOf (save)});

      buf = visitor.getCurrentBuffer();

      if (needsToken)
      {
         // skip leading null's and empty lines
         while (token != null)
         {
            if (token.getSourceCode() != null && token.getSourceCode().trim().length() != 0)
            {
               break;
            }
            token = token.getNext();
         }

         while (token != null)
         {
            buf.append (token.getSourceCode());
            token = token.getNext();
         }
      }

      buf = new StringBuffer();
      CodeGenFragment fragment = visitor.getFirstFragment();
      while (fragment != null)
      {
         buf.append (fragment.getSourceCode());
         fragment = fragment.getNextFragment();
      }

      return buf;
   }


   /**
    * Generate token for an element. Subclasses must override this method. For backward compatibility
    * this method is not abstract but invokes the old version.
    *
    * @param asgElement  No description provided
    * @param prevToken   No description provided
    * @param param       No description provided
    * @return            No description provided
    */
   public OOGenToken handlerGenerate (FElement asgElement, OOGenToken prevToken, Object param[])
   {
      return handlerGenerate ((ASGElement) asgElement, prevToken, param);
   }


   /**
    * @param asgElement
    * @param prevToken
    * @param param
    * @return            No description provided
    * @deprecated        use {@link #handlerGenerate(de.uni_paderborn.fujaba.metamodel.FElement,
    *      OOGenToken, Object[])}
    */
   public OOGenToken handlerGenerate (ASGElement asgElement, OOGenToken prevToken, Object param[])
   {
      // iterate through the handlers until work is done
      Iterator iter = this.iteratorOfHandlerChain();
      boolean done = false;
      OOGenToken result = null;

      while (!done && iter.hasNext())
      {
         CodeGenStrategyHandler nextHandler = (CodeGenStrategyHandler) iter.next();

         try
         {
            if (nextHandler.isResponsible ((FElement) asgElement))
            {
               result = nextHandler.generate (asgElement, prevToken, param, true);
               prevToken = result;
               done = !nextHandler.isContinueChain();
            }
         }
         catch (AbstractMethodError e)
         {
            log.error ("handler of class " + nextHandler.getClass().getName() + " does not implement new interface!");
            throw e;
         }

      }

      if (!done)
      {
         // no handler found
         throw new IllegalStateException ("incr=" + asgElement + ",prevToken="
            + prevToken + ",param=" + param);
      }

      return result;
   }


   /**
    * This method calls the generateSourceCode () method on incr with parameter prevToken.
    * The return value is the last token to go on with. It also deletes superfluous tokens.
    *
    * @param incr       the increment on which to call generateJava
    * @param prevToken  the previous token where to insert the new one
    * @param param      No description provided
    * @return           the last token of the list generated by the increment
    */
   public OOGenToken generateSourceCodeFor (FElement incr, OOGenToken prevToken, Object param[])
   {
      if (log.isDebugEnabled())
      {
         log.debug (
            this
            + ".generateSourceCodeFor(incr="
            + incr
            + ",class="
            +  (incr != null ? incr.getClass().getName() : "null")
            + ")");
      }

      // append code for incr to prevToken
      if (incr != null)
      {
         prevToken = handlerGenerate (incr, prevToken, param);
      }
      else
      {
         log.error ("incr is null !", new Throwable());
         return null;
      }

      // FIX ME: Since UMLRole tends to have no code (when an corresponding
      // attribute exists already, we need a work around here.
      if (incr.getLastOOGenToken() == null)
      {
         return prevToken;
      }
      else
      {
         return incr.getLastOOGenToken();
      }
   }


   /**
    * Describe <code>generateSourceCodeFor</code> method here.
    *
    * @param iter     an <code>Iterator</code> value
    * @param current  an <code>OOGenToken</code> value
    * @param param    No description provided
    * @return         an <code>OOGenToken</code> value
    */
   public OOGenToken generateSourceCodeFor (Iterator iter, OOGenToken current, Object param[])
   {
      if (log.isDebugEnabled())
      {
         log.debug (this + ".generateSourceCodeFor(iter=" + iter + ")");
      }

      while (iter.hasNext())
      {
         current = generateSourceCodeFor ((FElement) iter.next(), current, param);
      }
      return current;
   }


   /**
    * Describe <code>generateSourceCodeFor</code> method here.
    *
    * @param enumeration  an <code>Enumeration</code> value
    * @param current      an <code>OOGenToken</code> value
    * @param param        No description provided
    * @return             an <code>OOGenToken</code> value
    */
   public OOGenToken generateSourceCodeFor (Enumeration enumeration, OOGenToken current, Object param[])
   {
      if (log.isDebugEnabled())
      {
         log.debug (this + ".generateSourceCodeFor(enumeration=" + enumeration + ")");
      }

      while (enumeration.hasMoreElements())
      {
         current = generateSourceCodeFor ((FElement) enumeration.nextElement(), current, param);
      }
      return current;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr        No description provided
    * @param methodName  No description provided
    * @param param       No description provided
    * @return            No description provided
    */
   public final Object generateCode (FElement incr, String methodName, Object param[])
   {
      CodeGenFunction responsible = CodeGenFunction.findNextResponsible (getFunction(), methodName);
      if (responsible != null)
      {
         return responsible.generate (incr, methodName, param);
      }
      throw new IllegalStateException ("methodName=" + methodName +
         ",param=" + Arrays.asList (param));
   }


   /**
    * Checks if a token list is present and if not creates a new one. It throws an runtime
    * exception if the list only contains one token or the first or last token is null. If
    * the prevToken is not the previous token of firstToken then it cuts out the list and inserts
    * it behind prevToken.
    *
    * @param increment   an <code>FElement</code> value
    * @param prevToken   marks the new position for the list
    * @param firstToken  marks the first token in the list
    * @param lastToken   marks the last token in the list
    * @return            the 'new' first token
    */
   public static OOGenToken checkTokenList (
                                            FElement increment,
                                            OOGenToken prevToken,
                                            OOGenToken firstToken,
                                            OOGenToken lastToken)
   {
      if (firstToken == null)
      {
         firstToken = OOGenToken.createNewList (prevToken);
         lastToken = firstToken.getNext();
         increment.setFirstOOGenToken (firstToken);
         increment.setLastOOGenToken (lastToken);
      }

      if ( (firstToken == lastToken) ||  (firstToken == null) ||  (lastToken == null))
      {
         throw new RuntimeException ("Token list is not correct");
      }

      if (firstToken.getPrev() != prevToken)
      {
         OOGenToken.cutAndPaste (firstToken, lastToken, prevToken);
      }

      return firstToken;
   }

   //
   // begin delegation section
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param path  No description provided
    * @param name  No description provided
    */
   public void createFiles (String path, String name)
   {
      getCurrentVisitor().createFiles (path, name);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param path        No description provided
    * @param name        No description provided
    * @param writeFiles  No description provided
    */
   public void initVisitor (String path, String name, boolean writeFiles)
   {
      getCurrentVisitor().initVisitor (path, name, writeFiles);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void initBuffer()
   {
      getCurrentVisitor().initBuffer();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void writeFiles()
   {
      getCurrentVisitor().writeFiles();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param code  No description provided
    * @return      No description provided
    */
   public CodeGenStrategy append (StringBuffer code)
   {
      getCurrentVisitor().append (code);
      return this;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param code  No description provided
    * @return      No description provided
    */
   public CodeGenStrategy append (String code)
   {
      getCurrentVisitor().append (code);
      return this;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String createIndentString()
   {
      return getCurrentVisitor().createIndentString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text  No description provided
    * @return      No description provided
    */
   public String indentText (String text)
   {
      return getCurrentVisitor().indentText (text);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text  No description provided
    * @return      No description provided
    */
   public String indentText (StringBuffer text)
   {
      return getCurrentVisitor().indentText (text.toString());
   }


   /**
    * Get the currentBuffer attribute of the OOGenStrategyClient object
    *
    * @return   The currentBuffer value
    */
   public StringBuffer getCurrentBuffer()
   {
      return getCurrentVisitor().getCurrentBuffer();
   }

   //
   // end delegation section
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text  No description provided
    * @return      No description provided
    */
   public String createLineCommentary (String text)
   {
      return getCurrentVisitor().createLineCommentary (text);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param text  No description provided
    * @return      No description provided
    */
   public String createCommentary (String text)
   {
      return getCurrentVisitor().createCommentary (text);
   }


   /**
    * Returns the directory of the source file corresponding to theFile.
    *
    * @param theFile  the file whose directory is to be returned.
    * @return         the name of the directory.
    */
   public String getFilePath (FFile theFile)
   {
      String theFilePath = GeneralPreferences.get().getExportFolder();

      FPackage thePackage = theFile.getFPackage();

      if (thePackage != null)
      {
         theFilePath += File.separator + thePackage.getPackagePath();
      }

      return theFilePath;
   }


   /**
    * Returns the name of the source file corresponding to theFile.
    *
    * @param theFile  the file whose source file is to be returned
    * @return         the name of the source file without extension
    */
   public String getFileName (FFile theFile)
   {
      return getCurrentVisitor().getFileName (theFile);
   }


   /**
    * <pre>
    *                  0..1                        0..1
    * CodeGenStrategy ---------------------------------- CodeGenFactory
    *                  currentStrategy   currentFactory
    * </pre>
    */
   private CodeGenFactory currentFactory;


   /**
    * @param value  The new currentFactory value
    * @return       No description provided
    * @see          #currentFactory
    */
   public boolean setCurrentFactory (CodeGenFactory value)
   {
      if (this.currentFactory != value)
      {
         if (this.currentFactory != null)
         {
            CodeGenFactory oldValue = this.currentFactory;
            this.currentFactory = null;
            oldValue.setCurrentStrategy (null);
         }
         this.currentFactory = value;
         if (value != null)
         {
            this.currentFactory.setCurrentStrategy (this);
         }

         return true;
      }

      return false;
   }


   /**
    * @return   The currentFactory value
    * @see      #currentFactory
    */
   public CodeGenFactory getCurrentFactory()
   {
      return this.currentFactory;
   }


   /**
    * <pre>
    *                  0..1                          0..1
    * CodeGenStrategy ------------------------------------ CodeGenVisitor
    *                  currentStrategy     currentVisitor
    * </pre>
    */
   private CodeGenVisitor currentVisitor;


   /**
    * @param value  The new currentVisitor value
    * @return       No description provided
    * @see          #currentVisitor
    */
   public boolean setCurrentVisitor (CodeGenVisitor value)
   {
      if (this.currentVisitor != value)
      {
         if (this.currentVisitor != null)
         {
            CodeGenVisitor oldValue = this.currentVisitor;
            this.currentVisitor = null;
            oldValue.setCurrentStrategy (null);
         }
         this.currentVisitor = value;
         if (value != null)
         {
            this.currentVisitor.setCurrentStrategy (this);
         }

         return true;
      }

      return false;
   }


   /**
    * @return   The currentVisitor value
    * @see      #currentVisitor
    */
   public CodeGenVisitor getCurrentVisitor()
   {
      return this.currentVisitor;
   }


   /**
    * <pre>
    *                -------- 0..1       hasStrategies       0..1
    * CodeGenFactory | name |------------------------------------- CodeGenStrategy
    *                -------- codeGenFactory      codeGenStrategy
    * </pre>
    */
   private CodeGenFactory codeGenFactory;


   /**
    * Get the codeGenFactory attribute of the CodeGenStrategy object
    *
    * @return   The codeGenFactory value
    */
   public CodeGenFactory getCodeGenFactory()
   {
      return this.codeGenFactory;
   }


   /**
    * Sets the codeGenFactory attribute of the CodeGenStrategy object
    *
    * @param value  The new codeGenFactory value
    * @return       No description provided
    */
   public boolean setCodeGenFactory (CodeGenFactory value)
   {
      boolean changed = false;
      if (this.codeGenFactory != value)
      {
         if (this.codeGenFactory != null)
         {
            CodeGenFactory oldValue = this.codeGenFactory;
            this.codeGenFactory = null;
            oldValue.removeFromCodeGenStrategy (this);
         }
         this.codeGenFactory = value;
         if (value != null)
         {
            value.addToCodeGenStrategy (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name;


   /**
    * Get the value of name.
    *
    * @return   Value of name.
    */
   public String getName()
   {
      return this.name;
   }


   /**
    * Set the value of name.
    *
    * @param name  Value to assign to name.
    */
   public void setName (String name)
   {
      if ( (this.name == null && name != null) ||  (this.name != null && !this.name.equals (name)))
      {
         this.name = name;

      }
   }


   /**
    * <pre>
    *                 -------- 0..1        hasVisitors        0..1
    * CodeGenStrategy | name |------------------------------------- CodeGenVisitor
    *                 -------- codeGenStrategy      codeGenVisitor
    * </pre>
    */
   private FHashMap codeGenVisitor;


   /**
    * @param value  No description provided
    * @return       No description provided
    * @see          #codeGenVisitor
    */
   public boolean hasInCodeGenVisitor (CodeGenVisitor value)
   {
      return  (
          (this.codeGenVisitor != null)
         &&  (value != null)
         &&  (value.getName() != null)
         &&  (this.codeGenVisitor.get (value.getName()) == value));
   }


   /**
    * @param key  No description provided
    * @return     No description provided
    * @see        #codeGenVisitor
    */
   public boolean hasKeyInCodeGenVisitor (String key)
   {
      return  ( (this.codeGenVisitor != null) &&  (key != null) && this.codeGenVisitor.containsKey (key));
   }


   /**
    * @return   No description provided
    * @see      #codeGenVisitor
    */
   public Iterator iteratorOfCodeGenVisitor()
   {
      return  ( (this.codeGenVisitor == null) ? FEmptyIterator.get() : this.codeGenVisitor.values().iterator());
   }


   /**
    * @return   No description provided
    * @see      #codeGenVisitor
    */
   public Iterator keysOfCodeGenVisitor()
   {
      return  ( (this.codeGenVisitor == null) ? FEmptyIterator.get() : this.codeGenVisitor.keySet().iterator());
   }


   /**
    * @return   No description provided
    * @see      #codeGenVisitor
    */
   public Iterator entriesOfCodeGenVisitor()
   {
      return  ( (this.codeGenVisitor == null) ? FEmptyIterator.get() : this.codeGenVisitor.entrySet().iterator());
   }


   /**
    * @return   No description provided
    * @see      #codeGenVisitor
    */
   public int sizeOfCodeGenVisitor()
   {
      return  ( (this.codeGenVisitor == null) ? 0 : this.codeGenVisitor.size());
   }


   /**
    * @param key  No description provided
    * @return     The fromCodeGenVisitor value
    * @see        #codeGenVisitor
    */
   public CodeGenVisitor getFromCodeGenVisitor (String key)
   {
      return  ( ( (this.codeGenVisitor == null) ||  (key == null)) ? null : (CodeGenVisitor) this.codeGenVisitor.get (key));
   }


   /**
    * @param value  The object added.
    * @return       No description provided
    * @see          #codeGenVisitor
    */
   public boolean addToCodeGenVisitor (CodeGenVisitor value)
   {
      boolean changed = false;
      if ( (value != null) &&  (value.getName() != null))
      {
         if (this.codeGenVisitor == null)
         {
            this.codeGenVisitor = new FHashMap(); // or FTreeMap ()
         }
         CodeGenVisitor oldValue = (CodeGenVisitor) this.codeGenVisitor.put (value.getName(), value);
         if (oldValue != value)
         {
            if (oldValue != null)
            {
               oldValue.setCodeGenStrategy (null);
            }
            value.setCodeGenStrategy (this);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * @param value  No description provided
    * @return       No description provided
    * @see          #codeGenVisitor
    */
   public boolean removeFromCodeGenVisitor (CodeGenVisitor value)
   {
      boolean changed = false;
      if ( (this.codeGenVisitor != null) &&  (value != null) &&  (value.getName() != null))
      {
         CodeGenVisitor oldValue = (CodeGenVisitor) this.codeGenVisitor.get (value.getName());
         if (oldValue == value)
         {
            this.codeGenVisitor.remove (value.getName());
            value.setCodeGenStrategy (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * @param key  No description provided
    * @return     No description provided
    * @see        #codeGenVisitor
    */
   public boolean removeKeyFromCodeGenVisitor (String key)
   {
      boolean changed = false;
      if ( (this.codeGenVisitor != null) &&  (key != null))
      {
         CodeGenVisitor tmpValue = (CodeGenVisitor) this.codeGenVisitor.get (key);
         if (tmpValue != null)
         {
            this.codeGenVisitor.remove (key);
            tmpValue.setCodeGenStrategy (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * @see   #codeGenVisitor
    */
   public void removeAllFromCodeGenVisitor()
   {
      CodeGenVisitor tmpValue;
      Iterator iter = this.iteratorOfCodeGenVisitor();
      while (iter.hasNext())
      {
         tmpValue = (CodeGenVisitor) iter.next();
         this.removeFromCodeGenVisitor (tmpValue);
      }

   }


   /**
    * <pre>
    *                         0..n         handlerChain          0..1
    * CodeGenStrategyHandler ----------------------------------------- CodeGenStrategy
    *                         handlerChain   {ordered,}   chainMaster
    * </pre>
    */

   private FLinkedList handlerChain;


   /**
    * Access method for an one to n association.
    *
    * @param value             The object added.
    * @param successorHandler  The object added.
    * @return                  No description provided
    */
   public boolean addToBeforeHandlerChain (CodeGenStrategyHandler value,
                                           CodeGenStrategyHandler successorHandler)
   {
      boolean changed = false;
      if (value != null && !hasInHandlerChain (value))
      {
         if (this.handlerChain == null)
         {
            this.handlerChain = new FLinkedList(); // or FTreeSet () or FLinkedList ()
         }

         // find the successor index
         int i = this.handlerChain.indexOf (successorHandler);
         this.handlerChain.add (i, value);
         changed = true;

         if (changed)
         {
            value.setChainMaster (this);
         }
      }

      return changed;
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToHandlerChain (CodeGenStrategyHandler value)
   {
      boolean changed = false;
      if (value != null && !hasInHandlerChain (value))
      {
         if (this.handlerChain == null)
         {
            this.handlerChain = new FLinkedList(); // or FTreeSet () or FLinkedList ()
         }

         changed = this.handlerChain.add (value);

         if (changed)
         {
            value.setChainMaster (this);
         }
      }

      return changed;
   }


   /**
    * Get the firstOfHandlerChain attribute of the CodeGenStrategy object
    *
    * @return   The firstOfHandlerChain value
    */
   public CodeGenStrategyHandler getFirstOfHandlerChain()
   {
      if (handlerChain == null)
      {
         return null;
      }
      else
      {
         return (CodeGenStrategyHandler) handlerChain.getFirst();
      }
   }


   /**
    * Get the handlerChainAt attribute of the CodeGenStrategy object
    *
    * @param index  No description provided
    * @return       The handlerChainAt value
    */
   public CodeGenStrategyHandler getHandlerChainAt (int index)
   {
      if (index >= 0 && index < sizeOfHandlerChain())
      {
         return (CodeGenStrategyHandler) this.handlerChain.get (index);
      }
      else
      {
         throw new IllegalArgumentException ("getHandlerChainAt(" + index + ")");
      }
   }


   /**
    * Get the lastOfHandlerChain attribute of the CodeGenStrategy object
    *
    * @return   The lastOfHandlerChain value
    */
   public CodeGenStrategyHandler getLastOfHandlerChain()
   {
      if (handlerChain == null)
      {
         return null;
      }
      else
      {
         return (CodeGenStrategyHandler) handlerChain.getLast();
      }
   }


   /**
    * Get the nextIndexOfHandlerChain attribute of the CodeGenStrategy object
    *
    * @param object  No description provided
    * @param index   No description provided
    * @return        The nextIndexOfHandlerChain value
    */
   public CodeGenStrategyHandler getNextIndexOfHandlerChain (CodeGenStrategyHandler object, int index)
   {
      if (handlerChain == null)
      {
         return null;
      }
      else
      {
         return (CodeGenStrategyHandler) handlerChain.getNextOf (object, index);
      }
   }


   /**
    * Get the nextOfHandlerChain attribute of the CodeGenStrategy object
    *
    * @param object  No description provided
    * @return        The nextOfHandlerChain value
    */
   public CodeGenStrategyHandler getNextOfHandlerChain (CodeGenStrategyHandler object)
   {
      if (handlerChain == null)
      {
         return null;
      }
      else
      {
         return (CodeGenStrategyHandler) handlerChain.getNextOf (object);
      }
   }


   /**
    * Get the previousIndexOfHandlerChain attribute of the CodeGenStrategy object
    *
    * @param object  No description provided
    * @param index   No description provided
    * @return        The previousIndexOfHandlerChain value
    */
   public CodeGenStrategyHandler getPreviousIndexOfHandlerChain (CodeGenStrategyHandler object, int index)
   {
      if (handlerChain == null)
      {
         return null;
      }
      else
      {
         return (CodeGenStrategyHandler) handlerChain.getPreviousOf (object, index);
      }
   }


   /**
    * Get the previousOfHandlerChain attribute of the CodeGenStrategy object
    *
    * @param object  No description provided
    * @return        The previousOfHandlerChain value
    */
   public CodeGenStrategyHandler getPreviousOfHandlerChain (CodeGenStrategyHandler object)
   {
      if (handlerChain == null)
      {
         return null;
      }
      else
      {
         return (CodeGenStrategyHandler) handlerChain.getPreviousOf (object);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInHandlerChain (CodeGenStrategyHandler value)
   {
      return  ( (this.handlerChain != null) &&
          (value != null) &&
         this.handlerChain.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public int indexOfHandlerChain (CodeGenStrategyHandler elem)
   {
      return  ( (this.handlerChain == null)
         ? -1
         : this.handlerChain.indexOf (elem));
   }


   /**
    * Get the afterOfHandlerChain attribute of the CodeGenStrategy object
    *
    * @param leftObject   No description provided
    * @param rightObject  No description provided
    * @return             The afterOfHandlerChain value
    */
   public boolean isAfterOfHandlerChain (CodeGenStrategyHandler leftObject, CodeGenStrategyHandler rightObject)
   {
      if (handlerChain == null)
      {
         return false;
      }
      else
      {
         return handlerChain.isAfter (leftObject, rightObject);
      }
   }


   /**
    * Get the beforeOfHandlerChain attribute of the CodeGenStrategy object
    *
    * @param leftObject   No description provided
    * @param rightObject  No description provided
    * @return             The beforeOfHandlerChain value
    */
   public boolean isBeforeOfHandlerChain (CodeGenStrategyHandler leftObject, CodeGenStrategyHandler rightObject)
   {
      if (handlerChain == null)
      {
         return false;
      }
      else
      {
         return handlerChain.isBefore (leftObject, rightObject);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfHandlerChain()
   {
      return  ( (this.handlerChain == null)
         ? FEmptyIterator.get()
         : this.handlerChain.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param lowerBound  No description provided
    * @return            No description provided
    */
   public Iterator iteratorOfHandlerChain (CodeGenStrategyHandler lowerBound)
   {
      Iterator result = FEmptyIterator.get();
      if (handlerChain == null)
      {
         result = FEmptyIterator.get();
      }
      else if (handlerChain != null && lowerBound != null)
      {
         int index = handlerChain.indexOf (lowerBound) + 1;
         result = handlerChain.listIterator (index);
      }
      else if (handlerChain != null && lowerBound == null)
      {
         result = handlerChain.listIterator (0);
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public int lastIndexOfHandlerChain (CodeGenStrategyHandler elem)
   {
      return  ( (this.handlerChain == null)
         ? -1
         : this.handlerChain.lastIndexOf (elem));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromHandlerChain()
   {
      CodeGenStrategyHandler tmpValue;
      Iterator iter = this.iteratorOfHandlerChain();
      while (iter.hasNext())
      {
         tmpValue = (CodeGenStrategyHandler) iter.next();
         this.removeFromHandlerChain (tmpValue);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromHandlerChain (CodeGenStrategyHandler value)
   {
      boolean changed = false;
      if ( (this.handlerChain != null) &&  (value != null))
      {
         changed = this.handlerChain.remove (value);
         if (changed)
         {
            value.setChainMaster (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfHandlerChain()
   {
      return  ( (this.handlerChain == null)
         ? 0
         : this.handlerChain.size());
   }


   /**
    * Appends new handler as the next to last handler of the responsibility chain. A dummy
    * handler is always the last element of the chain.
    *
    * @param newHandler  No description provided
    */
   public void appendHandler (CodeGenStrategyHandler newHandler)
   {
      insertHandler (EndOfChainOOHandler.class, newHandler);
   }


   /**
    * Inserts new handler before a given successor handler of the responsibility chain.
    *
    * @param successorHandlerType  The type of the successor handler
    * @param newHandler            The handler to be added
    */
   public void insertHandler (Class successorHandlerType, CodeGenStrategyHandler newHandler)
   {
      if (newHandler == null)
      {
         throw new IllegalArgumentException ("CodeGenStrategy.insertHandler(): argument 'newHandler' is null!");
      }
      if (!CodeGenStrategyHandler.class.isAssignableFrom (successorHandlerType))
      {
         throw new IllegalArgumentException ("CodeGenStrategy.insertHandler(): argument 'successorHandlerType' is of wrong type!");
      }

      // look for a handler of type successorHandlerType
      CodeGenStrategyHandler successorHandler = null;
      boolean found = false;

      Iterator iter = this.iteratorOfHandlerChain();
      while (!found && iter.hasNext())
      {
         successorHandler = (CodeGenStrategyHandler) iter.next();

         found = successorHandlerType.equals (successorHandler.getClass());
      }

      if (found)
      {
         // add to handler chain
         addToBeforeHandlerChain (newHandler, successorHandler);
      }
   }


   /**
    * <pre>
    *                      0..1           0..1
    * OOGenStrategyClient --------------------- OOGenFunction
    *                      client     function
    * </pre>
    */
   private CodeGenFunction function;


   /**
    * Sets the function attribute of the OOGenStrategyClient object
    *
    * @param value  The new function value
    * @return       No description provided
    */
   public boolean setFunction (CodeGenFunction value)
   {
      if (this.function != value)
      {
         if (this.function != null)
         {
            CodeGenFunction oldValue = this.function;
            this.function = null;
            oldValue.setClient (null);
         }
         this.function = value;
         if (value != null)
         {
            this.function.setClient (this);
         }

         return true;
      }

      return false;
   }


   /**
    * Get the function attribute of the OOGenStrategyClient object
    *
    * @return   The function value
    */
   public CodeGenFunction getFunction()
   {
      return this.function;
   }


   /**
    * Append new function to the end of the responsibility chain.
    *
    * @param newFunction  No description provided
    */
   public void appendFunction (CodeGenFunction newFunction)
   {
      if (newFunction == null)
      {
         throw new IllegalArgumentException ("CodeGenFactor.appendFunction(): argument is null!");
      }

      // look for the endOfChain element
      CodeGenFunction endOfChain = getFunction();
      while (endOfChain.getSuccessor() != null)
      {
         endOfChain = endOfChain.getSuccessor();
      }

      // append newHandler as next to last element of the chain
      CodeGenFunction nextToLastFunction = endOfChain.getPrevFunction();

      if (nextToLastFunction == null)
      {
         setFunction (newFunction);
      }
      else
      {
         nextToLastFunction.setSuccessor (newFunction);
      }

      newFunction.setSuccessor (endOfChain);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param beforeFunction  No description provided
    * @param newFunction     No description provided
    */
   public void insertFunction (Class beforeFunction, CodeGenFunction newFunction)
   {
      if (newFunction == null)
      {
         throw new IllegalArgumentException ("CodeGenFactor.insertFunction(): argument is null!");
      }

      // look for the endOfChain element
      CodeGenFunction insertBefore = getFunction();
      while (insertBefore.getSuccessor() != null && !beforeFunction.equals (insertBefore.getClass()))
      {
         insertBefore = insertBefore.getSuccessor();
      }

      // append newHandler as next to last element of the chain
      CodeGenFunction prevFunction = insertBefore.getPrevFunction();

      if (prevFunction == null)
      {
         setFunction (newFunction);
      }
      else
      {
         prevFunction.setSuccessor (newFunction);
      }

      newFunction.setSuccessor (insertBefore);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      removeAllFromCodeGenVisitor();

      removeAllFromHandlerChain();

      CodeGenFactory tmpCodeGenFactory = getCodeGenFactory();

      if (tmpCodeGenFactory != null)
      {
         setCodeGenFactory (null);
      } // if
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "CodeGenStrategy[" + getName() + "]";
   }
}

/*
 * $Log: CodeGenStrategy.java,v $
 * Revision 1.45.2.1  2005/09/30 18:56:55  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
