/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.Arrays;
import java.util.LinkedList;
import de.uni_paderborn.fujaba.asg.ASGElement;

import de.uni_paderborn.fujaba.metamodel.FElement;


/**
 * Class CodeGenFunction
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.10 $ <h2> Associations </h2> <pre>
 *                0..1           0..1
 * OOGenFunction --------------------- OOGenStrategyClient
 *                function     client
 *
 *                0..1                  0..1
 * OOGenFunction ---------------------------- OOGenFunction
 *                prevFunction     successor
 * </pre>
 */
public abstract class CodeGenFunction
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int NORM = 0;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int PRE_ORDER = 1;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int IN_ORDER = 2;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int POST_ORDER = 3;


   /**
    * Default Constructor
    */
   public CodeGenFunction()
   {
      this (NORM);
   }


   /**
    * Constructor for class OOGenFunction
    *
    * @param mode  No description provided
    */
   public CodeGenFunction (int mode)
   {
      setMode (mode);
   }


   /**
    * UMLAttribute : 'mode : int '
    */
   private int mode;


   /**
    * Get the value of mode.
    *
    * @return   Value of mode.
    */
   public int getMode()
   {
      return this.mode;
   }


   /**
    * Set the value of mode.
    *
    * @param mode  Value to assign to mode.
    */
   public void setMode (int mode)
   {
      if (this.mode != mode)
      {
         this.mode = mode;
      }
   }


   /**
    * <pre>
    *                0..1           0..1
    * OOGenFunction --------------------- OOGenStrategyClient
    *                function     client
    * </pre>
    */
   private CodeGenStrategy client;


   /**
    * @param value  The new client value
    * @return       No description provided
    * @see          #client
    */
   public boolean setClient (CodeGenStrategy value)
   {
      if (this.client != value)
      {
         if (this.client != null)
         {
            CodeGenStrategy oldValue = this.client;
            this.client = null;
            oldValue.setFunction (null);
         }
         this.client = value;
         if (value != null)
         {
            this.client.setFunction (this);
         }

         return true;
      }

      return false;
   }


   /**
    * @return   The client value
    * @see      #client
    */
   public CodeGenStrategy getClient()
   {
      return this.client;
   }


   /**
    * Get the responsible attribute of the OOGenFunction object
    *
    * @param methodName  No description provided
    * @return            The responsible value
    */
   public abstract boolean isResponsible (String methodName);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param methodName  No description provided
    * @param param       No description provided
    * @return            No description provided
    */
   public final Object generateCode (String methodName, Object param[])
   {
      return null;
   }


   /**
    * Generate code for an element. Subclasses must override this method. For backward compatibility
    * this method is not abstract but invokes the old implementation.
    *
    * @param incr        No description provided
    * @param methodName  No description provided
    * @param param       No description provided
    * @return            No description provided
    */
   public Object generateCode (FElement incr, String methodName, Object param[])
   {
      return generateCode ((ASGElement) incr, methodName, param);
   }


   /**
    * @param incr
    * @param methodName
    * @param param
    * @return            No description provided
    * @deprecated        use {@link #generateCode(de.uni_paderborn.fujaba.metamodel.FElement,
    *      String, Object[])} instead
    */
   public Object generateCode (ASGElement incr, String methodName, Object param[])
   {
      throw new AbstractMethodError (getClass().getName() + " must implement generateCode(de.uni_paderborn.fujaba.metamodel.FElement, String, Object[])");
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr        No description provided
    * @param methodName  No description provided
    * @param param       No description provided
    * @param prevResult  No description provided
    * @return            No description provided
    */
   public Object generateCode (FElement incr, String methodName, Object param[], Object prevResult)
   {
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param function    No description provided
    * @param methodName  No description provided
    * @return            No description provided
    */
   public static CodeGenFunction findNextResponsible (CodeGenFunction function, String methodName)
   {
      while (function != null)
      {
         if (function.isResponsible (methodName))
         {
            return function;
         }
         function = function.getSuccessor();
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr        No description provided
    * @param methodName  No description provided
    * @param param       No description provided
    * @return            No description provided
    */
   public final Object generate (FElement incr, String methodName, Object param[])
   {
      Object resultA = null;
      Object resultB = null;

      if (!isResponsible (methodName))
      {
         CodeGenFunction next = findNextResponsible (this.getSuccessor(), methodName);
         if (next != null)
         {
            return next.generate (incr, methodName, param);
         }
         throw new IllegalStateException ("methodName=" + methodName +
            ",param=" + Arrays.asList (param));
      }

      if (mode == NORM)
      {
         return generateCode (incr,
            methodName,
            param);
      }
      else if (mode == PRE_ORDER)
      {
         resultA = generateCode (incr, methodName, param);
         CodeGenFunction next = findNextResponsible (this.getSuccessor(), methodName);
         if (next != null)
         {
            resultB = next.generate (incr,
               methodName,
               param);
         }

         return concat (resultA, resultB);
      }
      else if (mode == POST_ORDER)
      {
         resultA = generateCode (incr, methodName, param);

         CodeGenFunction next = findNextResponsible (this.getSuccessor(), methodName);
         if (next != null)
         {
            resultB = next.generate (incr,
               methodName,
               param);
         }

         return concat (resultB, resultA);
      }
      else if (mode == IN_ORDER)
      {
         CodeGenFunction next = findNextResponsible (this.getSuccessor(), methodName);
         if (next != null)
         {
            resultA = next.generate (incr,
               methodName,
               param);
         }

         return generateCode (incr,
            methodName,
            param,
            resultA);
      }
      else
      {
         throw new IllegalStateException ("methodName=" + methodName +
            ",param=" + Arrays.asList (param) +
            ",mode=" + mode);
      }
   }


   /**
    * Get the clientOfChain attribute of the OOGenFunction object
    *
    * @return   The clientOfChain value
    */
   public final CodeGenStrategy getClientOfChain()
   {
      if (getClient() != null)
      {
         return getClient();
      }
      else if (getPrevFunction() != null)
      {
         return getPrevFunction().getClientOfChain();
      }
      else
      {
         throw new Error ("chain is corrupt !");
      }
   }


   /**
    * create new <code>LinkedList</code> and append <code>a</code> and <code>b</code>
    *
    * @param a  an <code>Object</code> value
    * @param b  an <code>Object</code> value
    * @return   the new <code>LinkedList</code> value
    */
   public LinkedList concat (Object a, Object b)
   {
      LinkedList result = new LinkedList();

      add (result, a);
      add (result, b);

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theLinkedList  No description provided
    * @param theObject      No description provided
    */
   public void add (LinkedList theLinkedList, Object theObject)
   {
      if (theObject instanceof LinkedList)
      {
         theLinkedList.addAll ((LinkedList) theObject);
      }
      else
      {
         theLinkedList.add (theObject);
      }
   }


   /**
    * <pre>
    *                0..1                  0..1
    * OOGenFunction ---------------------------- OOGenFunction
    *                prevFunction     successor
    * </pre>
    */
   private CodeGenFunction successor;


   /**
    * @param value  The new successor value
    * @return       No description provided
    * @see          #successor
    */
   public boolean setSuccessor (CodeGenFunction value)
   {
      if (this.successor != value)
      {
         if (this.successor != null)
         {
            CodeGenFunction oldValue = this.successor;
            this.successor = null;
            oldValue.setPrevFunction (null);
         }
         this.successor = value;
         if (value != null)
         {
            this.successor.setPrevFunction (this);
         }

         return true;
      }

      return false;
   }


   /**
    * @return   The successor value
    * @see      #successor
    */
   public CodeGenFunction getSuccessor()
   {
      return this.successor;
   }


   /**
    * <pre>
    *                0..1                  0..1
    * OOGenFunction ---------------------------- OOGenFunction
    *                successor     prevFunction
    * </pre>
    */
   private CodeGenFunction prevFunction;


   /**
    * @param value  The new prevFunction value
    * @return       No description provided
    * @see          #prevFunction
    */
   public boolean setPrevFunction (CodeGenFunction value)
   {
      if (this.prevFunction != value)
      {
         if (this.prevFunction != null)
         {
            CodeGenFunction oldValue = this.prevFunction;
            this.prevFunction = null;
            oldValue.setSuccessor (null);
         }
         this.prevFunction = value;
         if (value != null)
         {
            this.prevFunction.setSuccessor (this);
         }

         return true;
      }

      return false;
   }


   /**
    * @return   The prevFunction value
    * @see      #prevFunction
    */
   public CodeGenFunction getPrevFunction()
   {
      return this.prevFunction;
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "CodeGenFunction[]";
   }
}

/*
 * $Log: CodeGenFunction.java,v $
 * Revision 1.10  2004/12/20 11:49:19  creckord
 * Fixed endless recursion in CodeGenFunction
 *
 */
