/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.*;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.basic.*;
import de.uni_paderborn.fujaba.fsa.swing.MethodComparator;
import de.uni_paderborn.fujaba.gui.TemplateBlockNameListener;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.UMLStereotypeManager;
import de.upb.tools.fca.FHashMap;


/**
 * @author    <a href="mailto:creckord@upb.de">Carsten Reckord</a>
 * @version   $Revision: 1.12.2.3 $
 */
public abstract class AccessorOOHandler extends OOGenStrategyHandler
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (AccessorOOHandler.class);


   /**
    * @param asgElement  No description provided
    * @param prevToken   No description provided
    * @param param       No description provided
    * @return            No description provided
    * @see               de.uni_paderborn.fujaba.codegen.CodeGenStrategyHandler#generateSourceCode(de.uni_paderborn.fujaba.metamodel.FElement,
    *      de.uni_paderborn.fujaba.codegen.OOGenToken, java.lang.Object[])
    */
   public OOGenToken generateSourceCode (FElement asgElement,
                                         OOGenToken prevToken, Object[] param)
   {
      generateAccessors (asgElement);
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean needToken()
   {
      return false;
   }


   /**
    * Get the templateFileName attribute of the AccessorOOHandler object
    *
    * @param element  No description provided
    * @return         The templateFileName value
    */
   protected abstract String getTemplateFileName (FElement element);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element    No description provided
    * @param accessors  No description provided
    * @return           No description provided
    */
   protected abstract FAttr createAccessorDescriptions (FElement element, Set accessors);


   /**
    * @param element element code is generated for
    * @param attr attr passed to {@link #generateMethod}
    * @return the parent of the generated method
    */
   public abstract FClass getTarget (FElement element, FAttr attr);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element  No description provided
    */
   protected void generateAccessors (FElement element)
   {
      TreeSet accessMethods = new TreeSet();

      FAttr attr = createAccessorDescriptions (element, accessMethods);

      Iterator methodDescriptions = accessMethods.iterator();
      while (methodDescriptions.hasNext())
      {
         MethodDescription description = (MethodDescription) methodDescriptions.next();
         generateMethod (element, attr, description);
      }
      removeGeneratedIncrs();
   }


   /**
    * @param element      No description provided
    * @param attr         No description provided
    * @param description  No description provided
    */
   protected void generateMethod (FElement element, FAttr attr,
                                  MethodDescription description)
   {
      FClass target = getTarget (element, attr);
      FMethod umlMethod = description.getMethodSignature();

      FMethod existingMethod = target.getFromFMethods (umlMethod.getFullMethodName());

      //FIXME: In the long run, legacy method handling should be removed...
      if (existingMethod != null &&  (existingMethod.isGenerated() ||
         existingMethod.getDisplayLevel() <= FDeclaration.CODE_DISPLAY_LEVEL))
      {
         //FIXME: This is a violation of the premise that the code generation
         //doesn't change the model
         existingMethod.removeYou();
         existingMethod = null;
      }

      if (existingMethod == null)
      {
         FHashMap templateParameters = description.getTemplateParameters();
         changeTemplateParameters (element, attr, description.getTemplateBlockName(), templateParameters);
         String code = createTemplateCode (element, attr,
            templateParameters,
            description.getTemplateBlockName());

         // Add code to method.
         if ( !isInterface( target ) )
         {
            umlMethod.setMethodBody (code, true);
         }
         umlMethod.setParent( target );
         getClientOfChain().generateSourceCodeFor (umlMethod, null, null);
         umlMethod.setParent( null );
      }
      else
      {
         //if (log.isInfoEnabled()) log.info ("Existing method: " + existingMethod.getFullMethodName());
         getClientOfChain().generateSourceCodeFor (existingMethod, null, null);
         //mark this method to exclude it from normal method code generation by
         //associating it with its accessed element
         AccessMethodRef.createAssociation (existingMethod, element);
      }
      umlMethod.removeYou();
   }

   private boolean isInterface( FClass target )
   {
      return target.hasInStereotypes( UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.INTERFACE) );
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param attr  No description provided
    */
   protected void generateAttrCode (FAttr attr)
   {
      OOGenStrategyClient client = (OOGenStrategyClient) getClientOfChain();

      if ( !isInterface( attr.getFParent() ) ) {
      // generate comment
      FCommentary fCommentary = attr.getFComment();
      if (fCommentary != null)
      {
         String commentaryText;
         if (attr.isParsed())
         {
            commentaryText = fCommentary.getText();
         }
         else
         {
            commentaryText = client.createFAttrCommentary (fCommentary);
         }
         client.appendFAttrCommentary (commentaryText);
      }

      String umlAttrDecl = client.createFAttrDeclaration (attr);
      client.appendFAttrDeclaration (umlAttrDecl);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element             No description provided
    * @param attr                No description provided
    * @param templateParameters  No description provided
    * @param blockName           No description provided
    * @return                    No description provided
    */
   public String createTemplateCode (FElement element, FAttr attr,
                                     FHashMap templateParameters, String blockName)
   {
      FClass target = getTarget (element, attr);
      String code = "";

      TemplateFile templateFile = TemplateManager.get().getTemplate (getTemplateFileName (element));

      // set codeBlock name
      String realBlockName = getTemplateBlockName (blockName, target, attr);

      // Get template block from file.
      TemplateCodeBlock templateCodeBlock = templateFile.getFromCodeBlocks (realBlockName);

      if (templateCodeBlock != null)
      {
         // Get the code.
         code = templateCodeBlock.getSourceCode (templateParameters);
      }
      return code;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private LinkedHashSet generatedIncrs;


   /**
    * Access method for an one to n association.
    *
    * @param incr  The object added.
    */
   public void addToGeneratedIncrs (FElement incr)
   {
      if (generatedIncrs == null)
      {
         generatedIncrs = new LinkedHashSet();
      }
      if (!generatedIncrs.contains (incr))
      {
         generatedIncrs.add (incr);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInGeneratedIncrs (FElement elem)
   {
      return  (generatedIncrs != null && generatedIncrs.contains (elem));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void removeGeneratedIncrs()
   {
      if (generatedIncrs != null)
      {
         //reverse order
         LinkedList reverseOrder = new LinkedList();
         Iterator iter = generatedIncrs.iterator();
         while (iter.hasNext())
         {
            FElement incr = (FElement) iter.next();
            reverseOrder.addFirst (incr);
         }
         generatedIncrs.clear();
         iter = reverseOrder.iterator();
         while (iter.hasNext())
         {
            FElement incr = (FElement) iter.next();
            incr.removeYou();
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element  No description provided
    */
   public void removeOldAccessors (FElement element)
   {
      TreeSet accessMethods = new TreeSet();

      FAttr attr = createAccessorDescriptions (element, accessMethods);

      Iterator methodDescriptions = accessMethods.iterator();
      while (methodDescriptions.hasNext())
      {
         MethodDescription description = (MethodDescription) methodDescriptions.next();
         removeOldMethod (element, attr, description);
      }
      removeGeneratedIncrs();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element      No description provided
    * @param attr         No description provided
    * @param description  No description provided
    */
   protected void removeOldMethod (FElement element, FAttr attr, MethodDescription description)
   {
      FClass target = getTarget (element, attr);
      FMethod umlMethod = description.getMethodSignature();

      FMethod existingMethod = target.getFromFMethods (umlMethod.getFullMethodName());
      if (existingMethod != null)
      {
         if (existingMethod.isGenerated() ||
            existingMethod.getDisplayLevel() <= FDeclaration.CODE_DISPLAY_LEVEL)
         {
            existingMethod.removeYou();
         }
         else
         {
            log.error ("Won't delete " + existingMethod.getFParent().getFullClassName() + "." +
               existingMethod.getName() + " because it seems to have been created manually");
         }
      }
   }


   /**
    * @param attr
    * @param blockName  No description provided
    * @param cls        No description provided
    * @return           The templateBlockName value
    */
   public static String getTemplateBlockName (String blockName, FClass cls, FAttr attr)
   {
      if ("accessor-set".equals (blockName) || "accessor-get".equals (blockName))
      {
         if (attr != null && attr.isStatic())
         {
            blockName += "-static";
         }
      }
      if ("postHook-propertyChange".equals (blockName))
      {
         if (cls.getFromAllFMethods ("getPropertyChangeSupport()") != null)
         {
            blockName += "-support";
         }
         else if (attr != null && !attr.getFAttrType().getProgLangType().equals (attr.getFAttrType().getName()))
         {
            blockName += "-primitive";
         }
      }
      if (templateBlockNameListeners != null)
      {
         Iterator it = templateBlockNameListeners.iterator();
         while (it.hasNext())
         {
            TemplateBlockNameListener listener = (TemplateBlockNameListener) it.next();
            blockName = listener.getTemplateBlockName (blockName, cls, attr);
         }
      }
      return blockName;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param templateParameters  No description provided
    * @param preHook             No description provided
    * @return                    No description provided
    */
   public static String appendPreHook (FHashMap templateParameters, String preHook)
   {
      return appendParameter (templateParameters, "$CHANGE_PREHOOK$", preHook);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param templateParameters  No description provided
    * @param postHook            No description provided
    * @return                    No description provided
    */
   public static String appendPostHook (FHashMap templateParameters, String postHook)
   {
      return appendParameter (templateParameters, "$CHANGE_POSTHOOK$", postHook);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param templateParameters  No description provided
    * @param preHook             No description provided
    * @return                    No description provided
    */
   public static String prependPreHook (FHashMap templateParameters, String preHook)
   {
      return prependParameter (templateParameters, "$CHANGE_PREHOOK$", preHook);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param templateParameters  No description provided
    * @param postHook            No description provided
    * @return                    No description provided
    */
   public static String prependPostHook (FHashMap templateParameters, String postHook)
   {
      return prependParameter (templateParameters, "$CHANGE_POSTHOOK$", postHook);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param templateParameters  No description provided
    * @param parameter           No description provided
    * @param value               No description provided
    * @return                    No description provided
    */
   protected static String appendParameter (FHashMap templateParameters, String parameter,
                                            String value)
   {
      StringBuffer parameterBuffer = new StringBuffer();
      String oldValue = (String) templateParameters.get (parameter);
      if (oldValue != null)
      {
         parameterBuffer.append (oldValue);
      }
      if (value != null)
      {
         parameterBuffer.append (value);
      }
      value = parameterBuffer.toString();
      templateParameters.put (parameter, value);
      return value;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param templateParameters  No description provided
    * @param parameter           No description provided
    * @param value               No description provided
    * @return                    No description provided
    */
   protected static String prependParameter (FHashMap templateParameters, String parameter,
                                             String value)
   {
      StringBuffer parameterBuffer = new StringBuffer();
      String oldValue = (String) templateParameters.get (parameter);
      if (value != null)
      {
         parameterBuffer.append (value);
      }
      if (oldValue != null)
      {
         parameterBuffer.append (oldValue);
      }
      value = parameterBuffer.toString();
      templateParameters.put (parameter, value);
      return value;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static HashSet templateBlockNameListeners;


   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public static void addToTemplateBlockNameListeners (TemplateBlockNameListener listener)
   {
      if (listener != null)
      {
         if (templateBlockNameListeners == null)
         {
            templateBlockNameListeners = new HashSet();
         }
         templateBlockNameListeners.add (listener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param listener  No description provided
    */
   public static void removeFromTemplateBlockNameListeners (TemplateBlockNameListener listener)
   {
      if (listener != null)
      {
         if (templateBlockNameListeners != null)
         {
            templateBlockNameListeners.remove (listener);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static LinkedHashSet globalTemplateParameterListeners;


   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public static void addToGlobalTemplateParameterListeners (TemplateParameterListener listener)
   {
      if (listener != null)
      {
         if (globalTemplateParameterListeners == null)
         {
            globalTemplateParameterListeners = new LinkedHashSet();
         }
         globalTemplateParameterListeners.add (listener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param listener  No description provided
    */
   public static void removeFromGlobalTemplateParameterListeners (TemplateParameterListener listener)
   {
      if (listener != null)
      {
         if (globalTemplateParameterListeners != null)
         {
            globalTemplateParameterListeners.remove (listener);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public static Iterator iteratorOfGlobalTemplateParameterListeners()
   {
      return  (globalTemplateParameterListeners == null ? Collections.EMPTY_SET.iterator() :
         globalTemplateParameterListeners.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private LinkedHashSet templateParameterListeners;


   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addToTemplateParameterListeners (TemplateParameterListener listener)
   {
      if (listener != null)
      {
         if (templateParameterListeners == null)
         {
            templateParameterListeners = new LinkedHashSet();
         }
         templateParameterListeners.add (listener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param listener  No description provided
    */
   public void removeFromTemplateParameterListeners (TemplateParameterListener listener)
   {
      if (listener != null)
      {
         if (templateParameterListeners != null)
         {
            templateParameterListeners.remove (listener);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element             No description provided
    * @param attr                No description provided
    * @param blockName           No description provided
    * @param templateParameters  No description provided
    */
   protected void changeTemplateParameters (FElement element,
                                            FAttr attr,
                                            String blockName,
                                            FHashMap templateParameters)
   {
      if (globalTemplateParameterListeners != null)
      {
         Iterator iter = globalTemplateParameterListeners.iterator();
         while (iter.hasNext())
         {
            TemplateParameterListener listener = (TemplateParameterListener) iter.next();
            listener.setupTemplateParameters (this, element, attr, blockName, templateParameters);
         }
      }
      if (templateParameterListeners != null)
      {
         Iterator iter = templateParameterListeners.iterator();
         while (iter.hasNext())
         {
            TemplateParameterListener listener = (TemplateParameterListener) iter.next();
            listener.setupTemplateParameters (this, element, attr, blockName, templateParameters);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.12.2.3 $ $Date: 2006/03/28 11:15:46 $
    */
   public final static class MethodDescription implements Comparable
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      protected final FMethod methodSignature;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      protected final String templateBlockName;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      protected final FHashMap templateParameters;


      /**
       * Constructor for class MethodDescription
       *
       * @param methodSignature     No description provided
       * @param templateBlockName   No description provided
       * @param templateParameters  No description provided
       */
      public MethodDescription (FMethod methodSignature, String templateBlockName,
                                FHashMap templateParameters)
      {
         this.methodSignature = methodSignature;
         this.templateBlockName = templateBlockName;
         this.templateParameters = templateParameters;
      }


      /**
       * Get the methodSignature attribute of the MethodDescription object
       *
       * @return   The methodSignature value
       */
      public FMethod getMethodSignature()
      {
         return methodSignature;
      }


      /**
       * Get the templateBlockName attribute of the MethodDescription object
       *
       * @return   The templateBlockName value
       */
      public String getTemplateBlockName()
      {
         return templateBlockName;
      }


      /**
       * Get the templateParameters attribute of the MethodDescription object
       *
       * @return   The templateParameters value
       */
      public FHashMap getTemplateParameters()
      {
         return templateParameters;
      }


      /**
       * @param o  No description provided
       * @return   No description provided
       * @see      java.lang.Comparable#compareTo(java.lang.Object)
       */
      public int compareTo (Object o)
      {
         MethodDescription description = (MethodDescription) o;
         return MethodComparator.get().compare (this.getMethodSignature(), description.getMethodSignature());
      }
   }
}

/*
 * $Log: AccessorOOHandler.java,v $
 * Revision 1.12.2.3  2006/03/28 11:15:46  cschneid
 * fixed accessor generation for interfaces, jnlp shortcut
 *
 * Revision 1.12.2.2  2005/09/30 18:56:54  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
