/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.basic;

import java.beans.*;
import java.util.*;

import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.IteratorConcatenation;


/**
 * This subclass of java.beans.PropertyChangeSupport is identical in functionality -- it sacrifices
 * thread-safety (not a Swing concern) for reduce memory consumption, which helps performance
 * (both big Swing concerns). Most of the overridden methods are only necessary because all
 * of PropertyChangeSupport's instance data is private, without accessor methods. Different
 * to SwingPropertyChangeSupport this class does not register a listener more than once for
 * a property. If the listener is registered for specific properties and shall be added for
 * all properties, the entries in the specific listener lists are deleted first.</p> Additionally
 * this class allows querying for the list of registered listeners according to Fujaba Styleguide.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.17.2.2 $
 */
public class FujabaPropertyChangeSupport extends PropertyChangeSupport
{
   /**
    * "listeners" lists all the generic listeners.
    */
   private transient Vector listeners;

   /**
    * "children" contains FujabaPropertyChangeSupports for individual properties
    */
   private transient Hashtable children;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Object source;

   // Serialization version ID
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   final static long serialVersionUID = 7162625831330845068L;


   /**
    * Constructs a FujabaPropertyChangeSupport object.
    *
    * @param sourceBean  The bean to be given as the source for any events.
    */
   public FujabaPropertyChangeSupport (Object sourceBean)
   {
      super (sourceBean);
      setSource (sourceBean);
   }


   /**
    * Sets the source attribute of the FujabaPropertyChangeSupport object
    *
    * @param source  The new source value
    * @return        No description provided
    */
   public boolean setSource (Object source)
   {
      if (this.source != source)
      {
         this.source = source;
         return true;
      }
      return false;
   }


   /**
    * Get the source attribute of the FujabaPropertyChangeSupport object
    *
    * @return   The source value
    */
   public Object getSource()
   {
      return source;
   }


   /**
    * Add a PropertyChangeListener to the listener list. The listener is registered for all
    * properties.
    *
    * @param listener  The PropertyChangeListener to be added
    */
   public void addPropertyChangeListener (PropertyChangeListener listener)
   {
      addToListeners (listener);
   }


   /**
    * if listener is not yet in listener-list it is added. All occurences of listener in child-lists
    * are removed
    *
    * @param listener  The object added.
    * @return          No description provided
    */
   public synchronized boolean addToListeners (PropertyChangeListener listener)
   {
      if (listeners == null || !listeners.contains (listener))
      {
         removeFromListeners (null, listener);

         if (listeners == null)
         {
            listeners = new Vector();
         }
         synchronized (listeners)
         {
            listeners.addElement (listener);
         }
         return true;
      }
      return false;
   }


   /**
    * Remove a PropertyChangeListener from the listener list. This removes a PropertyChangeListener
    * that was registered for all properties.
    *
    * @param listener  The PropertyChangeListener to be removed
    */

   public void removePropertyChangeListener (PropertyChangeListener listener)
   {
      removeFromListeners (listener);
   }


   /**
    * remove listener both from general listener list and child-lists
    *
    * @param listener  No description provided
    * @return          No description provided
    */
   public synchronized boolean removeFromListeners (PropertyChangeListener listener)
   {
      if (listeners == null)
      {
         return removeFromListeners (null, listener);
      }
      synchronized (listeners)
      {
         return  (listeners.removeElement (listener) || removeFromListeners (null, listener));
      }
   }


   /**
    * Add a PropertyChangeListener for a specific property. The listener will be invoked only
    * when a call on firePropertyChange names that specific property.
    *
    * @param propertyName  The name of the property to listen on.
    * @param listener      The PropertyChangeListener to be added
    */

   public void addPropertyChangeListener (String propertyName,
                                          PropertyChangeListener listener)
   {
      addToListeners (propertyName, listener);
   }


   /**
    * add listener if listsner is not already added as global listener or as listener for that
    * property
    *
    * @param propertyName  The object added.
    * @param listener      The object added.
    * @return              No description provided
    */
   public synchronized boolean addToListeners (String propertyName, PropertyChangeListener listener)
   {
      if (propertyName == null)
      {
         return addToListeners (listener);
      }

      if (!hasInListeners (listener) && !hasInListeners (propertyName, listener))
      {
         if (children == null)
         {
            children = new java.util.Hashtable();
         }
         FujabaPropertyChangeSupport child =
            (FujabaPropertyChangeSupport) children.get (propertyName);
         if (child == null)
         {
            child = new FujabaPropertyChangeSupport (source);
            children.put (propertyName, child);
         }
         child.addPropertyChangeListener (listener);
         return true;
      }
      return false;
   }


   /**
    * Remove a PropertyChangeListener for a specific property.
    *
    * @param propertyName  The name of the property that was listened on.
    * @param listener      The PropertyChangeListener to be removed
    */

   public void removePropertyChangeListener (
                                             String propertyName,
                                             PropertyChangeListener listener)
   {
      removeFromListeners (propertyName, listener);
   }


   /**
    * remove named listener from child-list. if propertyName is null, remove from all child-lists
    *
    * @param propertyName  No description provided
    * @param listener      No description provided
    * @return              No description provided
    */
   public synchronized boolean removeFromListeners (String propertyName, PropertyChangeListener listener)
   {
      if (children == null)
      {
         return false;
      }

      if (propertyName != null)
      {
         FujabaPropertyChangeSupport child =
            (FujabaPropertyChangeSupport) children.get (propertyName);
         if (child == null)
         {
            return false;
         }
         return child.removeFromListeners (listener);
      }
      else
      {
         boolean changed = false;
         Iterator childIter = iteratorOfChildren();
         while (childIter.hasNext())
         {
            changed = changed |  ((FujabaPropertyChangeSupport) childIter.next()).removeFromListeners (listener);
         }
         return changed;
      }
   }


   /**
    * Check if there are any listeners for a specific property.
    *
    * @param propertyName  the property name.
    * @return              true if there are ore or more listeners for the given property
    */
   public synchronized boolean hasListeners (String propertyName)
   {
      if (listeners != null && !listeners.isEmpty())
      {
         // there is a generic listener
         return true;
      }
      if (children != null)
      {
         FujabaPropertyChangeSupport child =
            (FujabaPropertyChangeSupport) children.get (propertyName);
         if (child != null)
         {
            // The child will always have a listeners Vector.
            return !child.listeners.isEmpty();
         }
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param listener  No description provided
    * @return          No description provided
    */
   public synchronized boolean hasInListeners (PropertyChangeListener listener)
   {
      return  (listeners != null && listeners.contains (listener));
   }


   /**
    * return true if listener is in child-list for propertyName, or in any child-list if propertyName
    * is null
    *
    * @param propertyName  No description provided
    * @param listener      No description provided
    * @return              No description provided
    */
   public synchronized boolean hasInListeners (String propertyName, PropertyChangeListener listener)
   {
      if (listener == null)
      {
         return false;
      }

      if (children == null)
      {
         return false;
      }

      if (propertyName != null)
      {
         FujabaPropertyChangeSupport child =
            (FujabaPropertyChangeSupport) children.get (propertyName);
         if (child != null)
         {
            // The child will always have a listeners Vector.
            return child.hasInListeners (listener);
         }
         return false;
      }
      else
      {
         Iterator childIter = iteratorOfChildren();
         while (childIter.hasNext())
         {
            if ( ((FujabaPropertyChangeSupport) childIter.next()).hasInListeners (listener))
            {
               return true;
            }
         }
         return false;
      }
   }


   /**
    * return true if listener is in global list or in child-list for propertyName, or in any
    * child-list if propertyName is null
    *
    * @param listener  No description provided
    * @return          No description provided
    */
   public boolean hasInAllListeners (PropertyChangeListener listener)
   {
      return  (hasInListeners (listener) || hasInListeners (null, listener));
   }


   /**
    * iterator of global listeners
    *
    * @return   No description provided
    */
   public synchronized Iterator iteratorOfListeners()
   {
      return  (listeners == null) ?
         FEmptyIterator.get() :
         listeners.iterator();
   }


   /**
    * iterator of all listeners in global list and child-lists
    *
    * @return   No description provided
    */
   public synchronized Iterator iteratorOfAllListeners()
   {
      if (listeners == null)
      {
         return iteratorOfListeners (null);
      }
      else
      {
         if (children == null)
         {
            return iteratorOfListeners();
         }

         return new IteratorConcatenation (iteratorOfListeners(), iteratorOfListeners (null));
      }
   }


   /**
    * iterator of listeners for propertyName or for all propertyNames if propertyName is null
    *
    * @param propertyName  No description provided
    * @return              No description provided
    */
   public synchronized Iterator iteratorOfListeners (String propertyName)
   {
      if (propertyName != null)
      {
         if (children != null)
         {
            FujabaPropertyChangeSupport child =
               (FujabaPropertyChangeSupport) children.get (propertyName);
            if (child != null)
            {
               return child.iteratorOfListeners();
            }
         }
         return FEmptyIterator.get();
      }
      else
      {
         Iterator result = null;

         Iterator childIter = iteratorOfChildren();
         while (childIter.hasNext())
         {
            FujabaPropertyChangeSupport child = (FujabaPropertyChangeSupport) childIter.next();
            if (child != null)
            {
               if (child.listeners.size() == 0)
               {
                  continue;
               }

               Iterator iter = child.iteratorOfListeners();

               if (result == null)
               {
                  result = iter;
               }
               else
               {
                  result = new IteratorConcatenation (result, iter);
               }
            }
         }

         if (result == null)
         {
            result = FEmptyIterator.get();
         }

         return result;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfListeners()
   {
      return  (listeners == null) ?
         0 : listeners.size();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfAllListeners()
   {
      return sizeOfListeners() + sizeOfListeners (null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param propertyName  No description provided
    * @return              No description provided
    */
   public int sizeOfListeners (String propertyName)
   {
      if (propertyName != null)
      {
         if (children != null)
         {
            FujabaPropertyChangeSupport child =
               (FujabaPropertyChangeSupport) children.get (propertyName);
            if (child != null)
            {
               return child.sizeOfListeners();
            }
         }
         return 0;
      }
      else
      {
         int result = 0;

         Iterator childIter = iteratorOfChildren();
         while (childIter.hasNext())
         {
            FujabaPropertyChangeSupport child = (FujabaPropertyChangeSupport) childIter.next();
            if (child != null)
            {
               result += child.listeners.size();
            }
         }
         return result;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfChildren()
   {
      return  (children == null) ?
         FEmptyIterator.get() :
         children.values().iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfChildren()
   {
      return  (children == null) ?
         FEmptyIterator.get() :
         children.keySet().iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfChildren()
   {
      return  (children == null) ?
         FEmptyIterator.get() :
         children.entrySet().iterator();
   }


   /**
    * Get the fromChildren attribute of the FujabaPropertyChangeSupport object
    *
    * @param propertyName  No description provided
    * @return              The fromChildren value
    */
   public FujabaPropertyChangeSupport getFromChildren (String propertyName)
   {
      return  (children == null) ?
         null :
         (FujabaPropertyChangeSupport) children.get (propertyName);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfChildren()
   {
      return  (children == null) ?
         0 : children.size();
   }


   /**
    * Report a bound property update to any registered listeners. No event is fired if old
    * and new are equal and non-null.
    *
    * @param propertyName  The programmatic name of the property that was changed.
    * @param oldValue      The old value of the property.
    * @param newValue      The new value of the property.
    */
   public void firePropertyChange (String propertyName,
                                   Object oldValue, Object newValue)
   {
      if (oldValue == newValue ||
          (oldValue != null && oldValue.equals (newValue)) ||
          (newValue != null && newValue.equals (oldValue)))
      {
         return;
      }

      FujabaPropertyChangeSupport child = null;
      synchronized (this)
      {
         if (children != null && propertyName != null)
         {
            child = (FujabaPropertyChangeSupport) children.get (propertyName);
         }
      }

      if (listeners != null || child != null)
      {
         // Only create an event if there's an interested receiver.
         PropertyChangeEvent evt = new PropertyChangeEvent (
            source, propertyName, oldValue, newValue);

         if (listeners != null)
         {
            synchronized (listeners)
            {
               ArrayList tmpListeners = new ArrayList (listeners.size());
               tmpListeners.addAll (listeners);
               for (int i = 0; i < tmpListeners.size(); i++)
               {
                  PropertyChangeListener target = (PropertyChangeListener) tmpListeners.get (i);
                  if (this.listeners.contains (target))
                  {
                     target.propertyChange (evt);
                  }
               }
            }
         }

         if (child != null)
         {
            child.firePropertyChange (evt);
         }
      }
   }


   /**
    * Fire an existing PropertyChangeEvent to any registered listeners. No event is fired if
    * the given event's old and new values are equal and non-null.
    *
    * @param evt  The PropertyChangeEvent object.
    */
   public void firePropertyChange (PropertyChangeEvent evt)
   {
      Object oldValue = evt.getOldValue();
      Object newValue = evt.getNewValue();
      String propertyName = evt.getPropertyName();

      if (oldValue == newValue ||
          (oldValue != null && oldValue.equals (newValue)))
      {
         return;
      }

      FujabaPropertyChangeSupport child = null;
      synchronized (this)
      {
         if (children != null && propertyName != null)
         {
            child = (FujabaPropertyChangeSupport) children.get (propertyName);
         }
      }

      if (listeners != null)
      {
         synchronized (listeners)
         {
            ArrayList tmpListeners = new ArrayList (listeners.size());
            tmpListeners.addAll (listeners);
            for (int i = 0; i < tmpListeners.size(); i++)
            {
               PropertyChangeListener target = (PropertyChangeListener) tmpListeners.get (i);
               if (this.listeners.contains (target))
               {
                  target.propertyChange (evt);
               }
            }
         }
      }
      if (child != null)
      {
         child.firePropertyChange (evt);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      source = null;
      if (listeners != null)
      {
         synchronized (listeners)
         {
            listeners.clear();
         }
      }
      if (children != null)
      {
         Iterator iter = iteratorOfChildren();
         while (iter.hasNext())
         {
            FujabaPropertyChangeSupport child = (FujabaPropertyChangeSupport) iter.next();
            child.removeYou();
         } // while
         children.clear();
      }
   }

}

/*
 * $Log: FujabaPropertyChangeSupport.java,v $
 * Revision 1.17.2.2  2006/02/07 16:39:23  lowende
 * Applied patch programmed by Dietrich: update problem of listeners fixed.
 *
 */
