/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.basic;

import java.io.File;
import java.util.Vector;

import javax.swing.*;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.action.OpenRecentProjectAction;
import de.uni_paderborn.fujaba.preferences.PreferencesProperties;
import de.upb.lib.userinterface.UserInterfaceManager;


/**
 * FileHistory provides a recently opened file list and a mechanism to update special Actions
 * (OpenRecentProjectAction) to open these files. These actions can be used to create a recent
 * file list in menus or in Toolbars. To do so, create an Action entry and a corresponding
 * Item that uses this entry like this: <pre>
 * <Action id="fileHistory0" class="de.uni_paderborn.fujaba.app.action.OpenRecentProjectAction" enabled="true">
 *   <Name>fileHistory0</Name> <Shortcut></Shortcut> <Mnemonic></Mnemonic> <ToolTip>Load recent
 * project</ToolTip> <Icon>de/uni_paderborn/fujaba/app/images/openRecent.gif</Icon> </Action>
 * <MenuSection id="openRecentSection"> <MenuItem actionId="fileHistory0"/> </MenuSection>
 * </pre> The Method updateAction updates these Actions identified by their name "fileHistory#"
 * with index # to open the files in the recently opened file list. Index starts with 0.
 *
 * @author    $Author: zuendorf $
 * @version   $Revision: 1.12.2.1 $
 */
public class FileHistory
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (FileHistory.class);

   /**
    * Holds all recently opened files as File Objects
    */
   private Vector fileVector;

   /**
    * Singleton of class FileHistory
    */
   private static FileHistory singleton;

   /**
    * Prefix for property file entries
    */
   private static String FILE_HISTORY = "FileHistory";

   /**
    * Key for PropertyOptions.
    */
   private static String FILE_HISTORY_KEY = "core/FileHistory.properties";


   /**
    * Constructor for class FileHistory
    */
   public FileHistory()
   {
      fileVector = new Vector();
      loadHistoryFromProperties();
   }


   /**
    * Get singleton of class FileHistory.
    *
    * @return   Singleton object of FileHistory
    */
   public static FileHistory get()
   {
      if (singleton == null)
      {
         singleton = new FileHistory();
      }
      return singleton;
   }


   /**
    * Adds a file to the file History
    *
    * @param file  File to add
    * @return      true if added (history changed), false if the file is still in history or
    *      not exists
    */
   public synchronized boolean addToHistory (File file)
   {
      if (file.exists() && !isInHistory (file))
      {
         try
         {
            //Clone the file here to make sure that changings to file do not affect history
            File newFile = new File (file.getCanonicalPath());
            //Add at first position of file history
            fileVector.insertElementAt (newFile, 0);
            return true;
         }
         catch (Exception e)
         {
            return false;
         }
      }
      else
      {
         return false;
      }
   }


   /**
    * Check if the provided file is in history
    *
    * @param file  File to check
    * @return      true if file is in history, false otherwise
    */
   public boolean isInHistory (File file)
   {
      for (int i = 0; i < fileVector.size(); i++)
      {
         File fileVFile = (File) fileVector.elementAt (i);

         try
         {
            if (file.getCanonicalPath().equals (fileVFile.getCanonicalPath()))
            {
               return true;
            }
         }
         catch (Exception e)
         {
            log.error ("FileHistory.isInHistory: getCanonicalPath throws:");
            e.printStackTrace();
         }
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public File firstOfHistory()
   {
      if (fileVector.size() > 0)
      {
         return (File) fileVector.get (0);
      }
      else
      {
         return null;
      }
   }

   //Most of the following methods are synchronized, wondering what would happen if
   //they where not...?

   /**
    * Loads the file history from property file. Called by constructor of FileHistory.
    */
   public synchronized void loadHistoryFromProperties()
   {
      PreferencesProperties props = PreferencesProperties.get (FILE_HISTORY_KEY);
      Vector fileNameVector = props.getVectorSetting (FILE_HISTORY);

      //delete old history
      fileVector = new Vector();

      for (int i = 0; i < fileNameVector.size(); i++)
      {
         File newFile = new File ((String) fileNameVector.elementAt (i));
         fileVector.add (newFile);
      }
   }


   /**
    * Saves the file history to properties. Should be called by ExitAction
    */
   public synchronized void saveHistoryToProperties()
   {
      PreferencesProperties props = PreferencesProperties.get (FILE_HISTORY_KEY);
      Vector fileNameVector = new Vector();

      try
      {
         for (int i = 0; i < Math.min (9, fileVector.size()); i++)
         {
            File file = (File) fileVector.elementAt (i);
            fileNameVector.add (file.getCanonicalPath());
         }

         props.putSetting (FILE_HISTORY, fileNameVector);
         props.save();
      }
      catch (Exception e)
      {
         log.error ("FileHistory.saveHistoryToProperties: getCanonicalPath() throws");
         e.printStackTrace();
      }
   }


   /**
    * Updates all fileHistory# Actions
    */
   public synchronized void updateActions()
   {
      UserInterfaceManager uiManager = UserInterfaceManager.get();

      int i = 0;
      AbstractAction action = uiManager.getFromActions ("fileHistory" + i);

      while (action != null)
      {
         if (action instanceof OpenRecentProjectAction)
         {
            OpenRecentProjectAction oRPAction = (OpenRecentProjectAction) action;
            if (i < fileVector.size())
            {
               File file = (File) fileVector.elementAt (i);
               oRPAction.setFile (file);
            }
            else
            {
               oRPAction.setFile (null);
            }
         }
         else
         {
            log.error ("FileHistory.updateActions: \n" +
               "Only OpenRecentProjectAction actions should have an id beginnig with fileHistory!");
         }

         i++;
         action = uiManager.getFromActions ("fileHistory" + i);
      }
   }


   /**
    * Checks all files in history if they exist and removes non existing files from the history.
    * It's called by OpenRecentProjectAction when the corresponding file is not found. Calling
    * this method might cause access to removable media like floppys, so it might take some
    * time.
    */
   public synchronized void removeNotExistingFiles()
   {
      int i = 0;
      while (i < fileVector.size())
      {
         File file = (File) fileVector.elementAt (i);

         if (!file.exists())
         {
            fileVector.removeElementAt (i);
            //fileVector was changed by deleting one item, so don't increment i
         }
         else
         {
            i++;
         }
      }
   }

}

/*
 * $Log: FileHistory.java,v $
 * Revision 1.12.2.1  2006/04/20 12:57:41  zuendorf
 * enhancement of the EditLink dialog to avoid frequent newby problems.
 * Editing the link name always creates a new assoc now.
 * New Assocs have unique roles names and more meaningful cardinalities now.
 *
 */
