/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.basic;

import java.awt.Dimension;
import java.awt.Insets;
import java.awt.Point;
import java.awt.geom.Point2D;
import java.beans.PropertyEditor;
import java.io.File;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.plaf.ColorUIResource;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.FujabaApp;
import de.uni_paderborn.fujaba.asg.ASGTransient;
import de.uni_paderborn.fujaba.gxl.GXLFilter;
import de.uni_paderborn.fujaba.metamodel.FType;
import de.uni_paderborn.fujaba.preferences.DebugPreferences;
import de.uni_paderborn.fujaba.views.Filter;
import de.uni_paderborn.fujaba.views.FilterClassLoader;
import de.uni_paderborn.fujaba.views.FilterManager;
import de.upb.tools.fca.FCollections;
import de.upb.tools.fca.FDuplicatedTreeMap;
import de.upb.tools.fca.FHashMap;
import de.upb.tools.fca.FLinkedList;
import de.upb.tools.fca.FTreeMap;
import de.upb.tools.fca.FTreeSet;


/**
 * Implementation of the BasicIncrement interface.
 *
 * <h2>Associations</h2>
 *
 * <pre>
 *                 0..1         Assoc         0..1 ------
 * BasicIncrement ---------------------------------| ID | UMLProject
 *                 objectHashTable      refProject ------
 * </pre>
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.190.2.6 $
 */
public class BasicIncrement implements Comparable, UniqueIdentifier
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (BasicIncrement.class);


   /**
    * Default constructor for BasicIncrement
    */
   public BasicIncrement() { }

   // ######################################################################

   /**
    * This attribute is set to true by the analyze engines in their newly generated incrs.
    * The set method automatically sets the needsToDiffed flag to false.
    */
   private boolean generated = false;


   /**
    * Sets the generated attribute of the BasicIncrement object
    *
    * @param value  The new generated value
    */
   public void setGenerated (boolean value)
   {
      this.generated = value;
   }


   /**
    * Get the generated attribute of the BasicIncrement object
    *
    * @return   The generated value
    */
   public boolean isGenerated()
   {
      return  (this.generated);
   }


   // ######################################################################

   /**
    * Wrapper for
    *
    * @param value  the tokenizer representing the value
    */
   public void setAttribValue (StringTokenizer value)
   {
      this.setAttribValue (value, null);
   }


   /**
    * sets the value of an attrib. Method is a wrapper for calling readAttributes when only
    * a single attrib is to be set.
    *
    * @param value    the tokenizer representing the value
    * @param objects  The new attribValue value
    * @see            #setAttribValue(StringTokenizer) to ommit the objects parameter
    */
   public void setAttribValue (StringTokenizer value, Hashtable objects)
   {
      FDuplicatedTreeMap attributes = new FDuplicatedTreeMap();
      attributes.put (value.nextToken(), value);
      readAttributes (objects, attributes, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient String ID = null;


   /**
    * Get the iD attribute of the BasicIncrement object
    *
    * @return   The iD value
    */
   public String getID()
   {
      if (ID == null)
      {
         ID = BasicIncrement.getUniqueID();
      }

      return ID;
   }


   /**
    * Access method for directly setting the ID, for use with loading the project, to keep
    * the IDs used the last time.
    *
    * @param ID  the new value for ID to be set.
    * @return    No description provided
    */
   public String setID (String ID)
   {
      this.ID = ID;
      return  (ID);
   }


   /**
    * It is important that every Object has an unique identifier. But the method hashCode()
    * is not unique in every case. So here are our own methods to create an unique ID. This
    * implementation is not very fine but I hope it will be very fast and that's very important!
    */
   private static transient StringBuffer uniqueID = new StringBuffer ("id0");


   /**
    * Sets the uniqueId attribute of the BasicIncrement class
    *
    * @param newUniqueID  The new uniqueId value
    */
   protected static void setUniqueId (StringBuffer newUniqueID)
   {
      uniqueID = newUniqueID;
   }


   /**
    * Get the uniqueID attribute of the BasicIncrement class
    *
    * @param increment  No description provided
    * @return           The uniqueID value
    */
   protected static synchronized String getUniqueID (boolean increment)
   {
      if (increment)
      {
         return getUniqueID();
      }
      else
      {
         return new String (uniqueID);
      }
   }


   /**
    * Get the uniqueID attribute of the BasicIncrement class
    *
    * @return   The uniqueID value
    */
   public static synchronized String getUniqueID()
   {
      boolean done = false;
      int pos = uniqueID.length() - 1;

      while (!done)
      {
         switch (uniqueID.charAt (pos))
         {
            case '0':
               uniqueID.setCharAt (pos, '1');
               done = true;
               break;
            case '1':
               uniqueID.setCharAt (pos, '2');
               done = true;
               break;
            case '2':
               uniqueID.setCharAt (pos, '3');
               done = true;
               break;
            case '3':
               uniqueID.setCharAt (pos, '4');
               done = true;
               break;
            case '4':
               uniqueID.setCharAt (pos, '5');
               done = true;
               break;
            case '5':
               uniqueID.setCharAt (pos, '6');
               done = true;
               break;
            case '6':
               uniqueID.setCharAt (pos, '7');
               done = true;
               break;
            case '7':
               uniqueID.setCharAt (pos, '8');
               done = true;
               break;
            case '8':
               uniqueID.setCharAt (pos, '9');
               done = true;
               break;
            case '9':
               uniqueID.setCharAt (pos, '0');
               pos--;
               break;
            case '.':
               uniqueID.insert (pos + 1, '1');
               done = true;
               break;
            default:
               throw new RuntimeException ("Ups");
         }
         if (pos < 2)
         {
            uniqueID.insert (2, '1');
            done = true;
         }
      }
      return new String (uniqueID);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param first   No description provided
    * @param second  No description provided
    * @return        No description provided
    */
   public static boolean lessUniqueId (String first, String second)
   {
      if (first.length() < second.length())
      {
         return true;
      }

      if (first.length() > second.length())
      {
         return false;
      }

      return first.compareTo (second) < 0;
   }


   /**
    * method from Compareble
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public int compareTo (Object obj)
   {
      if ( (obj != null && obj instanceof BasicIncrement))
      {
         return this.getID().compareTo ( ((BasicIncrement) obj).getID());
      }
      else if (obj == null)
      {
         return -1;
      }
      else
      {
         return  (this.hashCode() - obj.hashCode());
      }
   }

   // ####################################################################
   // (more) generic save-mechanism starts here

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected static transient boolean dontUseAccessMethodsInThisClass = false;

   /**
    * Cache the class information for speedup reasons.
    */
   private static transient FHashMap classInfos = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public static void resetClassInfos()
   {
      if (classInfos != null)
      {
         classInfos.clear();
         classInfos = null;
      }
   }


   /**
    * Returns a vector: first is the attribute list and second is the access method hash map
    *
    * @param clazz  No description provided
    * @return       The classFromCache value
    */
   private Vector getClassFromCache (Class clazz)
   {
      if (classInfos == null)
      {
         classInfos = new FHashMap();
      }

      // lookup cache hashmap
      Vector cacheEntry = (Vector) classInfos.get (clazz);
      if (cacheEntry == null)
      {
         // get fields of this class, its superclasses and all access methods
         Field[] tmpClassFields;
         Method[] tmpClassMethods;
         String name;
         Vector classFields = new Vector();
         FHashMap methodHashMap = new FHashMap();
         Class javaClass = clazz;

         while (javaClass != null)
         {
            // store all fields
            tmpClassFields = javaClass.getDeclaredFields();
            for (int i = 0; i < tmpClassFields.length; i++)
            {
               tmpClassFields[i].setAccessible (true);
               classFields.addElement (tmpClassFields[i]);
            }

            // store all methods
            tmpClassMethods = javaClass.getDeclaredMethods();
            for (int i = 0; i < tmpClassMethods.length; i++)
            {
               name = tmpClassMethods[i].getName();
               if (name.startsWith ("set") || name.startsWith ("add"))
               {
                  tmpClassMethods[i].setAccessible (true);
                  addToAddSetMethods (tmpClassMethods[i], methodHashMap);
               }
            }
            javaClass = javaClass.getSuperclass();
         }

         // add to cache classInfo
         cacheEntry = new Vector (2);
         cacheEntry.add (classFields);
         cacheEntry.add (methodHashMap);
         classInfos.put (clazz, cacheEntry);
      }
      return  (cacheEntry);
   }


   /**
    * Access method for an one to n association.
    *
    * @param newMethod  The object added.
    * @param map        The object added.
    */
   private final void addToAddSetMethods (Method newMethod, FHashMap map)
   {
      Vector newSet = (Vector) map.get (newMethod.getName());
      if (newSet == null)
      {
         newSet = new Vector (2);
         newSet.add (newMethod);
         map.put (newMethod.getName(), newSet);
      }
      else
      {
         newSet.add (newMethod);
      }
   }

   // These private variables are used for a comfortable handling
   // private OrderedSet setOBasicIncrements = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient StringBuffer data;


   /**
    * save attributes of current class object, only those which are not marked static or transient.
    *
    * @param data             No description provided
    * @param setOfNeighbours  No description provided
    */
   public void writeAttributes (StringBuffer data, FTreeSet setOfNeighbours)
   {
      // init the string buffer
      this.data = data;

      // Use reflection mechanism to get attributes and method of current class
      Class javaClass = this.getClass();
      Vector cacheEntry = getClassFromCache (javaClass);
      Vector classFields = (Vector) cacheEntry.get (0);

      // now store the fields into the string buffer
      Object fieldObject;
      Field currentField;
      Enumeration elements = classFields.elements();

      while (elements.hasMoreElements())
      {
         try
         {
            currentField = (Field) elements.nextElement();
            if (!checkStaticTransient (currentField))
            {
               currentField.setAccessible (true);
               fieldObject = currentField.get (this);

               // let's write into the stringBuffer data.
               callWriteToStringBuffer (fieldObject, currentField, setOfNeighbours);
            }
         }
         catch (IllegalAccessException accessExcept)
         {
            log.info ("Illegal access at writing attributes in "
               + this.getClass().getName());
         }
         catch (SecurityException accessExcept)
         {
            log.info ("Security Exception at writing attributes in "
               + this.getClass().getName()
               + ". Please check Security Manager.");
         }
         catch (NullPointerException nullExcept)
         {
            nullExcept.printStackTrace();
            // ignore these (null) attributes and skip to the next.
         }
      }
   }


   /**
    * check if the currentField is a static field
    *
    * @param currentField  No description provided
    * @return              No description provided
    */
   private boolean checkStatic (Field currentField)
   {
      int fieldModi = currentField.getModifiers();
      return  (Modifier.isStatic (fieldModi));
   }


   /**
    * check if the currentField is a static field
    *
    * @param currentField  No description provided
    * @return              No description provided
    */
   private boolean checkStaticTransient (Field currentField)
   {
      int fieldModi = currentField.getModifiers();
      return  (Modifier.isTransient (fieldModi) ||
         Modifier.isStatic (fieldModi));
   }


   /**
    * used by writeAttributes to call stringBuffer functions
    *
    * @param fieldObject      the attrib to be saved.
    * @param currentField     the field description of the attrib.
    * @param setOfNeighbours  to be saved incrs are inserted here and are saved later
    */
   private void callWriteToStringBuffer (Object fieldObject,
                                         Field currentField,
                                         FTreeSet setOfNeighbours)
   {
      String qualFieldName = getQualifiedFieldName (currentField);
      // let's look, what has to be done.
      if (fieldObject instanceof Collection)
      {
         writeToStringBuffer (qualFieldName, (Collection) fieldObject, setOfNeighbours);
      }
      else if (fieldObject instanceof Map)
      {
         writeToStringBuffer (qualFieldName, (Map) fieldObject, setOfNeighbours);
      }
      else if (fieldObject instanceof Filter)
      {
         if (fieldObject instanceof BasicIncrement)
         {
            writeToStringBuffer (qualFieldName + "Mode", "incr");
            writeToStringBuffer (qualFieldName, (BasicIncrement) fieldObject, setOfNeighbours);
         }
         else
         {
            writeToStringBuffer (qualFieldName + "Mode", "plain");

            Class clazz = fieldObject.getClass();
            writeToStringBuffer (qualFieldName + "Name",  ((Filter) fieldObject).getName());
            writeToStringBuffer (qualFieldName + "Type", clazz.getName());

            String file = null;
            ClassLoader loader = clazz.getClassLoader();
            if (! (loader instanceof FilterClassLoader))
            {
               loader = FilterClassLoader.get();
            }
            Iterator entries =  ((FilterClassLoader) loader).entriesOfResolvedClasses();
            while (entries.hasNext() && file == null)
            {
               Map.Entry entry = (Map.Entry) entries.next();
               if (entry.getValue() == clazz)
               {
                  file = (String) entry.getKey();
               }
            }
            writeToStringBuffer (qualFieldName + "File", file);
         }
      }
      else if (fieldObject instanceof BasicIncrement)
      {
         writeToStringBuffer (qualFieldName, (BasicIncrement) fieldObject, setOfNeighbours);
      }
      else if (fieldObject instanceof String)
      {
         writeToStringBuffer (qualFieldName, (String) fieldObject);
      }
      else if (fieldObject instanceof Boolean)
      {
         writeToStringBuffer (qualFieldName,  ((Boolean) fieldObject).booleanValue());
      }
      else if (fieldObject instanceof Float)
      {
         writeToStringBuffer (qualFieldName,  ((Float) fieldObject).floatValue());
      }
      else if (fieldObject instanceof Double)
      {
         writeToStringBuffer (qualFieldName,  ((Double) fieldObject).doubleValue());
      }
      else if (fieldObject instanceof Short)
      {
         writeToStringBuffer (qualFieldName,  ((Short) fieldObject).shortValue());
      }
      else if (fieldObject instanceof Byte)
      {
         writeToStringBuffer (qualFieldName,  ((Byte) fieldObject).byteValue());
      }
      else if (fieldObject instanceof Integer)
      {
         writeToStringBuffer (qualFieldName,  ((Integer) fieldObject).intValue());
      }
      else if (fieldObject instanceof Long)
      {
         writeToStringBuffer (qualFieldName,  ((Long) fieldObject).longValue());
      }
      else if (fieldObject instanceof Dimension)
      {
         writeToStringBuffer (qualFieldName + "Width",  ((Dimension) fieldObject).width);
         writeToStringBuffer (qualFieldName + "Height",  ((Dimension) fieldObject).height);
      }
      else if (fieldObject instanceof Insets)
      {
         writeToStringBuffer (qualFieldName + "Top",  ((Insets) fieldObject).top);
         writeToStringBuffer (qualFieldName + "Bottom",  ((Insets) fieldObject).bottom);
         writeToStringBuffer (qualFieldName + "Left",  ((Insets) fieldObject).left);
         writeToStringBuffer (qualFieldName + "Right",  ((Insets) fieldObject).right);
      }

      //
      // write instances of PropertyEditor
      //
      else if (fieldObject instanceof PropertyEditor)
      {
         writeToStringBuffer (qualFieldName + "Value",  ((PropertyEditor) fieldObject).getAsText());
         writeToStringBuffer (qualFieldName + "Class", fieldObject.getClass().getName());
      }

      //
      // write ColorUIResource
      //
      else if (fieldObject instanceof ColorUIResource)
      {
         writeToStringBuffer (qualFieldName + "Green",  ((ColorUIResource) fieldObject).getGreen());
         writeToStringBuffer (qualFieldName + "Red",  ((ColorUIResource) fieldObject).getRed());
         writeToStringBuffer (qualFieldName + "Blue",  ((ColorUIResource) fieldObject).getBlue());
      }

      //
      // write java.lang.reflect.Method
      //
      else if (fieldObject instanceof Method)
      {
         String methodName =  ((Method) fieldObject).getName();
         Class params[] =  ((Method) fieldObject).getParameterTypes();
         String className =  ((Method) fieldObject).getDeclaringClass().getName();
         writeToStringBuffer (qualFieldName + "Name", methodName);
         writeToStringBuffer (qualFieldName + "ClassName", className);
         int i = 0;
         int length = params.length;
         for (i = 0; i < length; i++)
         {
            writeToStringBuffer (qualFieldName + "Params", params[i].getName());
         }
      }
      else
      {
         if ( (fieldObject != null) &&  (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD)))
         {
            log.debug ("Attention! attributes of type " + fieldObject.getClass() + " will not be saved! In order to save them edit class de.uni_paderborn.fujaba.basic.BasicIncrement:");
            log.debug ("1. add by copy-paste an \"else if (fieldObject instanceof ...\" line to method \"private void callWriteToStringBuffer (Object fieldObject, Field currentField, FTreeSet setOfNeighbours)\"");
            log.debug ("2. add by copy-paste a method \"writeToStringBuffer\"");
            log.debug ("3. add by copy-paste a method \"readFromStringTokenizer\"");
         }
      }
   }


   /**
    * wrapper for
    *
    * @param objects     No description provided
    * @param attributes  No description provided
    * @see               #readAttributes(Hashtable, FDuplicatedTreeMap, boolean)
    */
   public void readAttributes (Hashtable objects, FDuplicatedTreeMap attributes)
   {
      this.readAttributes (objects, attributes, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static transient StringBuffer fieldNameStringBuffer = null;


   /**
    * Get the qualifiedFieldName attribute of the BasicIncrement class
    *
    * @param field  No description provided
    * @return       The qualifiedFieldName value
    */
   private static String getQualifiedFieldName (Field field)
   {
      if (fieldNameStringBuffer == null)
      {
         fieldNameStringBuffer = new StringBuffer();
      }
      else
      {
         fieldNameStringBuffer.delete (0, fieldNameStringBuffer.length());
      }
      fieldNameStringBuffer.append (field.getDeclaringClass().getName()).append ("::").append (field.getName());
      return fieldNameStringBuffer.toString();
   }


   /**
    * For backward compatibility: returns either the full qualified fieldname (declaringClass::fieldName)
    * if it is a key in the map or the normal fieldname otherwise.
    *
    * @param map          No description provided
    * @param field        No description provided
    * @param fieldSuffix  No description provided
    * @return             The correctFieldName value
    */
   private static String getCorrectFieldName (Map map, Field field, String fieldSuffix)
   {
      String fieldName = field.getName();
      String declaringClass = field.getDeclaringClass().getName();

      if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
      {
         FujabaDebug.println (DebugPreferences.DEBUG_LEVEL_SAVELOAD, "fieldName: " + fieldName);
         FujabaDebug.println (DebugPreferences.DEBUG_LEVEL_SAVELOAD, "declaringClass: " + declaringClass);
      }

      if (fieldNameStringBuffer == null)
      {
         fieldNameStringBuffer = new StringBuffer();
      }
      else
      {
         fieldNameStringBuffer.delete (0, fieldNameStringBuffer.length());
      }
      fieldNameStringBuffer.append (declaringClass).append ("::").append (fieldName).append (fieldSuffix);
      String fieldNameString = fieldNameStringBuffer.toString();

      if (map.containsKey (fieldNameString))
      {
         if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
         {
            FujabaDebug.println (DebugPreferences.DEBUG_LEVEL_SAVELOAD, "map contains " + fieldNameStringBuffer.toString());
         }
         return fieldNameString;
      }
      else
      {
         if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
         {
            FujabaDebug.println (DebugPreferences.DEBUG_LEVEL_SAVELOAD, "!!! map does not contain " + fieldNameStringBuffer.toString());
         }
         return fieldName;
      }
   }


   /**
    * read attributes for current class instance from a fpr-file, only those which are not
    * marked as transient or static
    *
    * @param objects     No description provided
    * @param attributes  No description provided
    * @param loadAll     No description provided
    */
   public void readAttributes (Hashtable objects, FDuplicatedTreeMap attributes,
                               boolean loadAll)
   {
      Class javaClass = this.getClass();
      Vector cacheEntry = getClassFromCache (javaClass);
      Vector classFields = (Vector) cacheEntry.get (0);
      FHashMap methodMap = (FHashMap) cacheEntry.get (1);
      //      log.info( "classFIelds: " + classFields);
      HashSet processedUnqualifiedAttributes = null; // used for checking of duplicate unqualified attributes for old projects

      // now store the fields into the string buffer
      Object fieldObject;
      Field currentField;
      Enumeration elements = classFields.elements();

      while (elements.hasMoreElements())
      {
         try
         {
            currentField = (Field) elements.nextElement();
            if (!checkStatic (currentField))
            {
               fieldObject = currentField.get (this);
               // let's mess with file data.

               String fieldName = getCorrectFieldName (attributes, currentField, "");
               if (processedUnqualifiedAttributes != null && processedUnqualifiedAttributes.contains (fieldName))
               {
                  log.error ("Problem in old Project: Duplicated Field entries!");
                  log.info ("Field " + currentField.getName() + " is defined more than once in the class hierarchy of " + this.getClass().getName());
                  log.info ("Skipping other initialization for this Field");
                  log.error ("Please save the project again.");
               }
               else if (loadAll || attributes.containsKey (fieldName))
               {
                  callReadFromStringBuffer (fieldObject, currentField,
                     objects, attributes, methodMap);
                  if (fieldName.indexOf ("::") == -1 && attributes.containsKey (fieldName))
                  {
                     if (processedUnqualifiedAttributes == null)
                     {
                        processedUnqualifiedAttributes = new HashSet();
                     }
                     processedUnqualifiedAttributes.add (fieldName);
                  }
               }
            }
         }
         catch (IllegalAccessException accessExcept)
         {
            log.info ("Illegal access at writing attributes in " +
               this.getClass().getName());
            accessExcept.printStackTrace();
         }
         catch (SecurityException accessExcept)
         {
            log.info ("Security Exception at writing attributes in "
               + this.getClass().getName() +
               ". Please check Security Manager.");
            accessExcept.printStackTrace();
         }
         catch (NullPointerException nullExcept)
         {
            // ignore these (null) attributes and skip to the next.
            nullExcept.printStackTrace();
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fieldObject              No description provided
    * @param currentField             No description provided
    * @param objects                  No description provided
    * @param attributes               No description provided
    * @param methodMap                No description provided
    * @throws IllegalAccessException  Exception description not provided
    */
   private void callReadFromStringBuffer (Object fieldObject,
                                          Field currentField,
                                          Hashtable objects,
                                          FDuplicatedTreeMap attributes,
                                          FHashMap methodMap)
       throws IllegalAccessException
   {
      // let's take care of what type of Object we have got.
      if (Collection.class.isAssignableFrom (currentField.getType()) ||
         fieldObject instanceof Collection ||
         Map.class.isAssignableFrom (currentField.getType()) ||
         fieldObject instanceof Hashtable)
      {
         // we have to handle a container
         putElements (currentField, fieldObject, objects,
            attributes, methodMap);
      }
      else
      {
         try
         {
            // we only have to set a single value
            setValue (currentField, fieldObject, objects, attributes, methodMap);
         }
         catch (Exception e)
         {
            log.info ("<BasicIncrement::callReadFromStringBuffer> Error setting field: " + currentField);
            //e.printStackTrace ();
         }

      }
   }


   /**
    * Sets the value attribute of the BasicIncrement object
    *
    * @param currentField  The new value value
    * @param fieldObject   The new value value
    * @param objects       The new value value
    * @param attributes    The new value value
    * @param methodMap     The new value value
    */
   private final void setValue (Field currentField, Object fieldObject,
                                Hashtable objects,
                                FDuplicatedTreeMap attributes,
                                FHashMap methodMap)
   {
      Object[] args = new Object[1];
      boolean foundSet = false;
      boolean methodIsInCache = true; // be optimistic

      Enumeration setMethods = null;

      Method setMethod = getFromFieldWriteMethods (this.getClass(),
         currentField);

      if (setMethod == null)
      { // if (1 == 1)

         // not yet cached
         methodIsInCache = false;

         // Catch the set method for the non-container attribs.
         setMethods = getAddSetMethods
             (getMethodName ("set", currentField.getName()), methodMap);
      }

      // retrieve the data which is to be set, we need the isAssignableFrom
      // to take care of the null-Fields and the instanceof to take
      // care of the interface-fields.
      if (BasicIncrement.class.isAssignableFrom (currentField.getType()) ||
          (fieldObject instanceof BasicIncrement) ||
         FType.class.isAssignableFrom (currentField.getType()))
      {
         args[0] = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, ""),
            (BasicIncrement) null, objects, attributes);
      }
      else if (String.class.isAssignableFrom (currentField.getType()) ||
         fieldObject instanceof String)
      {
         args[0] = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, ""), (String) null, attributes);
      }
      else if (fieldObject instanceof Boolean)
      {
         args[0] = Boolean.valueOf (readFromStringTokenizer (
            getCorrectFieldName (attributes, currentField, ""), false, attributes));
      }
      else if (fieldObject instanceof Float)
      {
         args[0] = new Float (readFromStringTokenizer (
            getCorrectFieldName (attributes, currentField, ""), 0.0f, attributes));
      }
      else if (fieldObject instanceof Short)
      {
         args[0] = new Short (readFromStringTokenizer (
            getCorrectFieldName (attributes, currentField, ""),  ((short) 0), attributes));
      }
      else if (fieldObject instanceof Double)
      {
         args[0] = new Double (readFromStringTokenizer (
            getCorrectFieldName (attributes, currentField, ""),  ((double) 0), attributes));
      }
      else if (fieldObject instanceof Integer)
      {
         args[0] = new Integer (readFromStringTokenizer (
            getCorrectFieldName (attributes, currentField, ""), 0, attributes));
      }
      else if (fieldObject instanceof Long)
      {
         args[0] = new Long (readFromStringTokenizer (
            getCorrectFieldName (attributes, currentField, ""), 0, attributes));
      }
      else if (fieldObject instanceof Byte)
      {
         args[0] = new Byte (readFromStringTokenizer (
            getCorrectFieldName (attributes, currentField, ""),  ((byte) 0), attributes));
      }
      else if (Dimension.class.isAssignableFrom (currentField.getType()) ||
         fieldObject instanceof Dimension)
      {
         int sizeW = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Width"),
            0, attributes);
         int sizeH = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Height"),
            0, attributes);
         args[0] = new Dimension (sizeW, sizeH);
         if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
         {
            log.debug ("newDimension=" + args[0]);
         }
      }
      else if (Insets.class.isAssignableFrom (currentField.getType()) ||
         fieldObject instanceof Insets)
      {
         int top = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Top"),
            0, attributes);
         int bottom = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Bottom"),
            0, attributes);
         int left = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Left"),
            0, attributes);
         int right = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Right"),
            0, attributes);

         args[0] = new Insets (top, left, bottom, right);
      }
      // try to read PropertyEditor Interface
      else if (PropertyEditor.class.isAssignableFrom (currentField.getType()) ||
          (fieldObject instanceof PropertyEditor &&
         ! (fieldObject instanceof BasicIncrement)))
      {
         // found PropertyEditor Interface, try to read value and class
         String value = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Value"),
            "", attributes);
         String theClazzName = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Class"),
            null, attributes);
         PropertyEditor newEditor = null;
         Class theClazz = null;

         if (theClazzName != null)
         {
            try
            {
               theClazz = ClassMap.get().forName (theClazzName);
               newEditor = (PropertyEditor) theClazz.newInstance();
            }
            catch (Exception exception)
            {
               if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
               {
                  log.debug (this + " not able to create an instance of "
                     + theClazzName);
                  log.debug (exception.getMessage());
                  log.debug (exception.toString());
               }
               newEditor = null;
            }

            if (newEditor != null && value != null)
            {
               newEditor.setAsText (value);
            }

            if (newEditor != null)
            {
               args[0] = newEditor;
               if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
               {
                  log.debug ("BasicIncrement.PropertyEditor.reading " + args[0]);
               }
            }
         }
      }

      //
      // read java.lang.reflect.Method
      //
      else if (Method.class.isAssignableFrom (currentField.getType()) ||
         fieldObject instanceof Method)
      {
         if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
         {
            log.debug ("BasicIncrement.Method.fieldObject=" + fieldObject);
            log.debug ("BasicIncrement.Method.currentField=" + currentField);
         }

         String className = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "ClassName"),
            "", attributes);
         String methodName = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Name"),
            "", attributes);
         Class params[] = null;
         Class tmpClass = null;

         Class theClazz = null;

         boolean success = true;
         Method theMethod = null;

         try
         {
            theClazz = ClassMap.get().forName (className);
         }
         catch (Exception exception)
         {
            if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
            {
               log.debug ("BasicIncrement.Method.setValue () can not get class :" + className + ".");
               log.debug (exception.getMessage());
               log.debug (exception.toString());
            }

            success = false;
         }

         FLinkedList tmpParams = (FLinkedList) attributes.values (getCorrectFieldName (attributes, currentField, "Params"));

         if (tmpParams != null && success)
         {
            Iterator iter = tmpParams.iterator();
            int size = tmpParams.size();
            int i = 0;
            StringTokenizer tokenizer = null;
            String nextToken = null;

            params = new Class[size];

            while (iter.hasNext() && success)
            {
               tokenizer = (StringTokenizer) iter.next();

               while (tokenizer.hasMoreTokens())
               {
                  nextToken = tokenizer.nextToken();

                  if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
                  {
                     log.debug ("BasicIncrement.Method.paramToken=" + nextToken);
                  }

                  try
                  {
                     tmpClass = ClassMap.get().forName (nextToken);
                     params[i] = tmpClass;
                  }
                  catch (Exception exception)
                  {
                     // try to get primitive class (expensive , i know !)
                     if (nextToken.equals ("boolean"))
                     {
                        params[i] = Boolean.TYPE;
                     }
                     else if (nextToken.equals ("int"))
                     {
                        params[i] = Integer.TYPE;
                     }
                     else if (nextToken.equals ("char"))
                     {
                        params[i] = Character.TYPE;
                     }
                     else if (nextToken.equals ("byte"))
                     {
                        params[i] = Byte.TYPE;
                     }
                     else if (nextToken.equals ("short"))
                     {
                        params[i] = Short.TYPE;
                     }
                     else if (nextToken.equals ("long"))
                     {
                        params[i] = Long.TYPE;
                     }
                     else if (nextToken.equals ("float"))
                     {
                        params[i] = Float.TYPE;
                     }
                     else if (nextToken.equals ("double"))
                     {
                        params[i] = Double.TYPE;
                     }
                     else
                     {
                        if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
                        {
                           log.debug ("BasicIncrement.setValue() : can not create Class!" + nextToken);
                           log.debug (exception.getMessage());
                           log.debug (exception.toString());
                        }
                        params = new Class[]{};
                        success = false;
                     }
                  }
               }
               i++;
            }
         }
         else
         {
            params = new Class[]{};
         }

         if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
         {
            log.debug ("BasicIncrement.Method.className=" + className);
            log.debug ("BasicIncrement.Method.methodName=" + methodName);
            log.debug ("BasicIncrement.Method.params=" + params);
         }

         if (success)
         {
            try
            {
               theMethod = theClazz.getMethod (methodName, params);
            }
            catch (Exception exception)
            {
               if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
               {
                  log.debug ("BasicIncrement.setValue() : can not get method " + methodName);
                  log.debug (exception.getMessage());
                  log.debug (exception.toString());
               }
               success = false;
            }
         }

         if (success)
         {
            if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
            {
               log.debug ("BasicIncrement.Method.setMethod : " + theMethod + "\n\n");
            }
            args[0] = theMethod;
         }
         else
         {
            if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
            {
               log.debug ("BasicIncrement.Method.setMethod failed !\n\n");
            }
            args[0] = null;
         }
      }
      else if (Filter.class.isAssignableFrom (currentField.getType()) ||
         fieldObject instanceof Filter)
      {
         String mode = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Mode"), "plain", attributes);
         if (mode.compareTo ("incr") == 0)
         {
            args[0] = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, ""),
               (BasicIncrement) null, objects, attributes);
         }
         else
         {
            String name = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Name"), null, attributes);
            String type = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "Type"), null, attributes);
            String file = readFromStringTokenizer (getCorrectFieldName (attributes, currentField, "File"), null, attributes);

            if (type != null)
            {
               try
               {
                  Class clazz = Class.forName (type);
                  args[0] = FilterManager.createInstance (clazz);
               }
               catch (Exception e)
               {
               }
            }

            if (args[0] == null && file != null)
            {
               File theFile = new File (file);
               if (theFile.exists())
               {
                  FilterManager.createInstance (theFile);
               }
            }

            if (args[0] == null && name != null)
            {
               args[0] = FilterManager.get().getFilter (name);
            }
         }
         if (args[0] == null)
         {
            if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
            {
               log.debug (
                  "Loading of Filter failed - Attrib not assigned: " + getCorrectFieldName (attributes, currentField, "")
                  + " ... " + this.getClass());
            }
         }

      }
      else
      {
         if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
         {
            log.debug (
               "Attrib not assigned: " + getCorrectFieldName (attributes, currentField, "")
               + " ... " + this.getClass());
         }
      }

      foundSet = false;
      Class declaringClass = currentField.getDeclaringClass();
      Class methodDeclaringClass;

      while (!foundSet &&  (methodIsInCache ||
          ( (setMethods != null)
         && setMethods.hasMoreElements()
         && !dontUseAccessMethodsInThisClass)))
      {
         try
         {
            if (!methodIsInCache)
            {
               setMethod = (Method) setMethods.nextElement();
            }
            methodDeclaringClass = setMethod.getDeclaringClass();

            if (methodDeclaringClass.isAssignableFrom (declaringClass))
            {
               if (!setMethod.isAccessible())
               {
                  setMethod.setAccessible (true);
               }

               setMethod.invoke (this, args);
               foundSet = true;

               if (!methodIsInCache)
               {
                  addToFieldWriteMethods (this.getClass(), currentField, setMethod);
               }
            }
         }
         catch (IllegalArgumentException argExcept)
         {
            if (setMethods == null || !setMethods.hasMoreElements())
            {
               log.info (this.getClass() + ": Id: " + this.getID());
               log.info (args[0].getClass());
               log.info (setMethod);

               argExcept.printStackTrace();
            }
         }
         catch (IllegalAccessException accExcept)
         {
            if (setMethods == null || !setMethods.hasMoreElements())
            {
               log.info (this.getClass() + ": Id: " + this.getID());
               log.info (args[0].getClass());
               log.info (setMethod);
               accExcept.printStackTrace();
            }
         }
         catch (InvocationTargetException invExcept)
         {
            if (setMethods == null || !setMethods.hasMoreElements())
            {
               log.info (this.getClass() + ": Id: " + this.getID());
               if (args[0] != null)
               {
                  log.info (args[0].getClass());
               } // if

               else
               {
                  log.info ("null");

               } // else

               log.info (setMethod);

               invExcept.printStackTrace();
            }
         }
         catch (Exception e)
         {
            log.info ("\nException: " + e);
            log.info ("Method:      " + setMethod);
            log.info ("this:        " + this);
            log.info ("arg:         " + args[0]);
            log.info ("fieldObject: " + fieldObject);
            log.info ("currentField.getType ():" + currentField.getType());
            e.printStackTrace();
         }
      }

      // we only have to do something, if we are out of set-methods
      if (foundSet == false)
      {
         // this attrib has no setMethod, so we must do it the direct
         // way
         try
         {
            currentField.set (this, args[0]);
         }
         catch (IllegalAccessException accExcept)
         {
            accExcept.printStackTrace();
         }
      }
   }


   /**
    * Get the methodName attribute of the BasicIncrement object
    *
    * @param prefix      No description provided
    * @param attribName  No description provided
    * @return            The methodName value
    */
   private final String getMethodName (String prefix, String attribName)
   {
      StringBuffer buf = new StringBuffer();
      buf.append (prefix);
      buf.append (Character.toUpperCase (attribName.charAt (0)));
      buf.append (attribName.substring (1));
      return new String (buf);
   }


   /**
    * Get the addSetMethods attribute of the BasicIncrement object
    *
    * @param addMethod  No description provided
    * @param methodMap  No description provided
    * @return           The addSetMethods value
    */
   private final Enumeration getAddSetMethods (String addMethod,
                                               FHashMap methodMap)
   {
      Vector methods = (Vector) methodMap.get (addMethod);
      if (methods == null)
      {
         return null;
      }
      else
      {
         return  (methods.elements());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param addMethods  No description provided
    * @return            No description provided
    */
   private final Method nextAddMethod (Enumeration addMethods)
   {
      if ( (addMethods != null) &&  (addMethods.hasMoreElements()))
      {
         return  ((Method) addMethods.nextElement());
      }
      else
      {
         return  (null);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fieldObject   No description provided
    * @param myToken       No description provided
    * @param currentField  No description provided
    * @param objects       No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   private final Object callReadFromStringTokenizer (Object fieldObject,
                                                     StringTokenizer myToken,
                                                     Field currentField,
                                                     Hashtable objects,
                                                     FDuplicatedTreeMap attributes)
   {
      if (Collection.class.isAssignableFrom (currentField.getType()))
      {
         return  (readFromStringTokenizer ((Collection) fieldObject, myToken, objects, attributes));
      }
      else if (Map.class.isAssignableFrom (currentField.getType()))
      {
         return  (readFromStringTokenizer ((Map) fieldObject, myToken, objects));
      }

      return null;
   }

   // caching the search for access methods. Store successful access
   // methods, using class and field as key
   // i.e fieldWriteMethods.get (class) returns another HashMap where
   // field serves as a key
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static transient FHashMap fieldWriteMethods = new FHashMap();


   /**
    * Access method for an one to n association.
    *
    * @param c  The object added.
    * @param f  The object added.
    * @param m  The object added.
    */
   private void addToFieldWriteMethods (Class c, Field f, Method m)
   {
      FHashMap entry = (FHashMap) fieldWriteMethods.get (c);
      if (entry == null)
      {
         entry = new FHashMap();
         fieldWriteMethods.put (c, entry);
      }

      entry.put (f, m);
   }


   /**
    * Get the fromFieldWriteMethods attribute of the BasicIncrement object
    *
    * @param c  No description provided
    * @param f  No description provided
    * @return   The fromFieldWriteMethods value
    */
   private Method getFromFieldWriteMethods (Class c, Field f)
   {
      if (fieldWriteMethods != null)
      {
         FHashMap entry = (FHashMap) fieldWriteMethods.get (c);

         if (entry != null)
         {
            return (Method) entry.get (f);
         }
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public static void resetFieldWriteMethods()
   {
      Iterator iter = fieldWriteMethods.keySet().iterator();

      while (iter.hasNext())
      {
         fieldWriteMethods.remove (iter.next());
      }
   }


   /**
    * insert an element from the tokenizer in the given fieldObject
    *
    * @param currentField             No description provided
    * @param fieldObject              No description provided
    * @param objects                  No description provided
    * @param attributes               No description provided
    * @param methodMap                No description provided
    * @throws IllegalAccessException  Exception description not provided
    */
   private final void putElements (Field currentField, Object fieldObject,
                                   Hashtable objects, FDuplicatedTreeMap attributes,
                                   FHashMap methodMap)
       throws IllegalAccessException
   {
      boolean methodIsInCache = true;

      Method addMethod = getFromFieldWriteMethods (this.getClass(), currentField);

      Enumeration addMethods = null;

      if (addMethod == null)
      { // if (1 == 1)

         // not yet known method, search for it.
         methodIsInCache = false;

         addMethods = getAddSetMethods
             (getMethodName ("addTo", currentField.getName()), methodMap);

         if (addMethods == null)
         {
            FujabaDebug.println ("OhOh, putElements did not find an addMethods iterator for "
               + getCorrectFieldName (attributes, currentField, ""));
         }

         // lets take the first possible access method
         addMethod = nextAddMethod (addMethods);
      }
      else
      {
         if (!addMethod.getDeclaringClass().isAssignableFrom (this.getClass()))
         {
            FujabaDebug.println ("RED ALERT::::::::: " + getCorrectFieldName (attributes, currentField, "")
               + " " + addMethod.getName());
         }
      }

      boolean retry = false;
      Object[] args = new Object[1];

      Iterator iter = FCollections.iterator (attributes.values (getCorrectFieldName (attributes, currentField, "")));

      while ( (iter.hasNext() || retry) &&  (addMethod != null) &&
         !dontUseAccessMethodsInThisClass)
      {
         try
         {
            while (iter.hasNext() || retry)
            {
               if (!retry)
               {
                  args[0] = callReadFromStringTokenizer (fieldObject,
                     (StringTokenizer) iter.next(), currentField,
                     objects, attributes);
               }
               retry = false; // only one retry per addMethod

               if (!addMethod.isAccessible())
               {
                  addMethod.setAccessible (true);
               }

               addMethod.invoke (this, args);

               if (!methodIsInCache)
               {
                  // that one worked, cache it
                  addToFieldWriteMethods (this.getClass(), currentField, addMethod);
               }
            }
         }
         catch (IllegalArgumentException argExcept)
         {
            Method oldMethod = addMethod;

            // lets ignore this method and flip to next, until we are out of
            // methods
            retry = true; // the current element needs still to be saved

            addMethod = nextAddMethod (addMethods);

            if (addMethod == null)
            {
               log.info (this.getClass() + ": Id: " + this.getID());
               log.info (args[0].getClass());
               log.info (oldMethod);
               argExcept.printStackTrace();
            }

         }
         catch (InvocationTargetException invExcept)
         {
            // lets ignore this method and flip to next, until we are out of methods
            retry = true; // the current element needs still to be saved
            addMethod = nextAddMethod (addMethods);
         }
      }

      // if this attrib has no access-method, lets do it the direct way
      // first don't forget the current element!
      if (retry)
      {
         if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
         {
            log.debug (
               "Did not find an addTo method for "
               + this.getClass().getName() + " : " +
               getCorrectFieldName (attributes, currentField, ""));
         }

         addElem (fieldObject, args[0], currentField);
      }

      while (iter.hasNext())
      {
         addElem (fieldObject, callReadFromStringTokenizer (fieldObject,
            (StringTokenizer) iter.next(), currentField,
            objects,
            attributes),
            currentField);
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param fieldObject   The object added.
    * @param arg           The object added.
    * @param currentField  The object added.
    */
   private void addElem (Object fieldObject, Object arg, Field currentField)
   {
      if (Map.class.isAssignableFrom (currentField.getType()))
      {
         if (fieldObject == null)
         {
            log.info ("BasicIncrement.addElem: " + this.getClass() + ", Id:" + this.getID() + ":Oops " + currentField.getName() + " " + arg);
         }
         else
         {
             ((Map) fieldObject).put ( ((KeyValuePair) arg).getKey(),
                ((KeyValuePair) arg).getValue());
         }
      }
      else if (Collection.class.isAssignableFrom
          (currentField.getType()))
      {
         if (fieldObject == null)
         {
            log.info ("BasicIncrement.addElem: " + this.getClass() + ", Id:" + this.getID() + ":Oops " + currentField.getName() + " " + arg);
         }
         else
         {
             ((Collection) fieldObject).add (arg);
         }
      }
   }

   // (more) generic save mechanism ends here
   // ###################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param data             No description provided
    * @param savedIncrements  No description provided
    */
   public final void writeClassToStringBuffer (StringBuffer data, FTreeSet savedIncrements)
   {
      writeClassToStringBuffer (data, savedIncrements, (GXLFilter) null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param data             No description provided
    * @param savedIncrements  No description provided
    * @param tmp              No description provided
    */
   public final void writeClassToStringBuffer (StringBuffer data, FTreeSet savedIncrements, GXLFilter tmp)
   {
      FTreeSet reachableIncrements = new FTreeSet (FujabaComparator.getLessBasicIncr());
      // write every class and their attributes to the StringBuffer
      writeClassToStringBuffer (data, savedIncrements, reachableIncrements, tmp);

      while (reachableIncrements.size() > 0)
      {
         Iterator iter = reachableIncrements.iterator();

         while (iter.hasNext())
         {
            Object obj = iter.next();
            if ( (obj instanceof BasicIncrement) &&
                (DebugPreferences.get().isSaveGenerated() ||
               ! ((BasicIncrement) obj).isGenerated()))
            {
               BasicIncrement incr = (BasicIncrement) obj;
               reachableIncrements.remove (incr);
               incr.writeClassToStringBuffer (data, savedIncrements, reachableIncrements, tmp);
            }
         }
      }
   }

   //some usefull methods for gxl-export

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param data                 No description provided
    * @param savedIncrements      No description provided
    * @param tmp                  No description provided
    * @param reachableIncrements  No description provided
    */
   private final void writeClassToStringBuffer (StringBuffer data,
                                                FTreeSet savedIncrements,
                                                FTreeSet reachableIncrements, GXLFilter tmp)
   {
      if (savedIncrements.contains (this) || this instanceof ASGTransient)
      {
         // do not save thic increment since
         //  either the increment is already saved
         //  or it implements the transient interface and should not be saved at all
         return;
      }

      if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
      {
         log.info ("Adding: " + FD.toString (this));
      }

      // do not dump this again
      savedIncrements.add (this);

      if (!savedIncrements.contains (this))
      {
         if (FD.isOn (DebugPreferences.DEBUG_LEVEL_SAVELOAD))
         {
            log.info ("Tried to add " + this.getID()
               + " cannot get it afterwards. " + this);
         }
      }

      if (tmp == null || tmp.verifyObject (this))
      {
         //write object identification
         ClassLoader loader = this.getClass().getClassLoader();
         String loaderID = FujabaApp.getPersistencySupport().getClassLoaderKey (loader);
         data = data.append ("*;" + this.getID() + ";" + this.getClass().getName() + ";" + loaderID + "\n");

         // call the 'abstract' method to write all attributes
         this.writeAttributes (data, reachableIncrements);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private boolean checkObjectsAttribValues()
   {
      return true;
   }


   /**
    * Save float attribute.
    *
    * @param decl   No description provided
    * @param value  No description provided
    */
   public final void writeToStringBuffer (String decl, float value)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }

      data.append ("~;").append (decl).append (";").append (Float.toString (value)).append ("\n");
   }


   /**
    * Save double attribute.
    *
    * @param decl   No description provided
    * @param value  No description provided
    */
   public final void writeToStringBuffer (String decl, double value)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }

      data.append ("~;").append (decl).append (";").append (Double.toString (value)).append ("\n");
   }


   /**
    * Save int attribute.
    *
    * @param decl   No description provided
    * @param value  No description provided
    */
   public final void writeToStringBuffer (String decl, byte value)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }

      data.append ("~;").append (decl).append (";").append (Byte.toString (value)).append ("\n");
   }


   /**
    * Save int attribute.
    *
    * @param decl   No description provided
    * @param value  No description provided
    */
   public final void writeToStringBuffer (String decl, int value)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }

      data.append ("~;").append (decl).append (";").append (Integer.toString (value)).append ("\n");
   }


   /**
    * Save long attribute.
    *
    * @param decl   No description provided
    * @param value  No description provided
    */
   public final void writeToStringBuffer (String decl, long value)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }

      data.append ("~;").append (decl).append (";").append (Long.toString (value)).append ("\n");
   }


   /**
    * Save boolean attribute.
    *
    * @param decl   No description provided
    * @param value  No description provided
    */
   public final void writeToStringBuffer (String decl, boolean value)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }

      if (value)
      {
         writeToStringBuffer (decl, "true");
      }
      else
      {
         writeToStringBuffer (decl, "false");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param strg  No description provided
    * @return      No description provided
    */
   private final String replaceEscapeSequences (String strg)
   {
      for (int pos = strg.indexOf ('\r'); pos > -1; pos = strg.indexOf ('\r'))
      {
         String substr =  (pos > 0 ? strg.substring (0, pos) : "");
         substr = substr.replace ('\n', '\1').replace (';', '\2');
         data = data.append (substr + '\1');
         if (strg.length() > pos + 1 && strg.charAt (pos + 1) == '\n')
         {
            pos++;
         }
         strg = strg.substring (pos + 1);
      }
      strg = strg.replace ('\n', '\1').replace (';', '\2');

      return strg;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param strg  No description provided
    * @return      No description provided
    */
   private final String restoreEscapeSequences (String strg)
   {
      return strg.replace ('\1', '\n').replace ('\2', ';');
   }


   /**
    * Save String Attribute.
    *
    * @param decl  No description provided
    * @param strg  No description provided
    */
   public final void writeToStringBuffer (String decl, String strg)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }

      if (strg != null)
      {
         data.append ("~;").append (decl).append (";");
         strg = replaceEscapeSequences (strg); //trinet
         data.append (strg).append ("\n");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl             No description provided
    * @param incr             No description provided
    * @param setOfNeighbours  No description provided
    */
   public final void writeToStringBuffer (String decl, BasicIncrement incr,
                                          FTreeSet setOfNeighbours)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }

      if ( (incr != null) &&  (DebugPreferences.get().isSaveGenerated() || !this.isGenerated()))
      {
         data.append ("~;").append (decl).append (";").append (incr.getID()).append ("\n");
         setOfNeighbours.add (incr);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl             No description provided
    * @param incr             No description provided
    * @param text             No description provided
    * @param setOfNeighbours  No description provided
    */
   public final void writeToStringBuffer (String decl, BasicIncrement incr, String text,
                                          FTreeSet setOfNeighbours)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }

      if ( (incr != null) &&  (text != null) &&  (DebugPreferences.get().isSaveGenerated() || !this.isGenerated()))
      {
         data.append ("~;").append (decl).append (";").append (text).append (";");
         data.append (incr.getID()).append ("\n");
         setOfNeighbours.add (incr);
      }
   }


   /**
    * Write a Point to the fpr file
    *
    * @param decl             field declaration
    * @param point            point to write to file
    * @param text             appended text
    * @param setOfNeighbours  neighbors (not used)
    */
   public final void writeToStringBuffer (String decl, Point point, String text,
                                          FTreeSet setOfNeighbours)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }

      if ( (point != null) &&  (text != null) &&  (DebugPreferences.get().isSaveGenerated() || !this.isGenerated()))
      {
         data.append ("~;").append (decl).append (";").append (text).append (";");
         data.append ("point").append (point.x).append (",").append (point.y).append ("\n");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl             No description provided
    * @param value            No description provided
    * @param text             No description provided
    * @param setOfNeighbours  No description provided
    */
   public final void writeToStringBuffer (String decl, String value,
                                          String text, FTreeSet setOfNeighbours)
   {
      if (data == null)
      {
         throw new RuntimeException (
            "Error: super method was not called in a "
            + "'writeAttributes' method\n in class "
            + this.toString());
      }

      if ( (value != null) &&  (text != null) &&  (DebugPreferences.get().isSaveGenerated() || !this.isGenerated()))
      {
         data.append ("~;").append (decl).append (";").append (text).append (";");
         data.append ("string").append (replaceEscapeSequences (value)).append ("\n");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl             No description provided
    * @param value            No description provided
    * @param text             No description provided
    * @param setOfNeighbours  No description provided
    */
   public final void writeToStringBuffer (String decl, Long value,
                                          String text, FTreeSet setOfNeighbours)
   {
      if (data == null)
      {
         throw new RuntimeException (
            "Error: super method was not called in a "
            + "'writeAttributes' method\n in class "
            + this.toString());
      }

      if ( (value != null) &&  (text != null) &&  (DebugPreferences.get().isSaveGenerated() || !this.isGenerated()))
      {
         data.append ("~;").append (decl).append (";").append (text).append (";");
         data.append ("long").append (value.toString()).append ("\n");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl             No description provided
    * @param enumeration      No description provided
    * @param setOfNeighbours  No description provided
    */
   public final void writeToStringBuffer (String decl, Enumeration enumeration,
                                          FTreeSet setOfNeighbours)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a"
            + "'writeAttributes' method\n in class " + this.toString());
      }
      if (enumeration != null && enumeration.hasMoreElements())
      {
         while (enumeration.hasMoreElements())
         {
            BasicIncrement incr = (BasicIncrement) enumeration.nextElement();
            writeToStringBuffer (decl, incr, setOfNeighbours);
         }
      }
   }


   /**
    * Method handles saving of collection containers like Vector or HashSet from the Java Foundation
    * classes.
    *
    * @param decl             the name of the attribute
    * @param collection       the default value (only needed to use overload mechanism when
    *      calling the writeTo... methods)
    * @param setOfNeighbours  incr referred by this incr and attrib are entered here.
    */
   public final void writeToStringBuffer (String decl, Collection collection,
                                          FTreeSet setOfNeighbours)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class "
            + this.toString());
      }
      if (collection != null)
      {
         writeToStringBuffer (decl, collection.iterator(), setOfNeighbours);
      }
   }


   /**
    * Method handles saving of map containers like HashMap or Hashtable from the Java Foundation
    * classes.
    *
    * @param decl             the name of the attribute
    * @param map              the value which should be saved
    * @param setOfNeighbours  incr referred by this incr and attrib are entered here.
    */
   public final void writeToStringBuffer (String decl, Map map,
                                          FTreeSet setOfNeighbours)
   {
      //log.info(decl);
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class "
            + this.toString());
      }
      if (map != null)
      {
         Iterator hashKeys = map.keySet().iterator();
         if (hashKeys == null || !hashKeys.hasNext())
         {
         }
         else
         {
            while (hashKeys.hasNext())
            {
               Object incr = null;
               Object tmp;
               String key;
               tmp = hashKeys.next();

               if (tmp instanceof String)
               {
                  key = (String) tmp;
               }
               else if (tmp instanceof BasicIncrement)
               {
                  BasicIncrement basic = (BasicIncrement) tmp;
                  key = basic.getID();
                  incr = map.get (basic);
               }
               else
               {
                  //dunno
                  key = null;
               }

               if (incr == null)
               {
                  incr = map.get (key);
               }

               if (incr instanceof BasicIncrement)
               {
                  writeToStringBuffer (decl, (BasicIncrement) incr, key, setOfNeighbours);
               }
               else if (incr instanceof Point)
               {
                  writeToStringBuffer (decl, (Point) incr, key, setOfNeighbours);
               }
               else if (incr instanceof String)
               {
                  writeToStringBuffer (decl, (String) incr, key, setOfNeighbours);
               }
               else if (incr instanceof Long)
               {
                  writeToStringBuffer (decl, (Long) incr, key, setOfNeighbours);
               }
               else
               {
                  if (log.isInfoEnabled())
                  {
                     log.info (incr);
                  }
                  if (log.isInfoEnabled())
                  {
                     log.info (incr.getClass());
                  }
                  throw new RuntimeException ("Error: cannot save unsupported type "
                     + incr.getClass().getName());
               }
            }
         }
      }
   }


   /**
    * This method handles the iterator returned by several methods of the container classes
    * in the Java Foundation classes.
    *
    * @param decl             the name of the container attribute the iterator belongs to.
    * @param iter             the iterator.
    * @param setOfNeighbours  incr referred by this incr and attrib are entered here.
    */
   public final void writeToStringBuffer (String decl, Iterator iter, FTreeSet setOfNeighbours)
   {
      if (data == null)
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + "'writeAttributes' method\n in class " + this.toString());
      }
      if (iter != null)
      {
         Object incr;
         while (iter.hasNext())
         {
            incr = iter.next();

            if (incr instanceof BasicIncrement)
            {
               writeToStringBuffer (decl, (BasicIncrement) incr, setOfNeighbours);
            }
            else
            {
               writeToStringBuffer (decl, incr.getClass().getName() + ";" + incr.toString());
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl          No description provided
    * @param defaultValue  No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   public final String readFromStringTokenizer (String decl,
                                                String defaultValue,
                                                FDuplicatedTreeMap attributes)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a"
            + " 'readAttributes' method\n in class "
            + this.toString());
      }
      StringTokenizer lineTok = (StringTokenizer) attributes.get (decl);
      if (lineTok != null)
      {
         String strg = "";
         if (lineTok.hasMoreTokens())
         {
            strg += lineTok.nextToken();
         }
         return restoreEscapeSequences (strg); //trinet
      }
      return defaultValue;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl          No description provided
    * @param defaultValue  No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   public final int readFromStringTokenizer (String decl, int defaultValue,
                                             FDuplicatedTreeMap attributes)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a"
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      StringTokenizer lineTok = (StringTokenizer) attributes.get (decl);
      if (lineTok != null)
      {
         return new Integer (lineTok.nextToken()).intValue();
      }
      return defaultValue;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl          No description provided
    * @param defaultValue  No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   public final long readFromStringTokenizer (String decl, long defaultValue,
                                              FDuplicatedTreeMap attributes)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a"
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      StringTokenizer lineTok = (StringTokenizer) attributes.get (decl);
      if (lineTok != null)
      {
         return new Long (lineTok.nextToken()).longValue();
      }
      return defaultValue;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl          No description provided
    * @param defaultValue  No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   public final byte readFromStringTokenizer (String decl, byte defaultValue,
                                              FDuplicatedTreeMap attributes)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a"
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      StringTokenizer lineTok = (StringTokenizer) attributes.get (decl);
      if (lineTok != null)
      {
         return new Byte (lineTok.nextToken()).byteValue();
      }
      return defaultValue;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl          No description provided
    * @param defaultValue  No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   public final float readFromStringTokenizer (String decl, float defaultValue,
                                               FDuplicatedTreeMap attributes)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a"
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      StringTokenizer lineTok = (StringTokenizer) attributes.get (decl);
      if (lineTok != null)
      {
         return new Float (lineTok.nextToken()).floatValue();
      }
      return defaultValue;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl          No description provided
    * @param defaultValue  No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   public final short readFromStringTokenizer (String decl, short defaultValue,
                                               FDuplicatedTreeMap attributes)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a"
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      StringTokenizer lineTok = (StringTokenizer) attributes.get (decl);
      if (lineTok != null)
      {
         return new Short (lineTok.nextToken()).shortValue();
      }
      return defaultValue;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl          No description provided
    * @param defaultValue  No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   public final double readFromStringTokenizer (String decl,
                                                double defaultValue,
                                                FDuplicatedTreeMap attributes)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a"
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      StringTokenizer lineTok = (StringTokenizer) attributes.get (decl);
      if (lineTok != null)
      {
         return new Double (lineTok.nextToken()).doubleValue();
      }
      return defaultValue;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl          No description provided
    * @param defaultValue  No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   public final boolean readFromStringTokenizer (String decl,
                                                 boolean defaultValue,
                                                 FDuplicatedTreeMap attributes)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a"
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      StringTokenizer lineTok = (StringTokenizer) attributes.get (decl);
      if (lineTok != null)
      {
         if (lineTok.nextToken().compareTo ("true") == 0)
         {
            return true;
         }
         else
         {
            return false;
         }
      }
      return defaultValue;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param decl          No description provided
    * @param defaultValue  No description provided
    * @param objects       No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   public final BasicIncrement readFromStringTokenizer (String decl,
                                                        BasicIncrement defaultValue,
                                                        Hashtable objects,
                                                        FDuplicatedTreeMap attributes)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      StringTokenizer lineTok = (StringTokenizer) attributes.get (decl);
      if (lineTok != null && lineTok.hasMoreElements())
      {
         String strg = lineTok.nextToken();
         if (strg.compareTo ("null") == 0)
         {
            return null;
         }
         else
         {
            return (BasicIncrement) objects.get (strg);
         }
      }
      return defaultValue;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param defaultValue  No description provided
    * @param lineTok       No description provided
    * @param objects       No description provided
    * @return              No description provided
    */
   public final BasicIncrement readFromStringTokenizer (FLinkedList defaultValue,
                                                        StringTokenizer lineTok,
                                                        Hashtable objects)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      if (lineTok != null)
      {
         String strg = lineTok.nextToken();
         if (strg.compareTo ("null") == 0)
         {
            return null;
         }
         else
         {
            return (BasicIncrement) objects.get (strg);
         }
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param defaultValue  No description provided
    * @param lineTok       No description provided
    * @param objects       No description provided
    * @return              No description provided
    */
   public final BasicIncrement readFromStringTokenizer (FTreeSet defaultValue,
                                                        StringTokenizer lineTok,
                                                        Hashtable objects)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      if (lineTok != null)
      {
         String strg = "null";
         try
         {
            if (lineTok.hasMoreElements())
            {
               strg = lineTok.nextToken();
            }
         }
         catch (NoSuchElementException e)
         {
            // Don't know why this exception is sometimes thrown, but it seems
            // that it can be ignored.
            e.printStackTrace();
            return null;
         }
         if (strg.equals ("null"))
         {
            return null;
         }
         //---- if a map changed to a set, the second value is the object-value
         if (lineTok.hasMoreElements())
         {
            strg = lineTok.nextToken();
         }
         return (BasicIncrement) objects.get (strg);
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param defaultValue  No description provided
    * @param lineTok       No description provided
    * @param objects       No description provided
    * @param attributes    No description provided
    * @return              No description provided
    */
   public final Object readFromStringTokenizer (Collection defaultValue,
                                                StringTokenizer lineTok,
                                                Hashtable objects,
                                                FDuplicatedTreeMap attributes)
   {
      if (lineTok != null)
      {
         String strg = "null";
         String value;

         try
         {
            if (lineTok.hasMoreElements())
            {
               strg = lineTok.nextToken();
            }
         }
         catch (NoSuchElementException e)
         {
            // Don't know why this exception is sometimes thrown, but it seems
            // that it can be ignored.
            e.printStackTrace();
            return null;
         }

         if (strg.equals ("null"))
         {
            return null;
         }

         // if a map changed to a set, the second value is the object-value
         if (lineTok.hasMoreElements())
         {
            strg = lineTok.nextToken();
         }

         if (strg.startsWith ("java.lang.String"))
         {
            value = strg.substring (17);

            return value;
         }

         else
         {
            if (strg.startsWith ("java.awt.geom.Point2D$Double"))
            {
               int first = strg.indexOf ("[");
               int last = strg.lastIndexOf ("]");
               Double xPoint = new Double (strg.substring (first + 1, strg.indexOf (",")));
               Double yPoint = new Double (strg.substring (strg.indexOf (",") + 2, last));

               return new Point2D.Double (xPoint.doubleValue(), yPoint.doubleValue());
            }
            else
            {
               if (strg.startsWith ("java.util.Vector[Point2D.Double["))
               {
                  Vector vec = new Vector();
                  int pos = strg.indexOf ("e[");
                  while ( (pos = strg.indexOf ("[", pos)) != -1)
                  {
                     pos++;
                     Double xPoint = new Double (strg.substring (pos, strg.indexOf (",", pos)));
                     Double yPoint = new Double (strg.substring (strg.indexOf (",", pos) + 2, strg.indexOf ("]", pos)));
                     vec.add (new Point2D.Double (xPoint.doubleValue(), yPoint.doubleValue()));
                  } // while

                  return vec;
               }
               else
               {
                  return (BasicIncrement) objects.get (strg);
               }
            }
         }
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param defaultValue  No description provided
    * @param lineTok       No description provided
    * @param objects       No description provided
    * @return              No description provided
    */
   public final KeyValuePair readFromStringTokenizer (Map defaultValue,
                                                      StringTokenizer lineTok,
                                                      Hashtable objects)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      if (lineTok != null)
      {
         String key = "null";
         String value = "null";

         if (lineTok.hasMoreTokens())
         {
            key = lineTok.nextToken();
         }
         else
         {
            log.info ("key is missing. Set to null ");
         }

         if (lineTok.hasMoreTokens())
         {
            value = lineTok.nextToken();
         }
         else
         {
            log.info ("value is missing for key: " + key);
         }

         if (key.equals ("null") || value.equals ("null"))
         {
            return null;
         }
         if ((BasicIncrement) objects.get (key) != null)
         {
            return new KeyValuePair ((BasicIncrement) objects.get (key), (BasicIncrement) objects.get (value));
         }
         else
         {
            if (objects.get (value) != null)
            {
               return new KeyValuePair (key, objects.get (value));
            }
            else if (value.startsWith ("point"))
            {
               int commaIndex = value.indexOf (',');
               if (commaIndex >= 0)
               {
                  String xString = value.substring (5, commaIndex);
                  String yString = value.substring (commaIndex + 1, value.length());
                  Point point = new Point (Integer.parseInt (xString), Integer.parseInt (yString));
                  return new KeyValuePair (key, point);
               }
            }
            else if (value.startsWith ("string"))
            {
               String string = value.substring (6, value.length());
               return new KeyValuePair (key, restoreEscapeSequences (string));
            }
            return new KeyValuePair (key, null);
         }
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param defaultValue  No description provided
    * @param lineTok       No description provided
    * @param objects       No description provided
    * @return              No description provided
    */
   public final KeyValuePair readFromStringTokenizer (FTreeMap defaultValue,
                                                      StringTokenizer lineTok,
                                                      Hashtable objects)
   {
      if (!checkObjectsAttribValues())
      {
         throw new RuntimeException ("Error: super method was not called in a "
            + " 'readAttribute' method\n in class "
            + this.toString());
      }
      if (lineTok != null)
      {
         String key = lineTok.nextToken();
         String value = "null";
         if (lineTok.hasMoreTokens())
         {
            value = lineTok.nextToken();
         }
         else
         {
            log.info ("value is missing for key: " + key);
         }

         if (key.equals ("null") || value.equals ("null"))
         {
            return null;
         }
         return new KeyValuePair (key, objects.get (value));
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      if (FD.isOn (DebugPreferences.DEBUG_LEVEL_REMOVEYOU))
      {
         FD.setRemoveYouPrinted (false);
         log.info (this.getClass() + "[" + getID() + "].removeYou() ");
      }

      if (FujabaDebug.SETATTRSTONULL)
      {
         this.data = null;
      }

   }
}

/*
 * $Log: BasicIncrement.java,v $
 * Revision 1.190.2.6  2005/12/15 16:22:38  lowende
 * Removed a few compile warnings.
 *
 */
