/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.asg;

import java.util.Iterator;
import java.util.Set;

import de.uni_paderborn.fujaba.metamodel.FDiagram;
import de.uni_paderborn.fujaba.metamodel.FElement;
import de.upb.tools.fca.*;


/**
 * A project stores a lot of different diagrams.
 *
 * <h2>Associations</h2>
 *
 * <pre>
 *             0..1                          n
 * ASGProject --------------------------------- ASGDiagram
 *             project                 diags
 *
 *             0..1   hasCurrentDiagram   0..1
 * ASGProject --------------------------------- ASGDiagram
 *             currentProject   currentDiagram
 * </pre>
 *
 * @author    Robert Wagner
 * @version   1.0
 */
public abstract class ASGProject extends ASGElement
{
   /**
    * the key to indicate changes in the diags role of the project.
    */
   public final static String DIAGS_PROPERTY = "diags";


   /**
    * returns the ASGElement specified by the given search key.
    *
    * @param id  the search key
    * @return    the ASGElement
    */
   public ASGElement searchID (String id)
   {
      ASGElement item = super.searchID (id);
      Iterator iter = iteratorOfDiags();
      while ( (item == null) &&  (iter.hasNext()))
      {
         item =  ((ASGElement) iter.next()).searchID (id);
      }
      return item;
   }


   /**
    * internal used collection to store all diagrams of this project. <pre>
    *             0..1                          n
    * ASGProject --------------------------------- ASGDiagram
    *             project                 diags
    * </pre>
    */
   private Set diags;


   /**
    * returns true if the given diagram is found in the set of diagrams of this project.
    *
    * @param diagram  the given diagram
    * @return         true if the diagram was found
    */
   public boolean hasInDiags (ASGDiagram diagram)
   {
      return  (diags == null ?
         false :
         diags.contains (diagram));
   }


   /**
    * returns an iterator over all diags of this project.
    *
    * @return   Iterator the iterator.
    */
   public Iterator iteratorOfDiags()
   {
      return  (diags == null ?
         FEmptyIterator.get() :
         diags.iterator());
   }


   /**
    * Adds a diagram to the project.
    *
    * @param diagram  the diagram
    * @return         No description provided
    */
   public boolean addToDiags (FDiagram diagram)
   {
      boolean changed = false;
      if (diagram != null &&
         !hasInDiags ((ASGDiagram) diagram))
      {
         if (diags == null)
         {
            diags = new FPropHashSet (this, DIAGS_PROPERTY);
         }

         diags.add (diagram);
          ((ASGDiagram) diagram).setProject (this);
         changed = true;
      }
      return changed;
   }


   /**
    * remove the given diagram from the project.
    *
    * @param diagram  the diagram
    */
   public void removeFromDiags (ASGDiagram diagram)
   {
      if (diagram != null &&
         hasInDiags (diagram))
      {
         diags.remove (diagram);
         diagram.setProject (null);
      }
   }


   /**
    * removes all diagrams of the project.
    */
   public void removeAllFromDiags()
   {
      ASGDiagram diagram = null;
      Iterator iter = iteratorOfDiags();

      while (iter.hasNext())
      {
         diagram = (ASGDiagram) iter.next();
         removeFromDiags (diagram);
      }
   }


   /**
    * <pre>
    *             0..1   hasCurrentDiagram   0..1
    * ASGDiagram --------------------------------- UMLProject
    *             currentDiagram       umlProject
    * </pre>
    */
   private transient ASGDiagram currentDiagram;


   /**
    * @param value  The new currentDiagram value
    * @return       Return Value description
    */
   public boolean setCurrentDiagram (ASGDiagram value)
   {
      boolean changed = false;
      if (this.currentDiagram != value)
      {
         if (this.currentDiagram != null)
         {
            ASGDiagram oldValue = this.currentDiagram;
            this.currentDiagram = null;
            oldValue.setCurrentProject (null);
         }
         this.currentDiagram = value;
         if (value != null)
         {
            value.setCurrentProject (this);
         }
         changed = true;
      }

      return changed;
   }


   /**
    * @return   The currentDiagram value
    */
   public ASGDiagram getCurrentDiagram()
   {
      return this.currentDiagram;
   }


   /**
    * method to help the garbage collector.
    */
   public void removeYou()
   {
      removeAllFromDiags();
      setCurrentDiagram (null);
      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      return null;
   }
}

/*
 * $Log: ASGProject.java,v $
 * Revision 1.14  2005/03/04 14:43:27  cschneid
 * changed ASGProject.diags to FHashSet, NAC fix, new libs
 *
 */
