/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.asg;

import java.util.*;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.metamodel.FDiagram;
import de.uni_paderborn.fujaba.metamodel.FElement;
import de.uni_paderborn.fujaba.packagediagrams.DiagramUsage;
import de.upb.tools.fca.*;
import de.upb.tools.pcs.CollectionChangeEvent;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *             0..1   hasCurrentDiagram   0..1
 * ASGProject --------------------------------- ASGDiagram
 *             currentProject   currentDiagram
 * </pre>
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.33.2.4 $ $Date: 2005/12/14 16:58:05 $
 */
public abstract class ASGDiagram extends ASGElement implements FDiagram
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (ASGDiagram.class);

   /**
    * property change event name for property attribute
    */
   public final static String PROJECT_PROPERTY = "project";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String ELEMENTS_PROPERTY_KEY = "elements";


   /**
    * Constructor for class ASGDiagram
    */
   public ASGDiagram()
   {
      super();
   }


   /**
    * Constructor for class ASGDiagram
    *
    * @param coobraPersistent  No description provided
    */
   protected ASGDiagram (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * Searches the ASG tree for a given id
    *
    * @param id  The id to search for.
    * @return    The Element with the given id, null if not found.
    */
   public ASGElement searchID (String id)
   {
      ASGElement item = super.searchID (id);
      Iterator iter = iteratorOfElements();
      while ( (item == null) &&  (iter.hasNext()))
      {
         ASGElement asgElement = (ASGElement) iter.next();
         if (asgElement == this ||  (asgElement instanceof ASGDiagram) && hasInDiagrams ((ASGDiagram) asgElement))
         {
            //whoops
            log.error ("Cycle in diagram detected!");
            if (asgElement == this)
            {
               removeFromElements (asgElement);
            }
         }
         else
         {
            item = asgElement.searchID (id);
         }
      }
      return item;
   }


   /**
    * List of all contained diagram items and its manipulating methods The FHashSet returns
    * the containted items in order of adding.
    */
   private FPropHashSet elements = null;


   /**
    * @return   number of elements contained in this diagram
    */
   public int sizeOfElements()
   {
      return  ( (this.elements == null)
         ? 0
         : this.elements.size());
   }


   /**
    * @param element  ASGElement of interest
    * @return         true when element is in elements attribute
    */
   public boolean hasInElements (FElement element)
   {
      return  (this.elements == null
         ? false
         : elements.contains (element));
   }


   /**
    * @return   iterator through elements (only ASGElements)
    */
   public Iterator iteratorOfElements()
   {
      return  (this.elements == null
         ? FEmptyIterator.get()
         : this.elements.iterator());
   }


   /**
    * add an ASGEelement to the elements attribute
    *
    * @param element  ASGElement to be added
    * @return         true when element has been newly added
    */
   public boolean addToElements (FElement element)
   {
      boolean changed = false;

      if (element != null && !hasInElements (element))
      {
         if (this.elements == null)
         {
            this.elements = new FPropHashSet (this, ELEMENTS_PROPERTY_KEY);
         }
         changed = this.elements.add (element);

         //todo: this property change will be removed
         if (!ELEMENTS_PROPERTY_KEY.equals (getElementKey()))
         {
            firePropertyChange (CollectionChangeEvent.get (this, getElementKey(), this.elements, null, element, CollectionChangeEvent.ADDED));
         }

         element.addToDiagrams (this);
      }

      return changed;
   }


   /**
    * @param entry  The object added.
    */
   public void addToElements (Map.Entry entry)
   {
      ASGElement element = (ASGElement) entry.getValue();
      addToElements (element);
   }


   /**
    * remove an ASGElement from the elements attribute
    *
    * @param element  what to remove
    * @return         true when element was removed (had in elements)
    */
   public boolean removeFromElements (FElement element)
   {
      boolean changed = false;

      if (element != null && hasInElements (element))
      {
         elements.remove (element);

         //todo: this property change event will be removed
         firePropertyChange (CollectionChangeEvent.get (this, getElementKey(), this.elements, element, null, CollectionChangeEvent.REMOVED));

         element.removeFromDiagrams (this);
         changed = true;
      }

      return changed;
   }


   /**
    * clear elements attribute
    */
   public void removeAllFromElements()
   {
      Iterator iter = iteratorOfElements();
      while (iter.hasNext())
      {
         removeFromElements ((ASGElement) iter.next());
      }
   }


   /**
    * the project role of the diagram.
    */
   private ASGProject project;


   /**
    * Get the project attribute of the UMLDiagram object
    *
    * @return   The project value
    */
   public ASGProject getProject()
   {
      return project;
   }


   /**
    * Sets the project attribute of the UMLDiagram object
    *
    * @param project  The new project value
    */
   public void setProject (ASGProject project)
   {
      if (this.project != project)
      {
         ASGProject oldValue = this.project;
         if (this.project != null)
         {
            this.project = null;
            oldValue.removeFromDiags (this);
         }
         this.project = project;
         if (project != null)
         {
            project.addToDiags (this);
         }
         firePropertyChange (PROJECT_PROPERTY, oldValue, project);
      }
   }


   /**
    * <pre>
    *             0..1   hasCurrentDiagram   0..1
    * ASGProject --------------------------------- ASGDiagram
    *             currentProject   currentDiagram
    * </pre>
    */
   private transient ASGProject currentProject;


   /**
    * @param value  The new currentProject value
    * @return       true when project was set
    * @see          #currentProject
    */
   public boolean setCurrentProject (ASGProject value)
   {
      boolean changed = false;
      if (this.currentProject != value)
      {
         if (this.currentProject != null)
         {
            ASGProject oldValue = this.currentProject;
            this.currentProject = null;
            oldValue.setCurrentDiagram (null);
         }
         this.currentProject = value;
         if (value != null)
         {
            value.setCurrentDiagram (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * @return   The currentProject value
    * @see      #currentProject
    */
   public ASGProject getCurrentProject()
   {
      return this.currentProject;
   }


   /**
    * remove all references to other objects
    */
   public void removeYou()
   {
      removeAllFromElements();
      setProject (null);
      setCurrentProject (null);
      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      return getProject();
   }


   /**
    * Get the elementKey for the elements attribute
    *
    * @return       "items"
    * @deprecated   the virtual items attribute will be removed - use elements instead
    */
   protected String getElementKey()
   {
      return "items";
   }


   /**
    * <pre>
    *             0..1    utility    1
    * ASGUtility ---------------------- ASGDiagram
    *             utility      diagram
    * </pre>
    */
   private ASGUtility utility;


   /**
    * Sets the utility attribute of the ASGDiagram object
    *
    * @param value  The new utility value
    * @return       true when utility was changed
    */
   public boolean setUtility (ASGUtility value)
   {
      boolean changed = false;
      if (this.utility != value)
      {
         if (this.utility != null)
         {
            ASGUtility oldValue = this.utility;
            this.utility = null;
            oldValue.setDiagram (null);
         }
         this.utility = value;
         if (value != null)
         {
            value.setDiagram (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * <pre>
    *               n    usages    0..1
    * DiagramUsage --------------------- UMLDiagram
    *               usages      diagram
    * </pre>
    */
   private Set usages;


   /**
    * Accessor for association usages (for package diagrams)
    *
    * @param value
    * @return       No description provided
    */
   public boolean hasInUsages (DiagramUsage value)
   {
      return  ( (this.usages != null) &&
          (value != null) &&
         this.usages.contains (value));
   }


   /**
    * Accessor for association usages (for package diagrams)
    *
    * @return   No description provided
    */
   public Iterator iteratorOfUsages()
   {
      return  ( (this.usages == null)
         ? FEmptyIterator.get()
         : this.usages.iterator());
   }


   /**
    * Accessor for association usages (for package diagrams)
    *
    * @return   No description provided
    */
   public int sizeOfUsages()
   {
      return  ( (this.usages == null)
         ? 0
         : this.usages.size());
   }


   /**
    * Accessor for association usages (for package diagrams)
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToUsages (DiagramUsage value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.usages == null)
         {
            this.usages = new FHashSet();
         }
         changed = this.usages.add (value);
         if (changed)
         {
            value.setDiagram (this);
         }
      }
      return changed;
   }


   /**
    * Accessor for association usages (for package diagrams)
    *
    * @param value
    * @return       No description provided
    */
   public boolean removeFromUsages (DiagramUsage value)
   {
      boolean changed = false;
      if ( (this.usages != null) &&  (value != null))
      {
         changed = this.usages.remove (value);
         if (changed)
         {
            value.setDiagram (null);
         }
      }
      return changed;
   }


   /**
    * Accessor for association usages (for package diagrams)
    */
   public void removeAllFromUsages()
   {
      DiagramUsage tmpValue;
      Iterator iter = this.iteratorOfUsages();
      while (iter.hasNext())
      {
         tmpValue = (DiagramUsage) iter.next();
         this.removeFromUsages (tmpValue);
      }
   }


   /**
    * Get the utility attribute of the ASGDiagram object
    *
    * @return   The utility value
    */
   public ASGUtility getUtility()
   {
      return this.utility;
   }


   /**
    * Deletes the diagram from the project. This method is called by <b>Delete is not implemented
    * for ASGDiagram, sub classes may override it to enable diagram deletion!</b>
    *
    * @see   de.uni_paderborn.fujaba.app.action.DeleteDiagramAction
    */
   public void delete()
   {
      // This method is not implemented for ASGDiagram, sub classes have to override it to enable diagram deletion!
   }
}

/*
 * $Log: ASGDiagram.java,v $
 * Revision 1.33.2.4  2005/12/14 16:58:05  lowende
 * Javadoc corrected
 *
 */
