/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app;

import java.io.IOException;
import java.net.URL;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;
import org.apache.log4j.Logger;
import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import de.uni_paderborn.lib.classloader.UPBClassLoader;


/**
 * @author    $Author: mksoft $
 * @version   $Revision: 1.23.2.2 $
 */
public class Version
{
   /**
    * log4j logging
    */
   final static transient Logger log = Logger.getLogger (Version.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int major = 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int minor = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int revision = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int build = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String release = "";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static volatile Version instance;


   /**
    * Constructor for class Version
    */
   private Version()
   {
      try
      {
         // parse the Version.xml document and set the major, minor and build number and release string
         VersionParser parser = new VersionParser();
         parser.parse ("version.xml");
      }
      catch (Exception ex)
      {
         log.error ("Could not parse version.xml.");
         ex.printStackTrace();
      }

   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public static Version get()
   {
      if (instance == null)
      {
         instance = new Version();
      }
      return instance;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      return getVersion() + " " + release + "  Build " + build;
   }


   /**
    * Get the version attribute of the Version object
    *
    * @return   The version value
    */
   public String getVersion()
   {
      return major + "." + minor + "." + revision;
   }


   /**
    * @return   the build version number
    */
   public int getBuild()
   {
      return build;
   }


   /**
    * Sets the build attribute of the Version object
    *
    * @param build  The new build value
    */
   void setBuild (int build)
   {
      this.build = build;
   }


   /**
    * @return   the major version number
    */
   public int getMajor()
   {
      return major;
   }


   /**
    * Sets the major attribute of the Version object
    *
    * @param major  The new major value
    */
   void setMajor (int major)
   {
      this.major = major;
   }


   /**
    * @return   the minor version number
    */
   public int getMinor()
   {
      return minor;
   }


   /**
    * Sets the minor attribute of the Version object
    *
    * @param minor  The new minor value
    */
   void setMinor (int minor)
   {
      this.minor = minor;
   }


   /**
    * Get the revision attribute of the Version object
    *
    * @return   The revision value
    */
   public int getRevision()
   {
      return revision;
   }


   /**
    * Sets the revision attribute of the Version object
    *
    * @param revision  The new revision value
    */
   void setRevision (int revision)
   {
      this.revision = revision;
   }


   /**
    * @return   the release string
    */
   public String getRelease()
   {
      return release;
   }


   /**
    * Sets the release attribute of the Version object
    *
    * @param release  The new release value
    */
   void setRelease (String release)
   {
      this.release = release;
   }


   /**
    * Class to parse the Version.xml document.
    *
    * @author    $Author: mksoft $
    * @version   $Revision: 1.23.2.2 $
    */
   class VersionParser extends DefaultHandler
   {
      /**
       * URL used in Plugin definitions (XML-files) to identify the DTD. DTD given as "SYSTEM"
       * (non-public) definition.
       */
      private final static String VERSION_SYSTEM_ID = "http://www.upb.de/cs/fujaba/DTDs/Version.dtd";

      /**
       * Alternative URL used in Plugin definitions (XML-files) to identify the DTD. DTD given
       * as "SYSTEM" (non-public) definition.
       */
      private final static String VERSION_SYSTEM_ID2 = "http://www.fujaba.de/DTDs/Version.dtd";

      /**
       * Name used in Plugin definitions (XML-files) to identify the DTD. DTD given as "PUBLIC"
       * definition.
       */
      private final static String VERSION_PUBLIC_ID = "-//Fujaba//Fujaba Plugin Definition//EN//1.0";

      /**
       */
      private final static String VERSION_DTD_RESOURCE = "DTDs/Version.dtd";

      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public final static String VERSION = "Version";
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public final static String MAJOR = "Major";
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public final static String MINOR = "Minor";
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public final static String REVISION = "Revision";
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public final static String BUILD = "Build";
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public final static String RELEASE = "Release";

      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private final static int VERSION_STATE = 0;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private final static int MAJOR_STATE = 1;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private final static int MINOR_STATE = 2;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private final static int REVISION_STATE = 3;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private final static int BUILD_STATE = 4;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private final static int RELEASE_STATE = 5;

      /**
       */
      private URL versionDTD;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private int state = -1;


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param xmlFile  No description provided
       */
      public void parse (String xmlFile)
      {
         try
         {
            SAXParserFactory factory = SAXParserFactory.newInstance();
            factory.setValidating (true);
            factory.setNamespaceAware (true);

            XMLReader xmlReader = factory.newSAXParser().getXMLReader();
            xmlReader.setContentHandler (this);
            xmlReader.setErrorHandler (this);
            xmlReader.setEntityResolver (this);

            URL url = UPBClassLoader.get().getResource (xmlFile);
            xmlReader.parse (new InputSource (url.toString()));
         }
         catch (SAXParseException e)
         {
            log.error ("Parse error in line " + e.getLineNumber() + "\n" + e.getMessage());
            e.printStackTrace();
         }
         catch (SAXException e)
         {
            log.error ("SAXException:\n" + e.getMessage());
            e.printStackTrace();
         }
         catch (ParserConfigurationException e)
         {
            log.error ("ParserConfigurationException:\n" + e.getMessage());
            e.printStackTrace();
         }
         catch (IOException e)
         {
            log.error ("File " + xmlFile + " can not be opened:\n" + e.getMessage());
            e.printStackTrace();
         }

      }


      /**
       * Method implements interface EntityResolver.
       *
       * @param publicId       No description provided
       * @param systemId       No description provided
       * @return               No description provided
       * @throws SAXException  Exception description not provided
       */
      public InputSource resolveEntity (String publicId, String systemId) throws SAXException
      {
         if (VERSION_PUBLIC_ID.equals (publicId) ||
            VERSION_SYSTEM_ID.equals (systemId) ||
            VERSION_SYSTEM_ID2.equals (systemId))
         {
            if (versionDTD == null)
            {
               versionDTD = UPBClassLoader.get().getResource (VERSION_DTD_RESOURCE);
            }
            if (versionDTD != null)
            {
               try
               {
                  return new InputSource (versionDTD.openStream());
               }
               catch (Exception e)
               {
                  if (log.isInfoEnabled())
                  {
                     log.info ("Could not solve SYSTEM or PUBLIC reference for DTD.");
                  }
                  throw new SAXException (e);
               }
            }
         }
         InputSource result = null;
         try
         {
            result = super.resolveEntity (publicId, systemId);
         }
         catch (Exception ex)
         {
         }
         return result;
      } // resolveEntity


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param uri    No description provided
       * @param name   No description provided
       * @param qName  No description provided
       * @param atts   No description provided
       */
      public void startElement (String uri, String name, String qName, Attributes atts)
      {
         try
         {

            if (qName.equals (VERSION))
            {
               state = VERSION_STATE;
            }
            else if (qName.equals (MAJOR))
            {
               state = MAJOR_STATE;
            }
            else if (qName.equals (MINOR))
            {
               state = MINOR_STATE;
            }
            else if (qName.equals (REVISION))
            {
               state = REVISION_STATE;
            }
            else if (qName.equals (BUILD))
            {
               state = BUILD_STATE;
            }
            else if (qName.equals (RELEASE))
            {
               state = RELEASE_STATE;
            }
         }
         catch (Exception ex)
         {
            ex.printStackTrace();
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param ch      No description provided
       * @param start   No description provided
       * @param length  No description provided
       */
      public void characters (char[] ch, int start, int length)
      {
         try
         {
            String data = new String (ch, start, length);

            switch (state)
            {
               case VERSION_STATE:
                  //do nothing
                  break;
               case MAJOR_STATE:
                  setMajor (new Integer (data).intValue());
                  break;
               case MINOR_STATE:
                  setMinor (new Integer (data).intValue());
                  break;
               case REVISION_STATE:
                  setRevision (new Integer (data).intValue());
                  break;
               case BUILD_STATE:
                  setBuild (new Integer (data).intValue());
                  break;
               case RELEASE_STATE:
                  setRelease (data);
                  break;
               default:
                  break;
            }
         }
         catch (Exception ex)
         {
            ex.printStackTrace();
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param uri    No description provided
       * @param name   No description provided
       * @param qName  No description provided
       */
      public void endElement (String uri, String name, String qName)
      {
         state = -1;
      }

   }

}

/*
 * $Log: Version.java,v $
 * Revision 1.23.2.2  2005/09/30 18:39:54  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
