/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app;

import java.awt.*;
import java.awt.event.*;
import java.util.Enumeration;
import java.util.EventObject;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;

import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.fsa.SelectionManager;
import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.uni_paderborn.fujaba.gui.FPopupMenu;
import de.uni_paderborn.fujaba.metamodel.FClassDiagram;
import de.uni_paderborn.fujaba.packagediagrams.PackageDiagram;
import de.uni_paderborn.fujaba.uml.UMLProject;
import de.upb.lib.userinterface.UserInterfaceManager;


/**
 * @author    $Author: cschneid $
 * @version   $Revision: 1.69.2.3 $
 */
public class TabProxy
{
   // --------------------------------------------------------------------------
   // CONSTRUCTORS
   // --------------------------------------------------------------------------

   /**
    * Constructor for class TabProxy
    *
    * @param tabTitle   No description provided
    * @param rootTitle  No description provided
    * @param tip        No description provided
    * @param icon       No description provided
    */
   public TabProxy (String tabTitle, String rootTitle, String tip, Icon icon)
   {
      setTitle (tabTitle);
      setRoot (new DefaultMutableTreeNode (rootTitle));
      setIcon (icon);
      setTip (tip);
   }


   // --------------------------------------------------------------------------
   // MANIPULATORS
   // --------------------------------------------------------------------------

   /**
    * Tree representation.
    */
   private DefaultMutableTreeNode root;


   /**
    * Sets the root attribute of the TabProxy object
    *
    * @param root  The new root value
    */
   public void setRoot (DefaultMutableTreeNode root)
   {
      if (this.root != root)
      {
         this.root = root;
      }
   }


   /**
    * Get the root attribute of the TabProxy object
    *
    * @return   The root value
    */
   public DefaultMutableTreeNode getRoot()
   {
      return this.root;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JTree tree;


   /**
    * Sets the tree attribute of the TabProxy object
    *
    * @param tree  The new tree value
    */
   public void setTree (JTree tree)
   {
      if (this.tree != tree)
      {
         this.tree = tree;
      }
   }


   /**
    * Return the tree component
    *
    * @return   The tree value
    */
   public JTree getTree()
   {
      return this.tree;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Component component;


   /**
    * UMLMethod: 'Read access method for attribute component : Void'
    *
    * @return   The component value
    */
   public Component getComponent()
   {
      return this.component;
   }


   /**
    * UMLMethod: 'Write access method for attribute component : Void'
    *
    * @param component  The new component value
    * @return           No description provided
    */
   public Component setComponent (Component component)
   {
      if (this.component != component)
      {
         this.component = component;
      } // if

      return this.component;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Icon icon;


   /**
    * UMLMethod: 'Read access method for attribute icon : Void'
    *
    * @return   The icon value
    */
   public Icon getIcon()
   {
      return icon;
   }


   /**
    * UMLMethod: 'Write access method for attribute icon : Void'
    *
    * @param icon  The new icon value
    * @return      No description provided
    */
   public Icon setIcon (Icon icon)
   {
      if (this.icon != icon)
      {
         this.icon = icon;
      } // if

      return this.icon;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String tip = "";


   /**
    * UMLMethod: 'Read access method for attribute tip : String'
    *
    * @return   The tip value
    */
   public String getTip()
   {
      return tip;
   }


   /**
    * UMLMethod: 'Write access method for attribute tip : String'
    *
    * @param tip  The new tip value
    * @return     No description provided
    */
   public String setTip (String tip)
   {
      if ( (this.tip == null) ||  (this.tip != null && !this.tip.equals (tip)))
      {
         this.tip = tip;
      } // if

      return this.tip;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String title = "";


   /**
    * UMLMethod: 'Read access method for attribute title : String'
    *
    * @return   The title value
    */
   public String getTitle()
   {
      return title;
   }


   /**
    * UMLMethod: 'Write access method for attribute title : String'
    *
    * @param title  The new title value
    * @return       No description provided
    */
   public String setTitle (String title)
   {
      if ( (this.title == null) ||  (this.title != null && !this.title.equals (title)))
      {
         this.title = title;
      } // if

      return this.title;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private TreeCellRenderer treeCellRenderer;


   /**
    * Get the treeCellRenderer attribute of the TabProxy object
    *
    * @return   The treeCellRenderer value
    */
   public TreeCellRenderer getTreeCellRenderer()
   {
      return this.treeCellRenderer;
   }


   /**
    * Sets the treeCellRenderer attribute of the TabProxy object
    *
    * @param treeCellRenderer  The new treeCellRenderer value
    */
   public void setTreeCellRenderer (TreeCellRenderer treeCellRenderer)
   {
      this.treeCellRenderer = treeCellRenderer;
   }

   // --------------------------------------------------------------------------
   // COMFORT
   // --------------------------------------------------------------------------

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void init()
   {
      setupTreeComponent();
      setComponent (setupScrollTreeComponent (getTree()));
   }


   /**
    * Access method for an one to n association.
    *
    * @param node  The object added.
    */
   public void addTreeNode (DefaultMutableTreeNode node)
   {
      this.root.add (node);
   }


   /**
    * Return true if this tab contains no entries except the root node.
    *
    * @return   The tabEmpty value
    */
   public boolean isTabEmpty()
   {
      return this.root.getChildCount() == 0 ? true : false;
   }


   /**
    * Setup tree component
    *
    * @return   No description provided
    */
   protected JTree setupTreeComponent()
   {
      final JTree tree = new JTree (new DefaultTreeModel (getRoot()));
      this.tree = tree;

      tree.setCellRenderer (this.treeCellRenderer);
      tree.setShowsRootHandles (false);
      tree.getSelectionModel().setSelectionMode (TreeSelectionModel.SINGLE_TREE_SELECTION);
      tree.putClientProperty ("JTree.lineStyle", "None");
      tree.setRootVisible (false);
      tree.setEditable (true);
      tree.setCellEditor (new FujabaTreeCellEditor (tree));

      ToolTipManager.sharedInstance().registerComponent (tree);

      TreeListener listener = new TreeListener();
      tree.addTreeSelectionListener (listener);

      MouseListener popupListener = new PopupListener();
      tree.addMouseListener (popupListener);

      return tree;
   } // setupTreeComponent


   /**
    * Add a scroll component to the tree.
    *
    * @param tree  No description provided
    * @return      No description provided
    */
   protected JScrollPane setupScrollTreeComponent (JTree tree)
   {
      JScrollPane scrollPane = new JScrollPane (tree);
      return scrollPane;
   }


   /**
    * Find obj in tree and return the tree path.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public TreePath findTreePath (Object obj)
   {
      TreePath path = null;
      TreeModel model = this.tree.getModel();

      try
      {
         if (obj != null)
         {
            Enumeration enumeration =  ((DefaultMutableTreeNode) model.getRoot()).breadthFirstEnumeration();
            while ( (path == null) &&  (enumeration.hasMoreElements()))
            {
               DefaultMutableTreeNode tmpNode = (DefaultMutableTreeNode) enumeration.nextElement();
               if (obj.equals (tmpNode.getUserObject()))
               {
                  path = new TreePath (tmpNode.getPath());
               }
            }
         }
      }
      catch (Exception e)
      {
      }

      return path;
   }


   // --------------------------------------------------------------------------
   // ASSOCS
   // --------------------------------------------------------------------------

   /**
    * <pre>
    *                 --------- 0..1      tabProxy       0..1
    * TabbedPaneProxy | title |------------------------------- TabProxy
    *                 --------- tabbedPaneProxy      tabProxy
    * </pre>
    */
   private TabbedPaneProxy tabbedPaneProxy;


   /**
    * Sets the tabbedPaneProxy attribute of the TabProxy object
    *
    * @param value  The new tabbedPaneProxy value
    * @return       No description provided
    */
   public boolean setTabbedPaneProxy (TabbedPaneProxy value)
   {
      boolean changed = false;
      if (this.tabbedPaneProxy != value)
      {
         if (this.tabbedPaneProxy != null)
         {
            TabbedPaneProxy oldValue = this.tabbedPaneProxy;
            this.tabbedPaneProxy = null;
            oldValue.removeFromTabProxy (this);
         }
         this.tabbedPaneProxy = value;
         if (value != null)
         {
            value.addToTabProxy (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * Get the tabbedPaneProxy attribute of the TabProxy object
    *
    * @return   The tabbedPaneProxy value
    */
   public TabbedPaneProxy getTabbedPaneProxy()
   {
      return this.tabbedPaneProxy;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      TabbedPaneProxy tmpTabbedPaneProxy = getTabbedPaneProxy();
      if (tmpTabbedPaneProxy != null)
      {
         setTabbedPaneProxy (null);
      }
   }


   // --------------------------------------------------------------------------
   // INNER CLASSES
   // --------------------------------------------------------------------------

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.69.2.3 $
    */
   private class TreeListener implements TreeSelectionListener
   {
      /**
       * Constructor for class TreeListener
       */
      TreeListener()
      {
         super();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void valueChanged (TreeSelectionEvent e)
      {
         if (getTabbedPaneProxy() == null)
         {
            return;
         }

         FrameMain frame = FrameMain.get();
         frame.setCursorWait();

         try
         {
            Object obj = e.getPath().getLastPathComponent();

            if (obj instanceof DefaultMutableTreeNode)
            {
               DefaultMutableTreeNode node = (DefaultMutableTreeNode) obj;
               obj = node.getUserObject();

               // is current tab a subtab of a diagram? then look for the parent diagram
               while ( (node.getParent() instanceof DefaultMutableTreeNode) && ! (obj instanceof ASGDiagram))
               {
                  node = (DefaultMutableTreeNode) node.getParent();
                  obj = node.getUserObject();
               }

               if (obj instanceof ASGDiagram)
               {
                  FrameMain.get().showDiagram ((ASGDiagram) obj);
               }
            }

            // need this for updates; for I don't know, if it's done elsewhere
            // it seems to be best done at changing between diagrams in tree
            UMLProject.get().refreshDisplay();
         }
         finally
         {
            frame.setCursorDefault();
         }
      } // valueChanged

   } // TreeListener


   /**
    * This class handles the mouse clicks on the 'project-tree' and shows a popup menu for
    * the UMLPackages
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.69.2.3 $
    */
   private class PopupListener extends MouseAdapter
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mouseReleased (MouseEvent e)
      {
         if (e.isPopupTrigger())
         {
            maybeShowPopup (e);
         }
      } // mouseReleased


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mouseClicked (MouseEvent e)
      {
         if (e.isPopupTrigger())
         {
            maybeShowPopup (e);
         }
      } // mouseClicked


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mousePressed (MouseEvent e)
      {
         if (e.isPopupTrigger())
         {
            maybeShowPopup (e);
         }
      } // mousePressed


      /**
       * Shows the ProjectDialog when user performs a right mouse click on the project tree
       * root.
       *
       * @param e  No description provided
       */
      private void maybeShowPopup (MouseEvent e)
      {
         LogicUnparseInterface logic = getLogicAt (e.getX(), e.getY());
         if ( (e.isMetaDown()) &&  (e.getComponent() instanceof JTree) && logic != null)
         {
            UserInterfaceManager uiManager = UserInterfaceManager.get();
            //System.out.println ("logic " + logic);
            JPopupMenu popup = uiManager.getFromPopupMenus (logic.getClass().getName());

            if (popup != null)
            {
               // set the popup source to the selected tree node
               SelectionManager.get().setPopupSource (tree, logic);

               FPopupMenu.show (popup, e.getComponent(), e.getX(), e.getY());
            }
         }
      } // mousePressed
   } // PopupListener


   /**
    * Find the logic represented by the gui element at a specific location in the tab.
    *
    * @param x  horizontal coordinate of location
    * @param y  vertical coordinate of location
    * @return   the logic that is represented, null if none was found
    */
   public LogicUnparseInterface getLogicAt (final int x, final int y)
   {
      LogicUnparseInterface logic = null;
      TreePath path = getTree().getPathForLocation (x, y);
      if ( (path != null) && path.getLastPathComponent() instanceof DefaultMutableTreeNode)
      {
         getTree().setSelectionPath (path);
         DefaultMutableTreeNode node = (DefaultMutableTreeNode) path.getLastPathComponent();
         Object obj = node.getUserObject();
         if (obj instanceof LogicUnparseInterface)
         {
            logic = (LogicUnparseInterface) obj;
         }
      }
      //System.out.println(x + "          "+y + "        "+logic);
      return logic;
   }


   /**
    * TreeCellEditor for editing the Nodes in the TreeView.
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.69.2.3 $
    */
   private class FujabaTreeCellEditor extends DefaultCellEditor
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private ASGDiagram editedDiagram;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private DefaultMutableTreeNode editedNode;


      /**
       * Constructor for class FujabaTreeCellEditor
       *
       * @param tree  No description provided
       */
      public FujabaTreeCellEditor (JTree tree)
      {
         super (new JTextField());
         addCellEditorListener (new FujabaCellEditorListener (this, tree));
      }


      /**
       * This method should return true, if the parameter diagram's node should be editable.
       *
       * @param diagram  The treenode's UserObject.
       * @return         true, if it should be editable, otherwise false.
       */
      private boolean isDiagramEditable (Object diagram)
      {
         return  ( (diagram instanceof FClassDiagram) ||
             (diagram instanceof PackageDiagram));
      }


      /**
       * Get the cellEditable attribute of the FujabaTreeCellEditor object
       *
       * @param event  No description provided
       * @return       The cellEditable value
       */
      public boolean isCellEditable (EventObject event)
      {
         JTree tree = event != null ? (JTree) event.getSource() : getTree();

         boolean superVote = super.isCellEditable (event);

         if (event instanceof MouseEvent)
         {
            MouseEvent mouse = (MouseEvent) event;

            TreePath path = tree.getPathForLocation (mouse.getX(), mouse.getY());

            Object userObject = null;

            try
            {

               Object object = path.getLastPathComponent();

               DefaultMutableTreeNode node = (DefaultMutableTreeNode) object;

               userObject = node.getUserObject();
            }
            catch (NullPointerException ex)
            {
            }

            return superVote && isDiagramEditable (userObject);
         }
         else
         {
            return superVote;
         }
      }


      /**
       * Get the treeCellEditorComponent attribute of the FujabaTreeCellEditor object
       *
       * @param tree        No description provided
       * @param value       No description provided
       * @param isSelected  No description provided
       * @param expanded    No description provided
       * @param leaf        No description provided
       * @param row         No description provided
       * @return            The treeCellEditorComponent value
       */
      public Component getTreeCellEditorComponent (JTree tree, Object value,
                                                   boolean isSelected,
                                                   boolean expanded,
                                                   boolean leaf, int row)
      {
         editedNode = (DefaultMutableTreeNode) value;
         if (isDiagramEditable ( ((DefaultMutableTreeNode) value).getUserObject()))
         {
            editedDiagram = (ASGDiagram)  ((DefaultMutableTreeNode) value).getUserObject();
         }
         else
         {
            editedDiagram = null;
         }

         // get icon from TreeCellRenderer
         Icon icon = null;
         int iconTextGap = 0;
         Component displayComp = tree.getCellRenderer().getTreeCellRendererComponent (tree, value, isSelected, expanded, leaf, row, false);
         if (displayComp instanceof JLabel)
         {
            icon =  ((JLabel) displayComp).getIcon();
            iconTextGap =  ((JLabel) displayComp).getIconTextGap();
         }

         final Component editingComp = super.getTreeCellEditorComponent (tree, value, isSelected, expanded, leaf, row);

         EditorContainer container = new EditorContainer (editingComp, icon, iconTextGap);
         container.add (editingComp);

         if (editingComp instanceof JTextField)
         {
            final JTextField textField = (JTextField) editingComp;
            textField.selectAll();
         }

         SwingUtilities.invokeLater (
            new Runnable()
            {
               public void run()
               {
                  editingComp.requestFocusInWindow();
               }
            });

         return container;
      }


      /**
       * Get the currentEditedDiagram attribute of the FujabaTreeCellEditor object
       *
       * @return   The currentEditedDiagram value
       */
      public ASGDiagram getCurrentEditedDiagram()
      {
         return editedDiagram;
      }


      /**
       * Get the currentEditedNode attribute of the FujabaTreeCellEditor object
       *
       * @return   The currentEditedNode value
       */
      public DefaultMutableTreeNode getCurrentEditedNode()
      {
         return editedNode;
      }
   }


   /**
    * Container responsible for placing the editingComponent.
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.69.2.3 $
    */
   private class EditorContainer extends Container
   {

      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private Component editingComponent;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private Icon editingIcon;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private int offset; // move editor component 18 points to the right to display icon correctly (which has suprisingly a width of 18)


      /**
       * Constructs an EditorContainer object.
       *
       * @param editingComponent  No description provided
       * @param icon              No description provided
       * @param iconTextGap       No description provided
       */
      public EditorContainer (Component editingComponent, Icon icon, int iconTextGap)
      {
         setLayout (null);
         this.editingIcon = icon;
         this.editingComponent = editingComponent;
         offset = iconTextGap;
         if (editingIcon != null)
         {
            offset += editingIcon.getIconWidth();
         }
      }


      /**
       * Overrides <code>Container.paint</code> to paint the node's icon and use the selection
       * color for the background.
       *
       * @param g  No description provided
       */
      public void paint (Graphics g)
      {

         // Then the icon.
         if (editingIcon != null)
         {
            int yLoc = Math.max (0,  (getSize().height -
               editingIcon.getIconHeight()) / 2);

            editingIcon.paintIcon (this, g, 0, yLoc);
         }

         super.paint (g);
      }


      /**
       * Lays out this Container. If editing, the editor will be placed at offset in the x
       * direction and 0 for y.
       */
      public void doLayout()
      {
         if (editingComponent != null)
         {
            Dimension cSize = getSize();

            editingComponent.getPreferredSize();
            editingComponent.setLocation (offset, 0);
            editingComponent.setBounds (offset, 0,
               cSize.width - offset,
               cSize.height);
         }
      }


      /**
       * Returns the preferred size for the Container. This will be the preferred size of the
       * editor offset by offset.
       *
       * @return   The preferredSize value
       */
      public Dimension getPreferredSize()
      {
         if (editingComponent != null)
         {
            Dimension pSize = editingComponent.getPreferredSize();

            pSize.width += offset + 5;

            if (editingIcon != null)
            {
               pSize.height = Math.max (pSize.height,
                  editingIcon.getIconHeight());
            }

            // Make sure height is at least 100.
            pSize.width = Math.max (pSize.width, 100);
            return pSize;
         }
         return new Dimension (0, 0);
      }
   }


   /**
    * This class deals with the setting the new names, when the user enters return.
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.69.2.3 $
    */
   private class FujabaCellEditorListener implements CellEditorListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private FujabaTreeCellEditor editor;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private JTree tree;


      /**
       * Constructor for class FujabaCellEditorListener
       *
       * @param editor  No description provided
       * @param tree    No description provided
       */
      public FujabaCellEditorListener (FujabaTreeCellEditor editor, JTree tree)
      {
         this.editor = editor;
         this.tree = tree;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void editingCanceled (ChangeEvent e)
      {
         // do nothing
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void editingStopped (ChangeEvent e)
      {
         ASGDiagram diagram = editor.getCurrentEditedDiagram();
         DefaultMutableTreeNode node = editor.getCurrentEditedNode();
         DefaultTreeModel model = (DefaultTreeModel) tree.getModel();

         diagram.setName ((String) editor.getCellEditorValue());
         node.setUserObject (diagram);
         model.nodeChanged (node);
      }
   }

}

/*
 * $Log: TabProxy.java,v $
 * Revision 1.69.2.3  2006/07/10 08:05:34  cschneid
 * sysouts removed, exceptions printed (instead of ignoring completely), caption changed to 'Fujaba 4'
 *
 */
