/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.net.URL;
import java.util.Vector;

import javax.swing.*;

import de.uni_paderborn.fujaba.preferences.ColorsPreferences;


/**
 * The ScrollPanel class implements a Panel that scrolls text blocks defined in a script from
 * bottom to the top of the panel. It extends from JPanel and implements Runnable for the scrolling
 * purposes. ScrollPanel uses the class Container to read a script. A script contains descriptions
 * of text blocks, a block starts with <BLOCK> and ends with </BLOCK> in the script. Every
 * block can hold an unlimited number of text lines which will be formated automatically during
 * the parsing of the script. The user can start a new line using the <CR> command, define
 * a Title with <TITLE> ... </TITLE> and set a subtitle line with <SUBTITLE> ... </SUBTITLE>
 * .
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.27 $
 */
public class ScrollPanel extends JPanel implements Runnable
{

   /**
    * Contains the starting point of the displayed block in pixel. startPos = 0 the top of
    * the block is at the top of the Panel startPos = this.getHeight the top of the block is
    * at the bottom of the panel (and so nothing is displayed) startPos = - getBlockHeight()
    * the bottom of the block is at the top of the panel (and so nothing is displayed)
    */
   int startPos;

   /**
    * Contains the current block that's scrolled through the panel
    */
   private int currBlock = 0;

   /**
    * Contains the delay (in ms) for the scrolling (scrolling speed, 20-30 ms is a good value)
    */
   private int delay;

   /**
    * The size of the scrolling area in the panel
    */
   private int scrollWidth, scrollHeight;

   /**
    * Absolute and relative mouse coordinates for the mouseScroller [TM]
    */
   int xmouse, ymouse, ydiff;

   /**
    * Flag for pause function
    */
   boolean pause = false;

   /**
    * Flag for the status symbol display (play, pause symbol) true if the mouse is in the panel
    * (and the status is displayed)
    */
   boolean inpanel = false;

   /**
    * Flag for initialisation purposes of mainImg and mainMap
    */
   private boolean initialize = true;

   /**
    * Contains the text shadow color
    */
   private Color shadowCol = new Color (220, 220, 220);

   /**
    * Contains the symbol color
    */
   private Color symbolCol = new Color (220, 220, 220);

   /**
    * Contains the background color
    */
   private Color backgrCol = new Color (255, 255, 255);

   /**
    * The Image of the scrolling area. JPanel supports DoubleBuffering, so we don't need to
    * DoubleBuffer the scroll area
    */
   private Image mainImg;

   /**
    * The Graphics object of mainImg
    */
   private Graphics mainMap;

   /**
    * Hard coded font styles for title, text and subtitle
    */
   private Font titleF = new Font ("Arial", Font.BOLD, 15);
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Font textF = new Font ("Arial", Font.PLAIN, 12);
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Font subtitleF = new Font ("Arial", Font.PLAIN, 10);

   /**
    * The thread to scroll the text
    */
   private Thread cthread;

   /**
    * Contains a BlockContainer that reads the script and holds all blocks of text
    */
   private BlockContainer cont;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String file;


   /**
    * Constructs a new ScrollPanel with the specified parameters
    *
    * @param width   the width of the ScrollPanel
    * @param height  the height of the ScrollPanel
    * @param delay   the scroll delay of the ScrollPanel in ms (20 - 35 ms is a good value)
    * @param file    the script file
    */
   public ScrollPanel (int width, int height, int delay, String file)
   {
      super();

      this.delay = delay;
      this.file = file;
      startPos = height;

      setBackground (ColorsPreferences.get().DEFAULT_BACKGROUND);
      setSize (width, height);
      setPreferredSize (new Dimension (width, height));
      setBorder (BorderFactory.createLoweredBevelBorder());

      addMouseListener (new ScrollPanelListener());
      addMouseMotionListener (new ScrollPanelMotionListener());

      scrollWidth = getWidth() - getInsets().left - getInsets().right;
      scrollHeight = getHeight() - getInsets().top - getInsets().bottom;

      cont = new BlockContainer (scrollWidth);

      cthread = new Thread (this);
      cthread.start();
   }


   /**
    * Starts the scroll thread
    */
   public void run()
   {
      boolean wasVisible = false;

      try
      {
         cont.readScript (file, titleF, textF, subtitleF);
      }
      catch (Exception e)
      {
         add (new Label (e.getMessage()));
         return;
      }

      cthread.setPriority (Thread.NORM_PRIORITY);
      while (cthread != null)
      {
         this.repaint();
         try
         {
            Thread.sleep (delay);
         }
         catch (InterruptedException e)
         {
         }

         if (!isParentVisible())
         {
            if (wasVisible)
            {
               cthread = null;
            }
         }
         else
         {
            wasVisible = true;
         }
      }
   }


   /**
    * Checks if the parent components of ScrollPanel are visible
    *
    * @return   True if parent components are visible
    */
   private boolean isParentVisible()
   {
      boolean visible = true;
      int depth = 0;
      int maxDepth = 5;
      Container parent = this;

      while ( (visible) &&  (depth < maxDepth) && ! (parent instanceof Dialog))
      {
         parent = parent.getParent();
         if (parent == null)
         {
            visible = false;
            break;
         }
         visible = parent.isVisible();
         depth++;
      }
      return visible;
   }


   /**
    * Clears the provided Graphics area
    *
    * @param g  Graphics object
    */
   private void clearimg (Graphics g)
   {
      g.setColor (backgrCol);
      g.fillRect (0, 0, scrollWidth, scrollHeight);
   }


   /**
    * Displays a block with the provided position in the ScrollPanel
    *
    * @param g          No description provided
    * @param block      No description provided
    * @param startYPos  No description provided
    */
   private void displayBlock (Graphics g, Block block, int startYPos)
   {
      FontMetrics fMetric;

      int yBlockPos = 0;

      int xPos;

      //draw symbols
      if (pause)
      { //Pause symbol
         g.setColor (symbolCol);
         g.fillRect (getWidth() - getInsets().right - 8, getHeight() - getInsets().bottom - 14, 4, 10);
         g.fillRect (getWidth() - getInsets().right - 16, getHeight() - getInsets().bottom - 14, 4, 10);
      }
      else if (inpanel)
      { //Play symbol
         int xCord[] = {getWidth() - getInsets().right - 16, getWidth() - getInsets().right - 4, getWidth() - getInsets().right - 16};
         int yCord[] = {getHeight() - getInsets().bottom - 16, getHeight() - getInsets().bottom - 10, getHeight() - getInsets().bottom - 4};

         g.setColor (symbolCol);
         g.fillPolygon (xCord, yCord, xCord.length);
      }

      //check and draw all lines of the current block
      for (int i = 0; i < block.getLineCount(); i++)
      {
         if (i <= block.getTitleCount())
         {
            g.setFont (titleF);
            fMetric = g.getFontMetrics();
         }
         else
            if (i == block.getSubtitleLine())
         {
            g.setFont (subtitleF);
            fMetric = g.getFontMetrics();
         }
         else
         {
            g.setFont (textF);
            fMetric = g.getFontMetrics();
         }

         yBlockPos = yBlockPos + fMetric.getHeight();

         if (yBlockPos + startYPos > -5)
         {
            if ( (yBlockPos + startYPos - fMetric.getHeight()) < getHeight())
            {
               xPos =  (getWidth() - fMetric.stringWidth (block.getLine (i))) / 2;
               g.setColor (shadowCol);
               g.drawString (block.getLine (i), xPos + 2, yBlockPos + startYPos + 2);
               g.setColor (ColorsPreferences.get().DEFAULT_FOREGROUND);
               g.drawString (block.getLine (i), xPos, yBlockPos + startYPos);
            }
         }
      }
   }


   /**
    * Scrolls all blocks in the Container in a loop
    *
    * @param g  Graphics object of the mainImg
    */
   private void scrollCont (Graphics g)
   {
      //something went wrong with the container
      if (cont == null || cont.getBlockCount() == 0)
      {
         g.setColor (ColorsPreferences.get().ERROR);
         g.drawString ("ERROR: Empty Container!", 5, 10);
         return;
      }

      //if not pause scroll the current block
      if (!pause)
      {
         startPos--;
      }

      //current block left the top of the ScrollPanel
      if (startPos < -cont.getBlock (currBlock).getBlockHeight())
      {
         currBlock++;
         if (currBlock == cont.getBlockCount())
         {
            currBlock = 0;
         }
         startPos = getHeight() - getInsets().bottom;
      }
      //current block left the bottom of the ScrollPanel (during a mouseScroll)
      else if (startPos >  (getHeight() - getInsets().bottom))
      {
         currBlock--;
         if (currBlock < 0)
         {
            currBlock = cont.getBlockCount() - 1;
         }
         startPos = -cont.getBlock (currBlock).getBlockHeight();
      }

      //display the current block at it's new position
      displayBlock (g, cont.getBlock (currBlock), startPos);
   }


   /**
    * Paints the ScrollPanel
    *
    * @param g  No description provided
    */
   public void paintComponent (Graphics g)
   {
      super.paintComponent (g);

      if (initialize)
      {
         mainImg = createImage (scrollWidth, scrollHeight);
         mainMap = mainImg.getGraphics();
         initialize = false;
      }
      else
      {
         clearimg (mainMap);
         scrollCont (mainMap);
         g.drawImage (mainImg, getInsets().left, getInsets().right, this);
      }
   }


   /**
    * MouseListener for mouseScroll.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.27 $
    */
   private class ScrollPanelListener implements MouseListener
   {

      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mouseEntered (MouseEvent e)
      {
         inpanel = true;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mouseExited (MouseEvent e)
      {
         inpanel = false;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mousePressed (MouseEvent e)
      {
         ymouse = e.getY();
         xmouse = e.getX();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mouseReleased (MouseEvent e)
      {
         ydiff = 0;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mouseClicked (MouseEvent e)
      {
         pause = !pause;
      }
   }


   /**
    * mouseMotionListener for mouseScroll.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.27 $
    */
   private class ScrollPanelMotionListener implements MouseMotionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mouseDragged (MouseEvent e)
      {
         int y = e.getY();

         pause = true;
         if (ydiff == 0)
         {
            ydiff = y;
         }
         else
         {
            startPos = startPos +  (y - ydiff);
            ydiff = y;
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void mouseMoved (MouseEvent e) { }
   }

}


/**
 * The class Block provides a container for a Textblock
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.27 $
 */
class Block
{


   /**
    * The vector that holds the formatted text lines
    */
   private Vector blockLines;

   /**
    * Contains the height in pixels
    */
   private int height = 0;

   /**
    * Contains the number of headlines
    */
   private int titleCount = -1;

   /**
    * Contains the number of subtitle lines
    */
   private int subtitleLine = -1;


   /**
    * Constructor for Block Creates the vector blockLines
    */
   public Block()
   {
      blockLines = new Vector();
   }


   /**
    * Add a line to the block
    *
    * @param s  The object added.
    */
   public void addLine (String s)
   {
      blockLines.add (s);
   }


   /**
    * Gets a line from the block
    *
    * @param i  No description provided
    * @return   the text line at index i
    */
   public String getLine (int i)
   {
      return (String) blockLines.get (i);
   }


   /**
    * Returns the number of lines in the block
    *
    * @return   number of lines
    */
   public int getLineCount()
   {
      return blockLines.size();
   }


   /**
    * Sets the block height in pixel
    *
    * @param h  The new blockHeight value
    */
   public void setBlockHeight (int h)
   {
      height = h;
   }


   /**
    * Gets the block height in pixel
    *
    * @return   height in pixel
    */
   public int getBlockHeight()
   {
      return height;
   }


   /**
    * Sets the number of title lines of the block
    *
    * @param c  The new titleCount value
    */
   public void setTitleCount (int c)
   {
      titleCount = c;
   }


   /**
    * Gets the number of title lines from the block
    *
    * @return   number of title lines
    */
   public int getTitleCount()
   {
      return titleCount;
   }


   /**
    * Sets the number of subscription lines from the block
    *
    * @param s  The new subtitleLine value
    */
   public void setSubtitleLine (int s)
   {
      subtitleLine = s;
   }


   /**
    * Gets the number of subscription lines from the block
    *
    * @return   number of subscr lines
    */
   public int getSubtitleLine()
   {
      return subtitleLine;
   }

}


/**
 * The class BlockContainer loads and stores a complete text for scrolling purposes
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.27 $
 */
class BlockContainer
{


   /**
    * The vector that contains all blocks of the container
    */
   private Vector blockTable;

   /**
    * Contains the width in pixel of all blocks in the container
    */
   private int width;

   /**
    * Start and ending point of the current word in the readbuffer
    */
   private int wordStart = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int wordEnd = 0;


   /**
    * Constructor of BlockContainer
    *
    * @param w  the width of all blocks in the container
    */
   public BlockContainer (int w)
   {
      width = w;
      blockTable = new Vector();
   }


   /**
    * scans the scriptLine buffer for the next word !!! should be rewritten in cause of ugly
    * implementation !!!
    *
    * @param scriptLine  No description provided
    * @return            The nextWord value
    */
   private String getNextWord (String scriptLine)
   {
      String s;

      wordEnd = scriptLine.indexOf (" ", wordStart);
      if (wordEnd == -1)
      {
         wordEnd = scriptLine.length();
      }
      s = scriptLine.substring (wordStart, wordEnd);
      wordStart = wordEnd + 1;
      return s;
   }


   /**
    * resets the scriptLine buffer scan !!! should be rewritten in cause of ugly implementation
    * !!!
    */
   private void resetGNW()
   {
      wordEnd = 0;
      wordStart = 0;
   }


   /**
    * Reads and parses a script file, formats the text and stores it into Blocks in BlockContainers
    * blockTable
    *
    * @param script      the script file
    * @param titleF      Font for the Titles
    * @param textF       Font for normal Text
    * @param subtitleF   Font for subscriptions
    * @return            No description provided
    * @throws Exception  Exception description not provided
    */
   public boolean readScript (String script, Font titleF, Font textF, Font subtitleF) throws Exception
   {
      int titleCount = -1;
      int lineWidth = 0;
      int blockHeight = 0;
      Block newBlock = null;
      String scriptLine;
      String word;
      String blockLine = "";
      boolean inBlock = false;
      boolean inTitle = false;
      BufferedImage fImg;
      Graphics fmap;
      FontMetrics fMetric;
      BufferedReader in = null;

      //get a fMetric to calculate lineWidth and blockHeight
      fImg = new BufferedImage (1, 1, BufferedImage.TYPE_INT_RGB);
      fmap = fImg.getGraphics();
      fmap.setFont (textF);
      fMetric = fmap.getFontMetrics();

      //open the script
      try
      {
         URL url = ClassLoader.getSystemClassLoader().getResource (script);

         in = new BufferedReader (new InputStreamReader (url.openStream()));
      }
      catch (Exception e)
      {
         throw new IOException ("Script does not exist:");
      }

      //linear parsing starts here
      while (true)
      {
         //get a line from the script
         try
         {
            scriptLine = in.readLine();
         }
         catch (Exception e)
         {
            throw new IOException ("Script Read ERROR");
         }

         //nothing to do?
         if (scriptLine == null)
         {
            in.close();
            return true;
         }

         scriptLine.trim();
         resetGNW();

         //start linear parsing of the current scripLine
         while (wordEnd < scriptLine.length())
         {
            word = getNextWord (scriptLine);
            if (inBlock)
            {

               if (word.startsWith ("</BLOCK>"))
               {
                  inBlock = false;

                  if (blockLine.length() > 0)
                  {
                     newBlock.addLine (blockLine);
                     blockHeight += fMetric.getHeight();
                  }
                  newBlock.setBlockHeight (blockHeight);
                  newBlock.setTitleCount (titleCount);
                  blockTable.add (newBlock);

                  blockLine = "";
                  lineWidth = 0;
                  blockHeight = 0;
                  titleCount = -1;
                  inTitle = false;
               }
               else
                  if (word.startsWith ("<TITLE>"))
               {
                  inTitle = true;
                  fmap.setFont (titleF);
                  fMetric = fmap.getFontMetrics();
               }
               else
                  if (word.startsWith ("</TITLE>") || word.startsWith ("<SUBTITLE>") || word.startsWith ("</SUBTITLE>") || word.startsWith ("<CR>"))
               {
                  if (word.startsWith ("<CR>") || blockLine.length() != 0)
                  {
                     newBlock.addLine (blockLine);
                     blockHeight += fMetric.getHeight();
                     blockLine = "";
                     lineWidth = 0;
                  }
                  if (inTitle)
                  {
                     titleCount++;
                  }

                  if (word.startsWith ("<SUBTITLE>"))
                  {
                     newBlock.setSubtitleLine (newBlock.getLineCount());
                     inTitle = false;
                     fmap.setFont (subtitleF);
                     fMetric = fmap.getFontMetrics();
                  }
                  else
                     if (word.startsWith ("</TITLE>") || word.startsWith ("</SUBTITLE>"))
                  {
                     inTitle = false;
                     fmap.setFont (textF);
                     fMetric = fmap.getFontMetrics();
                  }
               }
               else
                  if ( (lineWidth + fMetric.stringWidth (word + " ")) < width)
               {
                  blockLine += word + " ";
                  lineWidth += fMetric.stringWidth (word + " ");
               }
               else
               {
                  newBlock.addLine (blockLine);
                  blockLine = word + " ";
                  lineWidth = fMetric.stringWidth (word + " ");
                  blockHeight += fMetric.getHeight();
                  if (inTitle)
                  {
                     titleCount++;
                  }
               }

            }
            else if (word.startsWith ("<BLOCK>"))
            {
               inBlock = true;
               newBlock = new Block();
               fmap.setFont (textF);
               fMetric = fmap.getFontMetrics();
            }
         }
      }
   }


   /**
    * Returns the width in pixel of all blocks in the container
    *
    * @return   width in pixel
    */
   public int getWidth()
   {
      return width;
   }


   /**
    * Returns the block at the specified position from the container
    *
    * @param i  index of block to return
    * @return   the specified block
    */
   public Block getBlock (int i)
   {
      return (Block) blockTable.get (i);
   }


   /**
    * Returns the number of blocks in the container
    *
    * @return   The number of blocks in the container
    */
   public int getBlockCount()
   {
      return blockTable.size();
   }

}

/*
 * $Log: ScrollPanel.java,v $
 * Revision 1.27  2004/11/22 19:04:53  lowende
 * Some javadoc corrections.
 *
 */
