/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app;

import java.io.*;
import java.lang.reflect.Field;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import org.apache.log4j.Logger;


/**
 * Class RemoveFragmentFromProject
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.12.2.1 $
 */
public class RemoveFragmentFromProject
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (RemoveFragmentFromProject.class);


   /**
    * Default Constructor
    */
   public RemoveFragmentFromProject() { }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line  No description provided
    * @return      No description provided
    */
   public final StringTokenizer createLineTokenizer (String line)
   {
      StringTokenizer st = new StringTokenizer (line, ";");

      st.nextToken();

      return st;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param filename  No description provided
    * @param content   No description provided
    * @return          No description provided
    */
   public boolean writeFile (String filename, StringBuffer content)
   {
      File tmpFile = new File (filename);
      boolean success = true;

      try
      {
         BufferedWriter tmpWriter = new BufferedWriter (new OutputStreamWriter (new GZIPOutputStream (new FileOutputStream (tmpFile))));

         tmpWriter.write (content.toString(), 0, content.length());
         tmpWriter.close();
      }
      catch (IOException exception)
      {
         log.error ("can't write file " + filename);
         success = false;
      }

      return success;
   }


   /**
    * Get the classOfField attribute of the RemoveFragmentFromProject class
    *
    * @param className  No description provided
    * @param fieldName  No description provided
    * @return           The classOfField value
    */
   public String getClassOfField (String className,
                                  String fieldName)
   {
      try
      {
         Class currentClass = Class.forName (className);
         Field theField = null;
         while (theField == null && currentClass != java.lang.Object.class)
         {
            try
            {
               theField = currentClass.getDeclaredField (fieldName);
            }
            catch (NoSuchFieldException e)
            {
               theField = null;
               currentClass = currentClass.getSuperclass();
            }
            catch (SecurityException e)
            {
               log.error (e.getMessage());
               log.error (e.toString());
            }
         }

         if (currentClass == java.lang.Object.class)
         {
            return null;
         }

         return currentClass.getName();
      }
      catch (Exception exception)
      {
         log.error (exception.toString());
         log.error (exception.getMessage());

         throw new Error ("className=" + className +
            ",fieldName=" + fieldName);
      }

   }


   /**
    * Describe <code>removeFragmentsFor</code> method here.
    *
    * @param fileName  the filename.
    * @return          a stringbuffer which contains the project file without any fragments
    *      etc.
    */
   public StringBuffer removeFragmentsFor (String fileName)
   {
      File inputFile = new File (fileName);
      StringBuffer out = new StringBuffer();
      BufferedReader fr = null;
      boolean append = true;

      Vector toBeRemovedAttr = new Vector();
      Vector toBeRemovedClasses = new Vector();

      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLClass::fragments");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLClass::proxies");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLFile::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLFile::fragments");

      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLProject::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLBaseTypes::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLTransitionGuard::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLStereotype::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLTypeList::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLStopActivity::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLTransition::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLStatementActivity::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLActivityDiagram::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLStartActivity::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLAttrExprPair::displayLevel");
      toBeRemovedAttr.add ("de.uni_paderborn.fujaba.uml.UMLMethod::proxies");

      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.uml.UMLFragment");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.uml.UMLFragment$PackageDecl");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.uml.UMLFragment$ImportDecl");

      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisCanvasItem");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisObject");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisLine");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisColumn");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisComposite");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisGrab");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisRow");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisText");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisVisibilityIcon");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisCanvas");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisSeparator");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisFrame");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisResizeable");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisBend");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisIcon");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisPointingText");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.dis.DisConnector");

      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.sim.SimConnectorInstance");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.sim.SimObjectInstance");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.sim.SimConnector");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.sim.SimAttr");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.sim.SimAttrInstance");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.sim.SimObject");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.sim.SimTool");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.sim.SimState");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.sim.SimTopologyDiagram");

      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.pattern.DPAnnotation");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.pattern.DPAccessMethodAnnotation");
      toBeRemovedClasses.add ("de.uni_paderborn.fujaba.pattern.DPLine");

      try
      {
         try
         {
            fr = new BufferedReader (new InputStreamReader
                (new GZIPInputStream (new FileInputStream (inputFile))), 32768);
         }
         catch (IOException e)
         {
            fr = new BufferedReader (new FileReader (inputFile), 32768);
         }

         // check, if it is the correct FileFormat
         String line = fr.readLine();
         int lineNum = 0;

         StringBuffer header = new StringBuffer();
         int hashTableLength = 0;

         lineNum++;
         if (!line.startsWith ("# Fujaba-Project-File"))
         {
            throw new RuntimeException ("This is not a Fujaba project file");
         }

         header.append (line);
         header.append ("\n");

         boolean isFragment = false;
         boolean isHeader = true;

         String className = null;

         // Now parse the rest of the file
         while (fr.ready())
         {
            append = true;
            lineNum++;
            line = fr.readLine();
            //	    if (log.isInfoEnabled()) log.info ("line:" + lineNum + "=" + line );

            StringTokenizer lineTok = createLineTokenizer (line);

            switch (line.charAt (0))
            {
               case '#': // documentation

                  break;
               case '-': // file specific variables
                  String name = lineTok.nextToken();
                  if (name.equals ("HashTableLength"))
                  {
                     hashTableLength = Integer.parseInt (lineTok.nextToken());
                     append = false;
                     isHeader = false;
                  }
                  break;
               case '+': // hashtable entry
                  lineTok.nextToken();

                  // get classname
                  className = lineTok.nextToken();

                  if (toBeRemovedClasses.contains (className))
                  {
                     append = false;
                  }
                  className = null;

                  break;
               case '*': // object reference

                  lineTok.nextToken();
                  className = lineTok.nextToken();

                  if (toBeRemovedClasses.contains (className))
                  {
                     hashTableLength--;
                     isFragment = true;
                     append = false;
                  }
                  else
                  {
                     isFragment = false;
                  }
                  break;
               case '~': // attribute reference
                  if (isFragment)
                  {
                     append = false;
                  }
                  else
                  {
                     String shortAttrName = lineTok.nextToken();
                     String attrName = className + "::" + shortAttrName;
                     if (toBeRemovedAttr.contains (attrName))
                     {
                        append = false;
                     }
                     else
                     {
                        // create new attr name
                        String classOfField = getClassOfField (className,
                           shortAttrName);

                        if (classOfField != null)
                        {
                           line = "~;" + classOfField + "::" + shortAttrName;

                           while (lineTok.hasMoreTokens())
                           {
                              line += ";";
                              line += lineTok.nextToken();
                           }
                        }
                        else
                        {
                           append = false;
                        }

                     }
                  }

                  break;
               case '&': // diff definition start here

                  break;
            } // switch

            if (append)
            {
               StringBuffer use;
               if (isHeader)
               {
                  use = header;
               }
               else
               {
                  use = out;
               }
               use.append (line);
               if (fr.ready())
               {
                  use.append ("\n");
               }

            }

         } // while fpr file has more lines

         header.append ("-;HashTableLength;").append (hashTableLength).append ("\n");
         out.insert (0, header.toString());
      }
      catch (IOException exception)
      {
         log.error ("exception=" + exception);

      }
      return out;
   }


   /**
    * Main-Method.
    *
    * @param argv  argv[0] must contain the fileName.
    */
   public static void main (String argv[])
   {
      if (argv.length != 1)
      {
         log.error ("Usage:");
         log.error ("\tRemoveFragmentFromProject <filename>");
      }
      else
      {
         RemoveFragmentFromProject remove = new RemoveFragmentFromProject();
         StringBuffer result = remove.removeFragmentsFor (argv[0]);
         remove.writeFile (argv[0].substring (0, argv[0].length() - 7)
            + ".noFragment.fpr.gz", result);
      }
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "RemoveFragmentFromProject[]";
   }

}

/*
 * $Log: RemoveFragmentFromProject.java,v $
 * Revision 1.12.2.1  2005/09/30 18:39:54  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
