/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app;


import java.awt.Cursor;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.JarURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.Logger;
import org.apache.log4j.Priority;

import de.uni_paderborn.fujaba.basic.FileHistory;
import de.uni_paderborn.fujaba.coobra.RecoverDialog;
import de.uni_paderborn.fujaba.preferences.DebugPreferences;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.uni_paderborn.fujaba.preferences.LoggingPreferences;
import de.uni_paderborn.fujaba.preferences.PreferencesProperties;
import de.uni_paderborn.fujaba.project.PersistencySupport;
import de.uni_paderborn.fujaba.texteditor.TextEditor;
import de.uni_paderborn.lib.classloader.UPBClassLoader;
import de.upb.lib.plugins.PluginManager;


/**
 * The Fujaba application. Here you can find the main method.
 * 
 * @author $Author: lowende $
 * @version $Revision: 1.126.2.7 $
 */
public class FujabaApp
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger
         .getLogger(FujabaApp.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static volatile FujabaApp fujabaApp;

   /**
    * Plugin manager used by Fujaba. Note: do not initialize here, because preferences are queried
    * by the call to 'KernelInterfaceImpl.get()', but they have to be set up first!
    */
   private static PluginManager pluginManager;

   /**
    * Strategy for accessing plugin and class loader information during loading and saving.
    * 
    * TODO will be moved elsewhere in Fujaba 5
    */
   private static PersistencySupport persistencySupport = new FujabaPersistencySupport();


   /**
    * Get the persistencySupport attribute of the FujabaApp class
    * 
    * @return The persistencySupport value
    */
   public static PersistencySupport getPersistencySupport()
   {
      return persistencySupport;
   }


   /**
    * Sets the persistencySupport attribute of the FujabaApp class
    * 
    * @param value The new persistencySupport value
    */
   public static void setPersistencySupport(PersistencySupport value)
   {
      persistencySupport = value;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static transient URI FUJABA_BASE = findFujabaBase();

   /**
    * The URI of the properties file in which Fujaba remembers, which directory should be the
    * 'propertyDir'.
    */
   public static final String PROPERTIES_PATH = System.getProperty("user.home")
         + File.separator + "fujaba.properties";


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @return No description provided
    */
   private final static URI findFujabaBase()
   {
      URI uri = null;
      URL url = FujabaApp.class.getResource("FujabaApp.class");
      if (url != null)
      {
         try
         {
            URLConnection connection = url.openConnection();
            if (connection instanceof JarURLConnection)
            {
               JarURLConnection jarConnection = (JarURLConnection) connection;
               url = jarConnection.getJarFileURL();
               uri = new URI(url.toExternalForm());
               uri = uri.resolve("."); // get directory of jar file
            }
            else
            {
               uri = new URI(url.toExternalForm());

               StringBuffer parentBuffer = new StringBuffer();
               // go up once from /bin directory to project root
               // and once for each package
               StringTokenizer packTokens = new StringTokenizer(FujabaApp.class
                     .getName(), ".");
               while (packTokens.hasMoreTokens())
               {
                  parentBuffer.append("../");
                  packTokens.nextToken();
               }
               uri = uri.resolve(parentBuffer.toString());
            }
         }
         catch (IOException ioe)
         {
            return null;
         }
         catch (URISyntaxException use)
         {
            return null;
         }
      }
      return uri;
   }


   /**
    * Constructor for class FujabaApp
    */
   private FujabaApp()
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @return No description provided
    */
   public static FujabaApp get()
   {
      if (fujabaApp == null)
      {
         fujabaApp = new FujabaApp();
      }

      return fujabaApp;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static boolean started = false;


   /**
    * Sets the started attribute of the FujabaApp class
    */
   private static void setStarted()
   {
      started = true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @return No description provided
    */
   public static FujabaApp start()
   {
      FujabaApp fujabaApp = get();

      if (!started)
      {
         fujabaApp.run();
      }

      return fujabaApp;
   }


   /**
    * Initializes the propertyDir. This is the base-directory which Fujaba uses for storing
    * <code>PreferencesProperties</code>.
    * 
    * <pre>
    *
    * The propertyDir is determined by the following algorithm:
    *
    *  - if the user has specified a propertyDir in the commandline,
    *    this one will be used.
    *  -
    * If Fujaba can't determine the propertyDir itself, the user is bugged.
    * </pre>
    * 
    * 
    * @see PreferencesProperties#getPropertyDir()
    */
   private void initPropertyDir()
   {
      String propertyDir = null;

      Properties propertyDirProperties = new Properties();
      boolean bPropertyDirModified = false;

      // the propertyDirProperties should be stored in the following file
      File propertyDirPropertiesFile = new File(PROPERTIES_PATH);

      if (this.desiredPropertyDir != null)
      {
         // user has specified a propertyDir,
         // use it instead of the default one
         propertyDir = this.desiredPropertyDir;

         // should be saved later
         bPropertyDirModified = true;
      }
      else
      {
         // check, if we have already stored a propertyDir
         try
         {
            propertyDirProperties.load(new FileInputStream(
                  propertyDirPropertiesFile));
         }
         catch (Exception e)
         {
         }

         // does 'propertyDir' exist in file?
         String storedPropertyDir = null;
         if (propertyDirProperties != null)
         {
            storedPropertyDir = propertyDirProperties
                  .getProperty(PreferencesProperties.PROPERTY_DIR_PROPERTY);
         }

         if (storedPropertyDir != null)
         {
            propertyDir = storedPropertyDir;
         }
         else
         {
            // use default property dir
            propertyDir = PreferencesProperties.getProposedPropertyDir();

            // should be saved later
            bPropertyDirModified = true;
         }
      }

      // does propertyDir exist on disc?
      File propertyDirFile = new File(propertyDir);
      boolean validPropertyDirFile = (propertyDirFile != null
            && propertyDirFile.exists() && propertyDirFile.isDirectory());
      if (!validPropertyDirFile)
      {
         // propertyDir does not exist on disc, so we have to ask
         // the user, if it should be created or if another propertyDir should be used
         int iAnswer;
         boolean proceed = false;
         do
         {
            String title = "Fujaba " + Version.get().getVersion()
                  + " - Specify property directory";

            String message = null;
            // the check, if the file exists doesn't make sense in the
            // first run, but it does, if the user has browsed for an
            // existing directory and is asked again, if the
            // directory should be used
            if (propertyDirFile.exists())
            {
               message = "Do you want to use\n   '" + propertyDirFile
                     + "'\nas property directory?";
            }
            else
            {
               message = "Do you want to create\n   '" + propertyDirFile
                     + "'\nand use it as property directory?";
            }

            message += "\n\nOr do you want to browse for another property directory?\n\n";

            Object[] options = { "Yes", "Browse...", "Exit Fujaba" };
            Object initialValue = options[0];

            iAnswer = JOptionPane.showOptionDialog(null, message, title,
                  JOptionPane.DEFAULT_OPTION, JOptionPane.QUESTION_MESSAGE,
                  null, options, initialValue);

            switch (iAnswer)
            {
               case 0:
                  // user has answered 'yes',
                  // so we can use the propertyDir
                  PreferencesProperties.setPropertyDir(propertyDir);
                  break;
               case 1:
                  // open a directory browser and take the result as
                  // property dir
                  File choosePropertyDirFile;

                  if (propertyDirFile.exists())
                  {
                     choosePropertyDirFile = propertyDirFile;
                  }
                  else
                  {
                     choosePropertyDirFile = new File(System
                           .getProperty("user.home"));
                  }

                  // create a filechooser
                  JFileChooser fileChooser = new JFileChooser(
                        choosePropertyDirFile);
                  fileChooser
                        .setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

                  // open the filechooser - user MUST select a directory!
                  int chooserResult = fileChooser.showOpenDialog(null);

                  if (chooserResult == JFileChooser.APPROVE_OPTION)
                  {
                     propertyDirFile = fileChooser.getSelectedFile();
                  }

                  break;
               case JOptionPane.CLOSED_OPTION:
               case 2:
                  // the user wants to exit
                  exit(0);
                  break;
            }

            // repeat until the user has specified an existing directory
            // and clicked 'yes'
            proceed = (iAnswer == 0 && propertyDirFile != null);

            if (proceed)
            {
               // does the directory exist?
               if (!propertyDirFile.exists())
               {
                  // try to create the directory and all its parents
                  try
                  {
                     boolean success = propertyDirFile.mkdirs();
                     proceed = success;

                     if (log.isEnabledFor(Priority.ERROR))
                     {
                        if (!success)
                        {
                           System.err
                                 .println("error creating directory '"
                                       + propertyDirFile
                                       + "': is the directory valid and do you have write access?");
                        }
                     }
                     if (log.isEnabledFor(Priority.INFO))
                     {
                        if (success)
                        {
                           System.out.println("created directory '"
                                 + propertyDirFile + "'");
                        }
                     }
                  }
                  catch (Exception e)
                  {
                     if (log.isEnabledFor(Priority.ERROR))
                     {
                        log.error("error creating directory '"
                              + propertyDirFile + "': " + e.getMessage());
                     }
                     proceed = false;
                  }
               }
            }

            // only proceed, if the propertyDirFile really exists
            proceed = proceed && propertyDirFile.exists();
         }
         while (!proceed);

         // should be saved later
         bPropertyDirModified = true;
      }

      // now set the propertyDir for this application-run
      PreferencesProperties.setPropertyDir(propertyDirFile.getAbsolutePath());
      // make sure the propertyDir ends with a File.separator
      propertyDir = PreferencesProperties.getPropertyDir();

      if (log.isInfoEnabled())
      {
         log.info("using propertyDir '" + propertyDir + "'");
      }

      if (bPropertyDirModified)
      {
         // store propertyDir for next run
         propertyDirProperties.put(PreferencesProperties.PROPERTY_DIR_PROPERTY,
               propertyDir);

         try
         {
            propertyDirProperties
                  .store(new FileOutputStream(propertyDirPropertiesFile),
                        "Fujaba Properties, Do NOT edit!");

            if (log.isInfoEnabled())
            {
               log.info("propertyDir stored in '" + propertyDirPropertiesFile
                     + "'");
            }
         }
         catch (Exception e)
         {
            // inform user, if a value different to the
            // proposed propertyDir could not be stored
            if (!propertyDir.equals(PreferencesProperties
                  .getProposedPropertyDir()))
            {
               if (log.isEnabledFor(Priority.ERROR))
               {
                  log
                        .error("The propertyDir '"
                              + propertyDir
                              + "' could not be stored in file '"
                              + propertyDirPropertiesFile
                              + "': the propertyDir won't be recognized at the next fujaba-run and you will be asked again, which propertyDir to use.");
                  e.printStackTrace();
               }
            }
         }
      }

      // ////////////////////////////////////////////////////////////

      // ===== Configure log4j NOW so that framework ready if calls are made
      // to it

      // Turn off log4j default init, not necessary
      System.setProperty("log4j.defaultInitOverride", "true");

      // Trim file separator char to match how user.dir etc are stored
      if (propertyDir.endsWith(File.separator))
      {
         propertyDir = propertyDir.substring(0, propertyDir.length() - 1);
      }
      // Set the property dir as a system property so that it can be used in
      // the log4j config file:
      // /de/uni_paderborn/fujaba/logging/log4j-default-config.xml
      System.setProperty("fujaba.propertydir", propertyDir);

      // ===== end of log4j configuration
   }


   /**
    * this method initializes fujaba. It creates everything the user sees after calling fujaba.
    */
   public void run()
   {
      // before we do anything else, we initialize the property directory
      initPropertyDir();

      // create about box as soon as possible
      AboutBox aboutBox = null;

      if (this.getLoadFprFile() == null && this.isShowAboutBox())
      {
         aboutBox = new AboutBox(null, true);
         aboutBox.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
         aboutBox.setVisible(true);
      }

      // Note: log4J will be initialized by this call
      LoggingPreferences.get();

      // set value in DebugPreferences, after PreferencesProperties have been initialized
      DebugPreferences.get().setDebugMode(this.isDebugMode());

      // initialize fujaba's plugin manager
      // Note: 'KernelInterfaceImpl.get()' queries preferences, so
      // make sure they have been set up before this call!
      pluginManager = KernelInterfaceImpl.get().getManager();

      // set look and feel
      // todo: store classname instead of LaF-name, as getInstalledLookAndFeels takes a long time
      String lookAndFeel = GeneralPreferences.get().getLookAndFeel();
      String lafClassName = null;
      UIManager.LookAndFeelInfo[] installedLaFs = UIManager
            .getInstalledLookAndFeels();

      for (int i = 0; (i < installedLaFs.length) && (lafClassName == null); i++)
      {
         if (lookAndFeel.equals(installedLaFs[i].getName()))
         {
            lafClassName = installedLaFs[i].getClassName();
         }
      }
      try
      {
         UIManager.setLookAndFeel(lafClassName);
      }
      catch (Exception e)
      {
      }

      FrameMain frameMain = FrameMain.get();
      frameMain.init();

      try
      {
         if (this.getLoadFprFile() != null)
         {
            // a .fpr file is provided.
            if (log.isDebugEnabled())
            {
               log.debug("opening " + this.getLoadFprFile());
            }
            File myFile = new File(this.getLoadFprFile());

            if (myFile.exists())
            {
               if (log.isDebugEnabled())
               {
                  log.debug("loading " + this.getLoadFprFile());
               }
            }

            SwingUtilities.invokeLater(new ProjectLoader());
         }
         else
         {
            // plain project was already created in FrameMain.get().init() (s.a.)
         }

         if (aboutBox != null)
         {
            aboutBox.setVisible(false);
         }

         TextEditor.get();

         // show the Fujaba window
         frameMain.setVisible(!isInvisible());

         setStarted();

         // check for backup files from old sessions
         new RecoverDialog(frameMain).check();
      }
      catch (Throwable t)
      {
         if (log.isDebugEnabled())
         {
            log.debug("uncaught exception: " + t);
         }
         t.printStackTrace();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @author $Author: lowende $
    * @version $Revision: 1.126.2.7 $ $Date: 2006/04/03 14:37:33 $
    */
   class ProjectLoader implements Runnable
   {

      /**
       * Main processing method for the ProjectLoader object
       */
      public void run()
      {
         FrameMain frameMain = FrameMain.get();
         File myFile = new File(getLoadFprFile());
         frameMain.openFile(myFile);
      }

   }


   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public Vector javaFiles = new Vector();

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String loadFprFile = null;


   /**
    * Sets the loadFprFile attribute of the FujabaApp object
    * 
    * @param name The new loadFprFile value
    */
   public void setLoadFprFile(String name)
   {
      this.loadFprFile = name;
   }


   /**
    * Get the loadFprFile attribute of the FujabaApp object
    * 
    * @return The loadFprFile value
    */
   public String getLoadFprFile()
   {
      return this.loadFprFile;
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean showAboutBox = true;


   /**
    * Sets the showAboutBox attribute of the FujabaApp object
    * 
    * @param flag The new showAboutBox value
    */
   public void setShowAboutBox(boolean flag)
   {
      this.showAboutBox = flag;
   }


   /**
    * Get the showAboutBox attribute of the FujabaApp object
    * 
    * @return The showAboutBox value
    */
   public boolean isShowAboutBox()
   {
      return this.showAboutBox;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean invisible = false;


   /**
    * Sets the invisible attribute of the FujabaApp object
    * 
    * @param flag The new invisible value
    */
   public void setInvisible(boolean flag)
   {
      this.invisible = flag;
   }


   /**
    * Get the invisible attribute of the FujabaApp object
    * 
    * @return The invisible value
    */
   public boolean isInvisible()
   {
      return this.invisible;
   }


   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean maximize = false;


   /**
    * Sets the maximize attribute of the FujabaApp object
    * 
    * @param maximize The new maximize value
    */
   public void setMaximize(boolean maximize)
   {
      this.maximize = maximize;
   }


   /**
    * Get the maximize attribute of the FujabaApp object
    * 
    * @return The maximize value
    */
   public boolean isMaximize()
   {
      return maximize;
   }


   // ######################################################################

   private boolean debugMode = false;


   public void setDebugMode(boolean value)
   {
      this.debugMode = value;
   }


   public boolean isDebugMode()
   {
      return this.debugMode;
   }

   // ######################################################################

   private String desiredPropertyDir = null;


   private void setDesiredPropertyDir(String value)
   {
      this.desiredPropertyDir = value;
   }


   /**
    * <b>main</b> function of fujaba. the following parameters are allowed:
    * 
    * <pre>
    *  -noabout: do not show the fujaba about box.
    *  -invisible: makes frame invisible as soon as possible.
    *  -last: loads the last project.
    *  -debug: turns on the debug mode.
    *  -max: starts fujaba in full screen mode.
    *  -config <dir>: uses <dir> instead of $HOME/fujaba<version> as property dir.
    * </pre>
    * 
    * @param argv Command line parameters passed to the program.
    */
   public static void main(String argv[])
   {
      // initializing log4j. [ak]
      BasicConfigurator.configure();

      if (log.isInfoEnabled())
      {
         log.info("Welcome to Fujaba");
      }
      Runtime.getRuntime().traceMethodCalls(false);

      FujabaApp app = get();
      boolean parameterError = false;

      // Note: store arguments in corresponding attributes...
      // they will be applied later (in 'run()')
      for (int i = 0; i < argv.length && !parameterError; i++)
      {
         try
         {
            if (argv[i].equals("-noabout"))
            {
               app.setShowAboutBox(false);
            }
            else if (argv[i].equals("-invisible"))
            {
               app.setInvisible(true);
            }
            else if (argv[i].equals("-last"))
            {
               // open the last project
               File lastFile = FileHistory.get().firstOfHistory();
               if (log.isInfoEnabled())
               {
                  log.info("-last loads: " + lastFile.getPath());
               }
               app.setLoadFprFile(lastFile.getPath());
            }
            else if (argv[i].equals("-debug"))
            {
               app.setDebugMode(true);
            }
            else if (argv[i].equals("-max"))
            {
               app.setMaximize(true);
            }
            else if (argv[i].equals("-config"))
            {
               app.setDesiredPropertyDir(argv[i + 1]);
               i++;
            }
            else if (((argv[i].endsWith(".fpr"))
                  || ((argv[i].endsWith(".fpr.gz")))
                  || (argv[i].endsWith(".xml")) || (argv[i].endsWith(".xml.gz")))
                  && (app.getLoadFprFile() == null))
            {
               app.setLoadFprFile(argv[i]);
            }
            else
            {
               if (log.isEnabledFor(Priority.ERROR))
               {
                  log.error("Parameter is unknown: " + argv[i]);
               }
               parameterError = true;
            }
         }
         catch (IndexOutOfBoundsException iex)
         {
            if (log.isEnabledFor(Priority.ERROR))
            {
               log.error("Too few arguments for: " + argv[i]);
            }
            parameterError = true;
         }
      }
      if (parameterError)
      {
         if (log.isInfoEnabled())
         {
            log.info("The following parameters are valid:");
            log.info(" -noabout : do not show the fujaba about box.");
            log
                  .info(" -invisible : makes frame invisible as soon as possible.");
            log.info(" -last : loads the last project.");
            log.info(" -debug : turns on the debug mode.");
            log.info(" -max : starts fujaba in full screen mode.");
            log
                  .info(" -config <dir> : uses <dir> instead of $HOME/fujaba<version> as property dir.");
         }

         // quit fujaba
         return;
      }

      Thread.currentThread().setContextClassLoader(
            UPBClassLoader.get(UPBClassLoader.DEFAULT_CLASSLOADER));

      app.run();
   }


   /**
    * @return the associated PluginManager instance
    */
   public static PluginManager getPluginManager()
   {
      return pluginManager;
   }


   /**
    * sets the pluginManager attribute
    * 
    * @param manager
    */
   public static void setPluginManager(PluginManager manager)
   {
      pluginManager = manager;
   }


   /**
    * exit behaviour
    * 
    * @return true if Fujaba calls {@link System#exit} on exit (default)
    */
   public static boolean isAllowedToCallSystemExit()
   {
      return allowedToCallSystemExit;
   }


   /**
    * change the exit behaviour
    * 
    * @param allowedToCallSystemExit true if Fujaba should call {@link System#exit} on exit, false
    *           if not
    */
   public static void setAllowedToCallSystemExit(boolean allowedToCallSystemExit)
   {
      FujabaApp.allowedToCallSystemExit = allowedToCallSystemExit;
   }


   /**
    * true if Fujaba calls {@link System#exit} on exit
    */
   private static boolean allowedToCallSystemExit = true;


   /**
    * Calls System.exit or hides FrameMain
    * 
    * @param status exit code for {@link System#exit}
    * @see #isAllowedToCallSystemExit
    */
   public static void exit(int status)
   {
      if (isAllowedToCallSystemExit())
      {
         System.exit(status);
      }
      else
      {
         FrameMain.get().dispose();
      }
   }
}

/*
 * $Log: FujabaApp.java,v $
 * Revision 1.126.2.7  2006/04/03 14:37:33  lowende
 * Patch from HEAD: property dir will be saved dependent from Fujaba's installation dir.
 * Revision 1.126.2.6 2006/03/12 22:07:24 fklar now every
 * fujaba-installation can specify its preference-directory (as discussed in 'Fujaba-developers
 * Digest, Vol 17, Issue 1', 07/2005)
 * 
 */
