/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.beans.PropertyVetoException;
import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.*;

import javax.swing.*;
import javax.swing.border.BevelBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;

import org.apache.log4j.Logger;

import de.tu_bs.coobra.LocalRepository;
import de.uni_kassel.prop.*;
import de.uni_paderborn.fujaba.app.action.AppCloser;
import de.uni_paderborn.fujaba.asg.*;
import de.uni_paderborn.fujaba.basic.*;
import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.coobra.actions.RestoreAction;
import de.uni_paderborn.fujaba.coobra.actions.ShowPropertyEditorAction;
import de.uni_paderborn.fujaba.fsa.*;
import de.uni_paderborn.fujaba.fsa.swing.JDiagramRootPane;
import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.uni_paderborn.fujaba.fsa.unparse.UnparseManager;
import de.uni_paderborn.fujaba.gui.EditModeFactory;
import de.uni_paderborn.fujaba.gui.mdi.DesktopMenu;
import de.uni_paderborn.fujaba.gui.mdi.DesktopTaskBar;
import de.uni_paderborn.fujaba.messages.*;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.uni_paderborn.fujaba.preferences.PlugInsPreferences;
import de.uni_paderborn.fujaba.uml.*;
import de.uni_paderborn.lib.basic.ImageResourceManager;
import de.uni_paderborn.lib.classloader.UPBClassLoader;
import de.upb.lib.plugins.PluginManager;
import de.upb.lib.userinterface.ActionExecutionListenerEx;
import de.upb.lib.userinterface.UserInterfaceManager;
import de.upb.tools.fca.*;


/**
 * The main frame of Fujaba, in which all components are integrated.
 * <p/>
 * <h2>Associations</h2>
 * <p/>
 * <pre>
 *                  0..1   tabbedPaneProxy    0..1
 * TabbedPaneProxy -------------------------------- FrameMain
 *                  tabbedPaneProxy      frameMain
 * <p/>
 *                 0..n    decorators    0..1
 * FrameDecorator ---------------------------- FrameMain
 *                 decorators       frameMain
 * </pre>
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.609.2.15 $ $Date: 2006/07/10 08:05:34 $
 */
public class FrameMain extends JFrame
{
   /**
    * log4j logging
    */
   final static transient Logger log = Logger.getLogger (FrameMain.class);

   /**
    * Map from ASGDiagram to Point to store scrollbar positions per diagram.
    */
   private FHashMap scrollBarPositions = new FHashMap();

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JLabel statusLabel;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JLabel memoryLabel;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JSplitPane splitpane;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLProject umlProject;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JFileChooser fileChooser;

   /**
    * Directory of the plugins loaded by Fujaba and DOBS
    *
    * @deprecated   Use PluginsPreferences.getPluginDirs ()
    */
   public final static File PLUGIN_DIR = new File ((String) PlugInsPreferences.get().getPluginFolders().firstElement());
   /**
    * horizontal split pane on the left of the main frame
    */
   private JSplitPane leftSplitPane;
   /**
    * property editor (bottom left)
    */
   private JPropertyTable propertyEditor;
   /**
    * container for the property editor
    */
   private JPanel propertyEditorContainer;
   /**
    * caption for the property editor (shows currently selected object's class)
    */
   JLabel propertyEditorCaption;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JPanel messagePanel;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private MessageView messageView = new MessageView ("Messages");
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JSplitPane splitMessagesAndDesktop;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int MAX_DISPLAYED_STATUS_MESSAGES = 10;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String MESSAGE_CLASS_STATUS = "Status";


   /**
    * Get the editMode attribute of the FrameMain object
    *
    * @return   The editMode value
    */
   public EditModeFactory getEditMode()
   {
      return editModeFactory;
   }


   /**
    * Sets the editMode attribute of the FrameMain object
    *
    * @param editModeFactory  The new editMode value
    */
   public void setEditMode (EditModeFactory editModeFactory)
   {
      this.editModeFactory = editModeFactory;
   }


   /**
    * for (de)activate alternate editing modes
    */
   private EditModeFactory editModeFactory;

   /**
    * The singleton instance.
    */
   private static volatile FrameMain frame;


   /**
    * Initializes FrameMain
    */
   private FrameMain()
   {
      JOptionPane.setRootFrame (this);
      setTitle ("Fujaba 4");
      setDefaultCloseOperation (WindowConstants.DO_NOTHING_ON_CLOSE);
      setBackground (SystemColor.control);
      URL url = FrameMain.class.getResource ("images/Fujaba.gif");
      Image icon = ImageResourceManager.get().getImage (url);
      setIconImage (icon);
      addWindowListener (new AppCloser());

      // CoObRA changes
      filterFujabaProject.addExtension (".cxr");
      filterFujabaProject.addExtension (".cxri");
      filterFujabaProject.addExtension (".cxr.gz");

      // add xml and xml.gz to Fujaba project files.
      filterFujabaProject.addExtension (".xml");
      filterFujabaProject.addExtension (".xml.gz");
   }


   /**
    * Returns the singleton FrameMain.
    *
    * @return   No description provided
    */
   public static FrameMain get()
   {
      if (frame == null)
      {
         frame = new FrameMain();
      }

      return frame;
   }


   /**
    * Default view for messages.
    *
    * @return   message view
    */
   public MessageView getMessageView()
   {
      return messageView;
   }


   /**
    * Show the message view.
    */
   public void showMessageView()
   {
      if (splitMessagesAndDesktop.getDividerLocation() >= splitMessagesAndDesktop.getHeight() - 20)
      {
         splitMessagesAndDesktop.setDividerLocation (splitMessagesAndDesktop.getResizeWeight());
      }
   }


   /**
    * Hide the message view.
    */
   public void hideMessageView()
   {
      splitMessagesAndDesktop.setDividerLocation (Integer.MAX_VALUE);
   }


   /**
    * initializes FrameMain
    */
   public void init()
   {
      Container content = getContentPane();
      content.setLayout (new BorderLayout());

      JPanel mainPanel = new JPanel();
      mainPanel.setLayout (new BorderLayout());

      content.add (mainPanel, BorderLayout.CENTER);

      // initialize the project
      umlProject = UMLProject.createPlainProject();
      umlProject.setGui (this);

      // disallow undo in new project
      if (FujabaChangeManager.getVMRepository() != null)
      {
         FujabaChangeManager.getVMRepository().moveChangesIntoChangesBase();
      }

      // initialize plugin manager
      PluginManager pluginManager = FujabaApp.getPluginManager();

      // add the core definitions
      UserInterfaceManager uiManager = UserInterfaceManager.get();

      // adding the interface definitions
      uiManager.addToDocuments (UPBClassLoader.DEFAULT_CLASSLOADER, "de/uni_paderborn/fujaba/app/CoreUserInterface.xml");
      uiManager.addToDocuments (UPBClassLoader.DEFAULT_CLASSLOADER, "de/uni_paderborn/fujaba/uml/gui/UMLUserInterface.xml");

      if (GeneralPreferences.get().isRepositoryActivated())
      {
         uiManager.addToDocuments (UPBClassLoader.DEFAULT_CLASSLOADER, "de/uni_paderborn/fujaba/coobra/CoobraUserInterface.xml");
      }
      uiManager.addToDocuments (UPBClassLoader.DEFAULT_CLASSLOADER, "de/uni_paderborn/fujaba/packagediagrams/gui/PackageUserInterface.xml");

      // load the plugins
      pluginManager.scanPlugins (PlugInsPreferences.get().getPluginFolders());
      // scan in classpath jars
      Vector toScan = getResourcesFromJars ("fujabaPlugin.xml");
      toScan.addAll (getResourcesFromJars ("plugin.xml"));
      pluginManager.scanPlugins (toScan);

      pluginManager.loadPlugins();

      // loading the interface definitions
      uiManager.loadDocuments();

      if (!GeneralPreferences.get().isFPRActivated() && GeneralPreferences.get().isRepositoryActivated())
      {
         uiManager.getFromActions ("saveProject").putValue ("visible", Boolean.FALSE);
         uiManager.getFromActions ("saveProjectAs").putValue ("visible", Boolean.FALSE);
         setSaveAction (UserInterfaceManager.get().getFromActions ("coobra.store"));
      }

      uiManager.setEventSource (SelectionManager.get());

      setJMenuBar (uiManager.getFromMenuBars ("mainMenuBar"));

      //Actions for JMenuBarItems can have initializers (interface ElementInitializer) too
      //Now it's time to call them. OpenRecentProject Action uses ElementInitializer to
      //prepare menu items for FileHistory. >trinet
      uiManager.initMenu (getJMenuBar(), this);
      //Now we can update fileHistory JMenuItems...
      FileHistory.get().updateActions();

      setTabbedPaneProxy (new TabbedPaneProxy());
      getTabbedPaneProxy().instanceOfTabProxy (TabbedPaneProxy.TAB_PRJ);
      getTabbedPaneProxy().update();

      addToDecorators (new de.uni_paderborn.fujaba.uml.gui.FrameDecoratorUMLClassDiagram());
      addToDecorators (new de.uni_paderborn.fujaba.uml.gui.FrameDecoratorUMLActivityDiagram());
      addToDecorators (new de.uni_paderborn.fujaba.uml.gui.FrameDecoratorUMLStatechart());
      addToDecorators (new de.uni_paderborn.fujaba.packagediagrams.gui.FrameDecoratorPackageDiagram());

      leftSplitPane = new JSplitPane (JSplitPane.VERTICAL_SPLIT);
      leftSplitPane.setResizeWeight (0.7);
      leftSplitPane.setTopComponent (getTabbedPaneProxy().getTabbedPane());
      final JPanel rightPanel = new JPanel (new BorderLayout());
      rightPanel.add (desktop, BorderLayout.CENTER);
      JPanel rightBottomPanel = new JPanel (new BorderLayout());
      splitMessagesAndDesktop = new JSplitPane (JSplitPane.VERTICAL_SPLIT, rightPanel, rightBottomPanel);
      splitMessagesAndDesktop.setOneTouchExpandable (true);
      splitMessagesAndDesktop.setResizeWeight (0.8);
      hideMessageView();

      messagePanel = new JPanel (new BorderLayout());
      rightBottomPanel.add (messagePanel, BorderLayout.CENTER);
      messagePanel.add (messageView.getComponent(), BorderLayout.CENTER);

      splitpane = createSplitPane (leftSplitPane, splitMessagesAndDesktop);
      getJMenuBar().add (new DesktopMenu (desktop));
      rightPanel.add (new DesktopTaskBar (desktop, true), BorderLayout.SOUTH);

      propertyEditor = new JPropertyTable (true, "FujabaPropertyTable");

      URL url = getClass().getClassLoader().getResource ("de/uni_paderborn/fujaba/gui/propertyeditor.config");
      if (url != null)
      {
         try
         {
            propertyConfiguration.readConfig (url.openStream());
         }
         catch (IOException e)
         {
            log.error ("error while reading propertyeditor.config", e);
         }
      }
      else
      {
         log.error ("propertyeditor.config file not found");
      }

      JScrollPane scrollPE = new JScrollPane (propertyEditor);
      propertyEditorContainer = new JPanel();
      propertyEditorContainer.setLayout (new BorderLayout());
      propertyEditorContainer.add (scrollPE, BorderLayout.CENTER);
      propertyEditorCaption = new JLabel();
      propertyEditorContainer.add (propertyEditorCaption, BorderLayout.NORTH);
      showPropertyEditor (ShowPropertyEditorAction.isPropertyEditorActivated());

      SelectionManager.get().addToPopupSourceListeners (new PropertyEditorPopupSourceListener());

      JToolBar toolBar = UserInterfaceManager.get().getFromToolBars ("mainToolBar");
      toolBar.setVisible (true);
      desktop.setBackground (toolBar.getBackground());

      mainPanel.add (toolBar, BorderLayout.NORTH);
      mainPanel.add (splitpane, BorderLayout.CENTER);
      mainPanel.add (createStatusbar(), BorderLayout.SOUTH);

      StatusBarUpdater statusBarUpdater = new StatusBarUpdater();
      statusBarUpdater.start();

      filterFujabaProject.addExtension (".fpr.gz");
      filterFujabaProject.addExtension (".bak.fpr");

      centerFrame();
      UMLProject.get().setSaved (true);

      subscribeActionErrorListener();
   }


   /**
    * Get the resourcesFromJars attribute of the FrameMain object
    *
    * @param resName  No description provided
    * @return         The resourcesFromJars value
    */
   private Vector getResourcesFromJars (String resName)
   {
      Vector res = new Vector();

      try
      {
         Enumeration pluginXMLs = FrameMain.class.getClassLoader().getResources (resName);
         while (pluginXMLs.hasMoreElements())
         {
            URL url = (URL) pluginXMLs.nextElement();
            String path = url.toExternalForm();
            if (path.startsWith ("jar:"))
            {
               int indexOfMark = path.lastIndexOf ("!/");
               if (indexOfMark < 0)
               {
                  indexOfMark = path.length();
               }
               path = path.substring ("jar:".length(), indexOfMark);
            }
            if (path.startsWith ("file:"))
            {
               path = URLDecoder.decode (path.substring ("file:".length()), "UTF-8");
               if (path.endsWith (resName))
               {
                  path = path.substring (0, path.length() - resName.length());
               }
               res.add (path);
            }
         }
      }
      catch (IOException e)
      {
         e.printStackTrace();
      }

      return res;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void subscribeActionErrorListener()
   {
      UserInterfaceManager.get().addExecutionListener (
         new ActionExecutionListenerEx()
         {
            /**
             * This method will be called before actionPerformed() is executed.
             *
             * @param e  The action-defined event.
             */
            public void preActionNotify (ActionEvent e)
            {
            }


            /**
             * This method will be called after actionPerformed was executed if no exception was thrown.
             *
             * @param e  The action-defined event.
             */
            public void postActionNotify (ActionEvent e)
            {
            }


            /**
             * This method will be called after actionPerformed was executed if an exception/throwable was thrown.
             *
             * @param e          The action-defined event.
             * @param throwable  the exception/error that occured
             * @return           true if the error was handled (though each listener will be notified)
             */
            public boolean postActionNotify (ActionEvent e, Throwable throwable)
            {
               throwable.printStackTrace();
               String text = throwable.getClass().getName().indexOf ("RuntimeException") < 0 ? throwable.toString() : throwable.getMessage();
               ErrorMessage errorMessage = new ErrorMessage (text);
               if (throwable instanceof RuntimeExceptionWithContext)
               {
                  RuntimeExceptionWithContext ex = (RuntimeExceptionWithContext) throwable;
                  errorMessage.addToContext (ex.getContext());
               }
               getMessageView().addToMessages (errorMessage);
               showMessageView();
               errorMessage.showContext();
               return true;
            }
         });
   }


   // *********************** Various file filters *******************************

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private SchemaFilter filterFujabaProject = new SchemaFilter (".fpr", "Fujaba Project Files");


   /**
    * Get the filterFujabaProject attribute of the FrameMain object value.
    *
    * @return   The filterFujabaProject value
    */
   public SchemaFilter getFilterFujabaProject()
   {
      return filterFujabaProject;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private SchemaFilter filterJavaFiles = new SchemaFilter (".java", "Java Source Files (*.java)");


   /**
    * Get the filterJavaFiles attribute of the FrameMain object value
    *
    * @return   The filterJavaFiles value
    */
   public SchemaFilter getFilterJavaFiles()
   {
      return filterJavaFiles;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private SchemaFilter filterJarFiles = new SchemaFilter (".jar", "Java Archives (*.jar)");


   /**
    * Get the filterJarFiles attribute of the FrameMain object value
    *
    * @return   The filterJarFiles value
    */
   public SchemaFilter getFilterJarFiles()
   {
      return filterJarFiles;
   }


   /**
    * Get the fileChooser attribute of the FrameMain object.
    *
    * @return   The fileChooser value
    */
   public JFileChooser getFileChooser()
   {
      if (fileChooser == null)
      {
         fileChooser = new JFileChooser();
         File dir = new File (GeneralPreferences.get().getWorkspaceFolder());
         if (dir.isDirectory())
         {
            fileChooser.setCurrentDirectory (dir);
         }
         else
         {
            log.error ("Workspace folder not found: " + GeneralPreferences.get().getWorkspaceFolder());
         }
      }
      return fileChooser;
   }


   // *********************** Support for multiple frames *******************************

   /**
    * container for childwindows
    */
   JDesktopPane desktop = new JDesktopPane();

   /**
    * allows Fujaba to specify own popups in property editor
    */
   private static PropertyConfigurationAdapter propertyConfiguration;


   /**
    * Desktop pane for custom child windows.
    *
    * @return   the desktop pane
    */
   public JDesktopPane getDesktop()
   {
      return desktop;
   }


   /**
    * size of an inner diagram frame after creation
    */
   public static Dimension DEFAULT_INTERNAL_FRAME_SIZE = new Dimension (400, 300);

   /**
    * true while executing {@link #showDiagram(de.uni_paderborn.fujaba.asg.ASGDiagram)}
    */
   private boolean inShowDiagram;


   /**
    * Get the inShowDiagram attribute of the FrameMain object
    *
    * @return   The inShowDiagram value
    */
   public boolean isInShowDiagram()
   {
      return this.inShowDiagram;
   }


   /**
    * Switch the view to a specific diagram
    *
    * @param diag  what to view and select on the desktop
    */
   public void showDiagram (ASGDiagram diag)
   {
      if (!inShowDiagram)
      {
         inShowDiagram = true;
         try
         {
            final InternalFrame diagFrame = diag != null ? getInternalFrame (diag) : null;
            final ASGDiagram currentDiagram = UMLProject.get().getCurrentDiagram();

            if (diag != currentDiagram)
            {
               doEverythingOnExit (currentDiagram);

               SelectionManager.get().clear();

               // set the new current diagram
               UMLProject.get().setCurrentDiagram (diag);

               if (diag != null)
               {

                  // UMLDiagram uses JScrollPane
                  if (diagFrame.getDiagramComponent() != diagFrame.getScrollerPanel())
                  {
                     diagFrame.setDiagramComponent (diagFrame.getScrollerPanel());
                  }

                  diagFrame.getScrollerPanel().setViewportView (diagFrame.getDiagramRootPane());

                  FSAContainer pane = (FSAContainer) UnparseManager.get().unparse (diag,
                     diag.getFromFsaObjects ("@__null.entry"));
                  SelectionManager.get().setSelected (diag.getFirstFromFSAObjects(), true);

                  // Not every tree item has a diagram!
                  if ( (pane != null) &&  (pane.getJComponent() != null))
                  {
                     JDiagramRootPane rootPane = diagFrame.getDiagramRootPane();
                     rootPane.setContentPane (pane.getJComponent());
                     pane.getJComponent().setVisible (true);
                     rootPane.updateUI();
                  }
                  activateFrame (diagFrame);

                  doEverythingOnEntry (diag);
               }

               selectTreeItem (diag);
            }
         }
         finally
         {
            inShowDiagram = false;
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param frame  No description provided
    */
   private void activateFrame (InternalFrame frame)
   {
      if (frame != null)
      {
         try
         {
            frame.setVisible (true);
            frame.setSelected (true);
            if (frame.isIcon())
            {
               frame.setMaximum (true);
            }
         }
         catch (PropertyVetoException e)
         {
            //ok, if someone is offended - leave it
         }
      }
   }


   /**
    * map ASGDiagram -> InternalFrame
    */
   Map frames;


   /**
    * Find or create an internal frame for a diagram.
    *
    * @param diagram  what to find a frame for
    * @return         the InternalFrame that shows the diagram (never null)
    */
   public InternalFrame getInternalFrame (ASGDiagram diagram)
   {
      if (diagram == null)
      {
         throw new NullPointerException ("diagram cannot be null");
      }
      InternalFrame frame = frames != null ? (InternalFrame) frames.get (diagram) : null;
      if (frame == null)
      {
         final InternalFrame newFrame = new InternalFrame (diagram, this);
         frame = newFrame;
         frame.setVisible (true);
         frame.setSize (DEFAULT_INTERNAL_FRAME_SIZE);

         desktop.add (frame);
         if (frames == null)
         {
            frames = new FHashMap();
         }
         frames.put (diagram, frame);

         //activate or deactivate alternate editing modes (true/false)
         //has to happen after put, because activate calls getInternalFrame
         if (getEditMode() != null)
         {
            getEditMode().activate (diagram);
         }

         frame.pack();
         SwingUtilities.invokeLater (
            new Runnable()
            {
               /**
                * @see   Thread#run()
                */
               public void run()
               {
                  try
                  {
                     newFrame.setMaximum (true);
                  }
                  catch (PropertyVetoException e)
                  {
                     //ok, then leave it
                  }
               }
            });
      }
      return frame;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final InternalFrame frameDummy = new InternalFrame (null, this);


   /**
    * Currently selected internal frame.
    *
    * @return   currently selected internal frame
    */
   public InternalFrame getCurrentInternalFrame()
   {
      final JInternalFrame selectedFrame = desktop.getSelectedFrame();
      if (selectedFrame instanceof InternalFrame)
      {
         return (InternalFrame) selectedFrame;
      }
      else
      {
         return frameDummy;
      }
   }


   /**
    * close all internal frames (including diagrams)
    */
   public void closeAllInternalFrames()
   {
      closeAllDiagrams();

      final Component[] components = desktop.getComponents();
      for (int i = 0; i < components.length; i++)
      {
         Component component = components[i];
         if (component instanceof JInternalFrame)
         {
             ((JInternalFrame) component).dispose();
            desktop.remove (component);
         }
      }
   }


   /**
    * Close the internal frame of the diagram if one exists.
    * Do nothing otherwise.
    *
    * @param diag  The ASG diagram to be closed
    */
   public void closeInternalFrameOfDiag (ASGDiagram diag)
   {
      if (UMLProject.get().getCurrentDiagram() == diag)
      {
         selectTreeItem (null);
         UMLProject.get().setCurrentDiagram (null);
      }
      if (frames != null)
      {
         InternalFrame iFrame = (InternalFrame) frames.get (diag);
         if (iFrame != null)
         {
            final Component[] components = desktop.getComponents();
            for (int i = 0; i < components.length; i++)
            {
               Component component = components[i];
               if (component == iFrame)
               {
                  frames.remove (diag);
                  desktop.remove (iFrame);
                  iFrame.dispose();
               }
            }
         }
      }

   }


   /**
    * close all diagrams
    */
   public void closeAllDiagrams()
   {
      selectTreeItem (null);
      UMLProject.get().setCurrentDiagram (null);
      if (frames != null)
      {
         for (Iterator iterator = frames.values().iterator(); iterator.hasNext(); )
         {
            InternalFrame frame = (InternalFrame) iterator.next();
            frame.dispose();
         }
      }
   }


   // *********************** Support for property editor *******************************

   /**
    * Get the propertyEditor attribute of the FrameMain object value
    *
    * @return   The propertyEditor
    */
   public JPropertyTable getPropertyEditor()
   {
      return propertyEditor;
   }


   /**
    * show the current popup source in the property editor
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.609.2.15 $ $Date: 2006/07/10 08:05:34 $
    */
   private class PropertyEditorPopupSourceListener implements PopupSourceListener
   {
      /**
       * Called whenever the popupSource changes.
       *
       * @param newSource  new source
       */
      public void popupSourceChanged (LogicUnparseInterface newSource)
      {
         if (!getPropertyEditor().isPopupMenuVisible())
         {
            getPropertyEditor().removeAllFromShownObjects();
            if (newSource instanceof Iterator)
            {
               getPropertyEditor().addToShownObjects ((Iterator) newSource);
            }
            else
            {
               getPropertyEditor().addToShownObjects (newSource);
               propertyEditorCaption.setText (simpleClassName (newSource));
            }
         }
      }


      /**
       * @param element  a logic element
       * @return         name of the class of the logic element (unqualified), null if element
       *         is null
       */
      private String simpleClassName (LogicUnparseInterface element)
      {
         if (element != null)
         {
            String className = element.getClass().getName();
            if (className.lastIndexOf ('.') > 0)
            {
               className = className.substring (className.lastIndexOf ('.') + 1, className.length());
            }
            return className;
         }

         return "";
      }
   }

   static
   {
      // allow Fujaba to specify own popups in property editor
      propertyConfiguration =
         new PropertyConfigurationAdapter()
         {
            /**
             * query a popup menu for an object.
             *
             * @param object  subject of the popup menu
             * @return        the popup menu that should be displayed
             */
            public JPopupMenu getPopupMenu (Object object)
            {
               if (object instanceof FElement)
               {
                  if (object instanceof ASGElement)
                  {
                     SelectionManager.get().setPopupSource (null, (ASGElement) object);
                  }
                  return UserInterfaceManager.get().getFromPopupMenus (object.getClass().getName(), object);
               }

               return null;
            }
         };
      ObjectInspector.get().setConfig (propertyConfiguration);
   }


   /**
    * shows/hides the property editor
    *
    * @param show  when true the property editor is shown (otherwise hidden)
    */
   public void showPropertyEditor (boolean show)
   {
      leftSplitPane.setBottomComponent (show ? propertyEditorContainer : null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void writePropertyEditorConfig()
   {
      URL url = getClass().getResource ("../gui/propertyeditor.config");
      if (url != null)
      {
         try
         {
            final FileOutputStream out = new FileOutputStream (URLDecoder.decode (url.getFile(), "UTF-8"));
            propertyConfiguration.writeConfig (out);
            out.close();
         }
         catch (IOException e)
         {
            log.error ("error while writing propertyeditor.config", e);
         }
      }
      else
      {
         log.error ("propertyeditor.config file not found");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void centerFrame()
   {
      // get size of the primary display
      Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();

      // get insets of the primary display (e.g. space that Taskbar requires)
      GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
      GraphicsConfiguration defaultGC = ge.getDefaultScreenDevice().getDefaultConfiguration();
      Insets screenInsets = Toolkit.getDefaultToolkit().getScreenInsets (defaultGC);

      // adjust screenSize to size that is visible
      Dimension visibleScreenSize = new Dimension (screenSize.width -  (screenInsets.left + screenInsets.right), screenSize.height -  (screenInsets.top + screenInsets.bottom));

      Dimension dim = new Dimension();

      if (FujabaApp.get().isMaximize())
      {
         dim.width = visibleScreenSize.width;
         dim.height = visibleScreenSize.height;
      }
      else
      {
         dim.width = Math.min (1024, visibleScreenSize.width * 9 / 10);
         dim.height = Math.min (768, visibleScreenSize.height * 9 / 10);
      }

      pack();
      setSize (dim);

      // adjust location
      int newLocationX = visibleScreenSize.width / 2 - dim.width / 2 + screenInsets.left;
      int newLocationY = visibleScreenSize.height / 2 - dim.height / 2 + screenInsets.top;
      setLocation (newLocationX, newLocationY);

      setCursor (Cursor.getPredefinedCursor (Cursor.WAIT_CURSOR));
   } // centerFrame


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void show()
   {
      setCursor (Cursor.getPredefinedCursor (Cursor.DEFAULT_CURSOR));
      super.show();
   }


   /**
    * <pre>
    *                  0..1   tabbedPaneProxy    0..1
    * TabbedPaneProxy -------------------------------- FrameMain
    *                  tabbedPaneProxy      frameMain
    * </pre>
    */
   private TabbedPaneProxy tabbedPaneProxy;


   /**
    * Sets the tabbedPaneProxy attribute of the FrameMain object
    *
    * @param value  The new tabbedPaneProxy value
    * @return       No description provided
    */
   public boolean setTabbedPaneProxy (TabbedPaneProxy value)
   {
      boolean changed = false;
      if (this.tabbedPaneProxy != value)
      {
         if (this.tabbedPaneProxy != null)
         {
            TabbedPaneProxy oldValue = this.tabbedPaneProxy;
            this.tabbedPaneProxy = null;
            oldValue.setFrameMain (null);
         }
         this.tabbedPaneProxy = value;
         if (value != null)
         {
            value.setFrameMain (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * Get the tabbedPaneProxy attribute of the FrameMain object
    *
    * @return   The tabbedPaneProxy value
    */
   public TabbedPaneProxy getTabbedPaneProxy()
   {
      return this.tabbedPaneProxy;
   }


   /**
    * <pre>
    *                 0..n    decorators   0..1
    * FrameDecorator --------------------------- FrameMain
    *                 decorators      frameMain
    * </pre>
    */
   private FHashSet decorators;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInDecorators (FrameDecorator value)
   {
      return  ( (this.decorators != null) &&
          (value != null) &&
         this.decorators.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfDecorators()
   {
      return  ( (this.decorators == null)
         ? FEmptyIterator.get()
         : this.decorators.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfDecorators()
   {
      return  ( (this.decorators == null)
         ? 0
         : this.decorators.size());
   }


   /**
    * Access method for an one to n association. @@param value The object added.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToDecorators (FrameDecorator value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.decorators == null)
         {
            this.decorators = new FHashSet();
         }
         changed = this.decorators.add (value);
         if (changed)
         {
            value.setFrameMain (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromDecorators (FrameDecorator value)
   {
      boolean changed = false;
      if ( (this.decorators != null) &&  (value != null))
      {
         changed = this.decorators.remove (value);
         if (changed)
         {
            value.setFrameMain (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromDecorators()
   {
      FrameDecorator tmpValue;
      Iterator iter = this.iteratorOfDecorators();
      while (iter.hasNext())
      {
         tmpValue = (FrameDecorator) iter.next();
         this.removeFromDecorators (tmpValue);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      TabbedPaneProxy tmpTabbedPaneProxy = getTabbedPaneProxy();
      if (tmpTabbedPaneProxy != null)
      {
         setTabbedPaneProxy (null);
      }

      removeAllFromDecorators();
   }

   // ------------------------------------------------------------------------

   /**
    * Set a new UMLProject and reset the tree and other stuff.
    *
    * @param project  The new uMLProject value
    */
   public void setUMLProject (UMLProject project)
   {
      if (umlProject != project)
      { // new partner

         if (this.umlProject != null)
         { // inform old partner

            UMLProject oldUMLProject = this.umlProject;
            this.umlProject = null;
            oldUMLProject.setGui (null);
         }
         this.umlProject = project;
         if (project != null)
         { // inform new partner

            umlProject.setGui (this);
         }

         if (getTabbedPaneProxy() != null)
         {
            getTabbedPaneProxy().update();
         }
         showTitle();
      }
   }


   /**
    * Thread for updating the status bar.
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.609.2.15 $ $Date: 2006/07/10 08:05:34 $
    */
   private class DoUpdateStatusBar implements Runnable
   {
      /**
       * Main processing method for the DoUpdateStatusBar object
       */
      public void run()
      {
         setMemoryLabel();
      }
   }


   /**
    * Updates the text of the memory in the status bar.
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.609.2.15 $ $Date: 2006/07/10 08:05:34 $
    */
   private class StatusBarUpdater extends Thread
   {
      /**
       * Constructor for class StatusBarUpdater
       */
      public StatusBarUpdater()
      {
         this.setName (this.getClass().getName());
         try
         {
            setPriority (Thread.MIN_PRIORITY);
         }
         catch (Exception e)
         {
            //ok, don't lower the priority
            if (log.isInfoEnabled())
            {
               log.info ("Status update thread priority not lowered!");
            }
         }
      }


      /**
       * Main processing method for the StatusBarUpdater object
       */
      public void run()
      {
         DoUpdateStatusBar doUpdateStatusBar = new DoUpdateStatusBar();

         while (true)
         {
            SwingUtilities.invokeLater (doUpdateStatusBar);
            try
            {
               sleep (10000);
            }
            catch (InterruptedException except)
            {
            }
         }
      }
   }


   /**
    * Creates a status bar.
    *
    * @return   No description provided
    */
   public JPanel createStatusbar()
   {
      JPanel statusPanel = new JPanel();
      statusPanel.setBorder (new EmptyBorder (new Insets (3, 8, 3, 8)));

      GridBagLayout gridBag = new GridBagLayout();
      GridBagConstraints constraints = new GridBagConstraints();

      statusPanel.setLayout (gridBag);

      statusLabel = new JLabel();
      statusLabel.setBorder (new BevelBorder (BevelBorder.LOWERED));
      statusLabel.setFont (new Font ("SansSerif", 0, 10));
      constraints.gridwidth = GridBagConstraints.RELATIVE;
      constraints.fill = GridBagConstraints.BOTH;
      constraints.weightx = 1.0;
      gridBag.setConstraints (statusLabel, constraints);
      statusPanel.add (statusLabel);
      setStatusLabel ("Welcome to Fujaba Tool Suite!");

      memoryLabel = new JLabel();
      memoryLabel.setBorder (new BevelBorder (BevelBorder.LOWERED));
      memoryLabel.setFont (new Font ("SansSerif", 0, 10));
      memoryLabel.setToolTipText ("Memory Usage/Allocation");
      setMemoryLabel();
      constraints.gridwidth = 1;
      constraints.weightx = 0.0;
      gridBag.setConstraints (memoryLabel, constraints);
      statusPanel.add (memoryLabel);

      return statusPanel;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FLinkedList displayedStatusMessages = new FLinkedList();


   /**
    * Sets the current text of the status bar.
    *
    * @param text  The new statusLabel value
    */
   public void setStatusLabel (final String text)
   {
      if (statusLabel != null)
      {
         SwingUtilities.invokeLater (
            new Runnable()
            {
               public void run()
               {
                  statusLabel.setText (" " + text + " ");
                  statusLabel.repaint();
               }
            });
         if (text != null && !"".equals (text))
         {
            Message message = new Message (text);
            if (displayedStatusMessages.size() > MAX_DISPLAYED_STATUS_MESSAGES)
            {
               getMessageView().removeFromMessages ((Message) displayedStatusMessages.removeFirst());
            }
            message.setMessageCategory (MESSAGE_CLASS_STATUS);
            displayedStatusMessages.add (message);
            getMessageView().addToMessages (message);
         }
      }
   }


   /**
    * Convenience method to show a standard error text.
    *
    * @param text     error message text
    * @param context  what to highligh if message is clicked
    * @return         the nely created message (already added to message view)
    */
   public Message showError (String text, ASGElement context)
   {
      ErrorMessage message = new ErrorMessage (text);
      message.addToContext (context);
      message.setMessageCategory (MESSAGE_CLASS_STATUS);
      getMessageView().addToMessages (message);
      showMessageView();
      return message;
   }


   /**
    * Get the kiloByte attribute of the FrameMain object
    *
    * @param memory  No description provided
    * @return        The kiloByte value
    */
   private String getKiloByte (long memory)
   {
      String value;
      long mem = memory / 1024;

      if (mem >= 1024)
      {
         mem /= 1024;
         value = " MByte";
      }
      else
      {
         value = " KByte";
      }

      value = mem + value;

      return value;
   }


   /**
    * Sets the current free memory.
    */
   public void setMemoryLabel()
   {
      Runtime runtime = Runtime.getRuntime();

      long free = runtime.freeMemory();
      long total = runtime.totalMemory();

      String text = getKiloByte (total - free) + " of " + getKiloByte (total) + " allocated";

      memoryLabel.setText (" " + text + " ");
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diag  No description provided
    */
   public void doEverythingOnEntry (ASGDiagram diag)
   {
      doEverythingOnEntry (diag, true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diag              No description provided
    * @param saveScrollBarPos  No description provided
    */
   public void doEverythingOnEntry (ASGDiagram diag, boolean saveScrollBarPos)
   {
      Iterator iter = iteratorOfDecorators();
      while (iter.hasNext())
      {
         FrameDecorator decorator = (FrameDecorator) iter.next();
         decorator.enterDiagram (diag);
      }
      if (saveScrollBarPos && diag != null)
      {
         final InternalFrame diagFrame = getInternalFrame (diag);
         JScrollPane pane = diagFrame.getScrollerPanel();
         Point pos = (Point) scrollBarPositions.get (diag);
         if (pane != null)
         {
            JScrollBar bar = pane.getHorizontalScrollBar();
            if (bar != null)
            {
               if (pos != null)
               {
                  bar.setValue ((int) pos.getX());
               }
               else
               {
                  bar.setValue (0);
               }
            }
            bar = pane.getVerticalScrollBar();
            if (bar != null)
            {
               if (pos != null)
               {
                  bar.setValue ((int) pos.getY());
               }
               else
               {
                  bar.setValue (0);
               }
            }
         }
      }
   }


   /**
    * @return       the current diagram root
    * @see          de.uni_paderborn.fujaba.app.InternalFrame#getDiagramRoot()
    * @deprecated   please use getInternalFrame(yourdiagram).getDiagramRoot()
    */
   public FSALayeredPane getDiagramRoot()
   {
      return getCurrentInternalFrame().getDiagramRoot();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diag  No description provided
    */
   public void doEverythingOnExit (ASGDiagram diag)
   {
      doEverythingOnExit (diag, true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diag              No description provided
    * @param saveScrollBarPos  No description provided
    */
   public void doEverythingOnExit (ASGDiagram diag, boolean saveScrollBarPos)
   {
      Iterator iter = iteratorOfDecorators();
      while (iter.hasNext())
      {
         FrameDecorator decorator = (FrameDecorator) iter.next();
         decorator.leaveDiagram (diag);
      }
      if (saveScrollBarPos && diag != null)
      {
         JScrollPane pane = getInternalFrame (diag).getScrollerPanel();
         if (pane != null)
         {
            int scrollX = 0;
            int scrollY = 0;
            JScrollBar bar = pane.getHorizontalScrollBar();
            if (bar != null)
            {
               scrollX = bar.getValue();
            }
            bar = pane.getVerticalScrollBar();
            if (bar != null)
            {
               scrollY = bar.getValue();
            }
            scrollBarPositions.put (diag, new Point (scrollX, scrollY));
         }
      }
      UMLProject.get().refreshDisplay();
   }


   /**
    * Creates a horizontal JSplitPane.
    *
    * @param westComp  No description provided
    * @param eastComp  No description provided
    * @return          No description provided
    */
   protected JSplitPane createSplitPane (Component westComp, Component eastComp)
   {
      JSplitPane pane = new JSplitPane (JSplitPane.HORIZONTAL_SPLIT, westComp, eastComp);
      //pane.setContinuousLayout(true);
      pane.setOneTouchExpandable (true);
      pane.setPreferredSize (new Dimension (400, 300));

      if (pane.getDividerLocation() < 20)
      {
         pane.setDividerLocation (200);
      }

      return pane;
   }


   /**
    * Sets the rightComponent attribute of the FrameMain object
    *
    * @param comp  The new rightComponent value
    */
   protected void setRightComponent (Component comp)
   {
      getSplitPane().setRightComponent (comp);

      getSplitPane().setDividerLocation (200);
      getSplitPane().setLastDividerLocation (200);
   }


   /**
    * Get the rightComponent attribute of the FrameMain object
    *
    * @return   The rightComponent value
    */
   protected Component getRightComponent()
   {
      return getSplitPane().getRightComponent();
   }


   /**
    * Fetch the JSplitPane contained in this JPanel.
    *
    * @return   The splitPane value
    */
   public JSplitPane getSplitPane()
   {
      return splitpane;
   }


   /**
    * Find the hosting frame, for the file-chooser dialog.
    *
    * @return   The frame value
    */
   public JFrame getFrame()
   {
      return this;
   }


   /**
    * Sets the wait cursor for the whole frame.
    */
   public void setCursorWait()
   {
      getFrame().setCursor (Cursor.getPredefinedCursor (Cursor.WAIT_CURSOR));
   }


   /**
    * Sets the default cursor for the whole frame.
    */
   public void setCursorDefault()
   {
      getFrame().setCursor (Cursor.getPredefinedCursor (Cursor.DEFAULT_CURSOR));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param file  No description provided
    */
   public void openFile (File file)
   {
      if (file.exists() && file.isFile())
      {
         try
         {
            closeAllInternalFrames();
            getTabbedPaneProxy().getTabbedPane().setSelectedIndex (-1);
            UMLProject newProject = null;
            // This must not interfere with the setStatusBar thread.
            // Such an interference has had caused very hard headache.
            synchronized (this)
            {
               // distinguish between fpr and xml files
               String fileName = file.getName();

               if (FujabaChangeManager.getVMRepository() != null
                  && !FujabaChangeManager.getVMRepository().isServerRepository()
                  && !FujabaChangeManager.getVMRepository().isEmpty())
               {
                  setUMLProject (null);
                  UMLProject.setNewProject (null);
                  FujabaChangeManager.setVMRepository (new LocalRepository());
               }

               //to support old projects that discern between UMLAttr.umlVisibility and UMLAttr.visibility
               UMLAttr.setSyncVisibilities (false);
               try
               {
                  ASGUnparseInformation.resetPointsCalled();
                  if (fileName.endsWith (".fpr") || fileName.endsWith (".fpr.gz"))
                  {
                     newProject = UMLProject.loadProject (file);
                     //use default save
                     setSaveAction (null);
                  }
                  else if (fileName.endsWith (".cxr") || fileName.endsWith (".cxr.gz") || fileName.endsWith (".cxri"))
                  {
                     new RestoreAction().restoreWithProgressDialog (file.getPath());
                     setSaveAction (UserInterfaceManager.get().getFromActions ("coobra.store"));
                  }
                  else if (fileName.endsWith (".xml") || fileName.endsWith (".xml.gz"))
                  {
                     // open using XMLReflect
                     AbstractAction restoreAction = UserInterfaceManager.get().getFromActions ("XMLReflect.restoreXML");
                     if (restoreAction != null)
                     {
                        ActionEvent event = new ActionEvent ("directly", -1, file.getPath());
                        restoreAction.actionPerformed (event);
                        setSaveAction (UserInterfaceManager.get().getFromActions ("XMLReflect.storeXML"));
                     }
                  }
                  if (ASGUnparseInformation.isPointsCalled())
                  {
                     //FIXME: Old projects might need this to restore their layout,
                     //but this unparses all diagrams and needs massive amounts of RAM and time.
                     //repairUnparseInformation();
                  }
               }
               finally
               {
                  FileHistory.get().addToHistory (file);
                  FileHistory.get().updateActions();
                  UMLProject.get().setSaved (true);
                  UMLProject.get().setFile (file);
                  //to create new stereotypes for old projects
                  UMLStereotypeManager.get().initDefaultStereotypes();

                  //to fix visibility of attrs in old projects
                  UMLAttr.setSyncVisibilities (true);
               }
            }
            if (log.isDebugEnabled())
            {
               log.debug ("OpenAction: UMLProject.loadProject done ");
            }
            setUMLProject (newProject);
         }
         catch (Exception except)
         {
            except.printStackTrace();
            String message = except.getMessage();
            if ("".equals (message))
            {
               message = except.getClass().getName();
            }
            JOptionPane.showMessageDialog
                (getFrame(), message, "Warning: Can't load",
               JOptionPane.WARNING_MESSAGE);
         }
         finally
         {
            getTabbedPaneProxy().update();
            if (log.isDebugEnabled())
            {
               log.debug ("OpenAction: analyse engines resumed");
            }
         }
      }
   } // openfile


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void repairUnparseInformation()
   {
      for (Iterator it = UMLProject.get().iteratorOfDiags(); it.hasNext(); )
      {
         ASGDiagram diag = (ASGDiagram) it.next();
         FSAContainer pane = (FSAContainer) UnparseManager.get().unparse (diag,
            diag.getFromFsaObjects ("@__null.entry"));
         saveFSAProperties (pane);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param container  No description provided
    */
   private void saveFSAProperties (FSAContainer container)
   {
      for (Iterator it = container.iteratorOfChildren(); it.hasNext(); )
      {
         FSAObject child = (FSAObject) it.next();
         if (child instanceof FSAContainer)
         {
            saveFSAProperties ((FSAContainer) child);
         }
         else
         {
            child.saveFSAProperties();
         }
      }
      container.saveFSAProperties();
   }


   /**
    * Updates the tree items of the tabbed panes and builds the dtd tree.
    */
   public void createNewTreeItems()
   {
      getTabbedPaneProxy().update();
   } // createNewTreeItems


   /**
    * Selects a tree item and the associated tab containing this item.
    *
    * @param obj  The item which will be selected.
    */
   public void selectTreeItem (Object obj)
   {
      Runnable doUpdate = null;
      if (obj != null)
      {
         final TabProxy tabProxy = getTabbedPaneProxy().findTabProxy (obj);

         if (tabProxy != null)
         {
            final JTabbedPane pane = getTabbedPaneProxy().getTabbedPane();
            final TreePath newPath = tabProxy.findTreePath (obj);
            final JTree tree = tabProxy.getTree();
            final boolean update =  (newPath != null &&  (!newPath.equals (tree.getSelectionPath())));

            doUpdate =
               new Runnable()
               {
                  public void run()
                  {
                     pane.setSelectedComponent (tabProxy.getComponent());
                     if (update)
                     {
                        tree.clearSelection();
                        tree.setSelectionPath (newPath);
                     }
                  }
               };
         }
      }
      else
      {
         Component tmpSelectedComponent = getTabbedPaneProxy().getTabbedPane().getSelectedComponent();
         if (tmpSelectedComponent instanceof JScrollPane)
         {
            tmpSelectedComponent =  ((JScrollPane) tmpSelectedComponent).getViewport().getView();
         }
         final Component selectedComponent = tmpSelectedComponent;
         if (selectedComponent instanceof JTree)
         {
            doUpdate =
               new Runnable()
               {
                  public void run()
                  {
                      ((JTree) selectedComponent).setSelectionPath (null);
                  }
               };
         }
      }
      if (doUpdate != null)
      {
         if (SwingUtilities.isEventDispatchThread())
         {
            doUpdate.run();
         }
         else
         {
            try
            {
               SwingUtilities.invokeAndWait (doUpdate);
            }
            catch (InterruptedException ex)
            {
            }
            catch (InvocationTargetException ex)
            {
               throw new RuntimeException (ex);
            }
         }
      }
   } // selectTreeItem


   /**
    * Returns the selected node or null if no selection available.
    *
    * @return   The selected node.
    */
   public DefaultMutableTreeNode getSelectedTreeNode()
   {
      DefaultMutableTreeNode node = null;
      Component component = getTabbedPaneProxy().getTabbedPane().getSelectedComponent();

      if (component != null)
      {
         if (component instanceof JScrollPane)
         {
            component =  ((JScrollPane) component).getViewport().getView();
         }

         if (component instanceof JTree)
         {
            TreePath path =  ((JTree) component).getSelectionPath();
            if (path != null)
            {
               node = (DefaultMutableTreeNode) path.getLastPathComponent();
            }
         }
      }
      return node;
   } // getSelectedTreeNode


   /**
    * Set program title, project name and file name
    */
   public void showTitle()
   {
      final StringBuffer title = new StringBuffer ("Fujaba 4");

      String project = umlProject != null ? umlProject.getName() : null;
      if ( (project != null) &&  (!project.equals ("")))
      {
         title.append (" [");
         title.append (project);
         title.append ("]");
      }

      File file = UMLProject.get() != null ? UMLProject.get().getFile() : null;
      if (file != null)
      {
         String name = file.getName();

         if (!name.equals (""))
         {
            title.append (" - ");
            title.append (name);
         }
      }
      Runnable doUpdate =
         new Runnable()
         {
            public void run()
            {
               getFrame().setTitle (title.toString());
            }
         };
      if (SwingUtilities.isEventDispatchThread())
      {
         doUpdate.run();
      }
      else
      {
         SwingUtilities.invokeLater (doUpdate);
      }
   }


   /**
    * @param toolBar  toolbar to be added
    * @deprecated     please use {@link #getInternalFrame(de.uni_paderborn.fujaba.asg.ASGDiagram)}.addDiagramToolBar
    */
   public void addDiagramToolBar (JToolBar toolBar)
   {
      if (getCurrentInternalFrame() != null)
      {
         getCurrentInternalFrame().addDiagramToolBar (toolBar);
      }
   }


   /**
    * @param toolBar  toolbar to be removed
    * @deprecated     please use {@link #getInternalFrame(de.uni_paderborn.fujaba.asg.ASGDiagram)}.removeDiagramToolBar
    */
   public void removeDiagramToolBar (JToolBar toolBar)
   {
      if (getCurrentInternalFrame() != null)
      {
         getCurrentInternalFrame().removeDiagramToolBar (toolBar);
      }
   }


   /**
    * getter for field saveAction
    *
    * @return   the action that should be used by default to save a project
    */
   public AbstractAction getSaveAction()
   {
      if (saveAction == null)
      {
         saveAction = UserInterfaceManager.get().getFromActions ("saveProject");
      }
      return this.saveAction;
   }


   /**
    * store the value for field saveAction
    */
   private AbstractAction saveAction;


   /**
    * setter for field saveAction
    *
    * @param value  new value
    */
   public void setSaveAction (final AbstractAction value)
   {
      final AbstractAction oldValue = this.saveAction;
      if (oldValue != value)
      {
         this.saveAction = value;
         firePropertyChange ("saveAction", oldValue, value);
      }
   }
}

/*
 * $Log: FrameMain.java,v $
 * Revision 1.609.2.15  2006/07/10 08:05:34  cschneid
 * sysouts removed, exceptions printed (instead of ignoring completely), caption changed to 'Fujaba 4'
 *
 */
