/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app;

import java.awt.*;
import java.util.*;

import javax.swing.*;
import javax.swing.tree.*;

import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.basic.TreeNodeComparator;
import de.uni_paderborn.fujaba.metamodel.FElement;
import de.uni_paderborn.fujaba.uml.UMLDiagram;
import de.uni_paderborn.fujaba.views.ViewDiagram;
import de.uni_paderborn.lib.basic.ImageResourceManager;
import de.uni_paderborn.lib.classloader.UPBClassLoader;


/**
 * The task of this class is to provide some capsule for functions, which needed to be implemented
 * in the case of creating some new class of FUJABA diagrams, as they are a) creating / showing
 * a toolbar (every kind of diagram got one of its own) b) adding items of a diagram to the
 * tree node row at left side of FUJABA (diagram has to manage the order and kinds of items
 * and build up some tree structure of them) Methods and attributes of this class should be
 * helpful to manage those tasks in a more generic way, while abstract methods should provide
 * a reminder for childs of what to implement. <h2>Associations</h2> <pre>
 *            0..1      Assoc      0..n
 * FrameMain --------------------------- FrameDecorator
 *            frameMain      decorators
 * </pre>
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.48 $ Move to UML plugin
 */
public abstract class FrameDecorator
{
   /**
    * Checks wether decorator is responsible for the given diagram. Changes the visibility
    * of menus/toolbars. Called, if diagram is entered.
    *
    * @param diag  No description provided
    */
   public abstract void enterDiagram (ASGDiagram diag);


   /**
    * Checks wether decorator is responsible for the given diagram. Changes the visibility
    * of menus/toolbars. Called, if diagram was left.
    *
    * @param diag  No description provided
    */
   public abstract void leaveDiagram (ASGDiagram diag);


   /**
    * To be overridden by deriving classes and enables each type of diagram to set its own
    * location in Fujabas diagrams tree.
    *
    * @param diagEnum  No description provided
    * @return          The treeNode value
    */
   public abstract DefaultMutableTreeNode getTreeNode (Enumeration diagEnum);


   /**
    * Get the treeNode attribute of the FrameDecorator object
    *
    * @param diag  No description provided
    * @return      The treeNode value
    */
   public DefaultMutableTreeNode getTreeNode (ASGDiagram diag)
   {
      DefaultMutableTreeNode node = new DefaultMutableTreeNode (diag);
      TreeSet nodeSet = new TreeSet (new TreeNodeComparator());

      if (diag instanceof UMLDiagram)
      {
         UMLDiagram umlDiag = (UMLDiagram) diag;

         Iterator viewIter = umlDiag.iteratorOfViews();
         while (viewIter.hasNext())
         {
            ViewDiagram view = (ViewDiagram) viewIter.next();
            nodeSet.add (getTreeNode (view));
         }

         Iterator iter = nodeSet.iterator();
         while (iter.hasNext())
         {
            node.add ((DefaultMutableTreeNode) iter.next());
         }

      } // end of if ()

      return node;
   } // getTreeNode


   /**
    * Get the treeCellRenderer attribute of the FrameDecorator class
    *
    * @return   The treeCellRenderer value
    */
   public TreeCellRenderer getTreeCellRenderer()
   {
      return new FujabaTreeCellRenderer();
   } // getTreeCellRenderer


   /**
    * Get the destTabTitle attribute of the FrameDecorator object
    *
    * @return   The destTabTitle value
    */
   public String getDestTabTitle()
   {
      return TabbedPaneProxy.TAB_PRJ;
   } // getDestTabTitle


   /**
    * <pre>
    *            0..1      Assoc      0..n
    * FrameMain --------------------------- FrameDecorator
    *            frameMain      decorators
    * </pre>
    */
   private FrameMain frameMain;


   /**
    * Sets the frameMain attribute of the FrameDecorator object
    *
    * @param value  The new frameMain value
    * @return       No description provided
    */
   public boolean setFrameMain (FrameMain value)
   {
      boolean changed = false;
      if (this.frameMain != value)
      {
         if (this.frameMain != null)
         {
            FrameMain oldValue = this.frameMain;
            this.frameMain = null;
            oldValue.removeFromDecorators (this);
         }
         this.frameMain = value;
         if (value != null)
         {
            value.addToDecorators (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * Get the frameMain attribute of the FrameDecorator object
    *
    * @return   The frameMain value
    */
   public FrameMain getFrameMain()
   {
      return this.frameMain;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      FrameMain tmpFrameMain = getFrameMain();
      if (tmpFrameMain != null)
      {
         setFrameMain (null);
      }
   }

} // FrameDecorator.java


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.48 $
 */
class FujabaTreeCellRenderer extends DefaultTreeCellRenderer
{


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Icon diagramIcon;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Icon viewIcon;

   /**
    * icon for ASGElements in Trees
    */
   private Icon elementIcon;


   /**
    * Constructor of FujabaTreeCellRenderer. Sets the icons for the different tree nodes.
    */
   public FujabaTreeCellRenderer()
   {
      ImageResourceManager manager = ImageResourceManager.get();

      diagramIcon = manager.getImageIcon (UPBClassLoader.DEFAULT_CLASSLOADER, "de/uni_paderborn/fujaba/app/images/diagram.gif");

      // node for a view diagram
      viewIcon = manager.getImageIcon (UPBClassLoader.DEFAULT_CLASSLOADER, "de/uni_paderborn/fujaba/app/images/view.gif");

      // icon for an ASGElement
      elementIcon = manager.getImageIcon (UPBClassLoader.DEFAULT_CLASSLOADER, "de/uni_paderborn/fujaba/app/images/class.gif");

      // image for a closed node
      setClosedIcon (manager.getImageIcon (UPBClassLoader.DEFAULT_CLASSLOADER, "de/uni_paderborn/fujaba/app/images/closedFolder.gif"));

      // image for a opened node
      setOpenIcon (manager.getImageIcon (UPBClassLoader.DEFAULT_CLASSLOADER, "de/uni_paderborn/fujaba/app/images/openFolder.gif"));
   }


   /**
    * Get the treeCellRendererComponent attribute of the FujabaTreeCellRenderer object
    *
    * @param tree      No description provided
    * @param node      No description provided
    * @param sel       No description provided
    * @param expanded  No description provided
    * @param leaf      No description provided
    * @param row       No description provided
    * @param hasFocus  No description provided
    * @return          The treeCellRendererComponent value
    */
   public Component getTreeCellRendererComponent (JTree tree,
                                                  Object node,
                                                  boolean sel,
                                                  boolean expanded,
                                                  boolean leaf,
                                                  int row,
                                                  boolean hasFocus)
   {
      JLabel label = (JLabel) super.getTreeCellRendererComponent (tree, node, sel, expanded,
         leaf && node != tree.getModel().getRoot(), row, hasFocus);

      if (node instanceof DefaultMutableTreeNode)
      {
         Object value =  ((DefaultMutableTreeNode) node).getUserObject();

         if (value instanceof FujabaTreeNodeObject)
         {
            FujabaTreeNodeObject treeNode = (FujabaTreeNodeObject) value;
            label.setIcon (treeNode.getIcon());
            label.setToolTipText (treeNode.getToolTipText());
            String tags = treeNode.getHtmlTags();
            if ( (tags != null) &&  (tags.length() > 0))
            {
               tags = "<html>" + tags;
               String endTags = "";
               StringTokenizer token = new StringTokenizer (tags, "<>");
               while (token.hasMoreTokens())
               {
                  endTags = "</" + token.nextToken() + ">" + endTags;
               }
               String newText = tags + label.getText() + endTags;
               label.setText (newText);
            }
         }
         else if (value instanceof ASGDiagram)
         {
            if (label.isEnabled())
            {
               if (value instanceof ViewDiagram)
               {
                  if (viewIcon != null)
                  {
                     label.setIcon (viewIcon);
                  }
                  else if (diagramIcon != null)
                  {
                     label.setIcon (diagramIcon);
                  }
               }
               else if (diagramIcon != null)
               {
                  label.setIcon (diagramIcon);
               }
            }
            else
            {
               if (value instanceof ViewDiagram)
               {
                  if (viewIcon != null)
                  {
                     label.setDisabledIcon (viewIcon);
                  }
                  else if (diagramIcon != null)
                  {
                     label.setDisabledIcon (diagramIcon);
                  }
               }
               else if (diagramIcon != null)
               {
                  label.setDisabledIcon (diagramIcon);
               }
            }
         }
         else if (value instanceof FElement)
         {
            if (elementIcon != null)
            {
               label.setIcon (elementIcon);
            }
         }
      }

      return label;
   }
}

/*
 * $Log: FrameDecorator.java,v $
 * Revision 1.48  2004/10/20 17:49:23  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
