/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.upb.lib.userinterface.action;

import java.awt.event.ActionEvent;
import java.beans.PropertyChangeListener;
import java.util.Iterator;

import javax.swing.Action;
import javax.swing.JComponent;

import de.upb.lib.userinterface.*;


/**
 * Adapter for any Action loaded by this framework. All methods but actionPerformed are delegated
 * to the adapted action. The method actionPerformed changes the event source and event command.
 * This adapter action gets these parameters from the EventSource interface. This class has
 * package visibility.
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.4 $
 * @see       EventSource
 */
public class MultiplexAction implements AbstractMultiplexAction
{
   /**
    * The adapted action.
    */
   private Action action;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private ElementInitializer initializer;


   /**
    * @param initializer  The ElementInitializer
    */
   public void setElementInitializer (ElementInitializer initializer)
   {
      this.initializer = initializer;
   }


   /**
    * Sets the action that is called, if {@link #actionPerformed} is called.
    *
    * @param calledAction  The action to which the {@link #actionPerformed} method call is
    *      delegated.
    * @see                 de.upb.lib.userinterface.action.AbstractMultiplexAction#setAction(javax.swing.Action)
    */
   public void setAction (Action calledAction)
   {
      this.action = calledAction;
   }


   /**
    * Method is delegated to the adapted action.
    *
    * @param key    No description provided
    * @param value  No description provided
    */
   public void putValue (String key, Object value)
   {
      action.putValue (key, value);
   }


   /**
    * Method is delegated to the adapted action.
    *
    * @param key  No description provided
    * @return     The value value
    */
   public Object getValue (String key)
   {
      return action.getValue (key);
   }


   /**
    * Method is delegated to the adapted action.
    *
    * @param b  The new enabled value
    */
   public void setEnabled (boolean b)
   {
      action.setEnabled (b);
   }


   /**
    * Method is delegated to the adapted action.
    *
    * @return   The enabled value
    */
   public boolean isEnabled()
   {
      return action.isEnabled();
   }


   /**
    * Method is delegated to the adapted action.
    *
    * @param listener  The object added.
    */
   public void addPropertyChangeListener (PropertyChangeListener listener)
   {
      action.addPropertyChangeListener (listener);
   }


   /**
    * Method is delegated to the adapted action.
    *
    * @param listener  No description provided
    */
   public void removePropertyChangeListener (PropertyChangeListener listener)
   {
      action.removePropertyChangeListener (listener);
   }


   /**
    * A new event is generated by using the event source and action command from the EventSource
    * interface given by the UserInterfaceManager.
    *
    * @param event  No description provided
    * @see          UserInterfaceManager#getEventSource
    * @see          EventSource#getSource
    * @see          EventSource#getActionCommand
    */
   public void actionPerformed (ActionEvent event)
   {
      UserInterfaceManager uiManager = UserInterfaceManager.get();
      EventSource eventSource = uiManager.getEventSource();

      // get the source for the event
      Object source = null;
      if (eventSource != null)
      {
         source = eventSource.getSource();
      }
      else
      {
         source = event.getSource();
      }

      // get the action command for the event
      String actionCommand = null;
      if (eventSource != null)
      {
         actionCommand = eventSource.getActionCommand();
      }

      // if no actionCommand provided by the EventSource interface,
      // send the action key as action command
      if (actionCommand == null)
      {
         actionCommand = uiManager.getKeyForAction (action);
      }

      // create new action event
      ActionEvent actionEvent = new ActionEvent (source, ActionEvent.ACTION_PERFORMED, actionCommand);

      // notify about incoming action event
      Iterator preIter = uiManager.iteratorOfExecutionListeners();
      while (preIter.hasNext())
      {
         ActionExecutionListener listener = (ActionExecutionListener) preIter.next();
         listener.preActionNotify (actionEvent);
      }

      try
      {
         // perform the adapted action
         action.actionPerformed (actionEvent);
      }
      catch (Throwable e)
      {
         boolean handled = false;
         // notify about aborted action event
         Iterator postIter = uiManager.iteratorOfExecutionListeners();
         while (postIter.hasNext())
         {
            Object next = postIter.next();
            if (next instanceof ActionExecutionListenerEx)
            {
               ActionExecutionListenerEx listener = (ActionExecutionListenerEx) next;
               handled |= listener.postActionNotify (actionEvent, e);
            }
         }
         if (!handled)
         {
            if (e instanceof RuntimeException)
            {
               throw (RuntimeException) e;
            }
            else
            {
               throw new RuntimeException (e);
            }
         }
      }

      // notify about performed action event
      Iterator postIter = uiManager.iteratorOfExecutionListeners();
      while (postIter.hasNext())
      {
         ActionExecutionListener listener = (ActionExecutionListener) postIter.next();
         listener.postActionNotify (actionEvent);
      }
   } // actionPerformed


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param component  No description provided
    * @param source     No description provided
    */
   public void initialize (JComponent component, Object source)
   {
      if (this.initializer != null)
      {
         this.initializer.initialize (component, source);
      }

      if (this.action != null && this.action instanceof ElementInitializer)
      {
         ElementInitializer elInitializer = (ElementInitializer) this.action;
         elInitializer.initialize (component, source);
      }
   }

}

/*
 * $Log: MultiplexAction.java,v $
 * Revision 1.4  2005/02/17 12:32:49  cschneid
 * enable exception handling for actions
 *
 */
