/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.upb.lib.plugins.gui;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.*;

import de.uni_paderborn.lib.basic.ImageResourceManager;
import de.uni_paderborn.lib.classloader.UPBClassLoader;
import de.uni_paderborn.lib.javax.swing.progress.SwingWorker;
import de.uni_paderborn.lib.util.WindowUtility;
import de.upb.lib.plugins.*;


/**
 * @author    $Author: lowende $
 * @version   $Revision: 1.9 $
 */
public class PluginDownloadDialog extends JDialog
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String DOWNLOAD = "Download/Install";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String UPDATE = "Download/Install update";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String UP_TO_DATE = "Plug-In up to date";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JTree pluginPropertyTree;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static String[] infoLabels = new String[]
      {"Name", "Identification", "Version", "Required Fujaba", "Help URL", "Vendor", "Contact", "Download URL"};

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private JLabel pluginName = new JLabel();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JLabel pluginID = new JLabel();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JLabel pluginVersion = new JLabel();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JLabel pluginKernelNeeded = new JLabel();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JLabel pluginHelpURL = new JLabel();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JLabel pluginVendor = new JLabel();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JLabel pluginContactAdress = new JLabel();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JLabel pluginDownloadURL = new JLabel();

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JTextPane pluginShortDescription = new JTextPane();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JTextPane pluginDetailedDescription = new JTextPane();

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private DefaultListModel pluginDependencies = new DefaultListModel();

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private JButton downloadButton = new JButton (DOWNLOAD);

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private PluginManager manager;

   /**
    * Single instance of a thread that refreshes the "plugin data" tree.
    */
   RefreshPluginDataThread refreshPluginDataThread;


   /**
    * Constructor for class PluginDownloadDialog
    *
    * @param manager  No description provided
    * @param frame    No description provided
    */
   public PluginDownloadDialog (JFrame frame, PluginManager manager)
   {
      super (frame, "Manage Plug-Ins", true);
      this.manager = manager;

      // overall screen structure
      JPanel mainPanel = new JPanel (new BorderLayout());

      JButton closeButton = new JButton ("Close");
      closeButton.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               if (refreshPluginDataThread != null)
               {
                  refreshPluginDataThread.interrupt();
               }

               PluginDownloadDialog.this.dispose();
               ;
            }
         });

      this.getContentPane().setLayout (new BorderLayout());
      this.getContentPane().add (mainPanel, BorderLayout.CENTER);
      getRootPane().setDefaultButton (closeButton);

      // the left side

      // create an empty tree - data will be inserted later,
      // because we want to give early feedback to the user
      // and filling the tree with data may be very time-consuming

      DefaultMutableTreeNode treeRoot = new DefaultMutableTreeNode ("Fujaba Plug-ins");
      treeRoot.add (new DefaultMutableTreeNode ("retrieving plugin-data from the internet..."));
      pluginPropertyTree = new JTree (treeRoot);

      pluginPropertyTree.setShowsRootHandles (true);
      pluginPropertyTree.setRootVisible (false);
      pluginPropertyTree.setEditable (false);
      pluginPropertyTree.setCellRenderer (new PluginTreeCellRenderer (manager));
      pluginPropertyTree.addTreeSelectionListener (
         new TreeSelectionListener()
         {
            public void valueChanged (TreeSelectionEvent e)
            {
               valueChangedEventHandling (e);
            }
         });

      JScrollPane scrollPane = new JScrollPane (pluginPropertyTree);
      // let the scrollPane have minimal dimensions to show the tree
      scrollPane.setPreferredSize (new Dimension (300, 500));

      JPanel leftPanel = new JPanel (new BorderLayout());
      leftPanel.add (scrollPane, BorderLayout.CENTER);
      leftPanel.setBorder (new TitledBorder ("All available plug-ins are shown in the tree"));

      // legend
      leftPanel.add (PluginTreeCellRenderer.getLegend (manager), BorderLayout.SOUTH);

      // and the right side
      JPanel infoPanel = new JPanel (new GridBagLayout());
      infoPanel.setBackground (Color.WHITE);

      GridBagConstraints constraint = new GridBagConstraints();

      constraint.insets = new Insets (5, 5, 5, 5);
      constraint.gridx = 0;

      for (int i = 0; i < infoLabels.length; i++)
      {
         constraint.gridy = i;
         constraint.fill = GridBagConstraints.HORIZONTAL;
         constraint.anchor = GridBagConstraints.EAST;
         JLabel tmpLabel = new JLabel (infoLabels[i] + " : ");
         tmpLabel.setHorizontalAlignment (SwingConstants.RIGHT);
         infoPanel.add (tmpLabel, constraint);
      }

      constraint.gridx = 1;
      constraint.weightx = 100;
      constraint.fill = GridBagConstraints.HORIZONTAL;

      constraint.gridy = 0;
      infoPanel.add (pluginName, constraint);

      constraint.gridy = 1;
      infoPanel.add (pluginID, constraint);

      constraint.gridy = 2;
      infoPanel.add (pluginVersion, constraint);

      constraint.gridy = 3;
      infoPanel.add (pluginKernelNeeded, constraint);

      constraint.gridy = 4;
      infoPanel.add (pluginHelpURL, constraint);

      constraint.gridy = 5;
      infoPanel.add (pluginVendor, constraint);

      constraint.gridy = 6;
      infoPanel.add (pluginContactAdress, constraint);

      constraint.gridy = 7;
      infoPanel.add (pluginDownloadURL, constraint);

      JPanel outerInfoPanel = new JPanel (new BorderLayout());
      outerInfoPanel.setBorder (new TitledBorder ("General Plug-in Information"));
      outerInfoPanel.add (infoPanel, BorderLayout.NORTH);

      // description panels
      JPanel shortDescriptionPanel = new JPanel (new BorderLayout());
      shortDescriptionPanel.setBorder (new TitledBorder ("Plug-in Short Description"));
      pluginShortDescription.setEditable (false);
      shortDescriptionPanel.add (new JScrollPane (pluginShortDescription), BorderLayout.CENTER);

      JPanel dependenciesPanel = new JPanel (new BorderLayout());
      dependenciesPanel.setBorder (new TitledBorder ("Plug-in depends on"));
      JList dependencyList = new JList (pluginDependencies);
      dependencyList.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);
      dependenciesPanel.add (new JScrollPane (dependencyList), BorderLayout.CENTER);

      JPanel rightmiddlePanel = new JPanel (new BorderLayout());
      rightmiddlePanel.add (shortDescriptionPanel, BorderLayout.CENTER);
      rightmiddlePanel.add (dependenciesPanel, BorderLayout.EAST);

      JPanel detailedDescriptionPanel = new JPanel (new BorderLayout());
      detailedDescriptionPanel.setBorder (new TitledBorder ("Plug-in Detailed Description"));
      pluginDetailedDescription.setEditable (false);
      detailedDescriptionPanel.add (new JScrollPane (pluginDetailedDescription), BorderLayout.CENTER);

      JPanel descriptionPanel = new JPanel (new GridLayout (2, 1));
      descriptionPanel.add (rightmiddlePanel);
      descriptionPanel.add (detailedDescriptionPanel);

      // panel on the right side
      JPanel propertyPanel = new JPanel (new BorderLayout());
      propertyPanel.add (outerInfoPanel, BorderLayout.NORTH);
      propertyPanel.add (descriptionPanel, BorderLayout.CENTER);
      propertyPanel.setPreferredSize (new Dimension (650, 300));

      JSplitPane splitPane = new JSplitPane (JSplitPane.HORIZONTAL_SPLIT, leftPanel, propertyPanel);
      mainPanel.add (splitPane, BorderLayout.CENTER);

      JPanel buttonPanel = new JPanel (new FlowLayout (FlowLayout.RIGHT));

      buttonPanel.add (closeButton);
      buttonPanel.add (downloadButton);
      mainPanel.add (buttonPanel, BorderLayout.SOUTH);

      downloadButton.setEnabled (false);
      downloadButton.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               downloadButtonPressed (e);
            }
         });

      this.refreshPluginDataThread = new RefreshPluginDataThread (this);

      // add a window listener, that refreshes available plugins
      // if the dialog is shown the first time
      this.addWindowListener (
         new WindowListener()
         {

            public void windowOpened (WindowEvent e)
            {
               // ensure, that dialog-content is visible/painted, while
               // we are executing the worker code, so user sees an initial
               // picture ... for that we spawn a new SwingWorker thread and
               // return immediately

               // by now the thread will be executed only once,
               // but what if we want to retrieve data more then once?
               // do we have to close/interrupt the thread before
               // executing it again?
               if (refreshPluginDataThread != null)
               {
                  //refreshPluginDataThread.interrupt();
                  refreshPluginDataThread.start();
               }
            }


            public void windowClosing (WindowEvent e)
            {
            }


            public void windowClosed (WindowEvent e)
            {
            }


            public void windowIconified (WindowEvent e)
            {
            }


            public void windowDeiconified (WindowEvent e)
            {
            }


            public void windowActivated (WindowEvent e)
            {
            }


            public void windowDeactivated (WindowEvent e)
            {
            }
         });
   }


   /**
    * This inner classes refreshes the "available plugins" tree
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.9 $ $Date: 2006/02/07 17:04:12 $
    */
   private class RefreshPluginDataThread extends SwingWorker
   {
      /**
       * Reference to the PluginDownloadDialog window.
       */
      Window pluginDownloadWindow;


      /**
       * Constructor.
       *
       * @param pluginDownloadWindow  The plugin-download window.
       */
      public RefreshPluginDataThread (Window pluginDownloadWindow)
      {
         super();
         this.pluginDownloadWindow = pluginDownloadWindow;
      }


      /**
       * This Method is invoked by a new thread.
       *
       * @return   Nothing to care about.
       */
      public Object construct()
      {
         if (pluginDownloadWindow != null)
         {
            pluginDownloadWindow.setCursor (Cursor.getPredefinedCursor (Cursor.WAIT_CURSOR));
         }

         try
         {
            Thread.sleep (100);

            // re-create plugin-list
            try
            {
               pluginPropertyTree.setModel (createPluginTreeModel());

               pluginPropertyTree.revalidate();
               pluginPropertyTree.repaint();
            }
            finally
            {
               if (pluginDownloadWindow != null)
               {
                  pluginDownloadWindow.setCursor (Cursor.getPredefinedCursor (Cursor.DEFAULT_CURSOR));
               }
            }
         }
         catch (Exception except)
         {
            except.printStackTrace();
         }
         return "";
      } // construct


      /**
       * This Method is invoked after the layouter is ready.
       */
      public void finished()
      {
      }
   }


   /**
    * Centers and shows this dialog.
    */
   public void showCentered()
   {
      pack();
      WindowUtility.centerFrame (this);

      // this call pops up the modal dialog
      this.setVisible (true);
   }


   /**
    * Create a plugin tree model.
    *
    * @return   A TreeModel of all plugins that can be used by a JTree.
    */
   TreeModel createPluginTreeModel()
   {
      // just use default model with root node
      DefaultMutableTreeNode rootNode = new DefaultMutableTreeNode ("Fujaba Plug-ins");
      DefaultTreeModel model = new DefaultTreeModel (rootNode);

      DefaultMutableTreeNode allNode = new DefaultMutableTreeNode ("All Plug-ins");
      rootNode.add (allNode);

      Vector plugins = PluginDownloader.getDownloadablePlugins (manager);

      for (int i = 0; i < plugins.size(); i++)
      {
         Object tmpObj = plugins.elementAt (i);
         if (tmpObj instanceof Vector)
         {
            Vector entries = (Vector) tmpObj;

            // check if the entry has the expected format
            if (! (entries.elementAt (0) instanceof PluginList))
            {
               continue;
            }

            PluginList pluginList = (PluginList) entries.elementAt (0);

            // the first entry contains the name of the pluginlist
            // or an error message
            DefaultMutableTreeNode innerNode = new DefaultMutableTreeNode();
            innerNode.setUserObject (pluginList);
            rootNode.add (innerNode);

            for (int j = 1; j < entries.size(); j++)
            {
               PluginProperty property = (PluginProperty) entries.elementAt (j);
               if (innerNode != null)
               {
                  insertPluginInTree (innerNode, property);
               }
               insertPluginInTree (allNode, property);
            }
         }

      }

      // add installed plugins
      Vector installedPlugins = PluginDownloader.getInstalledPlugins (manager);

      DefaultMutableTreeNode installedNode = new DefaultMutableTreeNode (installedPlugins.elementAt (0));
      rootNode.add (installedNode);

      for (int j = 1; j < installedPlugins.size(); j++)
      {
         insertPluginInTree (installedNode, (PluginProperty) installedPlugins.elementAt (j));
      }

      // shrink tree
      for (int i = 0; i < rootNode.getChildCount(); i++)
      {
         TreeNode node = rootNode.getChildAt (i);
         for (int j = 0; j < node.getChildCount(); j++)
         {
            shrinkSubtree ((DefaultMutableTreeNode) node.getChildAt (j));
         }
      }

      return model;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param node  No description provided
    */
   private void shrinkSubtree (DefaultMutableTreeNode node)
   {
      for (int i = 0; i < node.getChildCount(); i++)
      {
         shrinkSubtree ((DefaultMutableTreeNode) node.getChildAt (i));
      }
      if (node.getChildCount() == 0 && node.getParent().getChildCount() == 1 && node.getLevel() != 2)
      {
         // ok, shrink the actual node by replacing the user object
         DefaultMutableTreeNode parent = (DefaultMutableTreeNode) node.getParent();
         parent.setUserObject (node.getUserObject());
         node.removeFromParent();
      }
   }


   /**
    * The innerNode parameter specifies the lowest level the property should be
    * inserted. Following the structure PluginName, Major.Minor, BuildNo, this
    * method inserts the plugin as a leaf of the tree. Inner nodes in the tree
    * will be sorted.
    *
    * @param property  The plugin property.
    * @param node      No description provided
    */
   private void insertPluginInTree (DefaultMutableTreeNode node, PluginProperty property)
   {
      // first level: PluginName
      String pluginName = property.getName();
      DefaultMutableTreeNode nameNode = getChildNode (node, pluginName);

      // second level: Major.Minor number
      String majorminor = "Version " + property.getMajorVersion() + "." + property.getMinorVersion();
      DefaultMutableTreeNode mmNode = getChildNode (nameNode, majorminor);

      // third level: Build number
      getChildNode (mmNode, property);
   }


   /**
    * Searches all children of node.
    *
    * @param node  Parent node.
    * @param obj   Object to be inserted.
    * @return      An existing or new node with user object obj
    */
   private DefaultMutableTreeNode getChildNode (DefaultMutableTreeNode node, Object obj)
   {
      int index = 0;
      DefaultMutableTreeNode result = null;

      for (int i = 0; index == 0 && i < node.getChildCount(); i++)
      {
         DefaultMutableTreeNode tmpNode = (DefaultMutableTreeNode) node.getChildAt (i);

         if (obj instanceof String)
         {
            String name = (String) obj;
            String userObj = (String) tmpNode.getUserObject();
            if (userObj.equals (name))
            {
               index = i; // leave loop, node has been found
               result = tmpNode;
            }
            if (userObj.compareTo (name) > 0)
            {
               index = i; // leave loop, node should be inserted here
            }
         }
         if (obj instanceof PluginProperty)
         {
            PluginProperty prop = (PluginProperty) obj;
            PluginProperty userObj = (PluginProperty) tmpNode.getUserObject();
            if (userObj.getBuildNumber() == prop.getBuildNumber())
            {
               index = i;
               result = tmpNode;
            }
            if (userObj.getBuildNumber() > prop.getBuildNumber())
            {
               index = i;
            }
         }
      }

      // recalculate insertion point
      if (result == null && index == 0)
      {
         index = node.getChildCount();
      }

      if (result == null)
      {
         result = new DefaultMutableTreeNode (obj);
         node.insert (result, index);
      }

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param e  No description provided
    */
   public void valueChangedEventHandling (TreeSelectionEvent e)
   {
      downloadButton.setEnabled (false);

      PluginProperty property = getSelectedProperty();

      if (property != null)
      {
         pluginName.setText (property.getName());
         pluginID.setText (property.getPluginID());
         pluginVersion.setText (property.getVersion());

         pluginKernelNeeded.setText ("Version " + property.getNeededKernelMajor() + "."
            + property.getNeededKernelMinor());
         if (property.getPluginHelp() == null)
         {
            pluginHelpURL.setText ("Help URL not available");
            pluginHelpURL.setForeground (Color.BLACK);
         }
         else
         {
            pluginHelpURL.setText (property.getPluginHelp());
            pluginHelpURL.setForeground (Color.BLUE);
         }
         pluginShortDescription.setText (formatText (property.getShortDescription()));

         pluginDependencies.removeAllElements();
         Iterator iter = property.iteratorOfDependsOn();
         while (iter.hasNext())
         {
            pluginDependencies.addElement (iter.next());
         }

         pluginDetailedDescription.setText (formatText (property.getDescription()));
         pluginVendor.setText (property.getVendor());
         pluginContactAdress.setText (property.getContact());
         pluginDownloadURL.setText (property.getSource());

         int state = PluginDownloader.getPluginState (property, manager);
         if (state == PluginDownloader.NEW_PLUGIN)
         {
            downloadButton.setEnabled (true);
            downloadButton.setText (DOWNLOAD);
         }
         else if (state == PluginDownloader.UPDATE_PLUGIN)
         {
            downloadButton.setEnabled (true);
            downloadButton.setText (UPDATE);
         }
         else
         {
            downloadButton.setText (UP_TO_DATE);
            downloadButton.setEnabled (false);
         }
      }

   }


   /**
    * Replaces LF and CR characters by spaces.
    *
    * @param text  No description provided
    * @return      No description provided
    * @author      joerg
    */
   private String formatText (String text)
   {
      StringBuffer tmpBuffer = new StringBuffer (text);

      // replace all line end delimiters
      replaceStrg (tmpBuffer, "\n\r", " ");
      replaceStrg (tmpBuffer, "\n", " ");
      replaceStrg (tmpBuffer, "\r", " ");

      // reduce double spaces to single space
      replaceStrg (tmpBuffer, "  ", " ");

      return tmpBuffer.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param buf      No description provided
    * @param pattern  No description provided
    * @param replace  No description provided
    */
   private void replaceStrg (StringBuffer buf, String pattern, String replace)
   {
      int index = buf.indexOf (pattern);
      while (index != -1)
      {
         buf.replace (index, index + pattern.length(), replace);
         index = buf.indexOf (pattern);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param e  No description provided
    */

   public void downloadButtonPressed (ActionEvent e)
   {
      PluginProperty selProp = getSelectedProperty();

      if (selProp != null)
      {
         try
         {
            DownloadProgress download = new DownloadProgress (new URL (selProp.getSource()),
               "Downloading Plug-in " + selProp.getName() + "     Version "
               + selProp.getVersion());
            download.show();
            download.download();
         }
         catch (MalformedURLException ex)
         {
            JOptionPane.showMessageDialog (null, ex.getMessage(), "Malformed URL",
               JOptionPane.ERROR_MESSAGE);
            System.err.println (ex.getMessage());
            ex.printStackTrace();
         }
      }
      else
      {
         JOptionPane.showMessageDialog (null, "No plug-in selected in the tree.", "Selection error",
            JOptionPane.ERROR_MESSAGE);

      }

   }


   /**
    * @return   null if no plugin is currently selected, the plugin otherwise
    */
   private PluginProperty getSelectedProperty()
   {
      TreePath selPath = pluginPropertyTree.getSelectionPath();

      if (selPath == null)
      {
         return null;
      }

      Object obj = selPath.getLastPathComponent();

      if (! (obj instanceof DefaultMutableTreeNode))
      {
         return null;
      }

      DefaultMutableTreeNode node = (DefaultMutableTreeNode) obj;

      if (! (node.getUserObject() instanceof PluginProperty))
      {
         return null;
      }

      return  ((PluginProperty) node.getUserObject());
   }

}


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.9 $ $Date: 2006/02/07 17:04:12 $
 */
class PluginTreeCellRenderer extends DefaultTreeCellRenderer
{


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private PluginManager manager = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static ImageIcon lightgreenspot = ImageResourceManager.get().getImageIcon (
      UPBClassLoader.DEFAULT_CLASSLOADER, "de/upb/lib/plugins/gui/images/lightgreenspot.gif");

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static ImageIcon greenspot = ImageResourceManager.get().getImageIcon (
      UPBClassLoader.DEFAULT_CLASSLOADER, "de/upb/lib/plugins/gui/images/greenspot.gif");

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static ImageIcon redspot = ImageResourceManager.get().getImageIcon (
      UPBClassLoader.DEFAULT_CLASSLOADER, "de/upb/lib/plugins/gui/images/redspot.gif");

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static ImageIcon yellowspot = ImageResourceManager.get().getImageIcon (
      UPBClassLoader.DEFAULT_CLASSLOADER, "de/upb/lib/plugins/gui/images/yellowspot.gif");

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static ImageIcon whitespot = ImageResourceManager.get().getImageIcon (
      UPBClassLoader.DEFAULT_CLASSLOADER, "de/upb/lib/plugins/gui/images/whitespot.gif");

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static ImageIcon orangespot = ImageResourceManager.get().getImageIcon (
      UPBClassLoader.DEFAULT_CLASSLOADER, "de/upb/lib/plugins/gui/images/orangespot.gif");

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static ImageIcon exclamation = ImageResourceManager.get().getImageIcon (
      UPBClassLoader.DEFAULT_CLASSLOADER, "de/upb/lib/plugins/gui/images/exclamation.gif");


   /**
    * Get the legend attribute of the PluginTreeCellRenderer class
    *
    * @param manager  No description provided
    * @return         The legend value
    */
   final static JPanel getLegend (PluginManager manager)
   {
      JPanel panel = new JPanel (new BorderLayout());

      JPanel versionPanel = new JPanel (new FlowLayout (FlowLayout.LEFT));
      KernelInterface kernel = manager.getKernelInterface();
      String version = kernel.getMajorVersion() + "." + kernel.getMinorVersion() + "." +
         kernel.getRevisionNumber();
      JLabel versionLabel = new JLabel ("Fujaba " + version);
      versionPanel.add (versionLabel);
      versionPanel.setBorder (new TitledBorder ("Fujaba Tool Suite Version"));

      panel.add (versionPanel, BorderLayout.NORTH);

      JPanel legendPanel = new JPanel (new GridLayout (5, 1));
      legendPanel.setBorder (new TitledBorder ("Legend"));

      JLabel tmpLabel = new JLabel ("Installed and running.", greenspot, SwingConstants.LEFT);
      tmpLabel.setToolTipText ("Plug-in is installed and has no version conflict.");
      legendPanel.add (tmpLabel);

      tmpLabel = new JLabel ("Newer Version available.", lightgreenspot, SwingConstants.LEFT);
      tmpLabel.setToolTipText ("An older version of this plug-in is already installed.");
      legendPanel.add (tmpLabel);

      tmpLabel = new JLabel ("Downloadable.", whitespot, SwingConstants.LEFT);
      tmpLabel.setToolTipText ("Plug-in is downloadable via internet without version conflict.");
      legendPanel.add (tmpLabel);

      tmpLabel = new JLabel ("Minor version conflict.", yellowspot, SwingConstants.LEFT);
      tmpLabel.setToolTipText ("Means that the kernel's minor version is lower than the needed one. Plug-in may run.");
      legendPanel.add (tmpLabel);

      tmpLabel = new JLabel ("Major version conflict.", redspot, SwingConstants.LEFT);
      tmpLabel.setToolTipText ("Means that the kernel's major version is lower than the needed one. Plug-in will properbly not run.");
      legendPanel.add (tmpLabel);

      panel.add (legendPanel, BorderLayout.CENTER);

      return panel;
   }


   /**
    *Constructor for class PluginTreeCellRenderer
    *
    * @param manager  No description provided
    */
   PluginTreeCellRenderer (PluginManager manager)
   {
      this.manager = manager;
   }


   /**
    * Get the treeCellRendererComponent attribute of the PluginTreeCellRenderer object
    *
    * @param tree      No description provided
    * @param value     No description provided
    * @param sel       No description provided
    * @param expanded  No description provided
    * @param leaf      No description provided
    * @param row       No description provided
    * @param hasFocus  No description provided
    * @return          The treeCellRendererComponent value
    */
   public Component getTreeCellRendererComponent (JTree tree, Object value, boolean sel,
                                                  boolean expanded, boolean leaf, int row, boolean hasFocus)
   {
      super.getTreeCellRendererComponent (tree, value, sel, expanded, leaf, row, hasFocus);

      DefaultMutableTreeNode node = null;

      if (value instanceof DefaultMutableTreeNode)
      {
         node = (DefaultMutableTreeNode) value;

         value = node.getUserObject();
      }

      // do we deal with a PluginList or a PluginProperty?
      if (value instanceof PluginList)
      {
         PluginList pluginList = (PluginList) value;

         this.setToolTipText (pluginList.getUrl());
         this.setText (pluginList.getName());

         // does an error occure with this plugin list?
         Exception parseException = pluginList.getParseException();
         if (parseException != null)
         {
            // display an icon that indicates an error
            this.setIcon (exclamation);

            // display the error message, too!
            String currentText = this.getText();
            String newText = currentText;
            if (!"".equals (currentText))
            {
               newText += ": ";
            }
            newText += parseException.getMessage();

            this.setText (newText);
         }
      }
      else if (value instanceof PluginProperty)
      {
         PluginProperty property = (PluginProperty) value;
         int state = PluginDownloader.getPluginState (property, manager);

         int level = node.getLevel();

         String text = "Build " + property.getBuildNumber();

         text =  (level < 4) ? "Version " + property.getMajor() + "." + property.getMinor() + " "
            + text : text;
         text =  (level < 3) ? property.getName() + " " + text : text;

         setText (text);

         // mark all currently installed plugins
         if (state == PluginDownloader.NEW_PLUGIN)
         {
            setIcon (whitespot);
         }
         else if (state == PluginDownloader.UPDATE_PLUGIN)
         {
            setText (getText() + " (newer version)");
            setIcon (lightgreenspot);
         }
         else
         {
            setText (getText() + " (already installed)");
            setIcon (greenspot);
         }

         // check plugin and kernel version and color spot appropriately

         int pluginMajor = property.getNeededKernelMajor();
         int pluginMinor = property.getNeededKernelMinor();

         int kernelMajor = manager.getKernelInterface().getMajorVersion();
         int kernelMinor = manager.getKernelInterface().getMinorVersion();

         if (pluginMajor == kernelMajor && pluginMinor > kernelMinor)
         {
            setIcon (orangespot);
         }

         if (pluginMajor != kernelMajor)
         {
            setIcon (redspot);
         }

      }

      return this;
   }
}

/*
 * $Log: PluginDownloadDialog.java,v $
 * Revision 1.9  2006/02/07 17:04:12  lowende
 * Applied patch send by Felix Klar: Connection error handling in plugin download dialog enhanced.
 *
 */
