/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.upb.lib.plugins;

import java.io.*;
import java.net.URL;
import java.util.*;
import java.util.zip.*;

import de.uni_paderborn.lib.java.io.ResourceLocator;
import de.upb.tools.fca.*;


/**
 * An instance of this class is responsible for the management of on plugin which is loaded
 * by the plugin manager.
 *
 * @author    Fujaba Development Group
 * @version   $Revision: 1.34 $
 */
public class PluginProperty implements ResourceLocator
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String pluginJarFile;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String pluginHelp = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int neededKernelMajor;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int neededKernelMinor;


   /**
    *Constructor for class PluginProperty
    *
    * @param requiredKernelMajor  No description provided
    * @param requiredKernelMinor  No description provided
    */
   public PluginProperty (int requiredKernelMajor, int requiredKernelMinor)
   {
      neededKernelMajor = requiredKernelMajor;
      neededKernelMinor = requiredKernelMinor;
   }


   /**
    * Get the pluginJarFile attribute of the PluginProperty object
    *
    * @return   The pluginJarFile value
    */
   public String getPluginJarFile()
   {
      return this.pluginJarFile;
   }


   /**
    * Sets the pluginJarFile attribute of the PluginProperty object
    *
    * @param pluginJarFile  The new pluginJarFile value
    */
   public void setPluginJarFile (String pluginJarFile)
   {
      if (pluginJarFile != null)
      {
         this.pluginJarFile = pluginJarFile;
      }
   }


   /**
    * the directory in which the plugin was found.
    *
    * @see   PluginProperty#getAbsolutePath
    * @see   PluginProperty#setAbsolutePath
    */
   private String absolutePath = "";


   /**
    * Get the absolutePath attribute of the PluginProperty object
    *
    * @return   The absolutePath value
    */
   public String getAbsolutePath()
   {
      return absolutePath;
   }


   /**
    * Sets the absolutePath attribute of the PluginProperty object
    *
    * @param newAbsolutePath  The new absolutePath value
    */
   public void setAbsolutePath (String newAbsolutePath)
   {
      if ( (this.absolutePath == null) ||  (this.absolutePath != null && !this.absolutePath.equals (newAbsolutePath)))
      {
         this.absolutePath = newAbsolutePath;
      }
   }


   /**
    * A detailed description for the plugin.
    *
    * @see   PluginProperty#getDescription
    * @see   PluginProperty#setDescription
    */
   private String description = "";


   /**
    * get a detailed description for the plugin.
    *
    * @return   the description
    * @see      PluginProperty#description
    */
   public String getDescription()
   {
      return this.description;
   }


   /**
    * set a new detailed description for the plugin.
    *
    * @param newDescription  the new description
    * @see                   PluginProperty#description
    */
   public void setDescription (String newDescription)
   {
      if ( (this.description == null) ||
          (this.description != null &&
         !this.description.equals (newDescription)))
      {
         this.description = newDescription;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String shortDescription;


   /**
    * Sets the shortDescription attribute of the PluginProperty object
    *
    * @param shortDescription  The new shortDescription value
    */
   public void setShortDescription (String shortDescription)
   {
      if ( (this.shortDescription == null) ||
          (this.shortDescription != null &&
         !this.shortDescription.equals (shortDescription)))
      {
         this.shortDescription = shortDescription;
      }
   }


   /**
    * Get the shortDescription attribute of the PluginProperty object
    *
    * @return   The shortDescription value
    */
   public String getShortDescription()
   {
      return this.shortDescription;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String source;


   /**
    * Sets the source attribute of the PluginProperty object
    *
    * @param source  The new source value
    */
   public void setSource (String source)
   {
      this.source = source;
   }


   /**
    * Get the source attribute of the PluginProperty object
    *
    * @return   The source value
    */
   public String getSource()
   {
      return source;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String vendor;


   /**
    * Sets the vendor attribute of the PluginProperty object
    *
    * @param vendor  The new vendor value
    */
   public void setVendor (String vendor)
   {
      if (vendor != null)
      {
         this.vendor = vendor;
      }
   }


   /**
    * Get the vendor attribute of the PluginProperty object
    *
    * @return   The vendor value
    */
   public String getVendor()
   {
      return this.vendor;
   }


   /**
    * Sets the contact attribute of the PluginProperty object
    *
    * @param contact  The new contact value
    */
   public void setContact (String contact)
   {
      if (contact != null)
      {
         this.contact = contact;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String contact;


   /**
    * Get the contact attribute of the PluginProperty object
    *
    * @return   The contact value
    */
   public String getContact()
   {
      return this.contact;
   }


   /**
    * the major version of the plugin.
    */
   private String majorVersion;


   /**
    * get the value of majorVersion.
    *
    * @return   the value of majorVersion
    */
   public String getMajorVersion()
   {
      return this.majorVersion;
   } // getMajorVersion


   /**
    * set the value of majorVersion.
    *
    * @param value  The new majorVersion value
    */
   public void setMajorVersion (String value)
   {
      if (this.majorVersion != value)
      {
         this.majorVersion = value;
      } // if ()
   } // setMajorVersion


   /**
    * Get the major attribute of the PluginProperty object
    *
    * @return   The major value
    */
   public int getMajor()
   {
      if (this.majorVersion != null)
      {
         Integer integer = new Integer (this.majorVersion);
         return integer.intValue();
      }
      else
      {
         return -1;
      }
   }


   /**
    * the minor version of the plugin.
    */
   private String minorVersion;


   /**
    * get the value of minorVersion.
    *
    * @return   the value of minorVersion
    */
   public String getMinorVersion()
   {
      return this.minorVersion;
   } // getMinorVersion


   /**
    * Get the minor attribute of the PluginProperty object
    *
    * @return   The minor value
    */
   public int getMinor()
   {
      if (this.minorVersion != null)
      {
         Integer integer = new Integer (this.minorVersion);
         return integer.intValue();
      }
      else
      {
         return -1;
      }
   }


   /**
    * set the value of minorVersion.
    *
    * @param value  The new minorVersion value
    */
   public void setMinorVersion (String value)
   {
      if (this.minorVersion != value)
      {
         this.minorVersion = value;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int buildNumber = 0;


   /**
    * Get the buildNumber attribute of the PluginProperty object
    *
    * @return   The buildNumber value
    */
   public int getBuildNumber()
   {
      return buildNumber;
   }


   /**
    * Sets the buildNumber attribute of the PluginProperty object
    *
    * @param buildNumber  The new buildNumber value
    */
   public void setBuildNumber (int buildNumber)
   {
      if (buildNumber > -1)
      {
         this.buildNumber = buildNumber;
      }
   }


   /**
    * Sets the buildNumber attribute of the PluginProperty object
    *
    * @param value  The new buildNumber value
    */
   public void setBuildNumber (String value)
   {
      try
      {
         setBuildNumber (Integer.parseInt (value));
      }
      catch (Exception ex)
      {
         setBuildNumber (0);
      }
   }


   /**
    * Get the version attribute of the PluginProperty object
    *
    * @return   The version value
    */
   public String getVersion()
   {
      return this.majorVersion + "." + this.minorVersion + "." + buildNumber;
   }


   /**
    * flag for the PluginManager to mark a plugin as dirty. A dirty plugin cannot be loaded.
    *
    * @see   PluginProperty#isDirty
    * @see   PluginProperty#setDirty
    */
   private boolean dirty = true;


   /**
    * get the information if the plugin can be loaded or not.
    *
    * @return   The dirty value
    * @see      PluginProperty#dirty
    */
   public boolean isDirty()
   {
      return this.dirty;
   }


   /**
    * mark the plugin as dirty or not.
    *
    * @param newDirty  The new dirty value
    * @see             PluginProperty#dirty
    */
   public void setDirty (boolean newDirty)
   {
      if (this.dirty != newDirty)
      {
         this.dirty = newDirty;
      }
   }


   /**
    * the plugin id for the plugin under which the plugin is stored in the plugin manager's
    * set of plugins.
    *
    * @see   PluginProperty#getPluginID
    * @see   PluginProperty#setPluginID
    */
   private String pluginID = "";


   /**
    * get the id of the plugin.
    *
    * @return   the pluginID of the plugin
    * @see      PluginProperty#pluginID
    */
   public String getPluginID()
   {
      return this.pluginID;
   }


   /**
    * set a new id for the plugin.
    *
    * @param newPluginID  the new pluginID
    * @see                PluginProperty#pluginID
    */
   public void setPluginID (String newPluginID)
   {
      if ( (this.pluginID == null) ||  (this.pluginID != null && !this.pluginID.equals (newPluginID)))
      {
         this.pluginID = newPluginID;
      }
   }


   /**
    * the plugin key to its classloader. In most cases == pluginId.
    *
    * @see   PluginProperty#getClassLoaderKey
    * @see   PluginProperty#setClassLoaderKey
    */
   private String classLoaderKey = "";


   /**
    * get the classLoaderKey of the plugin.
    *
    * @return   the classLoaderKey of the plugin
    * @see      PluginProperty#classLoaderKey
    */
   public String getClassLoaderKey()
   {
      return this.classLoaderKey;
   }


   /**
    * set a new classLoaderKey for the plugin.
    *
    * @param classLoaderKey  the new pluginID
    * @see                   PluginProperty#classLoaderKey
    */
   public void setClassLoaderKey (String classLoaderKey)
   {
      if ( (this.classLoaderKey == null) ||  (this.classLoaderKey != null && !this.classLoaderKey.equals (classLoaderKey)))
      {
         this.classLoaderKey = classLoaderKey;
      }
   }


   /**
    * The user readable name of the plugin.
    *
    * @see   PluginProperty#getName
    * @see   PluginProperty#setName
    */
   private String name = "";


   /**
    * Get the name attribute of the PluginProperty object
    *
    * @return   The name value
    */
   public String getName()
   {
      return this.name;
   }


   /**
    * Sets the name attribute of the PluginProperty object
    *
    * @param newName  The new name value
    */
   public void setName (String newName)
   {
      if ( (this.name == null) ||
          (this.name != null &&
         !this.name.equals (newName)))
      {
         this.name = newName;
      }
   }


   /**
    * the plugin's full qualified class name. This class must be instantiated by the class
    * loader to initialize the plugin.
    *
    * @see   PluginProperty#getRootClass
    * @see   PluginProperty#setRootClass
    */
   private String rootClass = "";


   /**
    * get the plugin's full qualified class name.
    *
    * @return   The rootClass value
    * @see      PluginProperty#rootClass
    */
   public String getRootClass()
   {
      return this.rootClass;
   }


   /**
    * set the plugin's full qualified class name.
    *
    * @param newRootClass  The new rootClass value
    * @see                 PluginProperty#rootClass
    */
   public void setRootClass (String newRootClass)
   {
      if ( (this.rootClass == null) ||
          (this.rootClass != null &&
         !this.rootClass.equals (newRootClass)))
      {
         this.rootClass = newRootClass;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FHashSet classPaths;


   /**
    * Access method for an one to n association.
    *
    * @param classPath  The object added.
    * @return           No description provided
    */
   public boolean addToClassPaths (String classPath)
   {
      boolean changed = false;
      if (classPath != null)
      {
         if (this.classPaths == null)
         {
            this.classPaths = new FHashSet();
         }
         changed = this.classPaths.add (classPath);
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfClassPaths()
   {
      return  ( (this.classPaths == null)
         ? FEmptyIterator.get()
         : this.classPaths.iterator());
   }


   /**
    * Get the inClassPaths attribute of the PluginProperty object
    *
    * @param classPath  No description provided
    * @return           The inClassPaths value
    */
   public boolean isInClassPaths (String classPath)
   {
      return  ( (this.classPaths != null) &&
          (classPath != null) &&
         this.classPaths.contains (classPath));
   }


   /**
    * <pre>
    *                 n      neededBy       n
    * PluginProperty ------------------------- PluginProperty
    *                 dependsOn      neededBy
    * </pre>
    */
   private FHashSet dependsOn;


   /**
    * Access method for an one to n association.
    *
    * @param pluginKey  The object added.
    * @return           No description provided
    */
   public boolean addToDependsOn (String pluginKey)
   {
      boolean changed = false;
      if (pluginKey != null)
      {
         if (this.dependsOn == null)
         {
            this.dependsOn = new FHashSet();
         }
         changed = this.dependsOn.add (pluginKey);
         if (changed)
         {
            // the next line causes an error during loading of plugins
            // addToNeededBy (this.getPluginID());
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pluginKey  No description provided
    * @return           No description provided
    */
   public boolean hasInDependsOn (String pluginKey)
   {
      return  ( (this.dependsOn != null) &&
          (pluginKey != null) &&
         this.dependsOn.contains (pluginKey));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfDependsOn()
   {
      return  ( (this.dependsOn == null)
         ? 0
         : this.dependsOn.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfDependsOn()
   {
      return  ( (this.dependsOn == null)
         ? FEmptyIterator.get()
         : this.dependsOn.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pluginKey  No description provided
    * @return           No description provided
    */
   public boolean removeFromDependsOn (String pluginKey)
   {
      boolean changed = false;
      if ( (this.dependsOn != null) &&  (pluginKey != null))
      {
         changed = this.dependsOn.remove (pluginKey);
         if (changed)
         {
            removeFromNeededBy (this.getPluginID());
         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromDependsOn()
   {
      String tmpKey;
      Iterator iter = this.iteratorOfDependsOn();
      while (iter.hasNext())
      {
         tmpKey = (String) iter.next();
         this.removeFromDependsOn (tmpKey);
      }
   }


   /**
    * <pre>
    *                n         jarFiles         n
    * PluginJarFile ------------------------------ PluginProperty
    *                jarFiles      pluginProperty
    * </pre>
    */
   private FHashSet jarFiles;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToJarFiles (PluginJarFile value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.jarFiles == null)
         {
            this.jarFiles = new FHashSet();
         }
         changed = this.jarFiles.add (value);
         if (changed)
         {
            value.setProperty (this);
         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInJarFiles (PluginJarFile value)
   {
      return  ( (this.jarFiles != null) &&
          (value != null) &&
         this.jarFiles.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfJarFiles()
   {
      return  ( (this.jarFiles == null)
         ? 0
         : this.jarFiles.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfJarFiles()
   {
      return  ( (this.jarFiles == null)
         ? FEmptyIterator.get()
         : this.jarFiles.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromJarFiles (PluginJarFile value)
   {
      boolean changed = false;
      if ( (this.jarFiles != null) &&  (value != null))
      {
         changed = this.jarFiles.remove (value);
         if (changed)
         {
            value.setProperty (null);
         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromJarFiles()
   {
      PluginJarFile tmpValue;
      Iterator iter = this.iteratorOfJarFiles();
      while (iter.hasNext())
      {
         tmpValue = (PluginJarFile) iter.next();
         this.removeFromJarFiles (tmpValue);
         tmpValue.removeYou();
      }
   }


   /**
    * <pre>
    *                  0..1       plugin        n
    * PluginInterface <--------------------------- PluginProperty
    *                  plugin      pluginProperty
    * </pre>
    */
   private PluginInterface plugin;


   /**
    * Get the plugin attribute of the PluginProperty object
    *
    * @return   The plugin value
    */
   public PluginInterface getPlugin()
   {
      return this.plugin;
   }


   /**
    * Sets the plugin attribute of the PluginProperty object
    *
    * @param value  The new plugin value
    * @return       No description provided
    */
   public boolean setPlugin (PluginInterface value)
   {
      boolean changed = false;
      if ( (this.plugin == null && value != null) ||
          (this.plugin != null && !this.plugin.equals (value)))
      {
         this.plugin = value;
         changed = true;
      }

      return changed;
   }


   /**
    * <pre>
    *                 n      neededBy       n
    * PluginProperty ------------------------- PluginProperty
    *                 neededBy      dependsOn
    * </pre>
    */
   private FHashSet neededBy;


   /**
    * Access method for an one to n association.
    *
    * @param pluginKey  The object added.
    * @return           No description provided
    */
   public boolean addToNeededBy (String pluginKey)
   {
      boolean changed = false;
      if (pluginKey != null)
      {
         if (this.neededBy == null)
         {
            this.neededBy = new FHashSet();
         }
         changed = this.neededBy.add (pluginKey);
         if (changed)
         {
            addToDependsOn (this.getPluginID());
         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pluginKey  No description provided
    * @return           No description provided
    */
   public boolean hasInNeededBy (String pluginKey)
   {
      return  ( (this.neededBy != null) &&
          (pluginKey != null) &&
         this.neededBy.contains (pluginKey));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfNeededBy()
   {
      return  ( (this.neededBy == null)
         ? 0
         : this.neededBy.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfNeededBy()
   {
      return  ( (this.neededBy == null)
         ? FEmptyIterator.get()
         : this.neededBy.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pluginKey  No description provided
    * @return           No description provided
    */
   public boolean removeFromNeededBy (String pluginKey)
   {
      boolean changed = false;
      if ( (this.neededBy != null) &&  (pluginKey != null))
      {
         changed = this.neededBy.remove (pluginKey);
         if (changed)
         {
            removeFromDependsOn (this.getPluginID());
         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromNeededBy()
   {
      String tmpKey;
      Iterator iter = this.iteratorOfNeededBy();
      while (iter.hasNext())
      {
         tmpKey = (String) iter.next();
         this.removeFromNeededBy (tmpKey);
      }
   }


   /**
    * <pre>
    *               ------------ 0..1   properties   0..1
    * PluginManager | pluginID |-------------------------- PluginProperty
    *               ----------- manager       properties
    * </pre>
    */
   private PluginManager manager;


   /**
    * Sets the manager attribute of the PluginProperty object
    *
    * @param value  The new manager value
    * @return       No description provided
    */
   public boolean setManager (PluginManager value)
   {
      boolean changed = false;
      if (this.manager != value)
      {
         if (this.manager != null)
         {
            PluginManager oldValue = this.manager;
            this.manager = null;
            oldValue.removeFromProperties (this);
         }
         this.manager = value;
         if (value != null)
         {
            value.addToProperties (this);
         }
         changed = true;
      }

      return changed;
   }


   /**
    * Get the manager attribute of the PluginProperty object
    *
    * @return   The manager value
    */
   public PluginManager getManager()
   {
      return this.manager;
   }


   /**
    * name of the menu xml file
    */
   private String xmlFile = null;


   /**
    * get the value of xmlFile.
    *
    * @return   the value of xmlFile
    */
   public String getXMLFile()
   {
      return this.xmlFile;
   }


   /**
    * set the value of xmlFile.
    *
    * @param value  The new xmlFile value
    */
   public void setXMLFile (String value)
   {
      if (this.xmlFile != value)
      {
         this.xmlFile = value;
      }
   }


   /**
    * Get the resource attribute of the PluginProperty object
    *
    * @param name  No description provided
    * @return      The resource value
    */
   public URL getResource (String name)
   {
      URL url = null;

      //first look in the parent directory of the absolute path
      //(the directory that contains the jar-file)
      if (getAbsolutePath() != null)
      {
         File absoluteFile = new File (getAbsolutePath());
         File parentDir = absoluteFile.getParentFile();

         File resourceFile = new File (parentDir, name);

         if (resourceFile.exists())
         {
            try
            {
               url = resourceFile.getAbsoluteFile().toURI().toURL();
               url.openStream();
            }
            catch (IOException ioEx)
            {
               url = null;
            }
         }
      }

      //then search all jar-Files
      if (url == null)
      {
         Iterator iter = iteratorOfJarFiles();
         while (iter.hasNext())
         {
            PluginJarFile pluginJarFile = (PluginJarFile) iter.next();
            File pluginFile = pluginJarFile.getFile();

            try
            {
               ZipFile zipFile = new ZipFile (pluginFile);
               ZipEntry zipEntry = zipFile.getEntry (name);

               try
               {
                  zipFile.close();
               }
               catch (IOException ex)
               {
               }

               if (zipEntry != null)
               {
                  url = pluginFile.getAbsoluteFile().toURI().toURL();

                  if (pluginFile.getName().toLowerCase().endsWith (".jar"))
                  {
                     url = new URL ("jar:" + url.toString() + "!/" + zipEntry.getName());
                  }
                  else
                  {
                     url = new URL ("zip:" + url.toString() + "!/" + zipEntry.getName());
                  }

                  url.openStream();
               }
            }
            catch (ZipException zEx)
            {
               url = null;
            }
            catch (IOException ioEx)
            {
               url = null;
            }
         }
      }

      //now try to open name as URL directly
      if (url == null)
      {
         try
         {
            url = new URL (name);
            url.openStream();
         }
         catch (IOException ioEx)
         {
            url = null;
         }
      }

      //now try to open name as file
      if (url == null)
      {
         File resourceFile = new File (name);

         if (resourceFile.exists())
         {
            try
            {
               url = resourceFile.getAbsoluteFile().toURI().toURL();
               url.openStream();
            }
            catch (IOException ioEx)
            {
               url = null;
            }
         }
      }

      return url;
   }


   /**
    * Get the resources attribute of the PluginProperty object
    *
    * @param name  No description provided
    * @return      The resources value
    */
   public Enumeration getResources (String name)
   {
      Hashtable resources = new Hashtable();
      URL url = null;

      //first look in the parent directory of the absolute path
      //(the directory that contains the jar-file)
      if (getAbsolutePath() != null)
      {
         File absoluteFile = new File (getAbsolutePath());
         File parentDir = absoluteFile.getParentFile();

         File resourceFile = new File (parentDir, name);

         if (resourceFile.exists())
         {
            try
            {
               url = resourceFile.getAbsoluteFile().toURI().toURL();
               url.openStream();
               resources.put (url, url);
            }
            catch (IOException ioEx)
            {
            }
         }
      }

      //then search all jar-Files
      Iterator iter = iteratorOfJarFiles();
      while (iter.hasNext())
      {
         PluginJarFile pluginJarFile = (PluginJarFile) iter.next();
         File pluginFile = pluginJarFile.getFile();

         try
         {
            ZipFile zipFile = new ZipFile (pluginFile);
            ZipEntry zipEntry = zipFile.getEntry (name);

            try
            {
               zipFile.close();
            }
            catch (IOException ex)
            {
            }

            if (zipEntry != null)
            {
               url = pluginFile.getAbsoluteFile().toURI().toURL();

               if (pluginFile.getName().toLowerCase().endsWith (".jar"))
               {
                  url = new URL ("jar:" + url.toString() + "!/" + zipEntry.getName());
               }
               else
               {
                  url = new URL ("zip:" + url.toString() + "!/" + zipEntry.getName());
               }

               url.openStream();
               resources.put (url, url);
            }
         }
         catch (ZipException zEx)
         {
         }
         catch (IOException ioEx)
         {
         }
      }

      //now try to open name as URL directly
      try
      {
         url = new URL (name);
         url.openStream();
         resources.put (url, url);
      }
      catch (IOException ioEx)
      {
      }

      //now try to open name as file
      File resourceFile = new File (name);
      if (resourceFile.exists())
      {
         try
         {
            url = resourceFile.getAbsoluteFile().toURI().toURL();
            url.openStream();
            resources.put (url, url);
         }
         catch (IOException ioEx)
         {
         }
      }

      return resources.elements();
   }


   /**
    * Get the resourceAsStream attribute of the PluginProperty object
    *
    * @param name  No description provided
    * @return      The resourceAsStream value
    */
   public InputStream getResourceAsStream (String name)
   {
      InputStream inputStream = null;

      URL url = getResource (name);
      if (url != null)
      {
         try
         {
            inputStream = url.openStream();
         }
         catch (IOException ioEx)
         {
         }
      }
      return inputStream;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      removeAllFromDependsOn();
      removeAllFromJarFiles();

      PluginInterface tmpPlugin = getPlugin();
      if (tmpPlugin != null)
      {
         setPlugin (null);
      }

      PluginManager tmpManager = getManager();
      if (tmpManager != null)
      {
         setManager (null);
      }
   } // removeYou


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator dependsIterCopy()
   {
      return  ((FHashSet) this.dependsOn.clone()).iterator();
   }


   /**
    * Get the pluginHelp attribute of the PluginProperty object
    *
    * @return   The pluginHelp value
    */
   public String getPluginHelp()
   {
      return pluginHelp;
   }


   /**
    * @param string
    */
   public void setPluginHelp (String string)
   {
      pluginHelp = string;
   }


   /**
    * Get the neededKernelMajor attribute of the PluginProperty object
    *
    * @return   The neededKernelMajor value
    */
   public int getNeededKernelMajor()
   {
      return neededKernelMajor;
   }


   /**
    * Get the neededKernelMinor attribute of the PluginProperty object
    *
    * @return   The neededKernelMinor value
    */
   public int getNeededKernelMinor()
   {
      return neededKernelMinor;
   }


   /**
    * @param i
    */
   public void setNeededKernelMajor (int i)
   {
      neededKernelMajor = i;
   }


   /**
    * @param i
    */
   public void setNeededKernelMinor (int i)
   {
      neededKernelMinor = i;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      return name + " Version " + majorVersion + "." + minorVersion + " build " + buildNumber;
   }

}

/*
 * $Log: PluginProperty.java,v $
 * Revision 1.34  2005/04/25 15:42:50  mm
 * plugins not specifying required kernel version are loaded again
 *
 */
