/*
 * The FUJABA [Just Draw It!] project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2002 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: fujaba@upb.de
 *
 */
package de.uni_paderborn.lib.util;

import java.lang.reflect.*;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;


/**
 * Visitor pattern implementation.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.6 $
 */
public class Visitor
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient Map visitorCache = new HashMap();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final transient Map visitMethodCache;


   /**
    * Constructor for class Visitor
    */
   public Visitor()
   {
      this (true);
   }


   /**
    * Constructor for class Visitor
    *
    * @param ignoreUnhandled  No description provided
    */
   public Visitor (boolean ignoreUnhandled)
   {
      this.ignoreUnhandled = ignoreUnhandled;
      synchronized (visitorCache)
      {
         Map visit = (Map) visitorCache.get (this.getClass());
         if (visit == null)
         {
            visit = new HashMap();
            visitorCache.put (this.getClass(), visit);
         }
         visitMethodCache = visit;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean ignoreUnhandled;


   /**
    * Get the ignoreUnhandled attribute of the Visitor object
    *
    * @return   The ignoreUnhandled value
    */
   public boolean isIgnoreUnhandled()
   {
      return ignoreUnhandled;
   }


   /**
    * Sets the ignoreUnhandled attribute of the Visitor object
    *
    * @param ignoreUnknown  The new ignoreUnhandled value
    */
   public void setIgnoreUnhandled (boolean ignoreUnknown)
   {
      this.ignoreUnhandled = ignoreUnknown;
   }


   /**
    * Visits an element.
    *
    * @param object  No description provided
    */
   public void visit (Object object)
   {
      Class clazz = object.getClass();
      try
      {
         Method method = getMethod (clazz);
         method.invoke (this, new Object[]{object});
      }
      catch (NoSuchMethodException e)
      {
         if (!ignoreUnhandled)
         {
            UnsupportedOperationException uoe = new UnsupportedOperationException ("No visitor method found for " + clazz.getName());
            uoe.initCause (e);
            throw uoe;
         }
      }
      catch (IllegalArgumentException e)
      {
         System.out.println (e.getMessage());
         e.printStackTrace();
      }
      catch (IllegalAccessException e)
      {
         System.out.println (e.getMessage());
         e.printStackTrace();
      }
      catch (InvocationTargetException e)
      {
         System.out.println (e.getMessage());
         e.printStackTrace();
      }
   }


   /**
    * Find a suitable visitor method.<p>
    *
    * A suitable visitor method is of the form <code>visit&lt;<i>classname</i> &gt;</code>
    * where <i>classname</i> is the name of the class on which the visitor is used, one of
    * its superclasses or one of its implemented interfaces. <p>
    *
    * This implementation uses the following search order:
    * <ol>
    *   <li> Use the name of the class itself</li>
    *   <li> Use the names of the interfaces implemented by the class in order of their appearance
    *   in the <code>implements</code> clause of the class</li>
    *   <li> Use the names of the super-interfaces of the implemented interfaces in breadth-first
    *   order</li>
    *   <li> Recursively use the name of the classes superclass as delivered by {@link java.lang.Class#getSuperclass}
    *   and start with 1. again</li>
    * </ol>
    *
    *
    * @param clazz                   The class for which a suitable visitor method is needed
    * @return                        The visitor method if one is found or null
    * @see                           #findMethod
    * @see                           java.lang.Class#getSuperclass
    * @throws NoSuchMethodException  Exception description not provided
    */
   protected Method getMethod (Class clazz) throws NoSuchMethodException
   {
      synchronized (visitMethodCache)
      {
         Method method = (Method) visitMethodCache.get (clazz);

         if (method == null)
         {
            Class tmpClass = clazz;
            if (!visitMethodCache.containsKey (clazz))
            {
               LinkedList classes = new LinkedList();
               StringBuffer buffer = new StringBuffer();

               while (method == null && tmpClass != null)
               {
                  classes.clear();
                  classes.add (tmpClass);
                  while (method == null && classes.size() > 0)
                  {
                     method = findMethod (classes, buffer);
                  }
                  if (method == null)
                  {
                     tmpClass = tmpClass.getSuperclass();
                  }
               }

               visitMethodCache.put (clazz, method);
            }
            if (method == null)
            {
               throw new NoSuchMethodException ("No visitor method found for " + clazz.getName());
            }
         }
         return method;
      }
   }


   /**
    * Find a suitable visitor method.<p>
    *
    * A suitable visitor method is of the form <code>visit&lt;<i>classname</i> &gt;</code>
    * where <i>classname</i> is the name of the class on which the visitor is used, one of
    * its superclasses or one of its implemented interfaces. <p>
    *
    * This method implements (together with getMethod) a breadth-first search beginning on
    * the first element of the provided list of classes. That class is checked first and if
    * no suitable method is found, all implemented interfaces of the class are enqueued in
    * the provided class list, which can be used for subsequent calls by getMethod for the
    * breadth-first search.
    *
    * @param classList  List of classes to search. Calling the method will dequeue the first
    *      class in the List and if no suitable method is found, enqueue all interfaces implemented
    *      by that class.
    * @param buffer     StringBuffer to use for efficiently building the method name
    * @return           a suitable visitor method or null if none was found
    * @see              #getMethod
    */
   protected Method findMethod (LinkedList classList, StringBuffer buffer)
   {
      Class clazz = (Class) classList.removeFirst();
      Method method = null;
      createMethodName (clazz, buffer);
      try
      {
         method = getClass().getMethod (buffer.toString(), new Class[]{clazz});
      }
      catch (NoSuchMethodException e)
      {
         Class[] interfaces = clazz.getInterfaces();
         for (int i = 0; i < interfaces.length; i++)
         {
            classList.add (interfaces[i]);
         }
      }
      return method;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param clazz   No description provided
    * @param buffer  No description provided
    */
   protected void createMethodName (Class clazz, StringBuffer buffer)
   {
      String methodName = clazz.getName();

      buffer.delete (0, buffer.length());
      buffer.append (methodName);
      int baseIndex = buffer.lastIndexOf (".") + 1;
      if (baseIndex > 0)
      {
         buffer.delete (0, baseIndex);
      }
      baseIndex = buffer.lastIndexOf ("$") + 1;
      if (baseIndex > 0)
      {
         buffer.delete (0, baseIndex);
      }
      buffer.insert (0, "visit");
   }
}

/*
 * $Log: Visitor.java,v $
 * Revision 1.6  2004/08/02 07:34:13  lowende
 * Getter method for class loader chain of UPBClassloader added. Javadoc corrected.
 *
 */
