/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.lib.javax.swing.progress;

import java.awt.*;
import java.awt.event.*;
import java.util.Vector;

import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.plaf.basic.BasicProgressBarUI;


/**
 * A Swing Progress-Bar.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.5 $
 */
public class ProgressDialog extends JDialog
{
   /**
    * the task which is observed.
    */
   Task task = null;

   /**
    * internal used timer to actualize the progress dialog.
    */
   Timer timer = null;

   /**
    * a progress dialog may have more than one progress bar.
    */
   Vector progressBars = null;

   /**
    * internal used layout.
    */
   private GridBagLayout gridBag = null;

   /**
    * internal used constraints.
    */
   private GridBagConstraints constraints = null;

   /**
    * internal used panel.
    */
   private JPanel panel = null;

   /**
    * is true when reading is finished.
    */
   boolean finishedRead = false;

   /**
    * flag to decide if the percentage status is shown or not.
    */
   boolean showPercent = false;
   /**
    * the title of the progress dialog.
    */
   String title = null;


   /**
    * Constructor for class ProgressDialog
    */
   public ProgressDialog()
   {
      super();
      initComponents();
      setModal (true);
      timer = new Timer (1000, createTimerListener());
   }


   /**
    * Constructor for class ProgressDialog
    *
    * @param parent  the dialog's parent.
    */
   public ProgressDialog (JDialog parent)
   {
      super (parent);
      initComponents();
      setModal (true);
      timer = new Timer (1000, createTimerListener());
   }


   /**
    * Constructor for class ProgressDialog
    *
    * @param parent  the dialog's parent
    */
   public ProgressDialog (Frame parent)
   {
      super (parent);
      initComponents();
      setModal (true);
      timer = new Timer (1000, createTimerListener());
   }


   /**
    * initialize the dialog. Set the internal stored components.
    */
   protected void initComponents()
   {
      getContentPane().removeAll();

      title = "Working - Please Wait";
      setTitle (title);
      // init content pane
      if (gridBag == null)
      {
         gridBag = new GridBagLayout();
      } // end of if ()

      if (constraints == null)
      {
         constraints = new GridBagConstraints();
         constraints.insets = new Insets (5, 5, 5, 5);
      } // end of if ()

      constraints.gridwidth = 1;
      constraints.gridheight = 1;
      constraints.weightx = 1;
      constraints.weighty = 1;
      constraints.gridx = 0;

      if (panel == null)
      {
         panel = new JPanel();
         panel.setLayout (gridBag);
      } // end of if ()
      panel.removeAll();

      if (progressBars == null)
      {
         progressBars = new Vector();
      } // end of if ()
      progressBars.clear();

      if (task != null)
      {
         int gridCount = 0;
         while (progressBars.size() < task.getBarCount())
         {
            JPanel progressPanel = createProgressPanel (progressBars);

            constraints.gridy = gridCount;
            panel.add (progressPanel, constraints);
            gridCount = gridCount + 1;
         } // end of while ()
      }
      getContentPane().add (panel);
      pack();
   }


   /**
    * internal used method to initialize progress bar panel of this progress dialog.
    *
    * @param progressBars  the progress bars of the dialog
    * @return              the panel which shows the progress bars.
    */
   private JPanel createProgressPanel (Vector progressBars)
   {
      // create Catalog section
      TitledBorder progressBorder = new TitledBorder (task.getMessage (progressBars.size()));
      JPanel progressBorderPanel = new JPanel (new BorderLayout());
      progressBorderPanel.setBorder (progressBorder);

      JProgressBar progressBar = new JProgressBar();
      progressBar.setMaximum (task.getLength (progressBars.size()));
      progressBar.setMinimum (0);
      progressBar.setValue (0);
      progressBar.setStringPainted (false);
      progressBar.setOrientation (SwingConstants.HORIZONTAL);
      progressBar.setForeground (new Color (0, 0, 127));
      progressBar.setUI (new ProgressDialogBarUI());

      progressBars.add (progressBar);
      progressBorderPanel.add (progressBar, BorderLayout.CENTER);

      progressBorderPanel.setPreferredSize (new Dimension (400, 45));

      return progressBorderPanel;
   }


   /**
    * returns an internal used listener to observe the timer.
    *
    * @return   the internal used listener
    */
   private ActionListener createTimerListener()
   {
      ActionListener listener =
         new ActionListener()
         {
            public void actionPerformed (ActionEvent event)
            {
               if (task == null ||  (task.isFinished() && finishedRead))
               {
                  timer.stop();
                  setVisible (false);
               } // end of if ()
               else
               {
                  for (int i = 0; i < progressBars.size(); i++)
                  {
                     JProgressBar progressBar = (JProgressBar) progressBars.get (i);
                     progressBar.setMaximum (task.getLength (i));
                     progressBar.setValue (task.getCurrent (i));
                     if (i == 0 && showPercent)
                     {
                        long percent = Math.round (progressBar.getPercentComplete() * 10000);
                        double value = (double) percent / 100;
                        String pString = " (" + value + "% completed)";
                        setTitle (title + pString);
                     } // end of if ()

                  } // end of for ()

                  if (task.isFinished())
                  {
                     finishedRead = true;
                  } // end of if ()

                  if (!isVisible())
                  {
                     setVisible (true);
                  } // end of if ()

               } // end of else
            }
         };
      return listener;
   }


   /**
    * start the given task and reset the progress dialog.
    *
    * @param task  the task which must be executed.
    */
   public void startTask (Task task)
   {
      finishedRead = false;
      if (this.task != null)
      {
         this.task.stop();
         timer.stop();
         for (int i = 0; i < progressBars.size(); i++)
         {
            JProgressBar progressBar = (JProgressBar) progressBars.get (i);
            progressBar.setValue (0);
            progressBar.setMaximum (0);
            progressBar.setString ("");
         } // end of for ()

      } // end of if ()

      this.task = task;
      initComponents();

      if (this.task != null)
      {
         this.task.start();
         timer.start();
         setVisible (true);
      } // end of if ()
      else
      {
         setVisible (false);
      } // end of else
   }


   /**
    * show the percentage status in the dialog.
    *
    * @param value  if true the percentage status is shown.
    */
   public void setShowPercent (boolean value)
   {
      showPercent = value;
   }


   /**
    * a special UI for the progress bar.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.5 $
    */
   class ProgressDialogBarUI extends BasicProgressBarUI
   {
      /**
       * Constructor for class ProgressDialogBarUI
       */
      ProgressDialogBarUI()
      {
         super();
      }


      /**
       * overwrites the paint method of PanelUI.
       *
       * @param g  the graphics object this UI paints
       * @param c  the component which is painted
       */
      public void paint (Graphics g, JComponent c)
      {
         setCellSpacing (5);
         setCellLength (15);
         super.paint (g, c);
      }
   }

}

/*
 * $Log: ProgressDialog.java,v $
 * Revision 1.5  2004/11/10 09:24:38  lowende
 * Removed some compile warnings.
 *
 */
