/**********************************************************************
 *
 *   FreeDoko a Doppelkopf-Game
 *    
 *   Copyright (C) 2001-2007  by Diether Knof and Borg Enders
 *
 *   This program is free software; you can redistribute it and/or 
 *   modify it under the terms of the GNU General Public License as 
 *   published by the Free Software Foundation; either version 2 of 
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details. 
 *   You can find this license in the file 'gpl.txt'.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *   MA  02111-1307  USA
 *
 *  Contact:
 *    Diether Knof dknof@gmx.de
 *    Borg Enders  borg@borgsoft.de
 *
 *********************************************************************/

#include "cards_information.of_player.weighting.heuristics.h"

#include "cards_information.of_player.h"

#include "../../card/hand_card.h"
#include "../../card/trick.h"
#include "ai.h"

#include "../../game/game.h"
#include "../../party/rule.h"

// For configuration values search for '*Value*'

#ifndef RELEASE
// whether to output the changes of the weightings
#define INFO_WEIGHTING_CHANGE
// the ai whose information are written
// (what the ai assumes of the other players)
#define DEBUG_AI 5
#endif


#ifdef INFO_WEIGHTING_CHANGE
#define CHANGE_WEIGHTING(card, value) \
  (((ai.no() == DEBUG_AI) && !ai.game().isvirtual()) \
   ? (cout << "weighting change: " << played_card.player().no() << ": " << setw(13) << card << ": " << setw(4) << value << "   (" << heuristic_name << ")\n") \
   : cout), \
(weightings[card] += value)
#else
#define CHANGE_WEIGHTING(card, value) \
  (void)heuristic_name, \
  weightings[card] += value
#endif

namespace CardsInformationOfPlayerWeightingHeuristics {

  /* Further ideas
   *
   * - opposite of 'choose pfund for partner'
   *   if a player has played his fox as startcard:
   *   o teams unknown: the player has a club queen and all higher cards
   * - club queen played
   *   o no second club queen
   */


  /**
   ** the player has accepted a poverty
   ** we will most surly have no color
   **
   ** @param	player       the player who has accepted the poverty
   ** @param	ai           the ai that analyses
   ** @param    weightings   the card weightings
   ** @param    trumpno      the number of returnes trumps
   **
   ** @return    -
   **
   ** @author    Diether Knof
   **
   ** @version   0.7.4
   **
   ** @todo      use the information of the returned cards
   **            (if trump, than there is a better chance)
   **/
  void poverty_returned(Player const& player,
			Ai const& ai,
			map<Card, int>& weightings,
			unsigned const trumpno)
  {
    /* Idea
     *
     * the player who accepts the poverty will have only trump
     */

    // the name of the heuristic
    char const* const heuristic_name = "poverty returned";

    // the game
    Game const& game = player.game();

    for (vector<Card>::const_iterator
	 c = game.rule().cards().begin();
	 c != game.rule().cards().end();
	 ++c) {
      if (!c->istrump(game)) {
	HandCard const played_card(player.hand(), *c);
	CHANGE_WEIGHTING(*c, -10 - 5 * static_cast<int>(trumpno)); // *Value*
      }
    } // for (c \in valid cards)

    return ;
  } // void poverty_returned(Player player, Ai ai, map<Card, int>& weightings, unsigned trumpno)

  /**
   ** a player has not started with a color ace
   ** If the player is startplayer and has not started with a color ace,
   ** it is assumed that he does not have one.
   ** Problem: He has so many cards of the color so that he knows he cannot
   **          get the trick
   **
   ** @param     played_card    the card played by the player
   ** @param     trick          the current trick
   ** @param     ai             the ai that analyses
   ** @param     weightings     the card weightings
   **
   ** @return    -
   **
   ** @author    Diether Knof
   **
   ** @version   0.7.3
   **
   ** @todo      marriage
   **/
  void
    no_color_ace(HandCard const& played_card,
		 Trick const& trick,
		 Ai const& ai,
		 map<Card, int>& weightings)
    {
      /* Idea
       *
       * A player would start with a color ace if he has one.
       * And in the first run of a color he would play his ace.
       */

      // the name of the heuristic
      char const* const heuristic_name = "no color ace";

      // the player of the card
      Player const& player = played_card.player();
      // the game
      Game const& game = player.game();

      // a normal game or a marriage
      if (!(   (game.type() == GAMETYPE::NORMAL)
	    || (game.type() == GAMETYPE::GENSCHER)
	    || (game.type() == GAMETYPE::MARRIAGE)))
	return ;

      // startplayer
      if (trick.startplayerno() == player.no()) {

        // In a marriage not the solo player,
        // if he determines the marriage with his card.
        if (   (game.type() == GAMETYPE::MARRIAGE)
            && (player.no() == game.soloplayer().no())
            && is_selector(played_card.tcolor(),
                           game.marriage_selector()) )
          return ;

        // has not played a color ace
        if (!(   (played_card.value() != Card::ACE)
              || played_card.istrump()))
          return ;

        // For all colors which have not run, yet, the player is assumed to not
        // have the ace.
        // ToDo: check also, that noone has thrown the color
        for (vector<Card::Color>::const_iterator
             c = ai.game().rule().card_colors().begin();
             c != ai.game().rule().card_colors().end();
             ++c)
          if (!Card(*c, Card::ACE).istrump(game))
            if (!ai.cards_information().played(Card(*c, Card::ACE)))
              CHANGE_WEIGHTING(Card(*c, Card::ACE),
                               -2 * (game.numberof(*c)
                                     - (game.rule()(Rule::NUMBER_OF_SAME_CARDS))
                                     )); // *Value*
      } else { // if !(startplayer)
        // color trick
        if (trick.startcard().istrump())
          return ;

        Card::TColor const color = trick.startcard().tcolor();

        // first color run
        if (!(ai.cards_information().color_runs(color) == 0))
          return ;

        // player has not played an ace
        if (played_card.value() == Card::ACE)
          return ;

        // winner card is no ace and no trump
        if (   (trick.winnercard().value() == Card::ACE)
            || trick.winnercard().istrump())
          return ;

        CHANGE_WEIGHTING(Card(color, Card::ACE), -10); // *Value*
      } // if !(startplayer

      return ;
    } // void no_color_ace(...)

  /**
   ** a player has served a color trick
   ** if the winnerplayer is not of the own team and there is no player in
   ** the own team, the player does have no card below the played one
   **
   ** @param     played_card    the card played by the player
   ** @param     trick          the current trick
   ** @param     ai             the ai that analyses
   ** @param     weightings     the card weightings
   **
   ** @return    -
   **
   ** @author    Diether Knof
   **
   ** @version   0.7.3
   **
   ** @todo      other gametypes
   **/
  void
    served_color(HandCard const& played_card,
                 Trick const& trick,
                 Ai const& ai,
                 map<Card, int>& weightings)
    {
      /* Idea:
       * If no team is known, the player plays the lowest card in a color trick
       */

      // the name of the heuristic
      char const* const heuristic_name = "served color";

      // the player of the card
      Player const& player = played_card.player();
      // the game
      Game const& game = player.game();

      // not startplayer
      if (trick.startplayerno() == player.no())
        return ;

      // color trick
      if (trick.startcard().istrump())
        return ;

      // color trick served
      if (played_card.tcolor() != trick.startcard().tcolor())
        return ;

      // it is not the winnerplayer
      if (trick.winnerplayer().no() == player.no())
        return ;

      // the winnerplayer is not of the team as the player
      // note: 'TeamInfo' makes the opposite assumption:
      //         a high card played is a sign for the same team
      if (ai.teaminfo(trick.winnerplayer()) != ::TEAM::UNKNOWN)
        if (maybe_to_team(ai.teaminfo(trick.winnerplayer()))
            == maybe_to_team(ai.teaminfo(player)))
          return ;

      // no team of the players behind is known
      for (unsigned c = game.playerno() - 1; c > trick.actcardno(); --c)
        if (ai.teaminfo(trick.player_of_card(c)) != TEAM::UNKNOWN)
          return ;

      switch (game.type()) {
      case GAMETYPE::NORMAL:
      case GAMETYPE::POVERTY:
      case GAMETYPE::GENSCHER:
      case GAMETYPE::MARRIAGE:
      case GAMETYPE::MARRIAGE_SOLO:
      case GAMETYPE::MARRIAGE_SILENT:
      case GAMETYPE::SOLO_CLUB:
      case GAMETYPE::SOLO_HEART:
      case GAMETYPE::SOLO_SPADE:
      case GAMETYPE::SOLO_DIAMOND:
        switch (played_card.value()) {
        case Card::ACE:
          // add value: 8 + point difference
          if (!Card(played_card.color(), Card::TEN).istrump(game))
            CHANGE_WEIGHTING(Card(played_card.color(), Card::TEN), -9); // *Value*
          if (!Card(played_card.color(), Card::KING).istrump(game))
            CHANGE_WEIGHTING(Card(played_card.color(), Card::KING), -15); // *Value*
          if (game.rule()(Rule::WITH_NINES))
            if (!Card(played_card.color(), Card::NINE).istrump(game))
              CHANGE_WEIGHTING(Card(played_card.color(), Card::NINE), -19); // *Value*
          break;
        case Card::TEN:
          // add value: 4 + point difference
          if (!Card(played_card.color(), Card::KING).istrump(game))
            CHANGE_WEIGHTING(Card(played_card.color(), Card::KING), -14); // *Value*
          if (game.rule()(Rule::WITH_NINES))
            if (!Card(played_card.color(), Card::NINE).istrump(game))
              CHANGE_WEIGHTING(Card(played_card.color(), Card::NINE), -14); // *Value*
          break;
        case Card::KING:
          // add value: 4 + point difference
          if (game.rule()(Rule::WITH_NINES))
            if (!Card(played_card.color(), Card::NINE).istrump(game))
              CHANGE_WEIGHTING(Card(played_card.color(), Card::NINE), -8); // *Value*
          break;
        default:
          break;
        } // switch (played_card.value())
        break;

      default:
        // ToDo
        break;
      } // switch (game.type())

      return ;
    } // void served_color(...)

  /**
   ** a player has jabbed a color trick in the first run
   ** The order of jabbing would be fox, trump ten, trump king, jacks, ...
   **
   ** @param     played_card    the card played by the player
   ** @param     trick          the current trick
   ** @param     ai             the ai that analyses
   ** @param     weightings     the card weightings
   **
   ** @return    -
   **
   ** @author    Diether Knof
   **
   ** @version   0.7.4
   **
   ** @todo      check, that no other player has jabbed
   ** @todo      other order for heart (without nines)
   **/
  void
    color_trick_jabbed_first_run(HandCard const& played_card,
                                 Trick const& trick,
                                 Ai const& ai,
                                 map<Card, int>& weightings)
    {
      /* Idea:
       * If the first color run is jabbed, the player will take low trump.
       */

      // the name of the heuristic
      char const* const heuristic_name = "color trick jabbed first run";

      // the player of the card
      Player const& player = played_card.player();
      // the game
      Game const& game = player.game();

      // normal game
      if (   GAMETYPE::is_solo(game.type())
          || (game.type() == GAMETYPE::POVERTY))
        return ;

      // not startplayer
      if (trick.startplayerno() == player.no())
        return ;

      // color trick
      if (trick.startcard().istrump())
        return ;

      // first run or last card
      if (   (ai.cards_information().color_runs(trick.startcard().color()) > 0)
          && !trick.isfull())
        return ;

      // color trick jabbed
      if (!played_card.istrump())
        return ;

      // the cards in the order to play
      list<pair<Card, int> > cards; // *Value*
      if (!game.swines_announced())
        cards.push_back(make_pair(Card(game.trumpcolor(), Card::ACE), -20));
      if (!(   (game.trumpcolor() == Card::HEART)
            && game.rule()(Rule::DOLLEN)))
        cards.push_back(make_pair(Card(game.trumpcolor(), Card::TEN), -10));
      if (!(   !game.rule()(Rule::WITH_NINES)
            && game.hyperswines_announced()) )
        cards.push_back(make_pair(Card(game.trumpcolor(), Card::KING), -9));
      cards.push_back(make_pair(Card(Card::DIAMOND, Card::JACK), -6));
      cards.push_back(make_pair(Card(Card::HEART,   Card::JACK), -6));
      cards.push_back(make_pair(Card(Card::SPADE,   Card::JACK), -6));
      cards.push_back(make_pair(Card(Card::CLUB,    Card::JACK), -4));
      if (   game.rule()(Rule::WITH_NINES)
          && !game.hyperswines_announced())
        cards.push_back(make_pair(Card(game.trumpcolor(), Card::NINE), -5));
      cards.push_back(make_pair(Card(Card::DIAMOND, Card::QUEEN), -5));
      cards.push_back(make_pair(Card(Card::HEART,   Card::QUEEN), -5));
      cards.push_back(make_pair(Card(Card::SPADE,   Card::QUEEN), -5));

      // search the former winnercard
      Trick former_trick(trick.startplayer());
      for (unsigned c = 0; c < trick.actcardno() - 1; ++c)
        former_trick += trick.card(c);

      // the former winnercard
      HandCard const& former_winnercard = former_trick.winnercard();

      for (list<pair<Card, int> >::const_iterator
           card = cards.begin();
           card != cards.end();
           ++card) {
        if (played_card == card->first)
          break;
        else if (former_winnercard.less(card->first))
          CHANGE_WEIGHTING(card->first, card->second);
      }

      return ;
    } // void color_trick_jabbed_first_run(...)

  /**
   ** the player has jabbed the trick as last player
   ** He will not have the cards between the highest card before
   ** and his played card
   **
   ** @param     played_card    the card played by the player
   ** @param     trick          the current trick
   ** @param     ai             the ai that analyses
   ** @param     weightings     the card weightings
   **
   ** @return    -
   **
   ** @author    Diether Knof
   **
   ** @version   0.7.4
   **/
  void
    last_player_jabbed_trump(HandCard const& played_card,
                             Trick const& trick,
                             Ai const& ai,
                             map<Card, int>& weightings)
    {
      /* Idea:
       * If the player has jabbed a trick,
       * he will not have a cards between the former winnercard and his card.
       */

      // the name of the heuristic
      char const* const heuristic_name = "last player jabbed trump";

      // the player of the card
      Player const& player = played_card.player();

      // last player in the trick
      if (!trick.isfull())
        return ;

      // winnerplayer is the player
      if (!(trick.winnerplayerno() == player.no()))
        return ;

      // the player does not show himself with a club queen
      if (   (played_card == Card::CLUB_QUEEN)
          && (player.game().type() == GAMETYPE::NORMAL)
          && (player.announcement() == ANNOUNCEMENT::NOANNOUNCEMENT) )
        return ;

      // search the former winnercard
      Trick former_trick(trick.startplayer());
      for (unsigned c = 0; c < trick.actcardno() - 1; ++c)
        former_trick += trick.card(c);

      // the former winnercard
      HandCard const& former_winnercard = former_trick.winnercard();

      // the former winnercard is a trump card
      if (!former_winnercard.istrump())
        return ;

      // the cards between
      list<Card> const cards_between
        = trick.game().cards_between(former_winnercard, played_card);

      for (list<Card>::const_iterator c = cards_between.begin();
           c != cards_between.end();
           ++c)
        CHANGE_WEIGHTING(*c, -4); // *Value*

      return ;
    } // void last_player_jabbed_trump(...)

  /**
   ** the player has served as last player
   ** If there are many points, he will not have a card higher
   ** than the winnercard
   **
   ** @param     played_card    the card played by the player
   ** @param     trick          the current trick
   ** @param     ai             the ai that analyses
   ** @param     weightings     the card weightings
   **
   ** @return    -
   **
   ** @author    Diether Knof
   **
   ** @version   0.7.3
   **/
  void
    last_player_served_trump(HandCard const& played_card,
                             Trick const& trick,
                             Ai const& ai,
                             map<Card, int>& weightings)
    {
      /* Idea:
       * If the player has not jabbed a trick with many points,
       * he will not have a card higher than the winnercard.
       */

      // the name of the heuristic
      char const* const heuristic_name = "last player served trump";

      // last player in the trick
      if (!trick.isfull())
        return ;

      // trick does not go to the team of the last player
      if (   (trick.winnerteam() != TEAM::UNKNOWN)
          && (::maybe_to_team(trick.winnerteam())
              == (ai.teaminfo(played_card.player())) ) )
        return ;

      // enough points in the trick
      if (!(trick.points() >= 16)) // *Value*
        return ;

      // trump trick
      if (!trick.startcard().istrump())
        return ;

      // the higher cards
      list<Card> const higher_cards
        = trick.game().higher_cards(trick.winnercard());
      // the modification value
      int const value = -((static_cast<int>(trick.points()) - 10)
                          / 4); // *Value*

      // ToDo: use 'TrickWeighting' for the value (if > 0)
      for (list<Card>::const_iterator c = higher_cards.begin();
           c != higher_cards.end();
           ++c)
        CHANGE_WEIGHTING(*c, value);

      return ;
    } // void last_player_served_trump(...)

  /**
   ** the player has no second diamond ace because he has not announced swines
   **
   ** @param     played_card    the card played by the player
   ** @param     trick          the current trick
   ** @param     ai             the ai that analyses
   ** @param     weightings     the card weightings
   **
   ** @return    -
   **
   ** @author    Diether Knof
   **
   ** @version   0.7.5
   **/
  void
    no_swines(HandCard const& played_card,
              Trick const& trick,
              Ai const& ai,
              map<Card, int>& weightings)
    {
      /* Idea:
       * If the player has played a fox and has not announced swines,
       * he has no second fox.
       * Same idea for hyperswines
       */

      // the name of the heuristic
      char const* const heuristic_name = "no swines";

      Game const& game = trick.game();

      if (   game.rule()(Rule::SWINES)
          && (played_card == game.swine())
          && !game.rule()(Rule::SWINE_ONLY_SECOND)
          && (ai.cards_information().of_player(ai).played(played_card) == 1)
          && (  (game.type() == GAMETYPE::POVERTY)
              ? game.rule()(Rule::SWINES_IN_POVERTY)
              : GAMETYPE::is_solo(game.type())
              ? game.rule()(Rule::SWINES_IN_SOLO)
              : true)) {
        CHANGE_WEIGHTING(played_card, -50); // *Value*
      }

      {
        // the name of the heuristic
        char const* const heuristic_name = "no hyperswines";
        if (   game.rule()(Rule::HYPERSWINES)
            && (played_card == game.hyperswine())
            && (ai.cards_information().of_player(ai).played(played_card) == 1)
            && game.swines_announced()
            && (   game.rule()(Rule::SWINES_AND_HYPERSWINES_JOINT)
                || (game.swines_owner()->no() != ai.no()))
            && (  (game.type() == GAMETYPE::POVERTY)
                ? game.rule()(Rule::HYPERSWINES_IN_POVERTY)
                : GAMETYPE::is_solo(game.type())
                ? game.rule()(Rule::HYPERSWINES_IN_SOLO)
                : true)) {
          CHANGE_WEIGHTING(played_card, -50); // *Value*
        }
      }

      return ;
    } // void no_swines(...)

  /**
   ** assumption: the player does not play a silent marriage
   **
   ** @param     played_card    the card played by the player
   ** @param     trick          the current trick
   ** @param     ai             the ai that analyses
   ** @param     weightings     the card weightings
   **
   ** @return    -
   **
   ** @author    Diether Knof
   **
   ** @version   0.7.5
   **/
  void
    no_silent_marriage(HandCard const& played_card,
                       Trick const& trick,
                       Ai const& ai,
                       map<Card, int>& weightings)
    {
      /* Idea:
       * If the player has played one club queen he will not have a second.
       * We assume he does not play a silent marriage
       */

      // the name of the heuristic
      char const* const heuristic_name = "no silent marriage";

      if (   (trick.game().type() == GAMETYPE::NORMAL)
          && (played_card == Card::CLUB_QUEEN)) {
        CHANGE_WEIGHTING(played_card, -10); // *Value*
      }

      return ;
    } // void no_silent_marriage(...)

  /**
   ** the soloplayer will probably start with a ten when he has both aces
   **
   ** @param     played_card    the card played by the player
   ** @param     trick          the current trick
   ** @param     ai             the ai that analyses
   ** @param     weightings     the card weightings
   **
   ** @return    -
   **
   ** @author    Diether Knof
   **
   ** @version   0.7.3
   **
   ** @todo      other gametypes
   **/
  void
    meatless_startcard(HandCard const& played_card,
                       Trick const& trick,
                       Ai const& ai,
                       map<Card, int>& weightings)
    {
      /* Idea:
       * If the soloplayer in a meatless solo starts with a lower card,
       * he probably will not have any higher one.
       */

      // the name of the heuristic
      char const* const heuristic_name = "meatless startcard";

      // the player of the card
      Player const& player = played_card.player();
      // the game
      Game const& game = player.game();

      if (game.type() != GAMETYPE::SOLO_MEATLESS)
        return ;

      if (player != game.soloplayer())
        return ;

      if (trick.startplayerno() != player.no())
        return ;



      for (vector<Card::Value>::const_iterator
           v = game.rule().card_values().begin();
           v != game.rule().card_values().end();
           ++v)
        if (*v > played_card.value())
          CHANGE_WEIGHTING(Card(played_card.color(), *v), -10); // *Value*


      return ;
    } // void meatless_startcard(...)

} // namespace CardsInformationOfPlayerWeightingHeuristics
