/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.war.cascade.handlers;

import com.jaspersoft.jasperserver.api.engine.common.service.ReportInputControlInformation;
import com.jaspersoft.jasperserver.api.metadata.common.domain.DataType;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ListOfValuesItem;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.war.cascade.CascadeResourceNotFoundException;
import com.jaspersoft.jasperserver.war.cascade.InputControlValidationException;
import com.jaspersoft.jasperserver.war.dto.InputControlOption;
import com.jaspersoft.jasperserver.war.dto.InputControlState;
import com.jaspersoft.jasperserver.war.dto.ReportInputControl;

import java.util.*;

/**
 * @author Yaroslav.Kovalchyk
 * @version $Id: SingleSelectListInputControlHandler.java 24202 2012-06-21 13:10:48Z afomin $
 */
public class SingleSelectListInputControlHandler extends BasicInputControlHandler {


    private ValuesLoader loader;

    public ValuesLoader getLoader() {
        return loader;
    }

    public void setLoader(ValuesLoader loader) {
        this.loader = loader;
    }

    @Override
    public ReportInputControl buildReportInputControl(InputControl inputControl, String uiType) throws CascadeResourceNotFoundException {
        final ReportInputControl reportInputControl = super.buildReportInputControl(inputControl, uiType);
        final Set<String> masterDependencies = loader.getMasterDependencies(inputControl);
        if (masterDependencies != null)
            reportInputControl.getMasterDependencies().addAll(masterDependencies);
        return reportInputControl;
    }

    @Override
    protected Object internalConvertParameterValueFromRawData(String[] rawData, InputControl inputControl, ReportInputControlInformation info) throws CascadeResourceNotFoundException, InputControlValidationException {
        Object value;
        try {
            value = super.internalConvertParameterValueFromRawData(rawData, inputControl, info);
        } catch (InputControlValidationException e) {
            // Explicitly set value for single select list to Null, thus we will recognize later that
            // this error doesn't need to be added to input control state
            e.getValidationError().setInvalidValue(null);
            throw e;
        }
        return value;
    }

    @Override
    protected void internalValidateValue(Object value, InputControl inputControl, ReportInputControlInformation info) throws InputControlValidationException, CascadeResourceNotFoundException {
        final DataType dataType = inputControl.getDataType() != null ? cachedRepositoryService.getResource(DataType.class, inputControl.getDataType()) : null;
        try {
            validateSingleValue(value, dataType);
        } catch (InputControlValidationException e) {
            // Explicitly set value for single select list to Null, thus we will recognize later that
            // this error doesn't need to be added to input control state
            e.getValidationError().setInvalidValue(null);
            throw e;
        }
    }

    /**
     * @param state        - state object to fill
     * @param inputControl - input control
     * @param dataSource   - data source
     * @param parameters   - incoming parameters
     * @param info         - input control information (contains input control default value)
     * @param parameterTypes
     * @throws CascadeResourceNotFoundException
     *          in case if some required for cascade logic resource isn't found.
     */
    @Override
    protected void fillStateValue(InputControlState state, InputControl inputControl, ResourceReference dataSource, Map<String, Object> parameters, ReportInputControlInformation info, Map<String, Class<?>> parameterTypes) throws CascadeResourceNotFoundException {
        List<ListOfValuesItem> values = loader.loadValues(inputControl, dataSource, parameters, parameterTypes, info);
        final DataType dataType = inputControl.getDataType() != null ? cachedRepositoryService.getResource(DataType.class, inputControl.getDataType()) : null;
        List<InputControlOption> options = new ArrayList<InputControlOption>();
        InputControlOption nothingOption = null;
        if (shouldAddNothinSelectedOption(inputControl)) {
            nothingOption = buildInputControlOption(NOTHING_SUBSTITUTION_LABEL, NOTHING_SUBSTITUTION_VALUE);
            options.add(nothingOption);
        }
        // default options will be collected to defaultOptions map
        // default values are collected to a map because of option object and typed value are both required
        Map<InputControlOption, Object> defaultOptions = new HashMap<InputControlOption, Object>();
        // selected values will be collected to selectedValues list
        List<Object> selectedValues = new ArrayList<Object>();
        final String controlName = inputControl.getName();
        Boolean isNothingSelected = isNothingSelected(controlName, parameters);
        Object incomingValue = parameters.get(controlName);
        Object defaultValue = info.getDefaultValue();
        if (values != null && !values.isEmpty()) {
            // iterate over values
            for (ListOfValuesItem currentItem : values) {
                Object currentItemValue = currentItem.getValue();
                if (currentItemValue instanceof String) {
                    // if incoming value is a String, then convert it to corresponding Object for correct further comparison
                    try {
                        currentItemValue = dataConverterService.convertSingleValue((String) currentItemValue, inputControl, info);
                    } catch (InputControlValidationException e) {
                        // this exception doesn't depend on user's input. So, throw technical runtime exception
                        throw new IllegalStateException(e.getValidationError().getDefaultMessage());
                    }
                }

                boolean optionIsValid = true;
                try {
                    validateSingleValue(currentItem.getValue(), dataType);
                } catch (InputControlValidationException e) {
                    optionIsValid = false;
                }
                // Add option only if it meets validation rules of the dataType
                if (optionIsValid) {
                    InputControlOption option = buildInputControlOption(currentItem.getLabel(), dataConverterService.formatSingleValue(currentItemValue, inputControl, info));
                    Boolean isSelected = !isNothingSelected && matches(currentItemValue, incomingValue);
                    option.setSelected(isSelected);
                    if (isSelected) {
                        // collect selected values to update parameters with selected data
                        selectedValues.add(currentItemValue);
                    }
                    // collect default options if no selected values found and not a case that nothing is selected
                    if (selectedValues.size() == 0 && !isNothingSelected && matches(currentItemValue, defaultValue)) {
                        // collect default options and values if no incoming values found yet
                        defaultOptions.put(option, currentItemValue);
                    }
                    options.add(option);
                }
            }
        }
        if (selectedValues.size() == 0 && !options.isEmpty()) {
            // incoming value isn't found in values list
            if (!defaultOptions.isEmpty()) {
                // default values found. So, they should be selected
                for (InputControlOption defaultOption : defaultOptions.keySet())
                    defaultOption.setSelected(true);
                // put default values to selected values list for further update of incoming parameters map
                selectedValues = new ArrayList<Object>(defaultOptions.values());
            } else if (inputControl.isMandatory() && !isNothingSelected) {
                // default values sre not found in values list
                // this control is mandatory, first value should be selected
                options.get(0).setSelected(true);
                selectedValues.add(values.get(0).getValue());
            }
        }
        state.setOptions(options);
        // update incoming parameters map with selected values
        if (selectedValues.size() == 0) {
            if (inputControl.isMandatory())
                doMandatoryValidation(null, state);// mandatory error should appears in this case
            // if control had some value but this value isn't in options (i.e. no such value in result set), then applyNothingSelected() should be called here too.
            applyNothingSelected(controlName, parameters);
            if (nothingOption != null)
                nothingOption.setSelected(true);
            internalApplyNothingSubstitution(controlName, parameters);
        } else {
            // selected values are exist, update incoming parameters
            parameters.put(controlName, getConcreteSelectedValue(selectedValues));
        }
    }


    protected Boolean shouldAddNothinSelectedOption(InputControl inputControl) {
        return !inputControl.isMandatory();
    }

    protected Boolean isNothingSelected(String inputControlName, Map<String, Object> incomingParameters) {
        return !incomingParameters.containsKey(inputControlName);
    }

    /**
     * @param selectedValues list of selected values
     * @return first item of selected values list (singleselect)
     */
    protected Object getConcreteSelectedValue(List<Object> selectedValues) {
        return selectedValues != null && !selectedValues.isEmpty() ? selectedValues.get(0) : null;
    }

    /**
     * Check if given value is selected
     *
     * @param valueToCheck value to check
     * @param selected     selected value
     * @return true if both values are null or equals
     */
    protected Boolean matches(Object valueToCheck, Object selected) {
        return (valueToCheck == null && selected == null) || (selected != null && selected.equals(valueToCheck))
                || (valueToCheck instanceof Number && selected instanceof Number
                    && ((Number)valueToCheck).doubleValue() == ((Number)selected).doubleValue());
    }

    protected InputControlOption buildInputControlOption(String label, String value) {
        return new InputControlOption(value, label);
    }
}
