/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.war.cascade.handlers;

import com.jaspersoft.jasperserver.api.engine.common.service.ReportInputControlInformation;
import com.jaspersoft.jasperserver.api.metadata.common.domain.DataType;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.war.cascade.CachedRepositoryService;
import com.jaspersoft.jasperserver.war.cascade.CascadeResourceNotFoundException;
import com.jaspersoft.jasperserver.war.cascade.InputControlValidationException;
import com.jaspersoft.jasperserver.war.cascade.handlers.converters.DataConverterService;
import com.jaspersoft.jasperserver.war.cascade.handlers.validators.InputControlValueValidator;
import com.jaspersoft.jasperserver.war.dto.InputControlState;
import com.jaspersoft.jasperserver.war.dto.MandatoryValidationRule;
import com.jaspersoft.jasperserver.war.dto.ReportInputControl;
import com.jaspersoft.jasperserver.war.dto.ValidationRule;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * @author Yaroslav.Kovalchyk
 * @version $Id: BasicInputControlHandler.java 24202 2012-06-21 13:10:48Z afomin $
 */
@Service
public class BasicInputControlHandler implements InputControlHandler {
    /**
     * currently the only multi typed input control is control of type singleValue.
     * multyDataTypeMapping is used to determine UI type of control in this case
     */
    public static final String MULTI_DATA_TYPE_UI_TYPE = "singleValue";

    @Resource
    private Map<String, String> multyDataTypeMapping;
    @Resource
    protected CachedRepositoryService cachedRepositoryService;
    @Resource
    private GenericTypeProcessorRegistry genericTypeProcessorRegistry;
    @Resource
    protected DataConverterService dataConverterService;
    @Resource
    protected MessageSource messageSource;

    @Override
    public void applyNothingSelected(String controlName, Map<String, Object> parameters) {
        parameters.remove(controlName);
    }

    @Override
    public boolean isNothingSelected(String[] rawData) {
        Boolean result = false;
        if (rawData != null) {
            List<String> rawDataList = Arrays.asList(rawData);
            result = rawDataList.contains(InputControlHandler.NOTHING_SUBSTITUTION_VALUE);
        }
        return result;
    }

    @Override
    public ReportInputControl buildReportInputControl(InputControl inputControl, String uiType) throws CascadeResourceNotFoundException {
        ReportInputControl control = new ReportInputControl();
        control.setId(inputControl.getName());
        control.setLabel(inputControl.getLabel());
        control.setUri(inputControl.getURI());
        control.setMandatory(inputControl.isMandatory());
        control.setReadOnly(inputControl.isReadOnly());
        control.setVisible(inputControl.isVisible());
        control.setType(getType(inputControl, uiType));
        final List<ValidationRule> validationRules = getValidationRules(inputControl);
        if (validationRules != null && !validationRules.isEmpty())
            control.setValidationRules(validationRules);
        return control;
    }

    protected String getType(InputControl inputControl, String uiType) throws CascadeResourceNotFoundException {
        String type = uiType;
        if (MULTI_DATA_TYPE_UI_TYPE.equals(uiType)) {
            byte dataType = cachedRepositoryService.getResource(DataType.class, inputControl.getDataType()).getType();
            type = multyDataTypeMapping.get(String.valueOf(dataType));
        }
        return type;

    }

    @Override
    public final InputControlState getState(InputControl inputControl, ResourceReference dataSource, Map<String, Object> parameters, Map<String, Class<?>> parameterTypes, ReportInputControlInformation info) throws CascadeResourceNotFoundException {
        InputControlState state = new InputControlState();
        state.setUri(inputControl.getURIString());
        state.setId(inputControl.getName());
        fillStateValue(state, inputControl, dataSource, parameters, info, parameterTypes);
        return state;
    }

    protected List<ValidationRule> getValidationRules(InputControl inputControl) {
        List<ValidationRule> validationRules = new ArrayList<ValidationRule>();
        if (inputControl.isMandatory()) {
            final MandatoryValidationRule mandatoryValidationRule = new MandatoryValidationRule();
            mandatoryValidationRule.setErrorMessage(getMessage(MandatoryValidationRule.ERROR_KEY));
            validationRules.add(mandatoryValidationRule);
        }
        return validationRules;
    }

    protected void fillStateValue(InputControlState state, InputControl inputControl, ResourceReference dataSource, Map<String, Object> parameters, ReportInputControlInformation info, Map<String, Class<?>> parameterTypes) throws CascadeResourceNotFoundException {
        // call formatting of value in case if this control parameter present in input parameters. This is needed to avoid wrong null substitution
        if (parameters.containsKey(inputControl.getName())) {
            final Object typedValue = parameters.get(inputControl.getName());
            if (inputControl.isMandatory())
                doMandatoryValidation(typedValue, state);
            state.setValue(typedValue != null ? dataConverterService.formatSingleValue(typedValue, inputControl, info) : "");
        } else
            internalApplyNothingSubstitution(inputControl.getName(), parameters);
    }

    /**
     * @param controlName
     * @param parameters
     * @deprecated this method need to be removed if real nothing substitution is implemented for single value and single select controls.
     */
    protected void internalApplyNothingSubstitution(String controlName, Map<String, Object> parameters) {
        parameters.put(controlName, null);
    }

    protected void doMandatoryValidation(Object typedValue, InputControlState state) {
        // if value isn't in error state and don't has a value, then put error message
        if (state.getError() == null && !hasValue(typedValue))
            state.setError(getMessage(MandatoryValidationRule.ERROR_KEY));
    }

    @Override
    public final Object convertParameterValueFromRawData(String[] rawData, InputControl inputControl, ReportInputControlInformation info) throws CascadeResourceNotFoundException, InputControlValidationException {
        // default value is used if no raw value provided
        Object value;
        if (rawData != null) {
            value = internalConvertParameterValueFromRawData(rawData, inputControl, info);
        } else {
            value = info.getDefaultValue();
        }

        internalValidateValue(value, inputControl, info);

        return value;
    }

    protected Boolean hasValue(Object value) {
        return value != null;
    }

    protected Object internalConvertParameterValueFromRawData(String[] rawData, InputControl inputControl, ReportInputControlInformation info) throws CascadeResourceNotFoundException, InputControlValidationException {
        return rawData != null && rawData.length > 0
                ? dataConverterService.convertSingleValue(rawData[0], inputControl, info) : null;
    }

    protected void internalValidateValue(Object value, InputControl inputControl, ReportInputControlInformation info) throws InputControlValidationException, CascadeResourceNotFoundException {
        final DataType dataType = inputControl.getDataType() != null ? cachedRepositoryService.getResource(DataType.class, inputControl.getDataType()) : null;
        try {
            validateSingleValue(value, dataType);
        } catch (InputControlValidationException e) {
            e.getValidationError().setInvalidValue(dataConverterService.formatSingleValue(value, inputControl, info));
            throw e;
        }
    }

    protected void validateSingleValue(Object value, DataType dataType) throws InputControlValidationException {
        if (value != null && dataType != null) {
            InputControlValueValidator<Object> validator = genericTypeProcessorRegistry.getTypeProcessor(value.getClass(), InputControlValueValidator.class, false);
            if (validator == null && value instanceof Comparable) {
                validator = genericTypeProcessorRegistry.getTypeProcessor(Comparable.class, InputControlValueValidator.class, false);
            }
            if (validator != null)
                validator.validateSingleValue(value, dataType);
        }
    }

    @Override
    public String[] formatValue(Object typedValue, InputControl inputControl, ReportInputControlInformation info) throws CascadeResourceNotFoundException {
        return new String[]{dataConverterService.formatSingleValue(typedValue, inputControl, info)};
    }

    protected String getMessage(String messageKey, Object... arguments) {
        return messageSource.getMessage(messageKey, arguments, LocaleContextHolder.getLocale());
    }
}
